/************************* * * * * * * * * * * * * ***************************
    Copyright (c) 1999-2005 Ryan Bobko
                       ryan@ostrich-emulators.com

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.     
************************** * * * * * * * * * * * * **************************/

#include "jreport.h"
#include "qhacc.h"
#include "qhaccutils.h"
#include "qhacctable.h"
#include "qhaccsegmenter.h"

#include <stdlib.h>

#include <map>
#include <memory>
#include <vector>
using namespace std;

// plugin factory calls
extern "C" {
	QHaccPlugin * create(){	return new JournalReport; }
	void destroy( JournalReport * p ){ delete p; }
}

JournalReport::JournalReport(){}
JournalReport::~JournalReport(){}

auto_ptr<QHaccResultSet> JournalReport::generate( uint jid,
																									QHaccResultSet *,
																									const QDate& start,
																									const QDate&  end,
																									QString& title ){	
	const MonCon& conv=engine->converter();
	// all-in-one journal creater/writer
	Journal led;
	if( jid==0 ){ 
		TableCol tcs[]={ TableCol( 0 ), TableCol( "All Journals" ),
										 TableCol( "" ) };
		led=Journal( tcs, 3 );
	}
	else led=engine->getL( jid );
	
	title=led.gets( QC::LNAME )+" Journal";

	QString sep=engine->getSP( "DATESEPARATOR" );
	int fmt=engine->getIP( "DATEFORMAT" );
	title+=" ("+Utils::stringFromDate( start, sep,  fmt )+" -";
	title+=" "+Utils::stringFromDate( end, sep,  fmt )+")";
	
	QString journ="\n";
	journ.append( QString( "%1 %2 R %3%4" ).arg( "Date",-10 ).arg( "Payee/Memo",-33 ).arg( "Credit", 10 ).arg( "Debit", 10 ) );

	uint rr=0;
	// keep a cache of accounts so we don't have to make many db calls
	auto_ptr<QHaccResultSet> ars=engine->getWhere( ACCOUNTS, TableGet(), 
																								 vector<TableSelect>(), rr );
	map<uint, Account> acctcache;
	for( uint i=0; i<rr; i++ ){
		const Account& acct=ars->at( i );
		acctcache[acct[QC::AID].getu()]=acct;
	}
 
	vector<TableSelect> crit;
	if( jid!=0 ) crit.push_back(  TableSelect( QC::TLID,	led[QC::LID] ) );
	crit.push_back( TableSelect( QC::TDATE, TableCol( start ),
															 TableSelect::GE ) );
	crit.push_back( TableSelect( QC::TDATE, TableCol( end ), TableSelect::LT ) );
	crit.push_back( TableSelect( QC::TTYPE, TableCol( QC::REGULAR ) ) );

	auto_ptr<QHaccResultSet> trans=engine->getWhere( TRANSACTIONS, crit, rr );
	QHaccTableIndex didx( trans.get(), QC::TDATE, CTDATE );

	for( uint i=0; i<rr; i++ ){
		const Transaction& t=trans->at( didx[i] );
		
		journ.append( "\n\n" );
		journ.append( Utils::stringFromDate( t.getd( QC::TDATE ), sep, fmt ) );
		journ.append( " "+t.gets( QC::TPAYEE ) );
		if( !t.gets( QC::TMEMO ).isEmpty() )
			journ.append( "\n            "+t.gets( QC::TMEMO ) );

		QHaccTable splits=engine->getTSplits( t[QC::TID].getu() );

		for( uint j=0; j<splits.rows(); j++ ){
			const Split& s=splits[j];
			// write the adjustments info
			int offset=0;
			Account acct=acctcache[s[QC::SACCTID].getu()];

			QString sum=conv.convert( s.gets( QC::SSUM ) );
			if( sum.startsWith( "-" ) ){
				sum=sum.mid( 1 );
				offset=20;
			}
			else offset=10;

			//make sure we have the proper format
			journ.append( QString( "\n              %1 %2 %3" ).arg( engine->getFNameOfA( acct ), -30 ).arg( s.getu( QC::SRECO )==QC::YREC ? "R" : "-" ).arg( sum, offset ) );
		}
	}

	
	// data is just one string
	TableRow row( 1 );
	row.set( 0, journ );
	auto_ptr<QHaccResultSet> data( new QHaccResultSet( 1 ) );
	data->add( row );
	return data;
}

auto_ptr<QHaccResultSet> JournalReport::generate( QHaccResultSet *,
																									vector<TableSelect>,
																									QString& title ){
	title="Program error: Cannot generate Journal Report from this function";
	return auto_ptr<QHaccResultSet>(new QHaccResultSet( 0 ) );
}

bool JournalReport::homeok( const QString& home, uint& jid,
														auto_ptr<QHaccResultSet>&,
														QDate& start, QDate& stop, QString& err ){
	Journal j=engine->getL( home );
	if( j.isNull() ){
		err="No journal: "+home;
		return false;
	}

	jid=j[QC::LID].getu();
	start=engine->min( TRANSACTIONS, QC::TDATE ).getd();
	stop=QDate::currentDate();
	return true;
}

QString JournalReport::writereport( const QString& title,
																		const QHaccResultSet * data ){
	QString ret=title;
	if( data->isEmpty() ) ret.append( "\n\tNo Transactions" );
	else ret.append( data->at( 0 ).gets( 0 ) );
	return ret;
}

void JournalReport::selected( QDate&, QDate&, bool& enableacctsel,
															bool& enablemultiselect ){
	enableacctsel=enablemultiselect=false;
}

const JournalInfo JournalReport::pinfo;
const PluginInfo& JournalReport::info() const { return pinfo; }
