/************************* * * * * * * * * * * * * ***************************
    Copyright (c) 1999-2005 Ryan Bobko
                       ryan@ostrich-emulators.com

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.     
************************** * * * * * * * * * * * * **************************/

#include "lfsqliteplugin.h"
#include "localfileplugin.h"
#include "qhacctable.h"
#include "qhaccutils.h"
#include "qhacc.h"

#include <qfile.h>

#include <qregexp.h>
#include <sys/stat.h>

// "factory" function calls
extern "C" {
	QHaccPlugin * create(){	return new LFSQLiteDBPlugin; }
	void destroy( LFSQLiteDBPlugin * p ){ delete p; }
}

const LFSQLiteInfo LFSQLiteDBPlugin::pinfo;

/*
 * This class is sort of a hybrid of LocalFilesDBPlugin and SQLiteDBPlugin
 * That is, it reads and writes data to the filesystem just like a LFDB, but 
 * operates on an in-memory SQLite database.
 */

LFSQLiteDBPlugin::LFSQLiteDBPlugin() : SQLiteDBPlugin(){ conn=0; }
LFSQLiteDBPlugin::~LFSQLiteDBPlugin(){ 
	if( conn )
#ifdef SQLITE3
		sqlite3_close( conn ); 
#else
		sqlite_close( conn ); 
#endif
}
 
const PluginInfo& LFSQLiteDBPlugin::info() const { return pinfo; }

bool LFSQLiteDBPlugin::save( QString& err ){
	bool ret=true;
  for( int i=0; i<QC::NUMTABLES; i++ ) {
		uint rr=0;
		auto_ptr<QHaccResultSet> rslt=getWhere( ( Table )i, TableSelect(), rr );
		QHaccTable tbl( *rslt );
		tbl.setName( QString( "t_" )+QC::TABLENAMES[i] );
		ret=( ret && 
					LocalFileDBPlugin::savet( tbl, home+"/"+QC::TABLENAMES[i], 
																		!engine->getBP( "KEEPFILEPERMS" ), err ) );
  }
	if( ret ) needsave=!ret;
  return ret;
}

bool LFSQLiteDBPlugin::load( QString& err ){
	bool ret=true;
	for( int i=0; i<QC::NUMTABLES; i++ ){
		Table t=( Table )i;
		QHaccTable tbl( Utils::tcols( t ), Utils::ttypes( t ) );
		tbl.setName( QString( "t_" )+QC::TABLENAMES[i] );
		ret=( ret &&
					LocalFileDBPlugin::loadt( tbl, home+"/"+QC::TABLENAMES[i], err ) );
		if( ret ){
			uint rr=tbl.rows();
			startLoad( t, rr );
			for( uint j=0; j<rr; j++ ) add( t, tbl[j] );
			stopLoad( t );
		}
	}

	needsave=false;
	return ret;
}

bool LFSQLiteDBPlugin::connect( QHacc * e, const QString& h, QString& err ){
	// make the db connection (actually, just open a memory space)
	if( conn )
#ifdef SQLITE3
		sqlite3_close( conn ); 
#else
		sqlite_close( conn ); 
#endif

	engine=e;
	home=h;
	char ** er=0;

#ifdef SQLITE3
	int eroor=sqlite3_open( ":memory:", &conn );
	if( eroor!=SQLITE_OK ){
		err=QString( sqlite3_errmsg( conn ) );
 		std::ostream * str=0;
		if( Utils::error( Utils::ERRFATAL, str ) ) *str<<err<<endl;
		return false;
	}
#else
	conn=sqlite_open( ":memory:", 0, er );
	if( conn==0 ){
 		std::ostream * str=0;
		if( Utils::error( Utils::ERRFATAL, str ) ) *str<<er<<endl;
		err=QString( *er );
		free( er );
		return false;
	}
#endif

	QString creater=SQLiteDBPlugin::screate( ":memory:" );
	creater=creater.mid( creater.find( ";" )+1 );


#ifdef SQLITE3
	if( sqlite3_exec( conn, creater, 0, 0, er )!=SQLITE_OK ){
#else
	if( sqlite_exec( conn, creater, 0, 0, er )!=SQLITE_OK ){
#endif
 		std::ostream * str=0;
		if( Utils::error( Utils::ERRFATAL, str ) ) *str<<er<<endl;
		err=QString( *er );
		free( er );		
		return false;
	}
	return true;
}

QString LFSQLiteDBPlugin::create( const QString& dbname ) const {
	return screate( dbname );
}

QString LFSQLiteDBPlugin::screate( const QString& dbname ){
	return LocalFileDBPlugin::screate( dbname );
}

bool LFSQLiteDBPlugin::dirty() const { return needsave; }
int LFSQLiteDBPlugin::run( const QString& stmt ){
	needsave=true;
	return SQLiteDBPlugin::run( stmt );
}

LFSQLiteInfo::LFSQLiteInfo(){
	description="SQLite/LocalFile Hybrid";
	stubby="LSQL";
	targ=LocalFileDBPlugin::pinfo.target();
	auto_ptr<QHaccResultSet> tp=LocalFileDBPlugin::pinfo.prefs();
	piprefs=tp;
}
