/************************* * * * * * * * * * * * * ***************************
    Copyright (c) 1999-2005 Ryan Bobko
                       ryan@ostrich-emulators.com

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.     
************************** * * * * * * * * * * * * **************************/

#include "qhaccdialogs.h"
#include "qhacc.h"
#include "qhaccutils.h"
#include "qhacctable.h"
#include "splitdialog.h"
#include "qhaccacctdlg.h"
#include "guiconstants.h"
#include "qhaccconstants.h"
#include "qhacclineedits.h"
#include "qhaccdialogs.moc"

#include <qlabel.h>
#include <qlayout.h>
#include <qlistbox.h>
#include <qsplitter.h>
#include <qcheckbox.h>
#include <qcombobox.h>
#include <qfontdialog.h>
#include <qpushbutton.h>
#include <qfiledialog.h>
#include <qmessagebox.h>
#include <qradiobutton.h>
#include <qbuttongroup.h>
#include <qinputdialog.h>
#include <qcolordialog.h> 
#include <qmultilineedit.h>

/* * * * * * * * * * * * * * * * */
/* * * * * JOURNAL DIALOG * * * * */
/* * * * * * * * * * * * * * * * */

JournalDlg::JournalDlg( QHacc * eng, const Journal& l, QWidget * p,
											const char * n ) : QDialog( p, n, false ){
	engine=eng;
	if( l.isNull() ){
		edit=false;
		int tcols=QC::LCOLS;
		TableCol * cts=new TableCol[tcols];
		for( int i=0; i<tcols; i++ ) cts[i]=TableCol( "" );

		journal=TableRow( cts, tcols );
		setCaption( tr( "Create Journal" ) );
		delete [] cts;
	}
	else{
		edit=true;
		journal=l;
		setCaption( tr( "Edit Journal" ) );
	}
	
	QGridLayout * layout=new QGridLayout( this, 5, 6 );

	lname=new QLineEdit( journal.gets( QC::LNAME ), this );
	QLabel * lbl=new QLabel( tr( "Name" ), this );
	layout->addMultiCellWidget( lbl, 0, 1, 0, 2 );
	layout->addMultiCellWidget( lname, 0, 1, 3, 6 );	

	ldesc=new QLineEdit( journal.gets( QC::LDESC ), this );
	lbl=new QLabel( tr( "Description" ), this );
	layout->addMultiCellWidget( lbl, 2, 3, 0, 2 );
	layout->addMultiCellWidget( ldesc, 2, 3, 3, 6 );

	QButton * but=new QPushButton( tr( "Accept" ), this );
	layout->addMultiCellWidget( but, 4, 5, 0, 2 );
	connect( but, SIGNAL( clicked() ), SLOT( accept() ) );

	but=new QPushButton( tr( "Dismiss" ), this );
	layout->addMultiCellWidget( but, 4, 5, 3, 6 );
	connect( but, SIGNAL( clicked() ), SLOT( close() ) );
}

JournalDlg::~JournalDlg(){}

void JournalDlg::qremove( QHacc * eng, const Journal& l, QWidget * p ){
	if( !l.isNull() ){
		if ( QMessageBox::warning( p, tr( "Delete " )+l.gets( QC::LNAME ),
															 tr( "Are you sure?" ), tr( "Delete Journal" ),
															 tr( "Reconsider" ) ) == 0 ) eng->removeL( l );
	}
}

void JournalDlg::accept(){
	Journal newy( journal );
	newy.set( QC::LNAME, lname->text() );
	newy.set( QC::LDESC, ldesc->text() );
	
	if( edit ) engine->updateL( journal, newy ); 
	else engine->addL( newy );

	QDialog::accept();
}


/* * * * * * * * * * * * * * * * * * * * * * */
/* * * * * SCHED/NAMED DIALOG BASE * * * * * */
/* * * * * * * * * * * * * * * * * * * * * * */
LeftDlg::LeftDlg( QHacc * e, QWidget * p, const char * n ) 
	: QDialog( p, n, false ) {
	
	engine=e;
	
  QGridLayout * layout=new QGridLayout( this, 10, 1 );

	splitter=new QSplitter( Horizontal, this );
	listbox=new QListBox( splitter );
	connect( listbox, SIGNAL( highlighted( const QString& ) ),
					 this, SLOT( listsel( const QString& ) ) );
	
	layout->addMultiCellWidget( splitter, 0, 8, 0, 0 );
	
	int row=9;
	//QPushButton * but=new QPushButton( "accept", this );
	//layout->addMultiCellWidget( but, row, row, 0, 0 );
	//connect( but, SIGNAL( clicked() ), SLOT( accept() ) );

	QPushButton * but=new QPushButton( "dismiss", this );
	layout->addMultiCellWidget( but, row, row, 0, 0 );
	connect( but, SIGNAL( clicked() ), SLOT( accept() ) );
}

LeftDlg::~LeftDlg(){}

void LeftDlg::addListItem( const QString& str ){ listbox->insertItem( str ); }
void LeftDlg::setListItems( const QStringList& strs ){
	listbox->clear();
	listbox->insertStringList( strs );
}

void LeftDlg::listsel( const QString& s ){ iselect( s ); }
void LeftDlg::iselect( const QString& ){}
void LeftDlg::remakeList(){ iremakeList(); }
void LeftDlg::iremakeList(){}
void LeftDlg::iaccept(){}
void LeftDlg::accept(){
	iaccept();
	QDialog::accept();
}

/* * * * * * * * * * * * * * * * * * */
/* * * * * SCHEDULE DIALOG * * * * * */
/* * * * * * * * * * * * * * * * * * */
const QString SchedFrame::NEVER=QT_TR_NOOP( "never" );

SchedFrame::SchedFrame( QHacc * e, QWidget * p,	const char * n ) 
	: QFrame( p, n ){

	engine=e;
	freq=new QLineEdit( this );
	freq->setAlignment( AlignCenter );
	connect( freq, SIGNAL( textChanged( const QString& ) ),
					 this, SLOT( texter( const QString& ) ) );
	onefreq=new QLineEdit( this );
	onefreq->setAlignment( AlignCenter );
	connect( onefreq, SIGNAL( textChanged( const QString& ) ),
					 this, SLOT( texter( const QString& ) ) );
  schedbox=new QButtonGroup( 4, Vertical );
	connect( schedbox, SIGNAL( clicked( int ) ), SLOT( toggle( int ) ) );
	
	non=new QRadioButton( tr( "never" ), this );
	day=new QRadioButton( tr( "every" ), this );
	mon=new QRadioButton( tr( "monthly" ), this );
	bim=new QRadioButton( tr( "bimonthly" ), this );
	qtr=new QRadioButton( tr( "quarterly" ), this );
	lastday=new QRadioButton( tr( "last day of month" ), this );
	one=new QRadioButton( tr( "once, in" ), this );

  QGridLayout * layout=new QGridLayout( this, 9, 3, 3, 3 );
	
	int row=0;
	title=new QLabel( this );
	layout->addMultiCellWidget( title, row, row, 0, 2 );
	row++;
	
	QRadioButton * qs[]={ non, day, mon, bim, qtr, lastday, one };

	for( uint i=0; i<7; i++ ){
		if( i==1 ){
			layout->addMultiCellWidget( freq, row, row, 1, 1 );
			layout->addMultiCellWidget( new QLabel( tr( "days" ), this ),
																	row, row, 2, 2 );
		}
		else if( i==6 ){
			layout->addMultiCellWidget( onefreq, row, row, 1, 1 );
			layout->addMultiCellWidget( new QLabel( tr( "days" ), this ),
																	row, row, 2, 2 );
		}

		layout->addMultiCellWidget( qs[i], row, row, 0, 0 );
		schedbox->insert( qs[i] );
		row++;
	}
	
	nextrun=new QLabel( tr( NEVER ), this );
	nextrun->setAlignment( AlignCenter );
	layout->addMultiCellWidget( new QLabel( tr( "next run" ), this ),
															row, row, 0, 0 );
	layout->addMultiCellWidget( nextrun, row, row, 1, 1 );
	row++;

	apply=new QPushButton( this );
	layout->addMultiCellWidget( apply, row, row, 1, 2 );
	connect( apply, SIGNAL( clicked() ), SLOT( accept() ) );
}

SchedFrame::~SchedFrame(){ delete schedbox; }

void SchedFrame::accept(){
	// there's no appropriate update function in the engine,
	// so we'll need to delete the current job and readd it
	bool nojob=job.isNull();
	if( nojob ){
		job=TableRow( QC::JCOLS );
		job.set( QC::JWHAT, TableCol( name ) );
		job.set( QC::JLASTRUN, TableCol( QDate::currentDate() ) );
	}
	else engine->removeJ( job.getu( QC::JID ) );

	int fr=getFreq();
	if( fr!=0 ){
		if( fr==QC::ONETIMEFREQ ){
			// onetime runs need to set lastrun in the future
			job.set( QC::JLASTRUN,
							 job[QC::JLASTRUN].getd().addDays( onefreq->text().toInt() ) );
		}
		job.set( QC::JFREQUENCY, TableCol( fr ) );
		engine->addJ( job );
	}
}

int SchedFrame::getFreq() const {
	QButton * but=schedbox->selected();
	if( but==day ){
		int f=freq->text().toInt();
		if( f<1 ) f=0;
		return f;
	}
	else if( but==mon ) return QC::MONTHFREQ;
	else if( but==bim ) return QC::BIMONTHFREQ;
	else if( but==qtr ) return QC::QUARTERLYFREQ;
	else if( but==one ) return QC::ONETIMEFREQ;
	else if( but==lastday ) return QC::LASTMONTHDAYFREQ;
	return 0;
}

void SchedFrame::setJob( const TableRow& t, const QString& nm ){
	job=t;
	name=nm;

	int bid=0;
	title->setText( tr( "Run" )+" "+nm+"..." );

	if( job.isNull() ){
		apply->setText( tr( "add" ) );
		//rem->setEnabled( false );
		freq->setText( "0" );
		onefreq->setText( "0" );
		bid=schedbox->id( non );
	}
	else{
		apply->setText( tr( "apply" ) );
		//rem->setEnabled( true );
		int nr=job.geti( QC::JFREQUENCY );

		QButton * but=0;
		if( nr==QC::MONTHFREQ ) but=mon;
		else if( nr==QC::BIMONTHFREQ ) but=bim;
		else if( nr==QC::QUARTERLYFREQ ) but=qtr;
		else if( nr==QC::ONETIMEFREQ ) but=one;
		else if( nr==QC::LASTMONTHDAYFREQ ) but=lastday;
		else{
			but=day;
			freq->setText( QString::number( nr ) );
		}
		bid=schedbox->id( but );
	}

	schedbox->setButton( bid );
	toggle( bid );
}

void SchedFrame:: toggle( int id ){
	QButton * but=schedbox->find( id );
	freq->setEnabled( but==day );
	onefreq->setEnabled( but==one );

	int nr=0;
	if( but==non ) nr=0;
	else if( but==day ) nr=freq->text().toInt();
	else if( but==mon ) nr=QC::MONTHFREQ;
	else if( but==bim ) nr=QC::BIMONTHFREQ;
	else if( but==qtr ) nr=QC::QUARTERLYFREQ;
	else if( but==lastday ) nr=QC::LASTMONTHDAYFREQ;
	else if( but==one )	nr=QC::ONETIMEFREQ;
	setNext( nr );
}

void SchedFrame::setNext( int fr ){
	QDate d;
	bool inny=true;
	if( job.isNull() ){
		d=QDate::currentDate();
		inny=false;
	}
	else d=job.getd( QC::JLASTRUN );
	
	if( fr==QC::MONTHFREQ )	d=d.addMonths( 1 );
	else if( fr==QC::BIMONTHFREQ ) d=d.addDays( d.daysInMonth()/2 );
	else if( fr==QC::QUARTERLYFREQ ) d=d.addMonths( 3 );
	else if( fr==QC::ONETIMEFREQ ){
		// lastrun IS nextrun for one-timers, but need to set "in XXX days" text
		int days=QDate::currentDate().daysTo( d );
		if( days<0 ) onefreq->setText( 0 );
		else onefreq->setText( QString::number( days ) );
	}
	else if( fr==QC::LASTMONTHDAYFREQ ){
		d=d.addMonths( 1 ).addDays( d.daysInMonth()+1-d.day() );
	}
	else if( fr==0 ) d=QDate();
	else d=d.addDays( fr );

	QString dstr;
	if( d.isValid() ){
		dstr=Utils::stringFromDate( d, engine->getSP( "DATESEPARATOR" ),
																engine->getIP( "DATEFORMAT" ) );
	}
	else dstr=tr( "never" );

	if( !inny ) dstr="("+dstr+")";
	nextrun->setText( dstr );
}

void SchedFrame::texter( const QString& s ){ setNext( s.toInt() ); }





SchedDlg::SchedDlg( QHacc * e, const QString& nm, QWidget * p,
										const char * n ) : LeftDlg( e, p, n ){

	ntrans=new QHaccTable();
	right=new SchedFrame( e, splitter );
	remakeList();
	resize( 300, 150 );	

	connect( engine, SIGNAL( addedT( const Transaction& ) ), 
					 SLOT( tmod( const Transaction& ) ) );
	connect( engine, SIGNAL( removedT( const Transaction& ) ),
					 SLOT( tmod( const Transaction& ) ) );
	connect( engine, SIGNAL( updatedT( const Transaction& ) ),
					 SLOT( tmod( const Transaction& ) ) );

	if( !nm.isEmpty() ) listbox->setSelected( new QListBoxText( nm ), true );
}

SchedDlg::~SchedDlg(){ delete ntrans; }

void SchedDlg::iaccept(){}
void SchedDlg::iselect( const QString& s ){
	setCaption( s );
	TableRow tr=engine->getJ( s );

	right->setEnabled( true );
	right->setJob( tr, s );
}

void SchedDlg::iremakeList(){
	//QString str=listbox->currentText();

	listbox->clear();

	vector<TableSelect> ts( 1, TableSelect( QC::NACCTID, TableCol( "0" ),
																					TableSelect::NE ) );
	uint rr=0;
	vector<TableSelect> v;
	auto_ptr<QHaccResultSet> rslt=engine->getWhere( NAMEDTRANS, v, rr );
	for( uint i=0; i<rr; i++ ){
		TableRow nt=rslt->at( i );
		Transaction t;
		QHaccTable splits;
		engine->getNT( nt.gets( QC::NNAME ), t, splits );
		if( engine->isResolvable( t, splits ) ){
			addListItem( nt.gets( QC::NNAME ) );
		}
	}

	ntrans=new QHaccTable( *rslt );

	right->setEnabled( false );
	//listbox->setSelected( new QListBoxText( str ), true );
}

void SchedDlg::tmod( const Transaction& t ){
	if( t[QC::XTTYPE]==QC::MEMORIZED ) remakeList();
}




/* * * * * * * * * * * * * * * * * * */
/* * * * * NAMEDTRANS DIALOG * * * * */
/* * * * * * * * * * * * * * * * * * */

const char * NamedFrame::NEWY=QT_TR_NOOP( "new" );

NamedFrame::NamedFrame( QHacc * e, uint ledgid, QWidget * p, const char * n ) 
	: QFrame( p, n ){
	
	lid=ledgid;
	engine=e;
	int row=0;
  QGridLayout * layout=new QGridLayout( this, 10, 5 ); // col, rows
	setCaption( tr( "Named Transaction" ) );

  name=new QHaccLineEdit( this );
	name->setEmptyText( "<"+tr( NEWY )+">" );
  QLabel * lbl=new QLabel( tr( "Transaction Name" ), this );
  layout->addMultiCellWidget( lbl, row, row, 0, 1 );
  layout->addMultiCellWidget( name, row, row, 2, 4 );
	row++;


	acctedit=new QHaccChoiceEdit( false, this );
	acctedit->ignoreCase( engine->getBP( "AUTOCOMPNOCASE" ) );
	QStringList strs;

	auto_ptr<QHaccResultSet> accts=engine->getAs( TableGet() );
	uint rr=accts->rows();
	for( uint i=0; i<rr; i++ ){
		Account acct=accts->at( i );
		QString aname=engine->getFNameOfA( acct );
		strs.append( aname );
	}
	acctedit->insertStringList( strs );
	acctedit->setEmptyText( "<"+tr( "any" )+">" );


  lbl=new QLabel( tr( "Parent Account" ), this );
  layout->addMultiCellWidget( lbl, row, row, 0, 1 );
  layout->addMultiCellWidget( acctedit, row, row, 2, 4 );	
	row++;

	num=new QHaccChoiceEdit( true, this );
	num->setEmptyText( "<"+tr( "num" )+">" );
  lbl=new QLabel( tr( "Number" ), this );
  layout->addMultiCellWidget( lbl, row, row, 0, 1 );
  layout->addMultiCellWidget( num, row, row, 2, 4 );	
	row++;

	payee=new QHaccChoiceEdit( true, this );
	payee->setEmptyText( "<"+tr( "payee" )+">" );
  lbl=new QLabel( tr( "Payee" ), this );
  layout->addMultiCellWidget( lbl, row, row, 0, 1 );
  layout->addMultiCellWidget( payee, row, row, 2, 4 );	
	row++;

	memo=new QHaccLineEdit( this );
	memo->setEmptyText( "<"+tr( "memo" )+">" );
  lbl=new QLabel( tr( "Memo" ), this );
  layout->addMultiCellWidget( lbl, row, row, 0, 1 );
  layout->addMultiCellWidget( memo, row, row, 2, 4 );	
	row++;

	shortcut=new QHaccAccelEdit( this );
	shortcut->setEmptyText( "<"+tr( "shortcut" )+">" );
  lbl=new QLabel( tr( "Keyboard Shortcut" ), this );
  layout->addMultiCellWidget( lbl, row, row, 0, 1 );
  layout->addMultiCellWidget( shortcut, row, row, 2, 4 );	
	row++;

	QHaccTable s;
  area=new AccountAreaFrame( engine, s, this );
  layout->addMultiCellWidget( area, row, row+4, 0, 4 );
	row+=5;

  apply=new QPushButton( this );
  layout->addMultiCellWidget( apply, row, row, 0, 2 );
  connect( apply, SIGNAL( clicked() ), SLOT( accept() ) );

  rem=new QPushButton( tr( "remove" ), this );
  layout->addMultiCellWidget( rem, row, row, 3, 4 );
  connect( rem, SIGNAL( clicked() ), SLOT( remove() ) );

	connect( acctedit, SIGNAL( textChanged( const QString& ) ), 
					 this, SLOT( cacct( const QString& ) ) );
}

NamedFrame::~NamedFrame(){}
void NamedFrame::cacct( const QString& str ){
	Account a=engine->getA( str );
	if( !a.isNull() ){
		num->clear();
		QStringList sts=QStringList::split( " ", a.gets( QC::ATRANSNUMS ), false );
		num->insertStringList( sts );
	}
}

void NamedFrame::accept(){
	const QString NAME=name->text();
	if( NAME==tr( NEWY ) ) return;

	bool add=mynt.isNull();
	TableRow nt;
	Transaction t;	
	QHaccTable s;
	if( add ){
		t=TableRow( QC::TCOLS );
		nt=TableRow( QC::NCOLS );
	}
	else nt=engine->getNT( mynt[QC::NNAME].gets(), t, s );
	nt.set( QC::NNAME, NAME );
	s=area->getSplits();

	// add one split for the main account
	uint aid=0;
	Account a=engine->getA( acctedit->text() );
	if( !a.isNull() ) aid=a[QC::AID].getu();

	Split spl( QC::SCOLS );
	spl.set( QC::SACCTID, TableCol( aid ) );
	spl.set( QC::SSUM, QC::REMAINDERVAL );
	s+=spl;

	t.set( QC::TPAYEE, TableCol( payee->text() ) );
	t.set( QC::TMEMO, TableCol( memo->text() ) );
	t.set( QC::TNUM, TableCol( num->text() ) );
	t.set( QC::TLID, TableCol( lid ) );
	t.set( QC::TVOID, TableCol( false ) );

	if( !shortcut->text().endsWith( "+" ) )
		nt.set( QC::NSHORTCUT, TableCol( shortcut->text() ) );
	nt.set( QC::NACCTID, aid );

	if( add )	engine->addNTForA( nt, t, s );
	else engine->updateNT( mynt[QC::NID].getu(), nt, t, s );
}

void NamedFrame::remove(){
	if( !mynt.isNull() ){
		const QString z( "" );
		acctedit->setText( z );
		num->setText( z );
		memo->setText( z );
		payee->setText( z );
		name->setText( z );
		area->clearSplits();
		shortcut->setText( z );
		engine->removeNT( mynt[QC::NNAME].gets() );
	}
}

void NamedFrame::setTrans( const QString& nm ){
	QHaccTable s;
	TableRow t;
	mynt=engine->getNT( nm, t, s );
	bool newy=( mynt.isNull() || nm==tr( NEWY ) );

	if( newy ){
		const QString z( "" );
		acctedit->setText( z );
		num->setText( z );
		memo->setText( z );
		payee->setText( z );
		name->setText( z );
		apply->setText( tr( "add" ) );
		shortcut->setText( z );
	}
	else{
		acctedit->setText( engine->getFNameOfA( mynt.getu( QC::NACCTID ) ) );
		num->setText( t.gets( QC::TNUM ) );
		memo->setText( t.gets( QC::TMEMO ) );
		payee->setText( t.gets( QC::TPAYEE ) );
		name->setText( nm );
		shortcut->setText( mynt[QC::NSHORTCUT].gets() );

		s.deleteWhere( TableSelect( QC::SACCTID, mynt.getu( QC::NACCTID ) ) );
		apply->setText( tr( "apply" ) );
  }

	area->clearSplits();
	uint sr=s.rows();
	for( uint i=0; i<sr; i++ ) area->addSplit( s[i] );
	for( uint i=sr; i<5; i++ ) area->addSplit( Split() );

	//acctedit->setEnabled( newy );
	name->setFocus();
}










NamedDlg::NamedDlg( QHacc * e, uint journalid, const QString& nm, QWidget * p,
										const char * n ) : LeftDlg( e, p, n ){

	right=new NamedFrame( e, journalid, splitter );
	iremakeList();
	resize( 520, 350 );

	if( !nm.isEmpty() ) listbox->setSelected( new QListBoxText( nm ), true );

	connect( engine, SIGNAL( addedT( const Transaction& ) ), 
					 SLOT( tmod( const Transaction& ) ) );
	connect( engine, SIGNAL( removedT( const Transaction& ) ),
					 SLOT( tmod( const Transaction& ) ) );
	connect( engine, SIGNAL( updatedT( const Transaction& ) ),
					 SLOT( tmod( const Transaction& ) ) );
}


NamedDlg::~NamedDlg(){}

void NamedDlg::iaccept(){}

void NamedDlg::iselect( const QString& s ){
	setCaption( s );

	right->setEnabled( true );
	right->setTrans( s );
}

void NamedDlg::tmod( const Transaction& t ){
	if( t[QC::XTTYPE]==QC::MEMORIZED ) remakeList();
}

void NamedDlg::iremakeList(){
	//QString str=listbox->currentText();

	listbox->clear();
	uint rr=0;
	vector<TableSelect> v;
	auto_ptr<QHaccResultSet> rslt=engine->getWhere( NAMEDTRANS, v, rr );
	for( uint i=0; i<rr; i++ ) addListItem( rslt->at( i ).gets( QC::NNAME ) );

	right->setEnabled( false );
	//listbox->setSelected( new QListBoxText( str ), true );
	addListItem( tr( NamedFrame::NEWY ) );
}

/* * * * * * * * * * * * * * * * * */
/* * * * * PITYPE Selector * * * * */
/* * * * * * * * * * * * * * * * * */
PluginSelector::PluginSelector( QHacc * engine, int pitype, QWidget * p, int d,
																const char * n ) : QFrame( p, n ){
	vector<PluginInfo> info=engine->getPluginInfo( pitype );
	// fill in the buttongroup, and keep track of which plugin goes
	// with which button id
	QHBoxLayout * lay=new QHBoxLayout( this );
	formater=new QButtonGroup( 3, Horizontal, this );
	lay->addWidget( formater );

	int nid=0;
	int idx=0;
	
	for( vector<PluginInfo>::iterator it=info.begin(); it!=info.end(); it++ ){
		// ignore Archive/Restore plugin
		if( it->isGUISelectable() ){
			QRadioButton * b=new QRadioButton( it->descr(), formater );
			int id=formater->insert( b );
			plugini[id]=( *it );
			if( d==-1 ){ // no default button passed in so select the native plugin
				if( it->stub()=="FILES" ) nid=id; // native format
			}
			else if( idx++==d ) nid=id; // else select if right plugin
		}
	}
	formater->setButton( nid );

}

PluginSelector::~PluginSelector(){}

void PluginSelector::setTitle( const QString& s ){ formater->setTitle( s ); }
int PluginSelector::getButtonID() const { 
#if QT_VERSION<320
	return formater->id( formater->selected() );
#else
	return formater->selectedId(); 
#endif
}
PluginInfo PluginSelector::getSelectedInfo() const {
#if QT_VERSION<320
	const int fid=formater->id( formater->selected() );
#else
	const int fid=formater->selectedId(); 
#endif

	map<int, PluginInfo>::const_iterator it=plugini.find( fid );
	return it->second;
}

/* * * * * * * * * * * * * * * * * * * */
/* * * * * Impport/Export Base * * * * */
/* * * * * * * * * * * * * * * * * * * */
HomeDlg::HomeDlg( QHacc * eng, int pitype, QWidget * p, const char * n )
	: QDialog( p, n, true ){
	
	engine=eng;

  pisel=new PluginSelector( engine, pitype, this );

  filename=new QLineEdit( this );

  fdlbl=new QLabel( tr( "Dataset Location" ), this );

  browsebtn=new QPushButton( tr( "Browse" ), this );
  connect( browsebtn, SIGNAL( clicked() ), SLOT( openFD() ) );

  okay=new QPushButton( tr( "Accept" ), this );
  connect( okay, SIGNAL( clicked() ), SLOT( accept() ) );

  cancel=new QPushButton( tr( "Dismiss" ), this );
  connect( cancel, SIGNAL( clicked() ), SLOT( close() ) );

}

HomeDlg::~HomeDlg(){}

void HomeDlg::openFD(){
	QString dir=queryHomeDlg();
  if( !dir.isEmpty() ) filename->setText( dir );
}

QString HomeDlg::queryHomeDlg(){
	QString dir, homer, cap, title;
	if( !filename->text().isEmpty() ) homer=filename->text();
	else homer=engine->getHome();

	cap=tr( "QHacc" );
	title=tr( "Select QHACC_HOME Location" );
	
	if( engine->getBP( "SIMPLEFILEDIALOG" ) ) 
		dir=QInputDialog::getText( cap, title, QLineEdit::Normal, homer, 0, this );
	else{
		const PluginInfo info=pisel->getSelectedInfo();
		TargetType targ=info.target();

		switch( targ ){
		case DATABASE:
			dir=QInputDialog::getText( cap, title, QLineEdit::Normal, homer, 
																 0, this );
			break;
		case DIRECTORY:
			dir=QFileDialog::getExistingDirectory( QString::null, this, title );
			break;
		case SINGLEFILE:
			dir=QFileDialog::getSaveFileName( QString::null, QString::null, 
																				this, title );
			break;
		default:
			dir=QInputDialog::getText( cap, title, QLineEdit::Normal, homer, 0, 
																 this );
		}
	}	

	return dir;
}

QString HomeDlg::home( bool * okay ) const {
	if( okay ) *okay=true; // there's no possible wrong QHACC_HOME here
	return pisel->getSelectedInfo().stub()+":"+filename->text();
}

auto_ptr<HomeDlg> HomeDlg::homedlg( QHacc * engine, QWidget * p,
																		const char * n ){
	auto_ptr<HomeDlg> ret( new QHaccHomeDlg( engine, p, n ) );
  ret->dolayout();
  return ret;
}


/* * * * * * * * * * * * * * * * * */
/* * * * * Database Dialog * * * * */
/* * * * * * * * * * * * * * * * * */

QHaccHomeDlg::QHaccHomeDlg( QHacc * eng, QWidget * p, const char * n )
  : HomeDlg( eng, QHacc::PIDATABASE, p, n ){}
QHaccHomeDlg::~QHaccHomeDlg(){}

void QHaccHomeDlg::dolayout(){
	setCaption( tr( "Select QHACC_HOME" ) );
	pisel->setTitle( tr( "Dataset Format" ) );
  QGridLayout * layout=new QGridLayout( this, 10, 10, 5 );
  layout->addMultiCellWidget( pisel, 0, 5, 0, 9 );

  int row=6;
  layout->addMultiCellWidget( fdlbl, row, row, 0, 2 );
  layout->addMultiCellWidget( filename, row, row, 3, 7 );
  layout->addMultiCellWidget( browsebtn, row, row, 8, 9 );

  row++;
  layout->addMultiCellWidget( okay, row, row, 0, 4 );
  layout->addMultiCellWidget( cancel, row, row, 5, 9 );
}

