/*
*  RAL -- Rubrica Addressbook Library
*  file: abook.c
*  
*  Copyright (C) Nicola Fragale <nicolafragale@libero.it>
*
*  This program is free software; you can redistribute it and/or modify
*  it under the terms of the GNU General Public License as published by
*  the Free Software Foundation; either version 3 of the License
*
*  This program is distributed in the hope that it will be useful,
*  but WITHOUT ANY WARRANTY; without even the implied warranty of
*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*  GNU General Public License for more details.
*
*  You should have received a copy of the GNU General Public License
*  along with this program; if not, write to the Free Software
*  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#include <time.h> 
#include <string.h>
#include <glib.h>
#include <glib-object.h>
#include <glib/gstdio.h>
#include <glib/gi18n-lib.h>

#include "rlib.h"
#include "manager.h"
#include "plugin.h"
#include "abook.h"
#include "card.h"
#include "personal.h"
#include "company.h"
#include "group_box.h"
#include "utils.h"
#include "error.h"
 

/*  property enumeration
 */
enum {
  ABOOK_NAME = 1,
  ABOOK_PATH,
  DELETED_CARDS,
  SELECTED_ID,
  MARKED_CARDS,
  PRINTABLE_CARDS,
  MODIFYED,
  FILE_FILTER,  
};


struct _RAbookPrivate {
  gchar*     name;          /* the addressbook (the file) name       */
  gchar*     path;          /* the addressbook (the file) path       */
  gchar*     filter;        /* filter/plugin setted from filechooser */

  gboolean   modifyed;      /* was addressbook modifyed?             */
  gulong     selected_id;   /* selected card id                      */
  gint       deleted;       /* cards marked deleted in addressbook   */
  gint       marked;        /* how many marked cards                 */
  gint       printable;     /* how many printable cards              */

  RGroupBox* box;

  GList*     cards;
  GList*     trash;
  GList*     iter;

  RPluginManager* manager;
  RPlugin*        plugin;    /* RPlugin */
  gpointer        engine;    /* ptr to the real plugin (a gobject) */

  gboolean dispose_has_run;
};


#define R_ABOOK_GET_PRIVATE(o)     (G_TYPE_INSTANCE_GET_PRIVATE((o),   \
			            R_ABOOK_TYPE, RAbookPrivate))


/*    signals enumeration 
 */
enum {
  OPEN_FAIL,            /* file not opened     */
  SAVE_FAIL,            /* file not saved      */

  ADDRESSBOOK_CHANGED,  /* addressbook changed */
  ADDRESSBOOK_READ,     /* file read           */ 
  ADDRESSBOOK_SAVED,    /* file saved          */
  ADDRESSBOOK_CLOSED,       

  CARD_READ,            /* card read                                        */ 
  CARD_ADDED,           /* card added to addressbook                        */
  CARD_DELETED,         /* card deleted from addressbook and moved to trash */
  CARD_DESTROYED,       /* card destroyed                                   */
  CARD_RECOVERED,       /* card recoverd from trash                         */
  CARD_REPLACED,        /* card replaced in addressbook                     */
  
  CARD_CUTTED,
  CARD_PASTED,
  NEED_NAME,

  LAST_SIGNAL 
};

static guint r_abook_signals[LAST_SIGNAL] = {0};

 
static void   r_abook_class_init    (RAbookClass* klass);
static void   r_abook_init          (RAbook* obj);

static void   r_abook_dispose       (RAbook* obj);
static void   r_abook_finalize      (RAbook* obj);

static void   r_abook_set_property  (GObject* obj, guint property_id,
				     GValue* value, GParamSpec* spec);
static void   r_abook_get_property  (GObject* obj, guint property_id,
				     GValue* value, GParamSpec* spec);



static gchar*   _r_get_file_extension   (gchar* string);
static gchar*   _r_get_plugin_extension (RPlugin* plugin);
static gboolean _r_try_all_plugins      (RAbook *abook, gchar* fname);
static void     _r_abook_add_card       (RAbook* abook, RCard* card, 
					 gboolean emit_signals);

static void     on_group_removed        (RGroupBox *box, gint grp_id, 
					 gpointer data);
static void     on_group_modifyed       (RGroupBox *box, gpointer grp,
					 gpointer data);





GType
r_abook_get_type()
{
  static GType r_abook_type = 0;
  
  if (!r_abook_type)
    {
      static const GTypeInfo r_abook_info =
	{
	  sizeof(RAbookClass),
	  NULL,
	  NULL,
	  (GClassInitFunc) r_abook_class_init,
	  NULL,
	  NULL,
	  sizeof(RAbook),
	  0,
	  (GInstanceInitFunc) r_abook_init
	};

      r_abook_type = g_type_register_static (G_TYPE_OBJECT, "RAbook",
					     &r_abook_info, 0);
    }
  
  return r_abook_type;
}


static void
r_abook_class_init(RAbookClass* klass)
{
  GObjectClass *class;
  GParamSpec* pspec;
  
  class  = G_OBJECT_CLASS (klass);
  
  class->dispose         = (GObjectFinalizeFunc) r_abook_dispose;
  class->finalize        = (GObjectFinalizeFunc) r_abook_finalize;
  class->set_property    = (gpointer) r_abook_set_property;
  class->get_property    = (gpointer) r_abook_get_property;
  
  klass->abook_open_file = NULL;
  klass->abook_save_file = NULL;
  klass->abook_overwrite = NULL;

  g_type_class_add_private (klass, sizeof(RAbookPrivate));

  /* class signals 
   */   
  /**
   * RAbook::open-fail:
   * @abook: the #RAbook object that receives the signal
   * @err: an #RError
   *
   * the "open-fail" signal is emitted if the given file can't be read
   */
  r_abook_signals[OPEN_FAIL] =
    g_signal_new("open_fail", 
		 R_ABOOK_TYPE,
		 G_SIGNAL_RUN_LAST,
		 G_STRUCT_OFFSET(RAbookClass, open_fail),
		 NULL,
		 NULL,
		 g_cclosure_marshal_VOID__INT,
		 G_TYPE_NONE,            /* return type */
		 1,                      /* params      */
		 G_TYPE_INT);            /* params type: error code */

  /**
   * RAbook::save-fail:
   * @abook: the #RAbook object that receives the signal
   * @err: an #RError
   *
   * the "save-fail" signal is emitted if the given file can't be read 
   */
  r_abook_signals[SAVE_FAIL] =
    g_signal_new("save_fail", 
		 R_ABOOK_TYPE,
		 G_SIGNAL_RUN_LAST,
		 G_STRUCT_OFFSET(RAbookClass, save_fail),
		 NULL,
		 NULL,
		 g_cclosure_marshal_VOID__INT,
		 G_TYPE_NONE,            /* return type */
		 1,                      /* params      */
		 G_TYPE_INT);            /* params type: error code */

  /**
   * RAbook::card-read:
   * @abook: the #RAbook object that receives the signal
   * @data: a pointer to the readed card 
   * 
   * The "card-read" signal is emitted after the addressbook has been readed
   */
  r_abook_signals[CARD_READ] = 
    g_signal_new("card_read", 
		 R_ABOOK_TYPE,
		 G_SIGNAL_RUN_LAST,
		 G_STRUCT_OFFSET(RAbookClass, card_read),
		 NULL,
		 NULL,
		 g_cclosure_marshal_VOID__POINTER,
		 G_TYPE_NONE,
		 1,
		 G_TYPE_POINTER);
   
  /**
   * RAbook::card-added:
   * @abook: the #RAbook object that receives the signal
   * @data: a pointer to the readed card 
   *
   * The "card-added" signal is emitted after a card is added to addressbook.
   * A pointer to the card is passed to the callback function
   */
  r_abook_signals[CARD_ADDED] =
    g_signal_new("card_added", 
		 R_ABOOK_TYPE,
		 G_SIGNAL_RUN_LAST,
		 G_STRUCT_OFFSET(RAbookClass, card_added),
		 NULL,
		 NULL,
		 g_cclosure_marshal_VOID__POINTER,
		 G_TYPE_NONE,
		 1,
		 G_TYPE_POINTER);
  
  /**
   * RAbook::card-deleted:
   * @abook: the #RAbook object that receives the signal
   *
   * The "card-removed" signal is emitted when a card is 
   * (deleted) removed from addressbook. The removed cards are 
   * marked as deleted, to really delete (remove) the card 
   * you must destroy it. You can recover (deleted) removed cards.
   * Destroyed cards are unrecoverable
   */
  r_abook_signals[CARD_DELETED] =
    g_signal_new("card_deleted", 
		 R_ABOOK_TYPE,
		 G_SIGNAL_RUN_LAST,
		 G_STRUCT_OFFSET(RAbookClass, card_deleted),
		 NULL,
		 NULL,
		 g_cclosure_marshal_VOID__POINTER,
		 G_TYPE_NONE,
		 1,
		 G_TYPE_POINTER);

  /**
   * RAbook::card-destroyed:
   * @abook: the #RAbook object that receives the signal
   *
   * The "card-destroyed" signal is emitted when a card is 
   * deleted from addressbook. Destroyed cards are unrecoverable.
   */
  r_abook_signals[CARD_DESTROYED] =
    g_signal_new("card_destroyed", 
		 R_ABOOK_TYPE,
		 G_SIGNAL_RUN_LAST,
		 G_STRUCT_OFFSET(RAbookClass, card_destroyed),
		 NULL,
		 NULL,
		 g_cclosure_marshal_VOID__STRING,
		 G_TYPE_NONE,
		 1,
		 G_TYPE_STRING);

  /**
   * RAbook::card-recovered:
   * @abook: the #RAbook object that receives the signal
   *
   * The "card-recovered" signal is emitted when a previously deleted 
   * marked card is unmarked. 
   */
  r_abook_signals[CARD_RECOVERED] =
    g_signal_new("card_recovered", 
		 R_ABOOK_TYPE,
		 G_SIGNAL_RUN_LAST,
		 G_STRUCT_OFFSET(RAbookClass, card_recovered),
		 NULL,
		 NULL,
		 g_cclosure_marshal_VOID__POINTER,
		 G_TYPE_NONE,
		 1,
		 G_TYPE_POINTER);

  /**
   * RAbook::card-replaced:
   * @abook: the #RAbook object that receives the signal
   * @data: a pointer to the readed card 
   *
   * The "card-replaced" signal is emitted on a card replacement.
   * A pointer to the new card is passed to the callback function
   */
  r_abook_signals[CARD_REPLACED] =
    g_signal_new("card_replaced", 
		 R_ABOOK_TYPE,
		 G_SIGNAL_RUN_LAST,
		 G_STRUCT_OFFSET(RAbookClass, card_replaced),
		 NULL,
		 NULL,
		 g_cclosure_marshal_VOID__POINTER,
		 G_TYPE_NONE,
		 1,
		 G_TYPE_POINTER);

  /**
   * RAbook::card-cutted
   * @abook: the #RAbook object that receives the signal
   * @data: a pointer to the cutted card 
   *
   * The "card-cutted" signal is emitted when a card in the 
   * addressbook is cutted.
   * A pointer to the cutted card is passed to the callback function
   */
  r_abook_signals[CARD_CUTTED] =
    g_signal_new("card_cutted", 
		 R_ABOOK_TYPE,
		 G_SIGNAL_RUN_LAST,
		 G_STRUCT_OFFSET(RAbookClass, card_cutted),
		 NULL,
		 NULL,
		 g_cclosure_marshal_VOID__POINTER,
		 G_TYPE_NONE,
		 1,
		 G_TYPE_POINTER);

  
  /**
   * RAbook::need-name:
   * @abook: the #RAbook object that receives the signal
   *
   * The "need-name" signal is emitted if user 
   * tries to save an addressbook without giving a filename to it
   */
  r_abook_signals[NEED_NAME] =
    g_signal_new("need_name", 
		 R_ABOOK_TYPE,
		 G_SIGNAL_RUN_LAST,
		 G_STRUCT_OFFSET(RAbookClass, need_name),
		 NULL,
		 NULL,
		 g_cclosure_marshal_VOID__VOID,
		 G_TYPE_NONE,
		 0,
		 G_TYPE_NONE);

  /**
   * RAbook::addressbook-read:
   * @abook: the #RAbook object that receives the signal
   *
   * The "addressbook-read" signal is emitted when the 
   * addressbook is read
   */
  r_abook_signals[ADDRESSBOOK_READ] =
    g_signal_new("addressbook_read", 
		 R_ABOOK_TYPE,
		 G_SIGNAL_RUN_LAST,
		 G_STRUCT_OFFSET(RAbookClass, addressbook_read),
		 NULL,
		 NULL,
		 g_cclosure_marshal_VOID__VOID,
		 G_TYPE_NONE,
		 0,
		 G_TYPE_NONE);

  /**
   * RAbook::addressbook-saved:
   * @abook: the #RAbook object that receives the signal
   *
   * The "addressbook_saved" signal
   */
  r_abook_signals[ADDRESSBOOK_SAVED] =
    g_signal_new("addressbook_saved", 
		 R_ABOOK_TYPE,
		 G_SIGNAL_RUN_LAST,
		 G_STRUCT_OFFSET(RAbookClass, addressbook_saved),
		 NULL,
		 NULL,
		 g_cclosure_marshal_VOID__VOID,
		 G_TYPE_NONE,
		 0,
		 G_TYPE_NONE);

  /**
   * RAbook::addressbook-changed:
   * @abook: the #RAbook object that receives the signal
   *
   * The "addressbook_changed" signal
   */
  r_abook_signals[ADDRESSBOOK_CHANGED] =
    g_signal_new("addressbook_changed", 
		 R_ABOOK_TYPE,
		 G_SIGNAL_RUN_LAST,
		 G_STRUCT_OFFSET(RAbookClass, addressbook_changed),
		 NULL,
		 NULL,
		 g_cclosure_marshal_VOID__VOID,
		 G_TYPE_NONE,
		 0,
		 G_TYPE_NONE);


  /**
   * RAbook::addressbook-closed:
   * @abook: the #RAbook object that receives the signal
   *
   * The "addressbook_closed" signal
   */
  r_abook_signals[ADDRESSBOOK_CLOSED] =
    g_signal_new("addressbook_closed", 
		 R_ABOOK_TYPE,
		 G_SIGNAL_RUN_LAST,
		 G_STRUCT_OFFSET(RAbookClass, addressbook_closed),
		 NULL,
		 NULL,
		 g_cclosure_marshal_VOID__VOID,
		 G_TYPE_NONE,
		 0,
		 G_TYPE_NONE);


  /**
   * ABook:addressbook-name
   *
   * address book's name
   */
  pspec = g_param_spec_string("addressbook-name",
			      "addressbook's name",
			      "the name (filename) of the addressbook",
			      NULL,
			      G_PARAM_CONSTRUCT | G_PARAM_READWRITE);
  g_object_class_install_property(class, ABOOK_NAME, pspec);     

  /**
   * ABook:addressbook-path
   *
   * address book's path
   */
  pspec = g_param_spec_string("addressbook-path",
			      "addressbook's path",
			      "the path of the addressbook",
			      NULL,
			      G_PARAM_CONSTRUCT | G_PARAM_READWRITE);
  g_object_class_install_property(class, ABOOK_PATH, pspec);


  /**
   * ABook:deleted-cards
   *
   * the number of cards marked as deleted in the addressbook
   */
  pspec = g_param_spec_int("deleted-cards",
			   "deleted cards",
			   "number of cards marked as deleted "
			   "in the addressbook",
			   G_MININT,
			   G_MAXINT,
			   0,
			   G_PARAM_READABLE);
  g_object_class_install_property(class, DELETED_CARDS, pspec);


  /**
   * ABook:selected-id
   *
   * the id of the selected card
   */
  pspec = g_param_spec_long("selected-id",
			    "selected card's id",
			    "The id of the selected card",
			    G_MINLONG,
			    G_MAXLONG,
			    0,
			    G_PARAM_CONSTRUCT | G_PARAM_READWRITE);
  g_object_class_install_property(class, SELECTED_ID, pspec);

  /**
   * ABook:addressbook-modifyed
   *
   * Check if the addressbook was modifyed
   */
  pspec = g_param_spec_boolean("addressbook-modifyed",
			       "addressbook modifyed",
			       "check if addressbook was modifyed",
			       FALSE,
			       G_PARAM_CONSTRUCT | G_PARAM_READWRITE);
  g_object_class_install_property(class, MODIFYED, pspec);

  /**
   * ABook:cards-marked
   *
   * Check if the all addressbook's cards are marked
   */
  pspec = g_param_spec_int("marked-cards",
			   "marked cards",
			   "check how many addressbooks's cards are marked",
			   0,
			   G_MAXINT,
			   0,
			   G_PARAM_CONSTRUCT | G_PARAM_READWRITE);
  g_object_class_install_property(class, MARKED_CARDS, pspec);

  /**
   * ABook:printable-cards
   *
   * how many cards are marked as printable
   */
  pspec = g_param_spec_int("printable-cards",
			   "printable cards",
			   "how many cards are marked printable",
			   0,
			   G_MAXINT,
			   0,
			   G_PARAM_CONSTRUCT | G_PARAM_READWRITE);
  g_object_class_install_property(class, MARKED_CARDS, pspec);

  /**
   * ABook:file-filter
   *
   * File filter. It indentifies the addressbook's fileformat and 
   * the plugin that will be loaded to manage this fileformat.
   */
  pspec = g_param_spec_string("file-filter",
			      "file filter",
			      "filter used to identify the file's type",
			      NULL,
			      G_PARAM_CONSTRUCT | G_PARAM_READWRITE);
  g_object_class_install_property(class, FILE_FILTER, pspec);
}



static void 
r_abook_set_property (GObject* obj, guint property_id,
		      GValue* value, GParamSpec* spec)
{
  RAbook *self = (RAbook*) obj;
  
  switch(property_id)
    {
    case ABOOK_NAME:
      if (self->priv->name)
	g_free(self->priv->name);
      self->priv->name = g_value_dup_string(value);
      break;
      
    case ABOOK_PATH:
      g_free(self->priv->path);
      self->priv->path = g_value_dup_string(value);
      break;
      
    case DELETED_CARDS:
      break;

    case SELECTED_ID: 
      self->priv->selected_id = g_value_get_long(value);
      break;
      
    case MODIFYED:
      self->priv->modifyed = g_value_get_boolean(value);
      break;

    case MARKED_CARDS:
      self->priv->marked = g_value_get_int(value);
      break;

    case PRINTABLE_CARDS:
      self->priv->printable = g_value_get_int(value);
      break;

    case FILE_FILTER:
      self->priv->filter = g_value_dup_string(value);
      break;

    default:
      break;    
    }
}


static void 
r_abook_get_property (GObject* obj, guint property_id,
		      GValue* value, GParamSpec* spec)
{
  RAbook *self = (RAbook*) obj;
  
  switch(property_id)
    {
    case ABOOK_NAME:
      g_value_set_string(value, self->priv->name);
      break;
      
    case ABOOK_PATH:
      g_value_set_string(value, self->priv->path);
      break;
      
    case DELETED_CARDS:
      g_value_set_int(value, self->priv->deleted);
      break;
      
    case MODIFYED:
      g_value_set_boolean(value, self->priv->modifyed);
      break;

    case MARKED_CARDS:
      g_value_set_int(value, self->priv->marked);
      break;

    case PRINTABLE_CARDS:
      g_value_set_int(value, self->priv->printable);
      break;

    case SELECTED_ID: 
      g_value_set_long(value, self->priv->selected_id);
      break;

    default:
      break;    
    }
}


static void
r_abook_init(RAbook* self)
{
  self->priv = R_ABOOK_GET_PRIVATE(self);

  self->priv->name        = NULL;
  self->priv->path        = NULL;
  self->priv->filter      = NULL;

  self->priv->modifyed    = FALSE;
  self->priv->selected_id = 0L;
  self->priv->deleted     = 0;
  self->priv->marked      = 0;
  self->priv->printable   = 0;

  self->priv->box         = r_lib_get_group_box();
 
  self->priv->cards       = NULL;
  self->priv->trash       = NULL;
  self->priv->iter        = NULL;

  self->priv->manager     = r_lib_get_manager();
  self->priv->plugin      = r_plugin_manager_get_plugin(self->priv->manager,
							"rubrica");
  self->priv->engine      = r_plugin_get_engine(self->priv->plugin);
  
  r_lib_admit (self);
  
  self->priv->dispose_has_run = FALSE;

  g_signal_connect(G_OBJECT (self->priv->box), "group_removed",
		   G_CALLBACK (on_group_removed), self); 
 
  g_signal_connect(G_OBJECT (self->priv->box), "group_modifyed",
		   G_CALLBACK (on_group_modifyed), self); 
}

 

static void 
r_abook_dispose (RAbook* self)
{
  RAbookPrivate* priv;

  g_return_if_fail(IS_R_ABOOK(self));
  
  priv = R_ABOOK_GET_PRIVATE(self);
  
  /* free the list's items (PersonalCard or CompanyCard) 
   */
  priv->iter = priv->cards;
  for (; priv->iter; priv->iter = priv->iter->next)
    {
      RCard* card = priv->iter->data;

      if (IS_R_CARD(card))
	r_card_free(R_CARD(card));
    }
  
  g_list_free(priv->cards);
  g_list_free(priv->trash);
  priv->cards = NULL;
  priv->trash = NULL;
  priv->iter  = NULL;
  
  r_plugin_free(priv->plugin);
  
  if (priv->dispose_has_run)
    return;

  priv->dispose_has_run = TRUE;
}


static void 
r_abook_finalize (RAbook* self)
{
  RAbookPrivate* priv;

  g_return_if_fail(IS_R_ABOOK(self));
  
  priv = R_ABOOK_GET_PRIVATE(self);

  r_utils_free_string(priv->name);
  r_utils_free_string(priv->path);
  r_utils_free_string(priv->filter);
}


static gchar* 
_r_get_file_extension (gchar* string)
{
  gint len;

  if (!string)
    return NULL;

  len = strlen(string);  
  for(; len; len --)
    {
      if (string[len] == '.')
	{
	  char* tmp;
	  
	  tmp = g_strdup(&string[len+1]);
	  	  
	  return tmp;
	}
    }  
  
  return NULL;
}

static gchar* 
_r_get_plugin_extension(RPlugin* plugin)
{
  GList *filters = NULL;
  RFilter* filter;
  gchar* ext = NULL;

  g_return_val_if_fail(IS_R_PLUGIN(plugin), NULL);
  
  filters = r_plugin_get_filters(plugin);
  
  if (filters && filters->data)
    {
      filter = (RFilter*) filters->data;
      g_object_get(filter, "filter-extension", &ext, NULL);
    }
  
  return ext;
}


static gboolean
_r_try_all_plugins(RAbook *abook, gchar* fname)
{
  RAbookClass* class;
  gint n, i = 0;

  g_return_val_if_fail(IS_R_ABOOK(abook), FALSE);

  class = R_ABOOK_GET_CLASS(abook);

  g_log(G_LOG_DOMAIN, G_LOG_LEVEL_INFO, " ");

  n = r_plugin_manager_how_many(abook->priv->manager);
  for (; i < n; i++)
    {
      RPlugin* plugin;
      gchar *name, *label;

      plugin = r_plugin_manager_get_nth_plugin(abook->priv->manager, i);
      g_object_get(plugin, "plugin-name", &name, "plugin-label", &label, NULL);
                 
      g_log(G_LOG_DOMAIN, G_LOG_LEVEL_INFO, "Trying plugin: %s (%s)", 
	    name, label);

      if (r_abook_load_plugin(abook, name))
	{
	  if(class->abook_open_file(abook, fname))
	    {
	      g_log(G_LOG_DOMAIN, G_LOG_LEVEL_INFO, "Used %s plugin", name);
	      g_log(G_LOG_DOMAIN, G_LOG_LEVEL_INFO, " ");
	      
	      return TRUE;
	    }
	}
      else
	g_log(G_LOG_DOMAIN, G_LOG_LEVEL_WARNING, "Can't load %s plugin", name);
    }

  return FALSE;
}


static void
_r_abook_add_card(RAbook* abook, RCard* card, gboolean emit_signals)
{
  GList* user_groups = NULL;
  RAbookPrivate* priv;
  gboolean deleted = FALSE;
  glong id;
		      
  g_return_if_fail(IS_R_ABOOK(abook));

  priv = R_ABOOK_GET_PRIVATE(abook);

  user_groups = r_card_get_groups_owned_by(R_CARD(card), "user");
  for (; user_groups; user_groups = user_groups->next)
    r_group_box_add_group(abook->priv->box, R_GROUP(user_groups->data));

  /* check if the card is marked deleted */
  g_object_get(card, "card-id", &id, "card-deleted", &deleted, NULL);
  if (deleted)
    {
      priv->deleted++;
      priv->trash = g_list_append(priv->trash, (gpointer) id);
    }
  
  priv->cards = g_list_append(priv->cards, card);
  priv->iter  = priv->cards;

  if (emit_signals)
    {
      g_signal_emit_by_name(abook, "card_added", card, G_TYPE_POINTER);  
      g_signal_emit_by_name(abook, "addressbook_changed", NULL, G_TYPE_NONE);
    }
}



static void 
on_group_removed  (RGroupBox *box, gint grp_id, gpointer data)
{
  RAbook *book = R_ABOOK(data);
  RAbookPrivate* priv = R_ABOOK_GET_PRIVATE(book);

  priv->modifyed = TRUE;
  g_signal_emit_by_name(book, "addressbook_changed", NULL, G_TYPE_NONE);
}


static void 
on_group_modifyed (RGroupBox *box, gpointer grp, gpointer data)
{
  RAbook *book = R_ABOOK(data);
  RAbookPrivate* priv = R_ABOOK_GET_PRIVATE(book);;
  
  priv->modifyed = TRUE;
  g_signal_emit_by_name(book, "addressbook_changed", NULL, G_TYPE_NONE);
}



/*   ***************************** Public *****************************
*/


/**
 * r_abook_new
 *
 * create a #RAbook
 *
 * Returns: a new #RAbook
 */
RAbook*
r_abook_new(void)
{
  RAbook* abook;

  if (!r_lib_is_running())
    g_error("\nLibral is not running");

  abook = g_object_new(r_abook_get_type(), NULL);
  
  return abook;
}


/**
 * r_abook_free
 * @abook: a #RAbook
 * 
 * free the #RAbook
 * The "addressbook_closed" signal is emitted 
 */
void
r_abook_free(RAbook* abook)
{
  g_return_if_fail(IS_R_ABOOK(abook));

  g_signal_emit_by_name(abook, "addressbook_closed", NULL, G_TYPE_NONE);

  g_object_unref(abook);   
}



/**
 * r_abook_open_file
 * @abook: a #RAbook
 * @fname: the name of the addressbook
 *
 * open the addressbook with given name
 *
 * Returns: %TRUE if file is opened successfully, %FALSE otherwise
 */
gboolean 
r_abook_open_file (RAbook* abook, gchar* fname)
{
  RAbookClass* class;
  RAbookPrivate* priv;
  gboolean result = FALSE;
  
  g_return_val_if_fail(IS_R_ABOOK(abook), FALSE);

  priv  = R_ABOOK_GET_PRIVATE(abook);
  class = R_ABOOK_GET_CLASS(abook);
  
  g_log(G_LOG_DOMAIN, G_LOG_LEVEL_INFO, " ");
  g_log(G_LOG_DOMAIN, G_LOG_LEVEL_INFO, "Opening %s ...", fname);
 
  if (g_ascii_strcasecmp(priv->filter, "autodetect") == 0)
    {
      g_log(G_LOG_DOMAIN,G_LOG_LEVEL_INFO, "Try to determine the file's type");
      
      result = _r_try_all_plugins(abook, fname);
    }
  else
    {
      g_log(G_LOG_DOMAIN, G_LOG_LEVEL_INFO, "Opening file with %s plugin",
	    priv->filter);       
      
      if (r_abook_load_plugin(abook, priv->filter))
	result = class->abook_open_file(abook, fname);        
    }

  if (result)
    {
      g_log(G_LOG_DOMAIN, G_LOG_LEVEL_INFO, "addressbook: %s", 
	    abook->priv->name);
      g_log(G_LOG_DOMAIN, G_LOG_LEVEL_INFO, "path: %s", abook->priv->path); 
      
      g_log(G_LOG_DOMAIN, G_LOG_LEVEL_INFO, "%s opened successfully", fname);
      g_log(G_LOG_DOMAIN, G_LOG_LEVEL_INFO, "Cards in this addressbook: %d", 
	    r_abook_get_items(R_ABOOK(abook)));
      g_log(G_LOG_DOMAIN, G_LOG_LEVEL_INFO, "Cards marked as deleted: %d", 
	    abook->priv->deleted);
      g_log(G_LOG_DOMAIN, G_LOG_LEVEL_INFO, " ");

      return TRUE;
    }

  // error occurred during file loading
  g_log(G_LOG_DOMAIN, G_LOG_LEVEL_WARNING, "%s open failed", fname);
  g_log(G_LOG_DOMAIN, G_LOG_LEVEL_INFO, " ");

  return FALSE;
}



/**
 * r_abook_save_file
 * @abook: a #RAbook
 * @fname: the name of file
 * 
 * save the addressbook with given filename
 *
 * Returns: %TRUE if file is saved successfully, %FALSE otherwise
 */
gboolean 
r_abook_save_file (RAbook* abook, gchar* fname, gint compression_rate)
{
  RAbookClass* class;
  RAbookPrivate* priv;
  gchar* ext = NULL, *tmp;
  gboolean result = FALSE;

  g_return_val_if_fail(IS_R_ABOOK(abook), FALSE);

  if (!fname)
    {
      g_log(G_LOG_DOMAIN, G_LOG_LEVEL_WARNING, "addressbook needs a filename");

      g_signal_emit_by_name(abook, "need_name", NULL, G_TYPE_NONE);
      
      return FALSE;
    }

  class = R_ABOOK_GET_CLASS(abook);
  priv  = R_ABOOK_GET_PRIVATE(abook);
  ext   = _r_get_file_extension(g_path_get_basename(fname));
  
  if (!ext)
    ext = _r_get_plugin_extension(abook->priv->plugin); 
  
  tmp = g_strdup_printf("%s.%s", fname, ext);
  g_free(fname);
  fname = tmp; 

  g_log(G_LOG_DOMAIN, G_LOG_LEVEL_INFO, 
	"Trying plugin %s to save the file", r_abook_get_plugin_name(abook));

  result = class->abook_save_file(abook, fname, compression_rate);   

  if (result)
    {
      g_signal_emit_by_name(abook, "addressbook_saved", NULL, G_TYPE_NONE);
      
      g_free(fname);
      return TRUE;
    } 
  else
    g_signal_emit_by_name(abook, "save_fail", WRITING_FILE, G_TYPE_INT);
  
  g_free(fname);  
  return FALSE;  
}


/**
 * r_abook_overwrite_file
 * @abook: a #RAbook
 * @backup: boolean
 *
 * overwrite an existing addressbook with a new one.
 * If backup is %TRUE the old addressbook is backuped.
 *
 * Returns: a %TRUE if old addressbook was successfully overwited, 
 * %FALSE otherwise
 **/
gboolean
r_abook_overwrite_file(RAbook* abook, gboolean backup, gint compression_rate)
{
  gchar *old, *new;
  RAbookClass* class;

  g_return_val_if_fail(IS_R_ABOOK(abook), FALSE);
  
  class = R_ABOOK_GET_CLASS(abook);
  
  if (backup)
    {
      old = g_strdup_printf("%s%s%s", abook->priv->path, 
			    G_DIR_SEPARATOR_S, abook->priv->name);

      if (!old || (g_ascii_strcasecmp(old, _("no name")) == 0))
	{
	  g_log(G_LOG_DOMAIN, G_LOG_LEVEL_WARNING, 
		"addressbook needs a filename");

	  g_signal_emit_by_name(abook, "need_name", NULL, G_TYPE_NONE);
	  
	  return FALSE;
	}      

      new = g_strdup_printf("%s~", old);
      g_rename(old, new);
      g_free(new);
      g_free(old);
    }
  
  /* write the file */
  if (class->abook_overwrite)
    return class->abook_overwrite(abook, compression_rate); 
  
  return FALSE;
}


/**
 * r_abook_copy
 * @abook: a #RAbook
 *
 * copy the addressbook pointed by abook
 *
 * Returns: a new #RAbook
 */
RAbook* 
r_abook_copy (RAbook* abook)
{
  RAbook* new;
  gpointer engine;
  RPlugin* plugin;
  RAbookClass* class = NULL;
  gpointer card;
  
  g_return_val_if_fail(IS_R_ABOOK(abook), NULL);
  
  new = r_abook_new();
  
  engine = r_abook_get_engine(abook);
  plugin = r_abook_get_plugin(abook);

  new->priv->engine = engine;
  new->priv->plugin = plugin;
  g_object_ref(engine);
  g_object_ref(plugin);
  
  class = R_ABOOK_GET_CLASS(new);
  if (class)
    {
      class->abook_open_file = r_plugin_get_handle(plugin, "read");
      class->abook_save_file = r_plugin_get_handle(plugin, "write");
      class->abook_overwrite = r_plugin_get_handle(plugin, "overwrite");
    }

  card = r_abook_get_card(abook);
  for (; card; card = r_abook_get_next_card(abook))
    {
      gpointer new_card; 

      new_card = r_card_copy(R_CARD(card));
      
      if (new_card)
	r_abook_add_card(new, new_card);
    }    
  r_abook_reset_book(abook);
  
  return new;
}


/**
 * r_abook_plugin_from_file
 * @abook: a #RAbook
 * @filename: gchar*
 *
 * Try to load the plugin using the file extension. If the file has a 
 * known extension the rigth plugin is loaded.
 * 
 * Returns: %TRUE if plugin is loaded, %FALSE otherwise
 **/
gboolean 
r_abook_plugin_from_file (RAbook* abook, gchar* filename)
{
  GList* filters = NULL;

  filters = r_plugin_manager_get_all_filters(abook->priv->manager);
  for (; filters; filters = filters->next)
    {
      gchar *name, *pattern;
      RFilter* filter = (RFilter*) filters->data;
      
      g_object_get(filter, "filter-name", &name, NULL);

      r_filter_reset(filter);
      pattern = r_filter_get_pattern(filter);
      for (; pattern; pattern = r_filter_get_next_pattern(filter))
	if (g_str_has_suffix(filename, pattern))
	  {
	    gchar* name;

	    g_object_get(filter, "filter-name", &name, NULL);
	    return r_abook_load_plugin (abook, name);
	  }
    }
  
  return FALSE;  
}



/**
 * r_abook_load_plugin
 * @abook: a #RAbook
 * @plugin_name: const gchar*
 *
 * Load the "plugin_name" plugin.
 * 
 * Returns: %TRUE if plugin is loaded, %FALSE otherwise
 **/
gboolean 
r_abook_load_plugin (RAbook* abook, const gchar* plugin_name)
{
  RPlugin* plugin;
  RAbookClass* class = NULL;

  g_return_val_if_fail(IS_R_ABOOK(abook), FALSE);
  g_return_val_if_fail(plugin_name != NULL, FALSE);
  
  class = R_ABOOK_GET_CLASS(abook);
  if (!class)
    {
      g_warning("\nR_ABOOK_GET_CLASS");
      
      return FALSE;
    }

  plugin = r_plugin_manager_get_plugin(abook->priv->manager, plugin_name);
  if (plugin)
    {
      abook->priv->plugin = plugin;
      abook->priv->engine = r_plugin_get_engine(plugin);

      class->abook_open_file = r_plugin_get_handle(plugin, "read");
      class->abook_save_file = r_plugin_get_handle(plugin, "write");
      class->abook_overwrite = r_plugin_get_handle(plugin, "overwrite");
      
      return TRUE;
    }    

  return FALSE;
}



/**
 * r_abook_get_plugin_extension
 * @abook: a #RAbook
 * 
 * Get the extension that is associate to the current active plugin
 *
 * Returns: a gchar* or NULL if an error occured
 **/
gchar*   
r_abook_get_plugin_extension (RAbook* abook)
{
  GList* filters = NULL;
  RFilter* filter;

  g_return_val_if_fail(IS_R_ABOOK(abook), NULL);
  
  filters = r_plugin_get_filters (abook->priv->plugin);
  filter  = (RFilter*) filters->data;
  r_filter_reset(filter);
  
  return r_filter_get_pattern(filter);
}


/**
 * r_abook_get_plugin_name
 * @abook: a #RAbook
 * 
 * get the name of the active plugin
 *
 * Returns: a gchar* 
 */
gchar* 
r_abook_get_plugin_name (RAbook* abook)
{
  g_return_val_if_fail(IS_R_ABOOK(abook), NULL);
  
  return r_plugin_get_name(R_PLUGIN(abook->priv->plugin));
}


/**
 * r_abook_get_engine
 * @abook: a #RAbook
 * 
 * Get the active engine (the real plugin)
 *
 * Currently are available those engines (plugins) with rubrica:
 * Rubrica        (manages the rubrica's file format)
 * VCard          (manages the vcard's file format)
 * CsvGmail       (manages the gmail's csv file format)
 * CsvThunderbird (managed the thunderbird's csv file format)
 * 
 * working on CsvOutlook and generic csv
 *
 * Returns: a %gpointer to the engine
 **/	 
gpointer 
r_abook_get_engine (RAbook* abook)
{
  g_return_val_if_fail(IS_R_ABOOK(abook), NULL);

  return abook->priv->engine;
}



/**
 * r_abook_get_r_plugin
 * @abook: a #RAbook
 * 
 * Get the currently active plugin (ptr to an #RPlugin struct)
 *
 * Returns: a %gpointer to the plugin
 **/	 
gpointer 
r_abook_get_plugin (RAbook* abook)
{
  g_return_val_if_fail(IS_R_ABOOK(abook), NULL);

  return abook->priv->plugin;
}


/**
 * r_abook_get_items
 * @abook: a #RAbook
 *
 * get the number of cards into addressbok
 *
 * Returns: an integer 
 */
gint
r_abook_get_items (RAbook* abook)
{
  g_return_val_if_fail(IS_R_ABOOK(abook), -1);

  return g_list_length(abook->priv->cards);
}

/**
 * r_abook_get_deleted
 * @abook: a #RAbook
 *
 * get the number of cards marked as deleted into addressbok
 *
 * Returns: an integer. If an error occurred -1 is returne
 */
gint 
r_abook_get_deleted (RAbook* abook)
{
  g_return_val_if_fail(IS_R_ABOOK(abook), -1);
  
  return abook->priv->deleted;
}



/**
 * r_abook_is_empty
 * @abook: a #RAbook
 *
 * is the addressbook empty?
 *
 * Returns: a boolean. %TRUE if addressbook is empty, %FALSE otherwise
 */
gboolean
r_abook_is_empty (RAbook* abook)
{
  g_return_val_if_fail(IS_R_ABOOK(abook), TRUE);
  
  return (r_abook_get_items(abook) == 0);
}



/**
 * r_abook_add_card
 * @abook: a #RAbook
 * @card: an #RCard object
 *
 * append the given card into the addressbook
 * The "card_added" and "addressbook_changed" signals are emitted
 */
void   
r_abook_add_card (RAbook* abook, RCard* card)
{
  _r_abook_add_card(abook, card, TRUE);
}



/**
 * r_abook_add_loaded_card
 * @abook: a #RAbook
 * @card: an #RCard object
 *
 * append a card read from disk into the addressbook
 */
void
r_abook_add_loaded_card (RAbook* abook, RCard* card)
{
  _r_abook_add_card(abook, card, FALSE);  
}



/**
 * r_abook_paste_card
 * @abook: a #RAbook
 * @card: an #RCard object
 *
 * peste the given card into the addressbook (append the card)
 */
void   
r_abook_paste_card (RAbook* abook, RCard* card)
{
  _r_abook_add_card(abook, card, TRUE);
}



/**
 * r_abook_empty_trash
 * @abook: a #RAbook
 *
 * Destroy all cards in the trash. The destroyed cards are unrecoverable
 * the "addressbook_changed" signal is emitted
 */ 
void
r_abook_empty_trash(RAbook* abook)
{
  GList* iter;

  g_return_if_fail(IS_R_ABOOK(abook));  
  
  for (iter = abook->priv->trash; iter; iter = iter->next)
    {
      RCard* card = NULL;

      card = r_abook_get_card_by_id(abook, (glong) iter->data);      
      g_object_set(card, "card-destroyed", TRUE, NULL);

      abook->priv->deleted--;        
    }

  g_list_free(abook->priv->trash);
  abook->priv->trash = NULL;
  
  g_signal_emit_by_name(abook, "addressbook_changed", NULL, G_TYPE_NONE); 
}


/**
 * r_abook_destroy_card
 * @abook: a #RAbook
 * @card: an #RCard object
 *
 * Destroy the given card. The destroyed cards are unrecoverable
 * The "card_destroyed" and "addressbook_changed" signals are emitted
 */ 
void
r_abook_destroy_card(RAbook* abook, RCard* card)
{
  gchar* name;

  g_return_if_fail(IS_R_ABOOK(abook));  
  g_return_if_fail(IS_R_CARD(card));

  g_object_get(card, "card-name", &name, NULL);
  g_object_set(card, "card-destroyed", TRUE, NULL);

  abook->priv->cards = g_list_remove(abook->priv->cards, card);      
  abook->priv->deleted--;   

  g_signal_emit_by_name(abook, "card_destroyed", name, G_TYPE_STRING);
  g_signal_emit_by_name(abook, "addressbook_changed", NULL, G_TYPE_NONE);
}


/**
 * r_abook_delete_card
 * @abook: a #RAbook
 * @card: an #RCard object
 *
 * mark the given card as delete. 
 * The id of the card is appended to the addressbook's trash
 * The "card_removed" and "addressbook_changed" signals are emitted
 */
void
r_abook_delete_card (RAbook* abook, RCard* card)
{
  gint id;

  g_return_if_fail(IS_R_ABOOK(abook));  
  g_return_if_fail(IS_R_CARD(card));
  
  g_object_set(card, "card-deleted", TRUE, NULL);
  g_object_get(card, "card-id", &id, NULL);

  abook->priv->deleted++;  
  abook->priv->trash = g_list_append(abook->priv->trash, GINT_TO_POINTER(id));

  g_signal_emit_by_name(abook, "card_deleted", card, G_TYPE_POINTER);
  g_signal_emit_by_name(abook, "addressbook_changed", NULL, G_TYPE_NONE);
}


/**
 * r_abook_recovery_card
 * @abook: a #RAbook
 * @card: an #RCard object
 *
 * unmark the card previously marked as deleted. 
 * The "card_recovered" and "addressbook_changed" signals are emitted
 *
 * Returns: %TRUE if card is successfully recovered, %FALSE otherwise
 */
gboolean 
r_abook_recovery_card (RAbook* abook, RCard* card)
{
  g_return_val_if_fail(IS_R_ABOOK(abook), FALSE);  
  g_return_val_if_fail(IS_R_CARD(card), FALSE);  
  

  g_object_set(card, "card-deleted", FALSE, NULL);
  abook->priv->deleted--;
  
  g_signal_emit_by_name(abook, "card_recovered", card, G_TYPE_POINTER); 
  g_signal_emit_by_name(abook, "addressbook_changed", NULL, G_TYPE_NONE);
  
  return TRUE;
}



/**
 * r_abook_replace_card
 * @abook: a #RAbook
 * @old: an #RCard object
 * @new: an #RCard object
 * 
 * replace the old card with the new one
 * The "card_replaced" and "addressbook_changed" signals are emitted
 */
void   
r_abook_replace_card (RAbook* abook, RCard* old, RCard* new)
{
  gint pos;
  GList* node = NULL;
  gpointer card = NULL;

  g_return_if_fail(IS_R_ABOOK(abook));
      
  pos  = g_list_index(abook->priv->cards, old);
  node = g_list_nth(abook->priv->cards, pos);
  if (node)
    {
      abook->priv->cards = g_list_remove_link(abook->priv->cards, node);
      
      card = node->data;
      r_card_free(R_CARD(card));
      
      g_list_free_1(node);

      abook->priv->cards = g_list_insert(abook->priv->cards, new, pos);
      
      g_signal_emit_by_name(abook, "addressbook_changed", NULL, G_TYPE_NONE);
      g_signal_emit_by_name(abook, "card_replaced", new, G_TYPE_POINTER);
    }
}



/**
 * r_abook_get_card
 * @abook: a #RAbook
 *
 * Get the first card in addressbook. 
 *
 * Returns: a pointer to the card or NULL if addressbook is empty
 */
gpointer 
r_abook_get_card (RAbook* abook)
{
  g_return_val_if_fail(IS_R_ABOOK(abook), NULL);

  if (abook->priv->iter)
    return abook->priv->iter->data;

  return NULL;
}



/**
 * r_abook_get_selected_card
 * @abook: a #RAbook
 * 
 * Get the selected card in the addressbook
 *
 * Returns:
 */
gpointer      
r_abook_get_selected_card (RAbook* abook)
{
  g_return_val_if_fail(IS_R_ABOOK(abook), NULL);
 
  return r_abook_get_card_by_id(abook, abook->priv->selected_id);
}


/**
 * r_abook_unselect_cards
 * @abook: a #RAbook
 * 
 * unselect card. Clear previpus selections
 */
void 
r_abook_unselect_cards (RAbook* abook)
{
  g_return_if_fail(IS_R_ABOOK(abook));

  abook->priv->selected_id = 0L;
}


/**
 * r_abook_get_card_by_id
 * @abook: a #RAbook
 * @id: the card id
 * 
 * get the card with "id" id
 *
 * Returns: a pointer to the card or NULL if addressbook doesn't own that card
 */
gpointer      
r_abook_get_card_by_id (RAbook* abook, glong id)
{
  gpointer card = NULL;

  g_return_val_if_fail(IS_R_ABOOK(abook), NULL);  
  g_return_val_if_fail(id > 0L, NULL);
   
  r_abook_reset_book(abook);
  for (card = r_abook_get_card(abook); card; 
       card = r_abook_get_next_card(abook))
    {
      glong card_id;
      
      g_object_get(R_CARD(card), "card-id", &card_id, NULL);
      if (id == card_id)
	{
	  abook->priv->selected_id = id;
	  
	  return card;      
	}
    }  

  return NULL;
}

/**
 * r_abook_get_group_box(RAbook* abook)
 * @abook: a #RAbook
 * 
 * get the default groups set
 * 
 * returns: a #gpointer (caller must cast to #RGroupSet*) or NULL
 */
gpointer 
r_abook_get_groups_box(RAbook* abook)
{
  g_return_val_if_fail(IS_R_ABOOK(abook), NULL);
  
  return (gpointer) abook->priv->box;
}




/**
 * r_abook_reset_book
 * @abook: a #RAbook
 * 
 * reset the private addressbook iterator. The iterator 
 * will point the head of cards's list.
 */
void     
r_abook_reset_book (RAbook* abook)
{
  g_return_if_fail(IS_R_ABOOK(abook));
 
  abook->priv->iter = abook->priv->cards;
}


/**
 * r_abook_get_next_card
 * @abook: a #RAbook
 *
 * get the next card in addressbook
 * 
 * Returns: a pointer to the card or NULL if the cards's end list is reached.
 */
gpointer
r_abook_get_next_card (RAbook* abook)
{
  g_return_val_if_fail(IS_R_ABOOK(abook), NULL);
 
  abook->priv->iter = g_list_next(abook->priv->iter);
  if (abook->priv->iter)
    return abook->priv->iter->data;
  else
    abook->priv->iter = g_list_last(abook->priv->cards);  

  return NULL;
}


/**
 * r_abook_get_prev_card
 * @abook: a #RAbook
 *
 * get the previous card in addressbook
 * 
 * Returns: a pointer to the card or NULL if the cards's head list is reached.
 */
gpointer 
r_abook_get_prev_card (RAbook* abook)
{
  g_return_val_if_fail(IS_R_ABOOK(abook), NULL);
 
  abook->priv->iter = g_list_previous(abook->priv->iter);
  if (abook->priv->iter)
    return abook->priv->iter->data;
  else 
    abook->priv->iter = abook->priv->cards;

  return NULL;
}


/**
 * r_abook_foreach_card
 * @abook: a #RAbook
 * @func: the #RFunc that will be called for each card in addressbook
 * @data: user's data to pass to func
 *
 * Call the function func() for each card in the #RAbook. 
 */
void 
r_abook_foreach_card (RAbook* abook, RFunc func, gpointer data)
{
  gpointer card;
     
  r_abook_reset_book(abook);
  card = r_abook_get_card(abook);
  for (; card; card = r_abook_get_next_card(abook))
    func(card, data);    
}


/**
 * r_abook_find_cards_by_group
 * @abook: a #RAbook
 * @group_name: the name of the group
 *
 * Find all cards that belongs to the given group. 
 * Found cards's ID has inserted in a GList. Caller must free returned GList
 * 
 * returns: a GList* or %NULL if no cards belong to group
 */
GList*   
r_abook_find_cards_by_group  (RAbook* abook, const gchar* group_name)
{
  gpointer data;
  GList* found = NULL;

  g_return_val_if_fail(IS_R_ABOOK(abook), NULL);
  g_return_val_if_fail(group_name != NULL, NULL);

  r_abook_reset_book(abook);
  data = r_abook_get_card(abook);
  for (; data; data = r_abook_get_next_card(abook))
    {
      glong id;

      if (!r_card_is_deleted(R_CARD(data)) && 
	  (r_card_belong_to_group(R_CARD(data), group_name) ||
	   (g_ascii_strcasecmp(group_name, "all groups") == 0)))
	{
	  g_object_get(R_CARD(data), "card-id", &id, NULL);
	  
	  found = g_list_append(found, GINT_TO_POINTER(id));
	}
    }
  
  return found;
}


/**
 * r_abook_find_cards_by_type
 * @abook: a #RAbook
 * @type: the type
 *
 * Find all cards that belongs to the given type. 
 * Found cards's ID has inserted in a GList. Caller must free returned GList
 * 
 * returns: a GList* or %NULL if no cards belong to the given type
 */
GList*   
r_abook_find_cards_by_type (RAbook* abook, const gchar* type)
{
  gpointer data;
  GList* found = NULL;

  g_return_val_if_fail(IS_R_ABOOK(abook), NULL);
  g_return_val_if_fail(type != NULL, NULL);  

  r_abook_reset_book(abook);
  data = r_abook_get_card(abook);
  for (; data; data = r_abook_get_next_card(abook))
    {
      gchar* card_type = NULL;
      gboolean deleted;
      glong id;
       
      g_object_get(R_CARD(data), "card-id", &id, 
		   "card-deleted", &deleted,
		   "card-type", &card_type, NULL);
      
      if (!card_type)
	{
	  gchar* name;
	  
	  g_object_get(R_CARD(data), "card-name", &name, NULL);
	  
	  g_log(G_LOG_DOMAIN, G_LOG_LEVEL_WARNING, 
		"This card (%s) has a wrong type.", name); 

	  continue;
	}

      if (!deleted)
	{
	  if (g_ascii_strcasecmp(type, "all") == 0)
	    found = g_list_append(found, GINT_TO_POINTER(id));
	  else
	    if (g_ascii_strcasecmp(type, card_type) == 0)
	      found = g_list_append(found, GINT_TO_POINTER(id));	    
	}
    }
  
  return found;
}



/**
 * r_abook_find_cards_by_rate
 * @abook: a #RAbook
 * @rate: an #RRate
 *
 * Find all cards that have the given rate
 * Found cards's ID has inserted in a GList. Caller must free returned GList
 * 
 * returns: a GList* or %NULL if no cards have the requested rate
 */
GList*   
r_abook_find_cards_by_rate (RAbook* abook, RRate rate)
{
  gpointer data;
  GList* found = NULL;

  g_return_val_if_fail(IS_R_ABOOK(abook), NULL);

  r_abook_reset_book(abook);
  data = r_abook_get_card(abook);
  for (; data; data = r_abook_get_next_card(abook))
    {
      glong id, card_rate;
      gboolean deleted;
       
      g_object_get(R_CARD(data), "card-id", &id, 
		   "card-deleted", &deleted,
		   "card-rate", &card_rate, NULL);

      if (!deleted && (card_rate == rate))
	found = g_list_append(found, GINT_TO_POINTER(id));
    }

  return found;
}


/**
 * r_abook_find_cards_by_genre
 * @abook: a #RAbook
 * @genre: a string
 *
 * Find all cards with the given genre ("male" or "female")
 * Found cards's ID has inserted in a GList. Caller must free returned GList
 * 
 * returns: a GList* or %NULL if no cards have the requested genre
 */
GList* 
r_abook_find_cards_by_genre (RAbook* abook, const gchar* genre)
{
  gpointer data;
  GList* found = NULL;

  g_return_val_if_fail(IS_R_ABOOK(abook), NULL);
  g_return_val_if_fail(genre != NULL, NULL);

  r_abook_reset_book(abook);
  data = r_abook_get_card(abook);
  for (; data; data = r_abook_get_next_card(abook))
    {
       gchar* type = NULL;
       gboolean deleted;
       glong id;
       
       g_object_get(R_CARD(data), "card-id", &id, "card-deleted", &deleted,
		    "card-type", &type, NULL);
       
       if (!deleted && (g_ascii_strcasecmp(type, "personal") == 0))
	 {
	   if (r_personal_card_belongs_to_genre(R_PERSONAL_CARD(data), genre))
	     found = g_list_append(found, GINT_TO_POINTER(id));
	 }   	    
    }
  
  return found;  
}


/**
 * r_abook_search
 * @abook: a #RAbook
 * @str: an string
 *
 * Search the given string in all cards's fields
 * 
 * returns: a GList* of cards's id or %NULL if no cards 
 * have the requested string
 */
GList*
r_abook_search (RAbook* abook, const gchar* str)
{
  gpointer data;
  GList* found = NULL;
  
  g_return_val_if_fail(IS_R_ABOOK(abook), NULL);
  g_return_val_if_fail(str != NULL, NULL);

  r_abook_reset_book(abook);
  data = r_abook_get_card(abook);
  for (; data; data = r_abook_get_next_card(abook))
    {
      glong id;
      
      g_object_get(R_CARD(data), "card-id", &id, NULL);
      
      if (r_card_search(R_CARD(data), str))
	found = g_list_append(found, GINT_TO_POINTER(id));	  
    }
  
  return found;
}


/**
 * r_abook_search_date
 * @abook: a #RAbook
 * @search_date: a time_t date
 * type: a #SearchType
 *
 * Search the given date 
 * 
 * returns: a GList* of cards's id or %NULL if no cards have the date
 */
GList*   
r_abook_search_date (RAbook* abook, gint search_date, SearchType type)
{
  gpointer data;
  GList* found = NULL;
  
  g_return_val_if_fail(IS_R_ABOOK(abook), NULL);
  g_return_val_if_fail(search_date > 0, NULL);

  r_abook_reset_book(abook);
  data = r_abook_get_card(abook);
  for (; data; data = r_abook_get_next_card(abook))
    {
      glong id;
      gint creation, changed;
      GDate *date1, *date2;      
      gboolean find = FALSE;
      
      g_object_get(R_CARD(data), "card-id", &id, 
		   "card-created", &creation, "card-changed", &changed, NULL);

      switch(type)
	{
	case SEARCH_ON_CREATION:
	  date1 = g_date_new();
	  date2 = g_date_new();
	  
	  g_date_set_time_t(date1, creation);
	  g_date_set_time_t(date2, search_date);

	  find = ((g_date_get_day(date1)   == g_date_get_day(date2))   &&
		  (g_date_get_month(date1) == g_date_get_month(date2)) &&
		  (g_date_get_year(date1)  == g_date_get_year(date2)));

	  g_date_free (date1);
	  g_date_free (date2);
	  break;
	  
	case SEARCH_ON_CHANGE:
	  date1 = g_date_new();
	  date2 = g_date_new();
	  
	  g_date_set_time_t(date1, changed);
	  g_date_set_time_t(date2, search_date);
	  
	  find = ((g_date_get_day(date1)   == g_date_get_day(date2))   &&
		  (g_date_get_month(date1) == g_date_get_month(date2)) &&
		  (g_date_get_year(date1)  == g_date_get_year(date2)));

	  g_date_free (date1);
	  g_date_free (date2);
	  break;
	  
	case SEARCH_CREATION_BEFORE:
	  find = (creation < search_date);
	  break;

	case SEARCH_CREATION_AFTER:
	  find = (creation > search_date);
	  break;
	  
	case SEARCH_CHANGE_BEFORE:
	  find = (changed < search_date);
	  break;

	case SEARCH_CHANGE_AFTER:
	  find = (changed > search_date);
	  break;

	default:
	  break;
	}
      
      if (find)
	found = g_list_append(found, GINT_TO_POINTER(id));	  
    }
  
  return found;
}


/**
 * r_abook_search_between
 * @abook: a #RAbook
 * @first: a time_t date
 * @second: a time_t date
 * type: a #SearchType
 *
 * Search the for a date between the first and the second date 
 * 
 * returns: a GList* of cards's id or %NULL if no cards have the date
 */
GList*   
r_abook_search_between (RAbook* abook, gint first, gint second,
			SearchType type)
{
  gpointer data;
  GList* found = NULL;
  
  g_return_val_if_fail(IS_R_ABOOK(abook), NULL);
  g_return_val_if_fail(first > 0, NULL);
  g_return_val_if_fail(second > 0, NULL);
 
  r_abook_reset_book(abook);
  data = r_abook_get_card(abook);
  for (; data; data = r_abook_get_next_card(abook))
    {
      glong id;
      gint creation, changed;
      gboolean find = FALSE;
      
      g_object_get(R_CARD(data), "card-id", &id, 
		   "card-created", &creation, "card-changed", &changed, NULL);

      switch(type)
	{
	case SEARCH_CREATION_BETWEEN:
	  find = ((creation >= first) && (creation <= second));
	  break;
	  
	case SEARCH_CHANGE_BETWEEN:
	  find = ((changed >= first) && (changed <= second)); 
	  break;

	default:
	  break;
	}
      
      if (find)
	found = g_list_append(found, GINT_TO_POINTER(id));	  
    }
  
  return found;
}

