/*
*  RAL -- Rubrica Addressbook Library
*  file: contact.c
*  
*  Copyright (C) Nicola Fragale <nicolafragale@libero.it>
*
*  This program is free software; you can redistribute it and/or modify
*  it under the terms of the GNU General Public License as published by
*  the Free Software Foundation; either version 3 of the License
*
*  This program is distributed in the hope that it will be useful,
*  but WITHOUT ANY WARRANTY; without even the implied warranty of
*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*  GNU General Public License for more details.
*
*  You should have received a copy of the GNU General Public License
*  along with this program; if not, write to the Free Software
*  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#include <stdlib.h>
#include <glib.h>
#include <glib-object.h>

#include "contact.h"
#include "date.h"
#include "utils.h"


enum {
  CONTACT_FIRST = 1,
  CONTACT_MIDDLE,
  CONTACT_LAST,
  CONTACT_NICK,
  CONTACT_PROFESSION,
  CONTACT_PREFIX,
  CONTACT_TITLE,
  CONTACT_GENRE,
  CONTACT_PHOTO
};


struct _RContactPrivate {
  gchar*   first;          /* first name  */
  gchar*   middle;         /* middle name */
  gchar*   last;           /* last name   */
  gchar*   nick;           /* nick name   */

  gchar*   prof;           /* profession  */
  gchar*   prefix;         /* prefix  (egr. sir. jr. I, II, ...)     */ 
  gchar*   title;          /* title   (sig. sig.ra, sig.na dott, ... */
  
  gchar*   genre;          /* male, female */

  RDate*   birthday;       /* */

  gchar*   photo;          /* contact's photo (uri) */

  gboolean dispose_has_run;
};


#define R_CONTACT_GET_PRIVATE(o)     (G_TYPE_INSTANCE_GET_PRIVATE((o),  \
			           R_CONTACT_TYPE, RContactPrivate))


static void r_contact_class_init (RContactClass* klass);
static void r_contact_init       (RContact* obj);

static void r_contact_dispose    (RContact* obj);
static void r_contact_finalize   (RContact* obj);

static void r_contact_set_property (GObject* obj, guint property_id, 
				    const GValue* value, GParamSpec* spec);
static void r_contact_get_property (GObject* obj, guint property_id,
				    GValue* value, GParamSpec* spec);


GType
r_contact_get_type()
{
  static GType contact_type = 0;
  
  if (!contact_type)
    {
      static const GTypeInfo contact_info =
	{
	  sizeof(RContactClass),
	  NULL,
	  NULL,
	  (GClassInitFunc) r_contact_class_init,
	  NULL,
	  NULL,
	  sizeof(RContact),
	  0,
	  (GInstanceInitFunc) r_contact_init
	};

      contact_type = g_type_register_static (G_TYPE_OBJECT, "RContact",
					     &contact_info, 0);
    }
  
  return contact_type;
}


static void
r_contact_class_init(RContactClass* klass)
{
  GObjectClass *class;
  GParamSpec* pspec;
  
  class  = G_OBJECT_CLASS (klass);

  class->dispose  = (GObjectFinalizeFunc) r_contact_dispose;
  class->finalize = (GObjectFinalizeFunc) r_contact_finalize;

  class->set_property = r_contact_set_property;
  class->get_property = r_contact_get_property;

  g_type_class_add_private (klass, sizeof(RContactPrivate));

  /**
   * RContact:first-name:
   *
   *  contact's fist name
   */
  pspec = g_param_spec_string("first-name", 
			      "first name",
			      "contact's given name",
			      NULL, 
			      G_PARAM_READWRITE);
  g_object_class_install_property (class, CONTACT_FIRST, pspec);

  /**
   * RContact:middle-name:
   *
   *  contact's middle name
   */
  pspec = g_param_spec_string("middle-name", 
			      "middle name",
			      "contact's middle name",
			      NULL, 
			      G_PARAM_READWRITE);
  g_object_class_install_property (class, CONTACT_MIDDLE, pspec);

  /**
   * RContact:last-name:
   *
   *  contact's last name
   */
  pspec = g_param_spec_string("last-name", 
			      "last name",
			      "contact's last name",
			      NULL, 
			      G_PARAM_READWRITE);
  g_object_class_install_property (class, CONTACT_LAST, pspec);

  /**
   * RContact:nick-name:
   *
   *  contact's nick name
   */
  pspec = g_param_spec_string("nick-name", 
			      "nick name",
			      "contact's nick name",
			      NULL, 
			      G_PARAM_READWRITE);
  g_object_class_install_property (class, CONTACT_NICK, pspec);

  /**
   * RContact:profession:
   *
   * contact's profession
   */
  pspec = g_param_spec_string("profession", 
			      "profession",
			      "contact's profession",
			      NULL, 
			      G_PARAM_READWRITE);
  g_object_class_install_property (class, CONTACT_PROFESSION, pspec);


  /**
   * RContact:prefix:
   *
   * contact's prefix
   */
  pspec = g_param_spec_string("prefix", 
			      "prefix",
			      "contact's prefix",
			      NULL, 
			      G_PARAM_READWRITE);
  g_object_class_install_property (class, CONTACT_PREFIX, pspec);

  /**
   * RContact:title:
   *
   * contact's title
   */
  pspec = g_param_spec_string("title", 
			      "title",
			      "contact's title",
			      NULL, 
			      G_PARAM_READWRITE);
  g_object_class_install_property (class, CONTACT_TITLE, pspec);


  /**
   * RContact:genre:
   *
   * contact's genre
   */
  pspec = g_param_spec_string("genre", 
			      "genre",
			      "contact's genre",
			      NULL, 
			      G_PARAM_READWRITE);
  g_object_class_install_property (class, CONTACT_GENRE, pspec);


  /**
   * RContact:photo:
   *
   * contact's photo
   */
  pspec = g_param_spec_string("photo", 
			      "photo",
			      "contact's photo",
			      NULL, 
			      G_PARAM_READWRITE);
  g_object_class_install_property (class, CONTACT_PHOTO, pspec);  
}


static void
r_contact_init(RContact* self)
{
  self->priv = R_CONTACT_GET_PRIVATE(self);

  self->priv->dispose_has_run = FALSE;
  self->priv->first           = NULL;
  self->priv->middle          = NULL;
  self->priv->last            = NULL;
  self->priv->nick            = NULL;
  self->priv->prof            = NULL;
  self->priv->prefix          = NULL;
  self->priv->title           = NULL;
  self->priv->genre           = NULL;
  self->priv->photo           = NULL;
  self->priv->birthday        = r_date_new();
}



static void 
r_contact_dispose (RContact* self)
{
  RContactPrivate* priv;

  g_return_if_fail(IS_R_CONTACT(self));
  
  priv = R_CONTACT_GET_PRIVATE(self);
  if (priv->dispose_has_run)
    return;

  priv->dispose_has_run = TRUE;
}


static void 
r_contact_finalize (RContact* self)
{
  RContactPrivate* priv;

  g_return_if_fail(IS_R_CONTACT(self));

  priv = R_CONTACT_GET_PRIVATE(self);
  r_utils_free_string(priv->first); 
  r_utils_free_string(priv->middle);
  r_utils_free_string(priv->last);
  r_utils_free_string(priv->nick);
  r_utils_free_string(priv->prof);
  r_utils_free_string(priv->prefix);
  r_utils_free_string(priv->title);
  r_utils_free_string(priv->genre);
  r_utils_free_string(priv->photo);

  r_date_free(priv->birthday);
}


static void 
r_contact_set_property (GObject* obj, guint property_id,
			const GValue* value, GParamSpec* spec)
{
  RContact* self = (RContact*) obj;
  RContactPrivate* priv = R_CONTACT_GET_PRIVATE(self);
  
  switch (property_id)
    {
   case CONTACT_FIRST:
     g_free(priv->first);
     priv->first = g_value_dup_string(value);
     break;
     
    case CONTACT_MIDDLE:
      g_free(priv->middle);
      priv->middle = g_value_dup_string(value);
      break; 
    
    case CONTACT_LAST:
      g_free(priv->last);
      priv->last = g_value_dup_string(value);
      break;

    case CONTACT_NICK:
      g_free(priv->nick);
      priv->nick = g_value_dup_string(value);
      break; 

    case CONTACT_PROFESSION:
      g_free(priv->prof);
      priv->prof = g_value_dup_string(value);     
      break; 

    case CONTACT_PREFIX:
      g_free(priv->prefix);
      priv->prefix = g_value_dup_string(value);
     break;

    case CONTACT_TITLE:
      g_free(priv->title);
      priv->title = g_value_dup_string(value);
     break;   

    case CONTACT_GENRE:
      g_free(priv->genre);
      priv->genre = g_value_dup_string(value);
     break;  

    case CONTACT_PHOTO:
      g_free(priv->photo);
      priv->photo = g_value_dup_string(value);
      break;    
      
    default:
      break;
    }
}



static void 
r_contact_get_property (GObject* obj, guint property_id,
			GValue* value, GParamSpec* spec)
{
  RContact* self = (RContact*) obj;
  RContactPrivate* priv = R_CONTACT_GET_PRIVATE(self);
  
  switch (property_id)
    {
   case CONTACT_FIRST:
     g_value_set_string(value, priv->first);
      break;
    
    case CONTACT_MIDDLE:
      g_value_set_string(value, priv->middle);
      break; 
    
    case CONTACT_LAST:
      g_value_set_string(value, priv->last);
      break;

    case CONTACT_NICK:
      g_value_set_string(value, priv->nick);
    break; 

    case CONTACT_PROFESSION:
      g_value_set_string(value, priv->prof);
     break; 

    case CONTACT_PREFIX:
      g_value_set_string(value, priv->prefix);
      break;

    case CONTACT_TITLE:
      g_value_set_string(value, priv->title);
      break;   

    case CONTACT_GENRE:
      g_value_set_string(value, priv->genre);
      break;   

    case CONTACT_PHOTO:
      g_value_set_string(value, priv->photo);
      break;    
       
    default:
      break;
    }
}


/**
 * r_contact_new
 *
 * create a new #RContact
 *
 * Returns: a #RContact*
 */
RContact*
r_contact_new(void)
{
  RContact* contact;

  contact = g_object_new(r_contact_get_type(), NULL);

  return contact;
}


/**
 * r_contact_free
 * @contact: a #RContact
 *
 * free memory owned by #RContact
 */
void
r_contact_free(RContact* contact)
{
  g_return_if_fail(IS_R_CONTACT(contact));

  g_object_unref(contact);   
}



RDate* 
r_contact_get_birthday (RContact* contact)
{
  g_return_val_if_fail(IS_R_CONTACT(contact), FALSE);
  
  return contact->priv->birthday;
}


/**
 * r_contact_set_birthday
 * @contact: a #RContact
 * @day: contact's birth day
 * @month: contact's birth month
 * @year: contact's birth year
 *
 * set contact's birthday
 */
void 
r_contact_set_birthday (RContact* contact, gint day, gint month, gint year)
{
  g_return_if_fail(IS_R_CONTACT(contact));

  g_object_set(contact->priv->birthday, "day", day, 
	       "month", month, "year", year, NULL);
}


/**
 * r_contact_get_birth_day
 * @contact: a #RContact
 *
 * get contact's day of birth. Caller must free the returned value
 *
 * returns: a gchar*
*/
gchar* 
r_contact_get_birth_day (RContact* contact)
{
  g_return_val_if_fail(IS_R_CONTACT(contact), NULL);
  
  return r_date_get_day(contact->priv->birthday);
}
 

/**
 * r_contact_get_bith_month
 * @contact: a #RContact
 *
 * get contact's month of birth. Caller must free the returned value
 *
 * returns: a gchar*
 */ 
gchar* 
r_contact_get_birth_month (RContact* contact)
{
  g_return_val_if_fail(IS_R_CONTACT(contact), NULL);

  return r_date_get_month(contact->priv->birthday);
}


/**
 * r_contact_get_birth_year
 * @contact: a #RContact
 *
 * get contact's year of birth. Caller must free the returned value
 *
 * returns: a gchar*
 */
gchar* 
r_contact_get_birth_year (RContact* contact)
{
  g_return_val_if_fail(IS_R_CONTACT(contact), NULL);
  
  return r_date_get_year(contact->priv->birthday);
}


/**
 * r_contact_get_birthdate
 * @contact: a #RContact
 *
 * get contact's birthday, or "unknown" if birthday 
 * wasn't setted. Caller must free the returned value
 *
 * returns: a gchar*
 */
gchar* 
r_contact_get_birth_date (RContact* contact)
{
  g_return_val_if_fail(IS_R_CONTACT(contact), "unknown");

  return r_date_get_human_date(contact->priv->birthday); 
}


gboolean  
r_contact_search (RContact* contact, const gchar* str)
{
  RContactPrivate* priv;

  g_return_val_if_fail(IS_R_CONTACT(contact), FALSE);
  g_return_val_if_fail(str != NULL, FALSE);

  priv = R_CONTACT_GET_PRIVATE(contact);

  if (priv->first && g_strrstr(priv->first, str))
    return TRUE; 

  if (priv->middle && g_strrstr(priv->middle, str))
    return TRUE; 

  if (priv->last && g_strrstr(priv->last, str))
    return TRUE; 

  if (priv->nick && g_strrstr(priv->nick, str))
    return TRUE; 

  if (priv->prof && g_strrstr(priv->prof, str))
    return TRUE; 

  if (priv->prefix && g_strrstr(priv->prefix, str))
    return TRUE; 

  if (priv->title && g_strrstr(priv->title, str))
    return TRUE;   

  return FALSE;
}



/**
 * r_contact_copy
 * @contact: a #RContact
 *
 * make a copy of given object
 *
 * returns: a new #RContact object
 */
RContact* 
r_contact_copy (RContact* contact)
{
  RContact* new;
  gchar *first, *middle, *last, *nick;
  gchar *prof, *prefix, *title, *genre, *photo;
  
  g_return_val_if_fail(IS_R_CONTACT(contact), NULL);

  new = r_contact_new();
  
  g_object_get(G_OBJECT(contact), 
	       "first-name", &first, "middle-name", &middle, 
	       "last-name", &last, "nick-name", &nick, "profession", &prof, 
	       "prefix", &prefix, "title", &title, "genre", &genre,
	       "photo", &photo, NULL);
  
  g_object_set(G_OBJECT(new), 
	       "first-name", first, "middle-name", middle, "last-name", last, 
	       "nick-name", nick, "profession",  prof, "prefix", prefix, 
	       "title", title, "genre", genre, "photo", photo, NULL);  

  return new;
}



/**
 * r_contact_have_data
 * @contact: a #RContact
 * 
 * check if contact has some property setted
 *
 * returns: %TRUE if contact has property setted, %FALSE otherwise
 */
gboolean 
r_contact_have_data (RContact* contact)
{
  RContactPrivate* priv;
  
  g_return_val_if_fail(IS_R_CONTACT(contact), FALSE);
  
  priv = R_CONTACT_GET_PRIVATE(contact);
  
  if (priv->first || priv->middle || priv->last  || priv->nick ||
      priv->prof  || priv->prefix || priv->title || priv->photo)
    return TRUE;

  return FALSE;
}


/**
 * r_contact_check
 * @contact: a #RContact
 * @property: an #RContact's property
 * @value: the property's value (if set)
 *
 * check if the given property is set. If property value is not
 * important can give %NULL as third parameter
 *
 * returns: %FALSE if the property is %NULL, otherwise it return %TRUE and
 * the content of the property is copied into value
 **/
gboolean  
r_contact_check (RContact* contact, const gchar* property, gchar** value)
{
  gchar* tmp;

  g_return_val_if_fail(IS_R_CONTACT(contact), FALSE);
  
  g_object_get(contact, property, &tmp, NULL);
  
  if (tmp)
    {
      if (value)
	*value = tmp;

      return TRUE;
    }
  
  return FALSE;
}
