/*
*  RAL -- Rubrica Addressbook Library
*  file: ref.c
*  
*  Copyright (C) Nicola Fragale <nicolafragale@libero.it>
*
*  This program is free software; you can redistribute it and/or modify
*  it under the terms of the GNU General Public License as published by
*  the Free Software Foundation; either version 3 of the License
*
*  This program is distributed in the hope that it will be useful,
*  but WITHOUT ANY WARRANTY; without even the implied warranty of
*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*  GNU General Public License for more details.
*
*  You should have received a copy of the GNU General Public License
*  along with this program; if not, write to the Free Software
*  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#include <glib.h>
#include <glib-object.h>

#include "csv_engine.h"


enum { 
  PROP_O,
  R_CSV_ENGINE_ENCODE,
  R_CSV_ENGINE_FIELD_SEPARATORS,
};


/*    signals enumeration 
 */
enum {
  RECORD_DECODED,
  LAST_SIGNAL   
};


struct _RCsvEnginePrivate {
  gchar* encode;

  gboolean dispose_has_run;
};

#define R_CSV_ENGINE_GET_PRIVATE(o)  (G_TYPE_INSTANCE_GET_PRIVATE((o),  \
                                      R_CSV_ENGINE_TYPE, RCsvEnginePrivate))


static guint engine_signals[LAST_SIGNAL] = {0};


static void r_csv_engine_class_init   (RCsvEngineClass* klass);
static void r_csv_engine_init         (RCsvEngine* obj);

static void r_csv_engine_dispose      (RCsvEngine* obj);
static void r_csv_engine_finalize     (RCsvEngine* obj);


static void r_csv_engine_set_property (GObject* obj, guint property_id,
				       const GValue* value, GParamSpec* spec);
static void r_csv_engine_get_property (GObject* obj, guint property_id,
				       GValue* value, GParamSpec* spec);



static void   free_decoded_data (GList* l);
static GList* decode_buffer     (gchar* buffer);


static void
free_decoded_data(GList* l)
{
  if (!l)
    return;

  if (l->next)
    free_decoded_data(l->next);
  
  g_free(l->data);
}



static GList*
decode_buffer(gchar* buffer)
{
  gchar** splits;
  gchar** alias;
  GList* ret = NULL;

  alias = splits = g_strsplit(buffer, ",", -1);

  if (!splits)
    return NULL;

  for(; *alias; alias++)
    ret = g_list_append(ret, g_strdup(*alias));

  g_strfreev(splits);

  return ret;
}


GType
r_csv_engine_get_type()
{
  static GType r_csv_engine_type = 0;
  
  if (!r_csv_engine_type)
    {
      static const GTypeInfo r_csv_engine_info =
	{
	  sizeof(RCsvEngineClass),
	  NULL,
	  NULL,
	  (GClassInitFunc) r_csv_engine_class_init,
	  NULL,
	  NULL,
	  sizeof(RCsvEngine),
	  0,
	  (GInstanceInitFunc) r_csv_engine_init
	};

      r_csv_engine_type = g_type_register_static (G_TYPE_OBJECT, 
						  "RCsvEngine",
						  &r_csv_engine_info, 0);
    }
  
  return r_csv_engine_type;
}


static void
r_csv_engine_class_init(RCsvEngineClass* klass)
{
  GObjectClass *parent_class;
  GParamSpec* pspec;
  
  parent_class  = G_OBJECT_CLASS (klass);

  parent_class->dispose      = (GObjectFinalizeFunc) r_csv_engine_dispose;
  parent_class->finalize     = (GObjectFinalizeFunc) r_csv_engine_finalize;

  parent_class->set_property = r_csv_engine_set_property;
  parent_class->get_property = r_csv_engine_get_property;
  
  g_type_class_add_private(klass, sizeof(RCsvEnginePrivate));

  /* class signals
   */
  engine_signals[RECORD_DECODED] =
    g_signal_new("record-decoded", 
		 R_CSV_ENGINE_TYPE,
		 G_SIGNAL_RUN_LAST,
		 G_STRUCT_OFFSET(RCsvEngineClass, record_decoded),
		 NULL,
		 NULL,
		 g_cclosure_marshal_VOID__POINTER,
		 G_TYPE_NONE,                /* return type */
		 1,                          /* params      */
		 G_TYPE_POINTER);            /* params type: error code */


  /* class property 
   */
  /**
   * RCsvEngine:encode
   *
   * The file's encode (uft8, utf16, ...)
   */
  pspec = g_param_spec_string("encode", 
			      "file encode", 
			      "file's encode",  
			      NULL,
			      G_PARAM_READWRITE);
  g_object_class_install_property(parent_class, R_CSV_ENGINE_ENCODE, pspec);  
}



static void
r_csv_engine_init(RCsvEngine* self)
{
  RCsvEnginePrivate* priv = R_CSV_ENGINE_GET_PRIVATE(self);

  priv->encode = g_strdup("utf8");
  
  priv->dispose_has_run = FALSE;
}

 

static void 
r_csv_engine_set_property (GObject* obj, guint property_id,
			   const GValue* value, GParamSpec* spec)
{
  RCsvEngine* self = (RCsvEngine*) obj;
  RCsvEnginePrivate* priv = R_CSV_ENGINE_GET_PRIVATE(self);
  const gchar* str = NULL;

  switch (property_id) 
    {
    case R_CSV_ENGINE_ENCODE:
      g_free(priv->encode);
      
      str = g_value_get_string(value);
      if (!str)
	priv->encode = g_strdup("utf8");
      else
	priv->encode = g_value_dup_string(value);
      break;
      
    default: 
      break; 
    } 
} 
 

static void 
r_csv_engine_get_property (GObject* obj, guint property_id,
			   GValue* value, GParamSpec* spec)
{
  RCsvEngine* self = (RCsvEngine*) obj;
  RCsvEnginePrivate* priv = R_CSV_ENGINE_GET_PRIVATE(self);
  
  switch (property_id) 
    {
    case R_CSV_ENGINE_ENCODE:
      g_value_set_string(value, priv->encode);
      break;
                
    default:
      break;  
    }  
}


static void 
r_csv_engine_dispose (RCsvEngine* self)
{
  RCsvEnginePrivate* priv;

  g_return_if_fail(IS_R_CSV_ENGINE(self));
  
  priv = R_CSV_ENGINE_GET_PRIVATE(self);
  if (priv->dispose_has_run)
    return;

  priv->dispose_has_run = TRUE;
}


static void 
r_csv_engine_finalize (RCsvEngine* self)
{
  RCsvEnginePrivate* priv;

  g_return_if_fail(IS_R_CSV_ENGINE(self));
  
  priv = R_CSV_ENGINE_GET_PRIVATE(self);
  g_free(priv->encode);
}



/*  Public
*/
RCsvEngine*
r_csv_engine_new(void)
{
  RCsvEngine* engine;

  engine = g_object_new(r_csv_engine_get_type(), NULL);
  
  return engine;  
}


void
r_csv_engine_free(RCsvEngine* engine)
{
  g_return_if_fail(IS_R_CSV_ENGINE(engine));

  g_object_unref(engine);
}



gboolean 
r_csv_engine_read_file(RCsvEngine* engine, const gchar* filename)
{
  RCsvEnginePrivate* priv;
  GIOChannel* channel = NULL;
  GIOStatus status;
  GError* error = NULL;
  GList* row;
  gchar* buffer;

  g_return_val_if_fail(IS_R_CSV_ENGINE(engine), FALSE);
  g_return_val_if_fail(filename != NULL, FALSE);

  priv = R_CSV_ENGINE_GET_PRIVATE(engine);

  // i test sull'esistenza del file a carico del chiamante
  
  channel = g_io_channel_new_file(filename, "r", &error);
  if (!channel)
    g_error("Channel: %s", error->message);
  
  status = g_io_channel_set_encoding(channel, priv->encode, &error);
  if (G_IO_STATUS_ERROR == status)
    g_error("Channel: %s", error->message);
  
  while((status = g_io_channel_read_line(channel, &buffer, NULL, NULL, &error))
	!= G_IO_STATUS_EOF)
    {            
      gint len;
      
      if (buffer)
	{
	  len = g_utf8_strlen(buffer, -1);
	  
	  if (buffer[len-1] == '\n')
	    buffer[len-1] = '\0';
	  
	  row = decode_buffer(buffer);
	  g_signal_emit_by_name(engine, "record_decoded", row, G_TYPE_POINTER);

	  free_decoded_data(row);
	  g_list_free(row);
	  g_free(buffer);
	  buffer = NULL;
	  row = NULL;
	}
    }

  if (status != G_IO_STATUS_EOF)
    {
      g_error("Channel: %s", error->message);
    }
  
  g_free(buffer);
  return TRUE;
}
