/*
*  RAL -- Rubrica Addressbook Library
*  file: date.c
*  
*  Copyright (C) Nicola Fragale <nicolafragale@libero.it>
*
*  This program is free software; you can redistribute it and/or modify
*  it under the terms of the GNU General Public License as published by
*  the Free Software Foundation; either version 3 of the License
*
*  This program is distributed in the hope that it will be useful,
*  but WITHOUT ANY WARRANTY; without even the implied warranty of
*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*  GNU General Public License for more details.
*
*  You should have received a copy of the GNU General Public License
*  along with this program; if not, write to the Free Software
*  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#include <time.h>
#include <stdlib.h>
#include <glib.h>
#include <glib-object.h>
#include <glib/gi18n-lib.h>

#include "date.h"
#include "utils.h"

#define BUFLEN 64

/*   property enumeration
*/
enum {
  DATE_KNOW = 1,
  DATE_DAY,
  DATE_MONTH,
  DATE_YEAR,
  DATE_HUMAN
};

struct _RDatePrivate {
  gboolean know;      /* is date khnown? */

  gint day;
  gint month;
  gint year;

  GDate* gdate;

  gchar* human_date;
  
  gboolean dispose_has_run;
};


#define R_DATE_GET_PRIVATE(o)     (G_TYPE_INSTANCE_GET_PRIVATE((o),  \
			           R_DATE_TYPE, RDatePrivate))

static void r_date_class_init     (RDateClass* klass);
static void r_date_init           (RDate* obj);

static void r_date_dispose        (RDate* obj);
static void r_date_finalize       (RDate* obj);


static void r_date_set_property   (GObject* obj, guint property_id,
				   GValue* value, GParamSpec* spec);

static void r_date_get_property   (GObject* obj, guint property_id,
				   GValue* value, GParamSpec* spec);

static gboolean date_is_valid     (RDate* date);
static void     update_human_date (RDate* date);


static gboolean 
date_is_valid (RDate* date)
{
  if (date->priv->day && date->priv->month && date->priv->year)
    return TRUE;
  
  return FALSE;
}

/*
  typedef guint8  GDateDay;    day of the month 
  Integer representing a day of the month; between 1 and 31. 
  G_DATE_BAD_DAY represents an invalid day of the month.

  typedef enum
  {
    G_DATE_BAD_MONTH = 0,
    G_DATE_JANUARY   = 1,
    ...
    G_DATE_DECEMBER  = 12
  } GDateMonth;   
  

  typedef enum
    {
      G_DATE_BAD_WEEKDAY  = 0,
      G_DATE_MONDAY       = 1,
      ...
      G_DATE_SUNDAY       = 7
    } GDateWeekday;

  #define G_DATE_BAD_DAY    0U
*/

static void 
update_human_date(RDate* date)
{
  gchar buffer[BUFLEN];
  RDatePrivate* priv;

  g_return_if_fail(IS_R_DATE(date));

  priv = R_DATE_GET_PRIVATE(date);
  
  if (date_is_valid(date))
    {
      g_date_set_dmy(priv->gdate, priv->day, priv->month, priv->year);
      priv->know = TRUE;
      
      if(g_date_strftime(buffer, BUFLEN, "%Ex", priv->gdate))
	{
	  g_free(priv->human_date);
	  
	  priv->human_date = g_strdup(buffer);
	}
    }
  else
    priv->know = FALSE;  
}



GType
r_date_get_type()
{
  static GType r_date_type = 0;
  
  if (!r_date_type)
    {
      static const GTypeInfo r_date_info =
	{
	  sizeof(RDateClass),
	  NULL,
	  NULL,
	  (GClassInitFunc) r_date_class_init,
	  NULL,
	  NULL,
	  sizeof(RDate),
	  0,
	  (GInstanceInitFunc) r_date_init
	};
      
      r_date_type = g_type_register_static (G_TYPE_OBJECT, "RDate", 
					    &r_date_info, 0);
    }
  
  return r_date_type;
}


static void
r_date_class_init(RDateClass* klass)
{
  GObjectClass *class;
  GParamSpec* pspec;
  
  class  = G_OBJECT_CLASS (klass);

  class->dispose  = (GObjectFinalizeFunc) r_date_dispose;
  class->finalize = (GObjectFinalizeFunc) r_date_finalize;
  
  class->set_property = (gpointer) r_date_set_property;
  class->get_property = (gpointer) r_date_get_property;

  g_type_class_add_private (klass, sizeof(RDatePrivate));

  /**
   * RDate:know:
   *
   * 
   */
  pspec = g_param_spec_boolean("know", 
			       "know", 
			       "is date known", 
			       FALSE,
			       G_PARAM_READWRITE);
  g_object_class_install_property (class, DATE_KNOW, pspec);

  /**
   * RDate:day:
   *
   * day
   */
  pspec = g_param_spec_int("day", 
			   "day", 
			   "the day", 
			   0, // 0 means error
			   31,
			   0,
			   G_PARAM_READWRITE);
  g_object_class_install_property (class, DATE_DAY, pspec);

  /**
   * RDate:month:
   *
   * month
   */
  pspec = g_param_spec_int("month", 
			   "month", 
			   "the month", 
			   0, // 0 means error
			   12,
			   0,
			   G_PARAM_READWRITE);
  g_object_class_install_property (class, DATE_MONTH, pspec);

  /**
   * RDate:year:
   *
   * year
   */
  pspec = g_param_spec_int("year", 
			   "year", 
			   "the year", 
			   0,
			   10000,
			   0,
			   G_PARAM_READWRITE);
  g_object_class_install_property (class, DATE_YEAR, pspec);

  /**
   * RDate:human-date:
   *
   * date in a human readable format
   */
  pspec = g_param_spec_string("human-date",
			      "human readable",
			      "data displayed in a human readable format",
			      NULL, 
			      G_PARAM_READABLE);
  g_object_class_install_property (class, DATE_HUMAN, pspec);
}


static void
r_date_init(RDate* self)
{
  self->priv = R_DATE_GET_PRIVATE(self);

  self->priv->know  = FALSE;
  self->priv->day   = 0;
  self->priv->month = 0;
  self->priv->year  = 0;
  self->priv->gdate = g_date_new();

  self->priv->human_date = g_strdup(_("unknown"));
  
  self->priv->dispose_has_run = FALSE;
}


static void 
r_date_dispose (RDate* self)
{
  g_return_if_fail(IS_R_DATE(self));

  if (self->priv->dispose_has_run)
    return;

  self->priv->dispose_has_run = TRUE;
}


static void 
r_date_finalize (RDate* self)
{
  g_return_if_fail(IS_R_DATE(self));

  r_utils_free_string(self->priv->human_date);
}


static void 
r_date_set_property (GObject* obj, guint property_id,
		     GValue* value,  GParamSpec* spec)
{
  RDate* self = (RDate *) obj;
  RDatePrivate* priv = R_DATE_GET_PRIVATE(self); 

  switch(property_id)
    {
    case DATE_KNOW:
      priv->know = g_value_get_boolean(value);
      break;
      
    case DATE_DAY:
      priv->day = g_value_get_int(value);
      update_human_date(self);
     break;
      
    case DATE_MONTH:
      priv->month = g_value_get_int(value);
      update_human_date(self);
      break;
      
    case DATE_YEAR:
      priv->year = g_value_get_int(value);
      update_human_date(self);
      break;
      
    default:
      break;
    }
  
  return;
}


static void 
r_date_get_property (GObject* obj, guint property_id,
		     GValue* value, GParamSpec* spec)
{
  RDate *self = (RDate *) obj;

  switch(property_id)
    {
    case DATE_KNOW:
      g_value_set_boolean(value, self->priv->know);
      break;
      
    case DATE_DAY:
      g_value_set_int(value, self->priv->day);
      break;
      
    case DATE_MONTH:
      g_value_set_int(value, self->priv->month);
      break;
      
    case DATE_YEAR:
      g_value_set_int(value, self->priv->year);
      break;
      
    case DATE_HUMAN:
      g_value_set_string(value, self->priv->human_date);
      break;
      
    default:
      break;
    }
}



/**
 * r_date_new
 *
 * create a new #RDate object
 *
 * Returns: a #RDate
 **/
RDate*
r_date_new(void)
{
  RDate* date;

  date = g_object_new(r_date_get_type(), NULL);

  return date;
}


/**
 * r_date_free
 * @date: a #RDate
 *
 * free the RDate object
 */
void
r_date_free(RDate* date)
{
  g_return_if_fail(IS_R_DATE(date));

  g_object_unref(date);   
}



/**
 * r_date_get_human_date
 * @date: a #RDate
 *
 * get the date as a locale string. Caller must free the returned value
 *
 * returns: a gchar* or "unknown", if date is not set
 */
gchar*       
r_date_get_human_date (RDate* date)
{
  g_return_val_if_fail(IS_R_DATE(date), g_strdup(_("unknown")));

  return g_strdup(date->priv->human_date);
}


/**
 * r_date_is_date_known
 * @date: a #RDate
 *
 * get the state of know date flag
 *
 * returns: a boolean
 */ 
gboolean 
r_date_is_known (RDate* date)
{
  g_return_val_if_fail(IS_R_DATE(date), FALSE);

  return date->priv->know;
}

gboolean 
r_date_is_valid (RDate* date)
{
  g_return_val_if_fail(IS_R_DATE(date), FALSE);
  
  return date_is_valid(date);
}


/**
 * r_date_copy
 * @date: a #RDate
 * 
 * copy the given #RDate
 * 
 * returns: a new allocated RDate*
 */
RDate* 
r_date_copy (RDate* date)
{
  RDate* new;
  gchar *day, *month, *year;
  gboolean known;

  g_return_val_if_fail(IS_R_DATE(date), NULL);

  new = r_date_new();
  
  g_object_get(date, "known", &known, "day", &day, 
	       "month", &month, "year", &year, NULL);
  
  g_object_set(new, "known", known, "day", day, 
	       "month", month, "year", year, NULL);  
  
  return new;
}


gchar* 
r_date_get_day (RDate* date)
{
  g_return_val_if_fail(IS_R_DATE(date), NULL);

  if(date->priv->day)
    return g_strdup_printf("%d", date->priv->day);

  return g_strdup("BadDay");
}

gchar* 
r_date_get_month (RDate* date)
{
  g_return_val_if_fail(IS_R_DATE(date), NULL);
  
  if(date->priv->month)
    return g_strdup_printf("%d", date->priv->month);
  
  return g_strdup("BadMonth");
}


gchar* 
r_date_get_year (RDate* date)
{
  g_return_val_if_fail(IS_R_DATE(date), NULL);
  
  if (date->priv->year)
    return g_strdup_printf("%d", date->priv->year);
  
  return g_strdup("BadYear");
}



/**
 * r_date_check
 * @date: a #RDate
 * @property: an #RDate's property
 * @value: the property's value (if set)
 *
 * check if the given property is set. 
 *
 * returns: %FALSE if the property is %NULL, otherwise it return %TRUE and
 * the content of the property is copied into value
 **/
gboolean 
r_date_check (RDate* date, const gchar* property, gchar** value)
{
  gchar* tmp;

  g_return_val_if_fail(IS_R_DATE(date), FALSE);
  
  g_object_get(date, property, &tmp, NULL);
  
  if (tmp)
    {
      if (value)
	*value = tmp;

      return TRUE;
    }
  
  return FALSE;  
}
