/*
*  RAL -- Rubrica Addressbook Library
*  file: groups.c
*  
*  Copyright (C) Nicola Fragale <nicolafragale@libero.it>
*
*  This program is free software; you can redistribute it and/or modify
*  it under the terms of the GNU General Public License as published by
*  the Free Software Foundation; either version 3 of the License
*
*  This program is distributed in the hope that it will be useful,
*  but WITHOUT ANY WARRANTY; without even the implied warranty of
*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*  GNU General Public License for more details.
*
*  You should have received a copy of the GNU General Public License
*  along with this program; if not, write to the Free Software
*  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#include <time.h>
#include <glib.h>
#include <glib-object.h>

#include "filter.h"
#include "utils.h"


enum { 
  FILTER_PROP_0,
  FILTER_NAME, 
  FILTER_MIME, 
  FILTER_EXTENSION,
};



struct _RFilterPrivate {
  gchar* name;  
  gchar* mime;
  gchar* extension;   // extension added to file's name

  GList* pattern;     // extensions' list

  GList* iter;

  gboolean dispose_has_run;
};

#define R_FILTER_GET_PRIVATE(o)     (G_TYPE_INSTANCE_GET_PRIVATE((o),  \
			             R_FILTER_TYPE, RFilterPrivate))


static void r_filter_class_init (RFilterClass* klass);
static void r_filter_init       (RFilter* obj);

static void r_filter_dispose    (RFilter* obj);
static void r_filter_finalize   (RFilter* obj);


static void r_filter_set_property (GObject* obj, guint property_id, 
				   const GValue* value, GParamSpec* spec);
static void r_filter_get_property (GObject* obj, guint property_id,
				   GValue* value, GParamSpec* spec);


GType
r_filter_get_type()
{
  static GType filter_type = 0;
  
  if (!filter_type)
    {
      static const GTypeInfo filter_info =
	{
	  sizeof(RFilterClass),
	  NULL,
	  NULL,
	  (GClassInitFunc) r_filter_class_init,
	  NULL,
	  NULL,
	  sizeof(RFilter),
	  0,
	  (GInstanceInitFunc) r_filter_init
	};

      filter_type = g_type_register_static (G_TYPE_OBJECT, "RFilter",
					    &filter_info, 0);
    }
  
  return filter_type;
}


static void
r_filter_class_init(RFilterClass* klass)
{
  GObjectClass *class; 
  GParamSpec* pspec;

  class           = G_OBJECT_CLASS (klass);
  class->dispose  = (GObjectFinalizeFunc) r_filter_dispose;
  class->finalize = (GObjectFinalizeFunc) r_filter_finalize;

  class->set_property = r_filter_set_property;
  class->get_property = r_filter_get_property;

  g_type_class_add_private (klass, sizeof(RFilterPrivate));

  /* class property 
   */
  /**
   * RFilter:filter-name
   *
   * the name of the filter
   */
  pspec = g_param_spec_string("filter-name",  
			      "filter name",  
			      "the name of the filter",   
			      NULL,     
			      G_PARAM_READWRITE);  
  g_object_class_install_property(class, FILTER_NAME, pspec);

  /**
   * RFilter:filter-extension
   *
   * extension added to file' name
   */
  pspec = g_param_spec_string("filter-extension",  
			      "filter extension",  
			      "file's extension",   
			      NULL,     
			      G_PARAM_READWRITE);  
  g_object_class_install_property(class, FILTER_EXTENSION, pspec);  

  /**
   * RFilter:filter-mime
   *
   * the name of the filter
   */
  pspec = g_param_spec_string("filter-mime",  
			      "filter mime",  
			      "file's mime type",   
			      NULL,     
			      G_PARAM_READWRITE);  
  g_object_class_install_property(class, FILTER_MIME, pspec);  
}


static void
r_filter_init(RFilter* self)
{
  self->priv = R_FILTER_GET_PRIVATE(self);

  self->priv->name      = NULL;
  self->priv->mime      = NULL;
  self->priv->extension = NULL;
  self->priv->pattern   = NULL;

  self->priv->dispose_has_run = FALSE;
}



static void 
r_filter_dispose (RFilter* self)
{
  GList* l;
  RFilterPrivate* priv;

  g_return_if_fail(R_IS_FILTER(self));
  
  priv = R_FILTER_GET_PRIVATE(self);
  
  if (priv->dispose_has_run)
    return;
  
  g_free(priv->name);
  g_free(priv->extension);
  g_free(priv->mime);

  for (l = priv->pattern; l; l = l->next)
    {
      if (l->data)
	g_free(l->data);      
    }
  g_list_free(priv->pattern);
  priv->pattern = NULL;
 
  priv->dispose_has_run = TRUE;
}


static void 
r_filter_finalize (RFilter* self)
{
  g_return_if_fail(R_IS_FILTER(self));  
}



static void 
r_filter_set_property (GObject* obj, guint property_id,
		       const GValue* value, GParamSpec* spec)
{
  RFilter* self = (RFilter*) obj;
  RFilterPrivate* priv = R_FILTER_GET_PRIVATE(self);
  
#ifdef ENABLE_DEBUG
  g_log(G_LOG_DOMAIN, G_LOG_LEVEL_DEBUG, "setting filter values");
  g_log(G_LOG_DOMAIN, G_LOG_LEVEL_DEBUG, "property: %d", property_id);  
  g_log(G_LOG_DOMAIN, G_LOG_LEVEL_DEBUG, "value: %s", 
	g_value_get_string(value));
  g_log(G_LOG_DOMAIN, G_LOG_LEVEL_DEBUG, " ");
#endif

  switch (property_id) 
    {
    case FILTER_NAME: 
      g_free(priv->name);
      priv->name = g_value_dup_string(value); 
      break; 
      
    case FILTER_EXTENSION:
      g_free(priv->extension);
      priv->extension = g_value_dup_string(value);
      break;

    case FILTER_MIME:
      g_free(priv->mime); 
      priv->mime = g_value_dup_string(value); 
      break;

    default:
      break;  
    }  
}


static void 
r_filter_get_property (GObject* obj, guint property_id,
		       GValue* value, GParamSpec* spec)
{
  RFilter* self = (RFilter*) obj;
  RFilterPrivate* priv = R_FILTER_GET_PRIVATE(self);
  
  switch (property_id) 
    {
    case FILTER_NAME: 
      g_value_set_string(value, priv->name); 
      break; 
      
    case FILTER_EXTENSION:
      g_value_set_string(value, priv->extension);
      break;

    case FILTER_MIME:
      g_value_set_string(value, priv->mime); 
      break;

    default:
      break;  
    }  
}



/*    *************************************  Public
*/

/**
 * r_filter_new
 *
 * create a new #RFilter
 *
 * returns: a RFilter*
 */
RFilter* 
r_filter_new(void) 
{
  RFilter* filter;

  filter = g_object_new(r_filter_get_type(), NULL);

  return filter;
}


/**
 * r_filter_free
 * @filter: a #RFilter
 *
 * free memory
 */
void
r_filter_free(RFilter* filter)
{
  g_return_if_fail(R_IS_FILTER(filter));

  g_object_unref(filter);   
}


/**
 * r_filter_copy
 * @filter: a #RFilter
 *
 * copy the given #RFilter
 *
 * Returns: a new #RFilter or %NULL if an error occurred
 */
RFilter* 
r_filter_copy (RFilter* filter)
{
  RFilter* new;
  gchar *name, *ext, *mime;
  gchar* pattern;

  g_return_val_if_fail(R_IS_FILTER(filter), NULL);

  new = r_filter_new();
  g_object_get(filter, 
	       "filter-name", &name, 
	       "filter-extension", &ext,
	       "filter-mime", &mime, NULL);
  g_object_set(new, 
	       "filter-name", name, 
	       "filter-extension", ext, 
	       "filter-mime", mime, NULL);
  
  r_filter_reset(filter);
  pattern = r_filter_get_pattern(filter);
  for (; pattern; pattern = r_filter_get_next_pattern(filter))
    r_filter_add_pattern(new, g_strdup(pattern));
  
  return new;
}


/**
 * r_filter_add_pattern
 * @filter: a #RFilter
 * @pattern: a gchar*
 *
 * append the given pattern to the filter's list
 */
void 
r_filter_add_pattern (RFilter* filter, gchar* pattern)
{
  RFilterPrivate* priv;

  g_return_if_fail(R_IS_FILTER(filter));
  g_return_if_fail(pattern != NULL);

  priv = R_FILTER_GET_PRIVATE(filter);

  priv->pattern = g_list_append(priv->pattern, pattern);
  if (!priv->iter)
    priv->iter = priv->pattern;
}



/**
 * r_filter_delete_pattern
 * @filter: a #RFilter
 * @pattern: a gchar*
 *
 * delete the given pattern from the filter's list
 *
 * Returns: %TRUE if pattern is succesfully deleted, %FALSE otherwise
 */
gboolean 
r_filter_delete_pattern(RFilter* filter, gchar* pattern)
{
  RFilterPrivate* priv;

  g_return_val_if_fail(R_IS_FILTER(filter), FALSE);
  g_return_val_if_fail(pattern != NULL, FALSE);
  
  priv = R_FILTER_GET_PRIVATE(filter);

  priv->iter = priv->pattern;
  for (; priv->iter; priv->iter = priv->iter->next)
    {
      if (g_ascii_strcasecmp(priv->iter->data, pattern) == 0)
	{
	  priv->pattern = g_list_remove_link(priv->pattern, priv->iter);
	  g_free(priv->iter->data);
	  g_list_free_1(priv->iter);
	  priv->iter = NULL;
	  
	  return TRUE;
	}
    }
  
  return FALSE;
}

  
/**
 * r_filter_reset
 * @filter: a #RFilter
 * 
 * reset the private filter's iterator
 */
void
r_filter_reset(RFilter* filter)
{
  RFilterPrivate* priv;

  g_return_if_fail(R_IS_FILTER(filter));

  priv = R_FILTER_GET_PRIVATE(filter);
  
  priv->iter = priv->pattern;   
}


/**
 * r_filter_get_pattern
 * @filter: a #RFilter
 *
 * get the first filter's pattern
 *
 * Returns: a gchar* or NULL if filter doesn't have patterns
 */
gchar* 
r_filter_get_pattern(RFilter* filter)
{
  RFilterPrivate* priv;

  g_return_val_if_fail(R_IS_FILTER(filter), NULL);

  priv = R_FILTER_GET_PRIVATE(filter);
  
  if (priv->iter)
    return (gchar*) priv->iter->data;
  
  return NULL;
}


/**
 * r_filter_get_next_pattern
 * @filter: a #RFilter
 *
 * get the next filter's pattern
 *
 * Returns: a gchar* or NULL if filter doesn't have more patterns
 */
gchar* 
r_filter_get_next_pattern(RFilter* filter)
{
  RFilterPrivate* priv;

  g_return_val_if_fail(R_IS_FILTER(filter), NULL);

  priv = R_FILTER_GET_PRIVATE(filter);
  
  priv->iter = g_list_next(priv->iter);
  if (priv->iter)
    return (gchar*) priv->iter->data;
  else
    priv->iter = priv->pattern;
  
  return NULL;
}
