/*
*  RAL -- Rubrica Addressbook Library
*  file: groups.c
*  
*  Copyright (C) Nicola Fragale <nicolafragale@libero.it>
*
*  This program is free software; you can redistribute it and/or modify
*  it under the terms of the GNU General Public License as published by
*  the Free Software Foundation; either version 3 of the License
*
*  This program is distributed in the hope that it will be useful,
*  but WITHOUT ANY WARRANTY; without even the implied warranty of
*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*  GNU General Public License for more details.
*
*  You should have received a copy of the GNU General Public License
*  along with this program; if not, write to the Free Software
*  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#include <time.h>
#include <glib.h>
#include <glib-object.h>

#include "group_box.h"
#include "group.h"

enum {
  GROUP_ADDED,
  GROUP_REMOVED,
  GROUP_MODIFYED,

  LAST_SIGNAL 
};

static guint r_group_box_signals[LAST_SIGNAL] = {0};


struct _RGroupBoxPrivate {
  GList* lst;
  GList* iter;
  GList* find;

  gboolean dispose_has_run;
};


static void r_group_box_class_init (RGroupBoxClass* klass);
static void r_group_box_init       (RGroupBox* obj);

static void r_group_box_dispose    (RGroupBox* obj);
static void r_group_box_finalize   (RGroupBox* obj);

/* Private
*/
static gboolean delete_group(RGroupBox* box, gchar* name);


GType
r_group_box_get_type()
{
  static GType group_box_type = 0;
  
  if (!group_box_type)
    {
      static const GTypeInfo group_box_info =
	{
	  sizeof(RGroupBoxClass),
	  NULL,
	  NULL,
	  (GClassInitFunc) r_group_box_class_init,
	  NULL,
	  NULL,
	  sizeof(RGroupBox),
	  0,
	  (GInstanceInitFunc) r_group_box_init
	};

      group_box_type = g_type_register_static (G_TYPE_OBJECT, 
					       "RGroupBox",
					       &group_box_info, 0);
    }
  
  return group_box_type;
}


static void
r_group_box_class_init(RGroupBoxClass* klass)
{
  GObjectClass *class; 

  class           = G_OBJECT_CLASS (klass);
  class->dispose  = (GObjectFinalizeFunc) r_group_box_dispose;
  class->finalize = (GObjectFinalizeFunc) r_group_box_finalize;


  /* class signals */
  /**
   * RGroupBox::group-added
   * @box: the #RGroupBox that receives the signal
   * @group: the group added to the box
   *
   * signal emitted when a new group is added to the group's box.
   */
  r_group_box_signals[GROUP_ADDED] =
    g_signal_new("group_added", 
		 R_GROUP_BOX_TYPE,
		 G_SIGNAL_RUN_LAST,
		 G_STRUCT_OFFSET(RGroupBoxClass, group_added),
		 NULL,
		 NULL,
		 g_cclosure_marshal_VOID__INT,
		 G_TYPE_NONE,            /* return type */
		 1,                      /* params      */
		 G_TYPE_POINTER);        /* params type */

  /**
   * RGroupbox::group-removed
   * @set: the #RGroupBox that receives the signal
   * @group: the group removed from the set
   *
   * signal emitted when a group is removed from the group's set.
   */
  r_group_box_signals[GROUP_REMOVED] =
    g_signal_new("group_removed", 
		 R_GROUP_BOX_TYPE,
		 G_SIGNAL_RUN_LAST,
		 G_STRUCT_OFFSET(RGroupBoxClass, group_removed),
		 NULL,
		 NULL,
		 g_cclosure_marshal_VOID__INT,
		 G_TYPE_NONE,            /* return type */
		 1,                      /* params      */
		 G_TYPE_INT);            /* params type */

  /**
   * RGroupbox::group-remamed
   * @set: the #RGroupBox that receives the signal
   * @group: the group remamed from the set
   *
   * signal emitted when a group is removed from the group's set.
   */
  r_group_box_signals[GROUP_MODIFYED] =
    g_signal_new("group_modifyed", 
		 R_GROUP_BOX_TYPE,
		 G_SIGNAL_RUN_LAST,
		 G_STRUCT_OFFSET(RGroupBoxClass, group_removed),
		 NULL,
		 NULL,
		 g_cclosure_marshal_VOID__INT,
		 G_TYPE_NONE,            /* return type */
		 1,                      /* params      */
		 G_TYPE_POINTER);       /* params type */
}


static void
r_group_box_init(RGroupBox* self)
{
  self->priv = g_new(RGroupBoxPrivate, 1);

  self->priv->lst  = NULL;
  self->priv->iter = NULL;
  self->priv->dispose_has_run = FALSE;
}



static void 
r_group_box_dispose (RGroupBox* self)
{
  GList* list;

  g_return_if_fail(IS_R_GROUP_BOX(self));
  
  if (self->priv->dispose_has_run)
    return;

  for (list = self->priv->lst; list; list = g_list_next(list))
    r_group_free(R_GROUP(list->data));   
  
  g_list_free(self->priv->lst);

  self->priv->dispose_has_run = TRUE;
}


static void 
r_group_box_finalize (RGroupBox* self)
{
  g_return_if_fail(IS_R_GROUP_BOX(self));
  
  g_free(self->priv);
  self->priv = NULL;
}


static gboolean 
delete_group(RGroupBox* box, gchar* name)
{
  gint id;

  box->priv->iter = box->priv->lst;
  for (; box->priv->iter; box->priv->iter = box->priv->iter->next) 
    { 
      RGroup* group = box->priv->iter->data;
      
      if (r_group_has_name(R_GROUP(group), name)) 
	{ 
	  g_object_get(group, "id", &id, NULL);

	  box->priv->lst = g_list_remove_link(box->priv->lst, box->priv->iter);
	  
	  r_group_free(R_GROUP(box->priv->iter->data));
	  
	  g_list_free_1(box->priv->iter);
	  box->priv->iter = NULL;	  

	  g_signal_emit_by_name(box, "group_removed", id, G_TYPE_INT);

	  return TRUE;
	}
    }
  
  return FALSE;
}


/*    Public
*/

/**
 * r_group_box_
 * @box: a #RGroupBox
 *
 * create a new #RGroupBox
 *
 * returns: a new allocated #RGroupBox*
 */
RGroupBox*
r_group_box_new(void)
{
  RGroupBox* box;

  box = g_object_new(r_group_box_get_type(), NULL);

  return box;
}


/**
 * r_group_box_free
 * @box: a #RGroupBox
 *
 * free memory owned by a #RGroupBox
 */
void
r_group_box_free(RGroupBox* box)
{
  g_return_if_fail(IS_R_GROUP_BOX(box));

  g_object_unref(box);   
}


/**
 * r_group_box_is_empty
 * @box: a #RGroupBox
 * 
 * Test if the given group_box is empty
 *
 * Returns: %TRUE if the box is empty, %FALSE otherwise
 */
gboolean 
r_group_box_is_empty (RGroupBox* box)
{
   g_return_val_if_fail(IS_R_GROUP_BOX(box), TRUE);  
   
   return (box->priv->lst == NULL);
}


/**
 * r_group_box_find
 * @box: a #RGroupBox
 * @group_name: a #RGroup's name
 *
 * find a group by name
 * 
 * returns: a gpointer to the group with the given name, or %NULL if 
 * box haven't the group. 
 */
gpointer  
r_group_box_find (RGroupBox* box, const gchar* group_name) 
{ 
  g_return_val_if_fail(IS_R_GROUP_BOX(box), NULL);  
  g_return_val_if_fail(group_name != NULL, NULL); 

  box->priv->find = box->priv->lst;
  for (; box->priv->find; box->priv->find = box->priv->find->next) 
    {
      RGroup* grp = box->priv->find->data;

      if (r_group_has_name(R_GROUP(grp), group_name)) 
	return (gpointer) grp;
    } 
  
  return NULL;
}


/**
 * r_group_box_find_groups_owned_by
 * @box: a #RGroupBox
 * @owner: a gchar*
 *
 * search for all groups owned by the given "owner"
 *
 * Returns: a #GList of #RGroup or NULL
 */
GList* 
r_group_box_find_groups_owned_by (RGroupBox* box, const gchar* owner)
{
  GList* ret = NULL;
 
  g_return_val_if_fail(IS_R_GROUP_BOX(box), NULL);  
  g_return_val_if_fail(owner != NULL, NULL); 
  
  box->priv->find = box->priv->lst;
  for (; box->priv->find; box->priv->find = box->priv->find->next) 
    {
      RGroup* grp = box->priv->find->data;

      if (r_group_has_owner(R_GROUP(grp), owner)) 
	ret = g_list_append(ret, grp);
    } 
  
  return ret;  
}


/**
 * r_group_box_owns_group
 * @box: a #RGroupBox
 * @group_name: a #RGroup's name
 *
 * test if box owns the given group
 *
 * returns: a gboolean, %TRUE if box owns group, %FALSE otherwise
 */
gboolean 
r_group_box_owns_group (RGroupBox* box, const gchar* group_name)
{
  g_return_val_if_fail(IS_R_GROUP_BOX(box), FALSE); 
  g_return_val_if_fail(group_name != NULL, FALSE); 

  if (r_group_box_find(box, group_name)) 
    return TRUE; 
  
  return FALSE;   
}


/**
 * r_group_box_add_group
 * @box: a #RGroupBox
 * @group: a #RGroup
 *
 * add the group to te box
 *
 * returns: a gboolean. %TRUE if group if added successfully, %FALSE otherwise
 */
gboolean               
r_group_box_add_group (RGroupBox* box, RGroup* group) 
{
  gchar* name; 

  g_return_val_if_fail(IS_R_GROUP_BOX(box), FALSE); 
  g_return_val_if_fail(IS_R_GROUP(group), FALSE); 

  /* test if group is already known 
   */
  g_object_get(group, "group-name", &name, NULL);
  if (r_group_box_owns_group(box, name))
    return TRUE;
  
  /* else add to the box
   */
  box->priv->lst = g_list_append(box->priv->lst, group);

  /* init group iterator */
  if (!box->priv->iter)
    box->priv->iter = box->priv->lst;  

  g_signal_emit_by_name(box, "group_added", group, G_TYPE_POINTER);

  return TRUE;
}


/**
 * r_group_box_delete_group_by_name
 * @box: a #RGroupBox
 * @name: group's name
 *
 * delete a group from box
 *
 * returns: a gboolean. %TRUE if group is deleted, %FALSE otherwise
 */
gboolean     
r_group_box_delete_group_by_name (RGroupBox* box, gchar* name)
{
  g_return_val_if_fail(IS_R_GROUP_BOX(box), FALSE); 
  g_return_val_if_fail(name != NULL, FALSE);
 
  return delete_group(box, name);
}


/**
 * r_group_box_delete_group
 * @box: a #RGroupBox
 * @group: a #RGroup
 *
 * delete a group from box
 *
 * returns: a gboolean. %TRUE if group is deleted, %FALSE otherwise
 */
gboolean     
r_group_box_delete_group (RGroupBox* box, RGroup* group)
{
  gchar* name;

  g_return_val_if_fail(IS_R_GROUP_BOX(box), FALSE); 
  g_return_val_if_fail(IS_R_GROUP(group), FALSE);

  g_object_get(group, "group-name", &name, NULL);
 
  return r_group_box_delete_group_by_name(box, name);
}


/**
 * r_group_box_modify_group_name
 * @box: a #RGroupBox
 * @oldname:
 * @newname:
 *
 * rename a group in the box
 *
 * returns: a gboolean. %TRUE if group is modifyed, %FALSE otherwise
 * If group is modifyed, the "group_modifyed" signal is emitted
 */
gboolean 
r_group_box_modify_group_name (RGroupBox* box, gchar* oldname, gchar* newname)
{
  RGroup* group;
  //  gint id;

  g_return_val_if_fail(IS_R_GROUP_BOX(box), FALSE); 
  g_return_val_if_fail(oldname != NULL, FALSE);
  g_return_val_if_fail(newname != NULL, FALSE);
  
  group = r_group_box_find(box, oldname);
  if (r_group_rename(group, newname))
    {
      g_object_set(group, "group-label", newname, NULL);
      //      g_object_get(group, "id", &id, NULL);

      g_signal_emit_by_name(box, "group_modifyed", group, G_TYPE_POINTER);

      return TRUE;
    }
  
  return FALSE;
}

/**
 * r_group_box_modify_group_pixmap
 * @box: a #RGroupBox
 * @group_name:
 * @newpixmap:
 *
 * modify the group's pixmap
 *
 * returns: a gboolean. %TRUE if group is modifyed, %FALSE otherwise. 
 * If group is modifyed, the "group_modifyed" signal is emitted
 */
gboolean   
r_group_box_modify_group_pixmap (RGroupBox* box, gchar* group_name,
				 gchar* newpixmap)
{
  RGroup* group;
  
  g_return_val_if_fail(IS_R_GROUP_BOX(box), FALSE); 
  g_return_val_if_fail(group_name != NULL, FALSE);
  g_return_val_if_fail(newpixmap != NULL, FALSE);
  
  group = r_group_box_find(box, group_name);
  if (r_group_change_pixmap (group, newpixmap))
    {
      g_signal_emit_by_name(box, "group_modifyed", group, G_TYPE_POINTER);

      return TRUE;
    }
  
  return FALSE;  
}


/**
 * r_group_box_disable_all
 * @box: a #RGroupBox
 *
 * disable all groups in box 
 */
void     
r_group_box_disable_all (RGroupBox* box)
{
  RGroup* group;

  g_return_if_fail(IS_R_GROUP_BOX(box)); 
  
  r_group_box_reset(box);
  group = r_group_box_get_group(box);
  for(; group; group = r_group_box_get_next_group(box))
    g_object_set(group, "enabled", FALSE, NULL);
}


/**
 * r_group_box_enable_group
 * @box: a #RGroupBox
 * @group: a #RGroup
 * 
 * enable the given group
 */
void     
r_group_box_enable_group (RGroupBox* box, RGroup* grp)
{
  gchar* name;
  RGroup* group = NULL;

  g_return_if_fail(IS_R_GROUP_BOX(box)); 
  g_return_if_fail(IS_R_GROUP(grp));

  g_object_get(grp, "group-name", &name, NULL);
  group = r_group_box_find(box, name);

  if (group)
    g_object_set(group, "enabled", TRUE, NULL);
}


/**
 * r_group_box_disable_group
 * @box: a #RGroupBox
 * @group: a #RGroup
 * 
 * disable the given group
 */
void     
r_group_box_disable_group (RGroupBox* box, RGroup* grp)
{
  gchar* name;
  RGroup* group = NULL;

  g_return_if_fail(IS_R_GROUP_BOX(box)); 
  g_return_if_fail(IS_R_GROUP(grp));
  
  g_object_get(grp, "group-name", &name, NULL);
  group = r_group_box_find(box, name);
  if (group)
    g_object_set(group, "enabled", FALSE, NULL); 
}




/**
 * r_group_box_merge_boxes
 * @box: a #RGroupBox
 * @second: a #RGroupBox
 *
 * merges the #RGroupBoxs
 *
 *returns: a RGroupBox*
 */
RGroupBox* 
r_group_box_merge_boxes (RGroupBox* box, RGroupBox* second)
{
  g_return_val_if_fail(IS_R_GROUP_BOX(box), NULL); 
  g_return_val_if_fail(IS_R_GROUP_BOX(second), box);
 
  box->priv->lst = g_list_concat(box->priv->lst, second->priv->lst);
  
  return box;
}


/**
 * r_group_box_reset
 * @box: a #RGroupBox
 *
 * reset the private #RGroupBox iterator to the head of group's list.
 */
void
r_group_box_reset(RGroupBox* box)
{
  g_return_if_fail(IS_R_GROUP_BOX(box));
  
  box->priv->iter = box->priv->lst;
}


/**
 * r_group_box_get_group
 * @box: a #RGroupBox
 *
 * get the first group in box
 *
 * returns: a gpointer or NULL if box is void
 */
RGroup* 
r_group_box_get_group (RGroupBox* box)
{
  g_return_val_if_fail(IS_R_GROUP_BOX(box), NULL); 
  
  if (box->priv->iter)
    return (RGroup*) box->priv->iter->data;
  
  return NULL;
}


/**
 * r_group_box_get_next_group
 * @box: a #RGroupBox
 *
 * get the next group in box. 
 *
 * returns: a gpointer or NULL if the groups's end list is reached.
 */
RGroup* 
r_group_box_get_next_group (RGroupBox* box)
{
  g_return_val_if_fail(IS_R_GROUP_BOX(box), NULL); 

  box->priv->iter = g_list_next(box->priv->iter);
  if (box->priv->iter)
    return (RGroup*) box->priv->iter->data;
  else
    box->priv->iter = box->priv->lst;

  return NULL;
}


/**
 * r_group_box_get_prev_group
 * @box: a #RGroupBox
 *
 * get the previous group in box.
 *
 * returns: a gpointer or NULL if the groups's head list is reached.
 */
RGroup* 
r_group_box_get_prev_group (RGroupBox* box)
{
  g_return_val_if_fail(IS_R_GROUP_BOX(box), NULL); 

  g_return_val_if_fail(IS_R_GROUP_BOX(box), NULL); 

  box->priv->iter = g_list_previous(box->priv->iter);
  if (box->priv->iter)
    return (RGroup*) box->priv->iter->data;
  else
    box->priv->iter = box->priv->lst;

  return NULL;
}







