/*
*  file: manager.c
*
*  
*  Copyright (C) 2005 Nicola Fragale <nicolafragale@libero.it>
*
*  This program is free software; you can redistribute it and/or modify
*  it under the terms of the GNU General Public License as published by
*  the Free Software Foundation; version 3 of the License
*
*  This program is distributed in the hope that it will be useful,
*  but WITHOUT ANY WARRANTY; without even the implied warranty of
*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*  GNU General Public License for more details.
*
*  You should have received a copy of the GNU General Public License
*  along with this program; if not, write to the Free Software
*  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#include <glib.h>
#include <glib-object.h>
#include <glib/gi18n-lib.h>
#include <gmodule.h>

#include <string.h>
#include <libintl.h>

#include "manager.h"
#include "plugin.h"
#include "types.h"


struct _RPluginManagerPrivate {
  GList* plugins;
  GList* filters;

  GHashTable* table;

  gboolean dispose_has_run;
};

typedef void (*RPluginInit) (RPlugin* plugin, gchar* file);

#define R_PLUGIN_MANAGER_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE((o),  \
                                         R_PLUGIN_MANAGER_TYPE,            \
                                         RPluginManagerPrivate))


static void r_plugin_manager_class_init (RPluginManagerClass* klass);
static void r_plugin_manager_init       (RPluginManager* obj);

static void r_plugin_manager_dispose    (RPluginManager* obj);
static void r_plugin_manager_finalize   (RPluginManager* obj);



GType
r_plugin_manager_get_type()
{
  static GType manager_type = 0;
  
  if (!manager_type)
    {
      static const GTypeInfo manager_info =
	{
	  sizeof(RPluginManagerClass),
	  NULL,
	  NULL,
	  (GClassInitFunc) r_plugin_manager_class_init,
	  NULL,
	  NULL,
	  sizeof(RPluginManager),
	  0,
	  (GInstanceInitFunc) r_plugin_manager_init
	};
      
      manager_type = g_type_register_static (G_TYPE_OBJECT, "PluginManager",
					     &manager_info, 0);
    }
  
  return manager_type;
}


static void
r_plugin_manager_class_init(RPluginManagerClass* klass)
{
  GObjectClass *class;
  
  class = G_OBJECT_CLASS (klass);

  class->dispose  = (GObjectFinalizeFunc) r_plugin_manager_dispose;
  class->finalize = (GObjectFinalizeFunc) r_plugin_manager_finalize;

  g_type_class_add_private (klass, sizeof(RPluginManagerPrivate));
}


static void
r_plugin_manager_init(RPluginManager* self)
{  
  RFilter* filter;

  filter = r_filter_new();
  g_object_set(filter, 
	       "filter-name", _("All files"), 
	       "filter-mime", "text/plain",
	       NULL);
  r_filter_add_pattern(filter, "*");

  self->priv = R_PLUGIN_MANAGER_GET_PRIVATE(self);
  
  self->priv->plugins  = NULL;
  self->priv->filters  = g_list_append(self->priv->filters, filter);
  self->priv->table    = g_hash_table_new(g_str_hash, g_str_equal);

  self->priv->dispose_has_run = FALSE;
}


static void 
r_plugin_manager_dispose (RPluginManager* manager)
{
  g_return_if_fail(IS_R_PLUGIN_MANAGER(manager));
  
  if (manager->priv->dispose_has_run)
    return;

  manager->priv->dispose_has_run = TRUE;
}


static void 
r_plugin_manager_finalize (RPluginManager* manager)
{
  g_return_if_fail(IS_R_PLUGIN_MANAGER(manager));  
}


/*    Public
*/

/**
 * r_plugin_manager_new
 *
 * create a new plugin manager
 *
 * Returns: a #RPluginManager*
 */
RPluginManager* 
r_plugin_manager_new(void)
{
  RPluginManager* manager;

  manager = g_object_new(r_plugin_manager_get_type(), NULL);

  return manager;
}


/**
 * r_plugin_manager_free
 * @manager: a #RPluginManager
 *
 * free the object
 */
void
r_plugin_manager_free(RPluginManager* manager)
{
  g_return_if_fail(IS_R_PLUGIN_MANAGER(manager));

  g_object_unref(manager);
}


/**
 * r_plugin_manager_scan_directory
 * @manager: a #RPluginManager
 * @dirname: a gchar*
 *
 * scan the given directory looking for RAL's plugins
 */
void 
r_plugin_manager_scan_directory (RPluginManager* manager, const gchar* dirname)
{
  GDir* dir;
  GError* error = NULL;
  const gchar* file;
  RPlugin* plugin;
  RFilter* filter;
  RPluginManagerPrivate* priv;

  g_return_if_fail(IS_R_PLUGIN_MANAGER(manager));
  g_return_if_fail(dirname != NULL);

  priv = R_PLUGIN_MANAGER_GET_PRIVATE(manager);

#ifdef ENABLE_DEBUG
  g_log(G_LOG_DOMAIN, G_LOG_LEVEL_DEBUG, " ");
  g_log(G_LOG_DOMAIN, G_LOG_LEVEL_DEBUG, "%s -- %s: %d", 
	__FILE__,__FUNCTION__,  __LINE__);
  g_log(G_LOG_DOMAIN, G_LOG_LEVEL_DEBUG, " ");
  g_log(G_LOG_DOMAIN, G_LOG_LEVEL_DEBUG, "enter r_plugin_manager_scan_directory");
  g_log(G_LOG_DOMAIN, G_LOG_LEVEL_DEBUG, "Scanning %s directory", dirname);
#endif

  dir = g_dir_open(dirname, 0, &error);
  if (error)
    {
      g_error("error opening directory: %s", error->message);
    }

  while ((file = g_dir_read_name(dir)) != NULL)
    {
#ifdef ENABLE_DEBUG
      g_log(G_LOG_DOMAIN, G_LOG_LEVEL_DEBUG, "Get file: %s", file);
#endif

      if (g_str_has_suffix(file, ".so"))
	{
#ifdef ENABLE_DEBUG
      g_log(G_LOG_DOMAIN, G_LOG_LEVEL_DEBUG, "Reading plugin: %s", file);
#endif
	  plugin = r_plugin_new();
	  
	  if (r_plugin_manager_load_plugin(plugin, file))
	    {
	      GList* filters = NULL, *l;
	      
	      priv->plugins = g_list_append(priv->plugins, plugin);
	      filters = r_plugin_get_filters(plugin);
	      
	      for (l = filters; l; l = l->next)
		{
		  filter = r_filter_copy(R_FILTER(l->data));
		  priv->filters = g_list_append(priv->filters, filter);
		}
	    }
	  else
	    g_warning(_("error loading plugin %s"), file);
	}      
    }
}



/**
 * r_plugin_manager_load_plugin
 * @plugin: a #RPlugin
 * @file: a const gchar*
 *
 * load the plugin from the given file
 *
 * Returns: %TRUE if plugin is successfully loaded, %FALSE otherwise
 */
gboolean
r_plugin_manager_load_plugin(RPlugin* plugin, const gchar* file)
{
  GModule* module;
  RPluginInit plugin_init;  
  RPluginInit plugin_fini; 
  gchar* plugin_file;

  g_return_val_if_fail(IS_R_PLUGIN(plugin), FALSE);

#ifdef ENABLE_DEBUG
  g_log(G_LOG_DOMAIN, G_LOG_LEVEL_DEBUG, "r_plugin_manager_load_plugin");
  g_log(G_LOG_DOMAIN, G_LOG_LEVEL_DEBUG, "Loading plugin: %s", file);
#else
  g_log(G_LOG_DOMAIN, G_LOG_LEVEL_INFO, "Loading plugin: %s", file);  
#endif

  plugin_file = g_strdup_printf("%s/%s", LIBRAL_PLUGINS_DIR, file);

  if ((module = g_module_open(plugin_file, G_MODULE_BIND_LAZY)) == NULL)
    {
      g_free(plugin_file);
      g_error(_("Opening plugin %s"), file);
      
      return FALSE;
    }

  if (!g_module_symbol(module, "plugin_init", (gpointer *) &plugin_init))
    {
      g_warning(_("Can't get initialization handle for %s plugin"), file);

      return FALSE;
    }
  
  if (!g_module_symbol(module, "plugin_fini", (gpointer *) &plugin_fini))
    {
      g_warning(_("Can't get initialization handle for %s plugin"), file);

      return FALSE;
    }
  
  plugin_init(plugin, plugin_file);
  g_object_set(plugin, "plugin-filename", plugin_file, NULL);
  g_free(plugin_file);

  return TRUE;
}


/**
 * r_plugin_manager_how_many
 * @manager: a #RPluginManager
 *
 * returns the number of plugins that have been read
 * 
 * Returns: a gint
 */
gint 
r_plugin_manager_how_many (RPluginManager* manager)
{
  g_return_val_if_fail(IS_R_PLUGIN_MANAGER(manager), 0);
 
  return g_list_length(manager->priv->plugins);
}


/**
 * r_plugin_manager_get_nth_plugin
 * @manager: a #RPluginManager
 * @n: a gint
 *
 * get the n-th plugin
 *
 * Returns: a #RPlugin* or %NULL
 */
RPlugin* 
r_plugin_manager_get_nth_plugin (RPluginManager* manager, gint n)
{
  g_return_val_if_fail(IS_R_PLUGIN_MANAGER(manager), NULL);
  g_return_val_if_fail(n >= 0, NULL);
  g_return_val_if_fail(n < g_list_length(manager->priv->plugins), NULL);
  
  return g_list_nth_data(manager->priv->plugins, n);
}


/**
 * r_plugin_manager_get_plugin
 * @manager: a #RPluginManager
 * @name: a gchar*
 *
 * get the plugin with the given name
 *
 * Returns: a #RPlugin* or %NULL if plugin is not found
 */
RPlugin* 
r_plugin_manager_get_plugin (RPluginManager* manager, const gchar* name)
{
  GList *iter;

  g_return_val_if_fail(IS_R_PLUGIN_MANAGER(manager), NULL);
  g_return_val_if_fail(name != NULL, NULL);
  
  iter = manager->priv->plugins;
  for (; iter; iter = g_list_next(iter))
    {
      RPlugin* plugin = (RPlugin*) iter->data;;
      gchar* plugin_name;
      
      if (plugin)
	{
	  g_object_get(plugin, "plugin-name", &plugin_name, NULL);
	  
	  if (g_ascii_strcasecmp(plugin_name, name) == 0)
	    return plugin;
	}
    }

  return NULL;
}


/**
 * r_plugin_manager_get_all_filters
 * @manager: a #RPluginManager
 * 
 * get all filters managed by installed plugins
 *
 * Returns: a #GList of #RFilter or %NULL
 */
GList* 
r_plugin_manager_get_all_filters (RPluginManager* manager)
{
  g_return_val_if_fail(IS_R_PLUGIN_MANAGER(manager), NULL);
  
  return manager->priv->filters;
}
