/*
*  RAL -- Rubrica Addressbook Library
*  file: net.c
*  
*  Copyright (C) Nicola Fragale <nicolafragale@libero.it>
*
*  This program is free software; you can redistribute it and/or modify
*  it under the terms of the GNU General Public License as published by
*  the Free Software Foundation; either version 3 of the License
*
*  This program is distributed in the hope that it will be useful,
*  but WITHOUT ANY WARRANTY; without even the implied warranty of
*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*  GNU General Public License for more details.
*
*  You should have received a copy of the GNU General Public License
*  along with this program; if not, write to the Free Software
*  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/


#include <glib.h>
#include <glib-object.h>

#include "net.h"
#include "utils.h"

enum {
  NET_URL = 1,
  NET_TYPE
};

typedef struct _NetConversion {
  gint   token;
  gchar* label;
} NetConversion;


struct _RNetAddressPrivate {
  gchar* url;
  gint   type;

  gboolean dispose_has_run;
};

#define R_NET_ADDRESS_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE((o), \
                                      R_NET_ADDRESS_TYPE, RNetAddressPrivate))

static NetConversion table[] = {
  {R_NET_ADDRESS_WEB,        "web"       },
  {R_NET_ADDRESS_EMAIL,      "email"     },
  {R_NET_ADDRESS_EKIGA,      "ekiga"     },
  {R_NET_ADDRESS_IRC,        "irc"       },
  {R_NET_ADDRESS_IRC_AIM,    "aim"       },
  {R_NET_ADDRESS_IRC_ICQ,    "icq"       },
  {R_NET_ADDRESS_IRC_JABBER, "jabber"    },
  {R_NET_ADDRESS_IRC_YAHOO,  "yahoo"     },
  {R_NET_ADDRESS_IRC_MSN,    "msn"       },
  {R_NET_ADDRESS_WORK_WEB,   "work web"  },
  {R_NET_ADDRESS_WORK_EMAIL, "work email"},
  {R_NET_ADDRESS_UNKNOWN,    "unknown"   },
};



static void r_net_address_class_init (RNetAddressClass* klass);
static void r_net_address_init       (RNetAddress* obj);

static void r_net_address_dispose    (RNetAddress* obj);
static void r_net_address_finalize   (RNetAddress* obj);


static void r_net_address_set_property (GObject* obj, guint property_id,
					const GValue* value, GParamSpec* spec);
static void r_net_address_get_property (GObject* obj, guint property_id,
					GValue* value, GParamSpec* spec);


GType
r_net_address_get_type()
{
  static GType r_net_address_type = 0;
  
  if (!r_net_address_type)
    {
      static const GTypeInfo r_net_address_info =
	{
	  sizeof(RNetAddressClass),
	  NULL,
	  NULL,
	  (GClassInitFunc) r_net_address_class_init,
	  NULL,
	  NULL,
	  sizeof(RNetAddress),
	  0,
	  (GInstanceInitFunc) r_net_address_init
	};

      r_net_address_type = g_type_register_static (G_TYPE_OBJECT, 
						   "RNetAddress",
						   &r_net_address_info, 0);
    }
  
  return r_net_address_type;
}


static void
r_net_address_class_init(RNetAddressClass* klass)
{
  GObjectClass *class;
  GParamSpec* pspec;
  
  class  = G_OBJECT_CLASS (klass);

  class->dispose  = (GObjectFinalizeFunc) r_net_address_dispose;
  class->finalize = (GObjectFinalizeFunc) r_net_address_finalize;
  
  class->set_property = r_net_address_set_property;
  class->get_property = r_net_address_get_property;

  g_type_class_add_private (klass, sizeof(RNetAddressPrivate));

  /**
   * RNetAddress:url:
   *
   * the net address uri
   */
  pspec = g_param_spec_string("url",
			      "url",
			      "the net address uri",
			      NULL,
			      G_PARAM_CONSTRUCT | G_PARAM_READWRITE);
  g_object_class_install_property(class, NET_URL, pspec);

  /**
   * RNetAddress:url-type:
   *
   * the uri's type, see #RNetAddressType for valid values
   */
  pspec = g_param_spec_int("url-type",
			   "url's type",
			   "url's type",
			   R_NET_ADDRESS_WEB,
			   R_NET_ADDRESS_UNKNOWN,
			   R_NET_ADDRESS_UNKNOWN,
			   G_PARAM_CONSTRUCT | G_PARAM_READWRITE);
  g_object_class_install_property(class, NET_TYPE, pspec);
}



static void
r_net_address_init(RNetAddress* self)
{
  RNetAddressPrivate* priv = R_NET_ADDRESS_GET_PRIVATE(self);
  
  priv->url  = NULL;
  priv->dispose_has_run = FALSE;
}


static void 
r_net_address_dispose (RNetAddress* self)
{
  RNetAddressPrivate* priv;

  g_return_if_fail(IS_R_NET_ADDRESS(self));
  
  priv = R_NET_ADDRESS_GET_PRIVATE(self);
  if (priv->dispose_has_run)
    return;

  priv->dispose_has_run = TRUE;
}


static void 
r_net_address_finalize (RNetAddress* self)
{
  RNetAddressPrivate* priv;

  g_return_if_fail(IS_R_NET_ADDRESS(self));

  priv = R_NET_ADDRESS_GET_PRIVATE(self);
  r_utils_free_string(priv->url);
}


static void 
r_net_address_set_property (GObject* obj, guint property_id,
			    const GValue* value, GParamSpec* spec)
{
  RNetAddress* self = (RNetAddress*) obj;
  RNetAddressPrivate* priv = R_NET_ADDRESS_GET_PRIVATE(self);;
  
  switch (property_id)
    {
    case NET_URL:
      g_free(priv->url);
      priv->url = g_value_dup_string(value);
      break;

    case NET_TYPE:
      priv->type = g_value_get_int(value);
      break;

    default:
      break;
    }
}


static void 
r_net_address_get_property (GObject* obj, guint property_id,
			    GValue* value, GParamSpec* spec)
{
  RNetAddress* self = (RNetAddress*) obj;
  RNetAddressPrivate* priv = R_NET_ADDRESS_GET_PRIVATE(self);;
  
  switch (property_id)
    {
    case NET_URL:
      g_value_set_string(value, priv->url);
      break;

    case NET_TYPE:
      g_value_set_int(value, priv->type);
      break;

    default:
      break;
    }
}


/**
 * r_net_address_new
 *
 * create a new #RNetAddress
 *
 * returns: a #RNetAddress*
 */
RNetAddress*
r_net_address_new(void)
{
  RNetAddress* net; 

  net = g_object_new(r_net_address_get_type(), NULL); 

  return net; 
}


/**
 * r_net_address_free
 * @net: a #RNetAddress
 *
 * free the memory owned by net
 */
void
r_net_address_free(RNetAddress* net)
{
  g_return_if_fail(IS_R_NET_ADDRESS(net));

  g_object_unref(net);   
}



gboolean        
r_net_address_search (RNetAddress *net, const gchar* str)
{
  RNetAddressPrivate* priv;

  g_return_val_if_fail(IS_R_NET_ADDRESS(net), FALSE);
  g_return_val_if_fail(str != NULL, FALSE);

  priv = R_NET_ADDRESS_GET_PRIVATE(net);
  if (g_strrstr(priv->url, str))
    return TRUE;

  return FALSE;  
}




/**
 * r_net_address_copy
 * @net: a #RNetAddress
 *
 * copy the given net address.
 *
 * returns: a new allocated #RNetAddress*
 */
RNetAddress* 
r_net_address_copy (RNetAddress *net)
{
  RNetAddress* new;
  RNetAddressType type;
  gchar *url;

  g_return_val_if_fail(IS_R_NET_ADDRESS(net), NULL);

  new = r_net_address_new();
  
  g_object_get(G_OBJECT(net), "url", &url, "url-type", &type, NULL);  
  g_object_set(G_OBJECT(new), "url", url,  "url-type", type, NULL);

  return new;
}


/**
 * r_net_address_decode_type
 * @type: a #RNetAddressType
 * 
 * decode the #RNetAddressType given type
 *
 * returns: a gchar*
 */
gchar* 
r_net_address_decode_type (RNetAddressType type)
{
  if ((type < R_NET_ADDRESS_WEB) || (type > R_NET_ADDRESS_UNKNOWN))
    return "unknown";

  return table[type].label;
}


/**
 * r_net_address_get_decoded_type
 * @type: gchar*
 * 
 * decode the string returned by r_net_address_get_url_type()
 *
 * returns: a #RNetAddressType
 */
RNetAddressType 
r_net_address_encode_type (gchar* type)
{
  RNetAddressType ret = R_NET_ADDRESS_UNKNOWN;

  if (!type)
    return ret;

  if (g_ascii_strcasecmp(type, "web") == 0)
    ret = R_NET_ADDRESS_WEB;
  if (g_ascii_strcasecmp(type, "work web") == 0)
    ret = R_NET_ADDRESS_WORK_WEB;
  else if (g_ascii_strcasecmp(type, "email") == 0)
    ret = R_NET_ADDRESS_EMAIL;
  else if (g_ascii_strcasecmp(type, "work email") == 0)
    ret = R_NET_ADDRESS_WORK_EMAIL;
  else if (g_ascii_strcasecmp(type, "ekiga") == 0)
    ret = R_NET_ADDRESS_EKIGA;
  else if (g_ascii_strcasecmp(type, "irc") == 0)
    ret = R_NET_ADDRESS_IRC;
  else if (g_ascii_strcasecmp(type, "aim") == 0)
    ret = R_NET_ADDRESS_IRC_AIM;
  else if (g_ascii_strcasecmp(type, "jabber") == 0)
    ret = R_NET_ADDRESS_IRC_JABBER;
  else if (g_ascii_strcasecmp(type, "icq") == 0)
    ret = R_NET_ADDRESS_IRC_ICQ;
  else if (g_ascii_strcasecmp(type, "yahoo") == 0)
    ret = R_NET_ADDRESS_IRC_YAHOO;
  else if (g_ascii_strcasecmp(type, "msn") == 0)
    ret = R_NET_ADDRESS_IRC_MSN;
  
  return ret;
}


/**
 * r_net_address_check
 * @net: a #RNetAddress
 * @property: an #RNetAddress's property
 * @value: the property's value (if set)
 *
 * check if the given property is set. 
 *
 * returns: %FALSE if the property is %NULL, otherwise it return %TRUE and
 * the content of the property is copied into value
 **/
gboolean 
r_net_address_check (RNetAddress* net, const gchar* property, gchar** value)
{
  gchar* tmp;

  g_return_val_if_fail(IS_R_NET_ADDRESS(net), FALSE);
  
  g_object_get(net, property, &tmp, NULL);
  
  if (tmp)
    {
      if (value)
	*value = tmp;

      return TRUE;
    }
  
  return FALSE;  
}
