/*
*  RAL -- Rubrica Addressbook Library
*  file: notes.c
*  
*  Copyright (C) Nicola Fragale <nicolafragale@libero.it>
*
*  This program is free software; you can redistribute it and/or modify
*  it under the terms of the GNU General Public License as published by
*  the Free Software Foundation; either version 3 of the License
*
*  This program is distributed in the hope that it will be useful,
*  but WITHOUT ANY WARRANTY; without even the implied warranty of
*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*  GNU General Public License for more details.
*
*  You should have received a copy of the GNU General Public License
*  along with this program; if not, write to the Free Software
*  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#include <stdlib.h>
#include <glib.h>
#include <glib-object.h>
#include <glib/gi18n-lib.h>

#include "notes.h"
#include "date.h"
#include "utils.h"

enum {
  NOTES_HAS_PARTNER = 1,
  NOTES_PARTNER,
  NOTES_CHILDREN,
  NOTES_OTHER_NOTES,                // hobbies and other notes
  NOTES_PUBKEY
};


struct _RNotesPrivate {
  gboolean has_partner;      /* have a partner?    */

  gchar* partner;            /* partner's name     */
  gchar* notes;              /* other notes        */
  gchar* pubkey;             /* public key         */

  RDate* birthday;           /* partner's birthday */
  RDate* anniversary;        /* their anniversary  */

  gboolean dispose_has_run;
};



static void r_notes_class_init (RNotesClass* klass);
static void r_notes_init       (RNotes* obj);

static void r_notes_dispose    (RNotes* obj);
static void r_notes_finalize   (RNotes* obj);


static void r_notes_set_property (GObject* obj, guint property_id,
				  const GValue* value, GParamSpec* spec);
static void r_notes_get_property (GObject* obj, guint property_id,
				  GValue* value, GParamSpec* spec);

GType
r_notes_get_type()
{
  static GType notes_type = 0;
  
  if (!notes_type)
    {
      static const GTypeInfo notes_info =
	{
	  sizeof(RNotesClass),
	  NULL,
	  NULL,
	  (GClassInitFunc) r_notes_class_init,
	  NULL,
	  NULL,
	  sizeof(RNotes),
	  0,
	  (GInstanceInitFunc) r_notes_init
	};

      notes_type = g_type_register_static (G_TYPE_OBJECT, "RNotes",
					   &notes_info, 0);
    }
  
  return notes_type;
}


static void
r_notes_class_init(RNotesClass* klass)
{
  GObjectClass *class;
  GParamSpec* pspec;
  
  class  = G_OBJECT_CLASS (klass);

  class->dispose  = (GObjectFinalizeFunc) r_notes_dispose;
  class->finalize = (GObjectFinalizeFunc) r_notes_finalize;
  
  class->set_property = r_notes_set_property;
  class->get_property = r_notes_get_property;

  /**
   * RNotes:has-partner:
   * 
   * contact has got a partner
   */
  pspec = g_param_spec_boolean("has-partner", 
			       "has partner", 
			       "contact has got a partner", 
			       FALSE, 
			       G_PARAM_CONSTRUCT | G_PARAM_READWRITE);
  g_object_class_install_property(class, NOTES_HAS_PARTNER, pspec);

  /**
   * RNotes:partner-name:
   * 
   * partner's name (first name last name)
   */
  pspec = g_param_spec_string("partner-name", 
			      "partner",
			      "partner's name",
			      NULL, 
			      G_PARAM_CONSTRUCT | G_PARAM_READWRITE);
  g_object_class_install_property(class, NOTES_PARTNER, pspec);

  /**
   * RNotes:notes:
   * 
   * contact's hobbies and other notes
   */
  pspec = g_param_spec_string("other-notes", 
			      "other notes", 
			      "contact's hobbies and other notes",
			      NULL, 
			      G_PARAM_CONSTRUCT | G_PARAM_READWRITE);
  g_object_class_install_property(class, NOTES_OTHER_NOTES, pspec);

  /**
   * RNotes:pubkey:
   * 
   * contact's public key
   */
  pspec = g_param_spec_string("pubkey", 
			      "pubkey", 
			      "contact's public key", 
			      NULL, 
			      G_PARAM_CONSTRUCT | G_PARAM_READWRITE);
  g_object_class_install_property(class, NOTES_PUBKEY, pspec);
}


static void
r_notes_init(RNotes* self)
{
  self->priv = g_new(RNotesPrivate, 1);
  self->priv->dispose_has_run = FALSE;
  
  self->priv->has_partner = FALSE;
  self->priv->partner     = NULL;
  self->priv->notes       = NULL;
  self->priv->pubkey      = NULL;
  
  self->priv->birthday    = r_date_new();
  self->priv->anniversary = r_date_new();
}



static void 
r_notes_dispose (RNotes* self)
{
  g_return_if_fail(IS_R_NOTES(self));

  if (self->priv->dispose_has_run)
    return;

  self->priv->dispose_has_run = TRUE;
}


static void 
r_notes_finalize (RNotes* self)
{
  g_return_if_fail(IS_R_NOTES(self));

  r_utils_free_string(self->priv->partner);
  r_utils_free_string(self->priv->notes);
  r_utils_free_string(self->priv->pubkey);
}


static void 
r_notes_set_property (GObject* obj, guint property_id,
		      const GValue* value, GParamSpec* spec)
{
  RNotes *self = (RNotes *) obj;
  
  switch(property_id)
    {
    case NOTES_HAS_PARTNER:
      self->priv->has_partner = g_value_get_boolean(value);
      break;

    case NOTES_PARTNER:
      g_free(self->priv->partner);
      self->priv->partner = g_value_dup_string(value);
      break;
    
    case NOTES_OTHER_NOTES:
      g_free(self->priv->notes);
      self->priv->notes = g_value_dup_string(value);
      break;

    case NOTES_PUBKEY:
      g_free(self->priv->pubkey);
      self->priv->pubkey = g_value_dup_string(value);
      break;
      
    default:
      break;
    }
}


static void 
r_notes_get_property (GObject* obj, guint property_id,
		      GValue* value, GParamSpec* spec)
{
  RNotes *self = (RNotes *) obj;
  
  switch(property_id)
    {
    case NOTES_HAS_PARTNER:
      g_value_set_boolean (value, self->priv->has_partner);
      break;
      
    case NOTES_PARTNER:
      g_value_set_string (value, self->priv->partner);
      break;
    
    case NOTES_OTHER_NOTES:
      g_value_set_string (value, self->priv->notes);
      break;

    case NOTES_PUBKEY:
      g_value_set_string (value, self->priv->pubkey);
     break;
      
    default:
      break;
    }
}


/*  Public
*/

/**
 * r_notes_new
 *
 * create a new #RNotes
 *
 * returns: a RNotes*
 */
RNotes*
r_notes_new(void)
{
  RNotes* notes;

  notes = g_object_new(r_notes_get_type(), NULL);

  return notes;
}


/**
 * r_notes_free
 * @notes: a #RNotes
 *
 * free memory owned by notes
 */
void
r_notes_free(RNotes* notes)
{
  g_return_if_fail(IS_R_NOTES(notes));

  g_object_unref(notes);   
}


/**
 * r_notes_set_know_birthday
 * @notes: a #RNotes
 * @know: gboolean
 *
 * set the know birthday flag
 */
void 
r_notes_set_know_birthday (RNotes* notes, gboolean know)
{
  g_return_if_fail(IS_R_NOTES(notes));
  
  g_object_set(notes->priv->birthday, "know", know, NULL);
}


/**
 * r_notes_set_know_anniversary
 * @notes: a #RNotes
 * @know: gboolean
 *
 * set the know anniversary flag
 */
void 
r_notes_set_know_anniversary (RNotes* notes, gboolean know)
{
  g_return_if_fail(IS_R_NOTES(notes));

  g_object_set(notes->priv->anniversary, "know", know, NULL);
}



/**
 * r_notes_set_birthday 
 * @notes: a #RNotes
 * @day:  
 * @month: 
 * @year: 
 *
 * set the partner's birthday
 */
void 
r_notes_set_birthday (RNotes* notes, gint day, gint month, gint year)
{
  g_object_set(notes->priv->birthday, "day", day, "month", month, 
	       "year", year, NULL);
}


/**
 * r_notes_set_anniversary
 * @notes: a #RNotes
 * @day:  
 * @month: 
 * @year: 
 *
 * set the contact's and his/her partner anniversary
 */
void 
r_notes_set_anniversary (RNotes* notes, gint day, gint month, gint year)
{
  g_object_set(notes->priv->anniversary, "day", day, "month", month, 
	       "year", year, NULL);
}

/**
 * r_notes_known_birthday
 * @notes: a #RNotes
 *
 * get the known_birthday flag value
 *
 * returns: gboolean. %TRUE if partner's birthday is known, %FALSE otherwise
 */
gboolean 
r_notes_know_birthday (RNotes* notes)
{
  g_return_val_if_fail(IS_R_NOTES(notes), FALSE);

  return r_date_is_known(R_DATE(notes->priv->birthday));
}


/**
 * r_notes_know_anniversary
 * @notes: a #RNotes
 *
 * get the known_anniversary flag value
 *
 * returns:gboolean. %TRUE if contact and his/her partner's 
 * anniversary is known, %FALSE otherwise
 */
gboolean 
r_notes_know_anniversary (RNotes* notes)
{
  g_return_val_if_fail(IS_R_NOTES(notes), FALSE);

  return r_date_is_known(R_DATE(notes->priv->anniversary));
}


gboolean 
r_notes_have_data (RNotes* notes)
{
  g_return_val_if_fail(IS_R_NOTES(notes), FALSE);
  
  if (notes->priv->partner || notes->priv->notes || 
      notes->priv->pubkey  || notes->priv->has_partner)
    return TRUE;
  
  return FALSE;
}



RDate*   
r_notes_get_birthday (RNotes* notes)
{
  g_return_val_if_fail(IS_R_NOTES(notes), NULL);

  return notes->priv->birthday;
}


RDate*   
r_notes_get_anniversary (RNotes* notes)
{
  g_return_val_if_fail(IS_R_NOTES(notes), NULL);
  
  return notes->priv->anniversary;
}


/**
 * r_notes_get_birth_day
 * @notes: a #RNotes
 *
 * get the partner's birth day. Caller must free the returned value
 *
 * returns: a gchar*
 */
gchar*   
r_notes_get_birth_day (RNotes* notes)
{
  g_return_val_if_fail(IS_R_NOTES(notes), NULL);
  
  return r_date_get_day(notes->priv->birthday);
}


/**
 * r_notes_get_birth_month
 * @notes: a #RNotes
 *
 * get the partner's birth month. Caller must free the returned value
 *
 * returns: a gchar*
 */
gchar*   
r_notes_get_birth_month (RNotes* notes)
{
  g_return_val_if_fail(IS_R_NOTES(notes), NULL);
 
  return r_date_get_month(notes->priv->birthday);
}


/**
 * r_notes_get_birth_year
 * @notes: a #RNotes
 *
 * get the partner's birth year. Caller must free the returned value
 *
 * returns: a gchar*
 */
gchar*   
r_notes_get_birth_year (RNotes* notes)
{
  g_return_val_if_fail(IS_R_NOTES(notes), NULL);
 
  return r_date_get_year(notes->priv->birthday);
}


/**
 * r_notes_get_anniversary_day 
 * @notes: a #RNotes
 *
 * get the anniversary day. Caller must free the returned value
 *
 * returns: a gchar*
 */
gchar*   
r_notes_get_anniversary_day (RNotes* notes)
{
  g_return_val_if_fail(IS_R_NOTES(notes), NULL);
  
  return r_date_get_day(notes->priv->anniversary);
}


/**
 * r_notes_get_anniversary_month
 * @notes: a #RNotes
 *
 * get the anniversary month. Caller must free the returned value
 *
 * returns: a gchar*
 */
gchar*   
r_notes_get_anniversary_month (RNotes* notes)
{
  g_return_val_if_fail(IS_R_NOTES(notes), NULL);
 
  return r_date_get_month(notes->priv->anniversary);
}


/**
 * r_notes_get_anniversary_year
 * @notes: a #RNotes
 *
 * get the anniversary year. Caller must free the returned value
 *
 * returns: a gchar*
 */
gchar*   
r_notes_get_anniversary_year (RNotes* notes)
{
  g_return_val_if_fail(IS_R_NOTES(notes), NULL);
 
  return r_date_get_year(notes->priv->anniversary);
}


/**
 * r_notes_get_birth_date
 * @notes: a #RNotes
 *
 * get partner's birthday, or unknown. Caller must free the returned value
 *
 * returns: a gchar*
 */
gchar*
r_notes_get_birth_date (RNotes* notes)
{
  g_return_val_if_fail(IS_R_NOTES(notes), g_strdup(_("unknown")));
  
  return r_date_get_human_date(R_DATE(notes->priv->birthday));
}


/**
 * r_notes_get_anniversary_date
 * @notes: a #RNotes
 *
 * get contact and partner's anniversary, or unknown. 
 * Caller must free the returned value
 *
 * returns: a gchar*
 */
gchar*
r_notes_get_anniversary_date (RNotes* notes)
{
  g_return_val_if_fail(IS_R_NOTES(notes), g_strdup(_("unknown")));

  return r_date_get_human_date(R_DATE(notes->priv->anniversary));
}


/**
 * r_notes_append_other_notes
 * @notes: a #RNotes
 * @other_note:
 *
 * append to other note 
 */
void r_notes_append_other_notes (RNotes* notes, gchar* other_note)
{
  gchar* tmp = NULL;
  gchar* new;

  g_return_if_fail(IS_R_NOTES(notes));
  g_return_if_fail(other_note != NULL);
  
  g_object_get(notes, "other-notes", &tmp, NULL);

  if (tmp && (g_ascii_strcasecmp(tmp, "") != 0))
    new = g_strdup_printf("%s\n%s", other_note, tmp);
  else
    new = g_strdup_printf("%s\n", other_note);

  g_object_set(notes, "other-notes", tmp, NULL);
}



/**
 * r_notes_check
 * @notes: a #RNotes
 * @property: an #RNotes's property
 * @value: the property's value (if set)
 *
 * check if the given property is set. 
 *
 * returns: %FALSE if the property is %NULL, otherwise it return %TRUE and
 * the content of the property is copied into value
 **/
gboolean 
r_notes_check (RNotes* notes, const gchar* property, gchar** value)
{
  gchar* tmp;

  g_return_val_if_fail(IS_R_NOTES(notes), FALSE);
  
  g_object_get(notes, property, &tmp, NULL);
  
  if (tmp)
    {
      if (value)
	*value = tmp;

      return TRUE;
    }
  
  return FALSE;  
}



gboolean 
r_notes_search (RNotes* notes, const gchar* str)
{
  g_return_val_if_fail(IS_R_NOTES(notes), FALSE);
  g_return_val_if_fail(str != NULL, FALSE);

  if (notes->priv->partner && g_strrstr (notes->priv->partner, str))
    return TRUE;

  if (notes->priv->notes && g_strrstr (notes->priv->notes, str))
    return TRUE; 
  
  if (notes->priv->pubkey && g_strrstr (notes->priv->pubkey, str))
    return TRUE;

  return FALSE;
}

/**
 * r_notes_copy
 * @notes: a #RNotes
 *
 * copy the given notes
 *
 * returns: a new allocated #RNotes* 
 */
RNotes* 
r_notes_copy (RNotes* notes)
{
  RNotes* new;
  RDate* date;
  gchar *partner, *other, *pubkey;
  gint day, month, year;
  gboolean has_partner;

  g_return_val_if_fail(IS_R_NOTES(notes), NULL);

  new = r_notes_new();

  g_object_get(notes, "has-partner", &has_partner, "partner-name", &partner, 
	       "other-notes", &other, "pubkey", &pubkey, NULL);

  g_object_set(new, "has-partner", has_partner, "partner-name", partner, 
	       "other-notes", other, "pubkey", pubkey, NULL);
  
  r_notes_set_know_birthday(new, r_notes_know_birthday(notes));
  r_notes_set_know_anniversary(new, r_notes_know_anniversary(notes));
 
  date = r_notes_get_birthday(notes);
  g_object_get(date, "day", &day, "month", &month, "year", &year, NULL);
  r_notes_set_birthday (new, day, month, year);
  
  date =  r_notes_get_anniversary (notes); 
  g_object_get(date, "day", &day, "month", &month, "year", &year, NULL);
  r_notes_set_anniversary(new, day, month, year);

  return new;
}
