/*
*  RAL -- Rubrica Addressbook Library
*  file: personal.c
*  
*  Copyright (C) Nicola Fragale <nicolafragale@libero.it>
*
*  This program is free software; you can redistribute it and/or modify
*  it under the terms of the GNU General Public License as published by
*  the Free Software Foundation; either version 3 of the License
*
*  This program is distributed in the hope that it will be useful,
*  but WITHOUT ANY WARRANTY; without even the implied warranty of
*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*  GNU General Public License for more details.
*
*  You should have received a copy of the GNU General Public License
*  along with this program; if not, write to the Free Software
*  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#include <time.h>
#include <glib.h>
#include <glib-object.h>

#include "types.h"
#include "card.h"
#include "personal.h"
#include "contact.h"
#include "work.h"
#include "notes.h"
#include "utils.h"


/*   property enumeration

enum {
  PERSONAL_CONTACT = 1,
  PERSONAL_WORK,
  PERSONAL_NOTES
};
*/

struct _RPersonalCardPrivate {

  RContact* contact;
  RWork*    work;
  RNotes*   notes;

  gboolean dispose_has_run;
};



static void           r_personal_card_class_init (RPersonalCardClass* klass);
static void           r_personal_card_init       (RPersonalCard* obj);

static void           r_personal_card_dispose    (RPersonalCard* obj);
static void           r_personal_card_finalize   (RPersonalCard* obj);


static void           r_personal_card_free       (RPersonalCard* self);

static gboolean       r_personal_card_search     (RPersonalCard* self,
						  const gchar* str);
static RPersonalCard* r_personal_card_copy       (RPersonalCard* self);
static RInfos*        r_personal_card_get_infos  (RPersonalCard* self);

/*
static void           r_personal_set_property    (GObject* obj, 
						  guint property_id,
						  const GValue* value, 
						  GParamSpec* spec);

static void           r_personal_get_property    (GObject* obj, 
						  guint property_id,
						  GValue* value, 
						  GParamSpec* spec);
*/

static RCardClass* parent_class;


GType
r_personal_card_get_type()
{
  static GType personal_card_type = 0;
  
  if (!personal_card_type)
    {
      static const GTypeInfo personal_card_info =
	{
	  sizeof(RPersonalCardClass),
	  NULL,
	  NULL,
	  (GClassInitFunc) r_personal_card_class_init,
	  NULL,
	  NULL,
	  sizeof(RPersonalCard),
	  0,
	  (GInstanceInitFunc) r_personal_card_init
	};
      
      personal_card_type = g_type_register_static (R_CARD_TYPE,
						   "RPersonalCard",
						   &personal_card_info, 0);
    }
  
  return personal_card_type;
}


static void
r_personal_card_class_init(RPersonalCardClass* klass)
{
  GObjectClass *obj_class;
  //  GParamSpec* pspec;
  
  parent_class                  = R_CARD_CLASS(klass);
  parent_class->free            = (gpointer) r_personal_card_free;
  parent_class->copy            = (gpointer) r_personal_card_copy;  
  parent_class->search          = (gpointer) r_personal_card_search;

  parent_class->get_infos       = (gpointer) r_personal_card_get_infos;
  parent_class->print_card      = NULL;

  obj_class = G_OBJECT_CLASS (klass);
  obj_class->dispose  = (GObjectFinalizeFunc) r_personal_card_dispose;
  obj_class->finalize = (GObjectFinalizeFunc) r_personal_card_finalize;
  
  //  obj_class->set_property = r_personal_set_property;
  //  obj_class->get_property = r_personal_get_property;


  /**
   * RPersonal:contact
   *

  pspec = g_param_spec_object("contact", 
			      "contact's personal data", 
			      "the personal data of the contact "
			      "(first name, last name, etc", 
			      R_CONTACT_TYPE,
			      G_PARAM_READWRITE);
  g_object_class_install_property (obj_class, PERSONAL_CONTACT, pspec);  
   */
  /**
   * RPersonal:work
   *

  pspec = g_param_spec_object("work", 
			      "contact's work", 
			      "the work of the contact. "
			      "His assignment, his role etc", 
			      R_WORK_TYPE,
			      G_PARAM_READWRITE);
  g_object_class_install_property (obj_class, PERSONAL_WORK, pspec);  
   */

  /**
   * RPersonal:
   *
 
  pspec = g_param_spec_object("notes", 
			      "contact's notes", 
			      "notes about the contact", 
			      R_NOTES_TYPE,
			      G_PARAM_READWRITE);
  g_object_class_install_property (obj_class, PERSONAL_NOTES, pspec);  
  */
}

/*
static void 
r_personal_set_property (GObject* obj, guint property_id,
			 const GValue* value, GParamSpec* spec)
{
  RPersonalCard *self = (RPersonalCard*) obj;
  
  switch(property_id)
    {
    case PERSONAL_CONTACT:
      self->priv->contact = g_value_get_object (value);
      break;
      
    case PERSONAL_WORK:
      self->priv->work = g_value_get_object (value);
      break;

    case PERSONAL_NOTES:
      self->priv->notes = g_value_get_object (value);
      break;
      
    default:
      break;
    }
}

static void 
r_personal_get_property (GObject* obj, guint property_id,
			 GValue* value, GParamSpec* spec)
{
  RPersonalCard *self = (RPersonalCard*) obj;
  
  switch(property_id)
    {
    case PERSONAL_CONTACT:
      g_value_set_object(value, self->priv->contact); 
      break;
      
    case PERSONAL_WORK:
      g_value_set_object(value, self->priv->work); 
      break;

    case PERSONAL_NOTES:
      g_value_set_object(value, self->priv->notes); 
      break;
      
    default:
      break;
    }
}
*/

static void
r_personal_card_init(RPersonalCard* self)
{
  self->priv = g_new(RPersonalCardPrivate, 1);

  self->priv->contact = NULL;
  self->priv->work    = NULL;
  self->priv->notes   = NULL;

  self->priv->dispose_has_run = FALSE;  
}



static void 
r_personal_card_dispose (RPersonalCard* self)
{
  g_return_if_fail(IS_R_PERSONAL_CARD(self));
  
  if (self->priv->dispose_has_run)
    return;
  
  /*    free the objects owned by personal card
   */
  r_contact_free(R_CONTACT(self->priv->contact));
  r_work_free(R_WORK(self->priv->work));
  r_notes_free(R_NOTES(self->priv->notes));  

  self->priv->dispose_has_run = TRUE;
}


static void 
r_personal_card_finalize (RPersonalCard* self)
{
  g_return_if_fail(IS_R_PERSONAL_CARD(self));
  
  g_free(self->priv);
  self->priv = NULL;
}


static void
r_personal_card_free(RPersonalCard* self)
{
  g_return_if_fail(IS_R_PERSONAL_CARD(self));

  g_object_unref(G_OBJECT(self)); 
}





RInfos* 
r_personal_card_get_infos (RPersonalCard* card)
{
  RInfos  *infos;
  RContact *contact = NULL;
  RWork    *work    = NULL;
  
  infos = g_malloc0(sizeof(RInfos));
  if (!infos)
    g_error("Unable to get enougth memory");

  contact = r_personal_card_get_contact(card);
  work    = r_personal_card_get_work(card);
  
  if (contact)
    g_object_get(contact, 
		 "first-name", &infos->first, 
		 "last-name",  &infos->last, 
		 "profession", &infos->prof, NULL);
  else
    infos->first = infos->last = infos->prof = g_strdup(NULL);
  
  if (work)
    g_object_get(work, "assignment", &infos->assignment, NULL);
  else
    infos->assignment = g_strdup(NULL);

  return infos;
}



/**
 * r_personal_card_search
 * @card: a #RPersonalCard
 * @str: a string
 *
 * search a string (a substring) into the card. Search is performed
 * on all data setted in card (personal data, addresses, net addresses, etc)
 *
 * returns: %TRUE if string (substring) was find, %FALSE otherwise
 */
static gboolean       
r_personal_card_search (RPersonalCard* self, const gchar* str)
{
  gchar* name;
  gpointer data;
  
  g_object_get(R_CARD(self), "card-name", &name, NULL);
  
  if (g_strrstr (name, str)) // cerca l'ago (str) nel pagliaio (haystack)
    return TRUE;
  
  /* search str in card's addresses */
  r_card_reset_address(R_CARD(self));
  data = r_card_get_address(R_CARD(self));
  for (; data; data = r_card_get_next_address(R_CARD(self)))
    if (r_address_search(R_ADDRESS(data), str))
      return TRUE;

  /* search str in card's net addresses */
  r_card_reset_net_address(R_CARD(self));
  data = r_card_get_net_address(R_CARD(self));
  for (; data; data = r_card_get_next_net_address(R_CARD(self)))
    if (r_net_address_search(R_NET_ADDRESS(data), str))
      return TRUE;  

  /* search str in card's telephone */
  r_card_reset_telephone(R_CARD(self));
  data = r_card_get_telephone(R_CARD(self));
  for (; data; data = r_card_get_next_telephone(R_CARD(self)))
    if (r_telephone_search(R_TELEPHONE(data), str))
      return TRUE;    
  
  if (r_contact_search(R_CONTACT(self->priv->contact), str))
    return TRUE;
  
  if (self->priv->work && r_work_search(R_WORK(self->priv->work), str))
    return TRUE;

  if (r_notes_search(R_NOTES(self->priv->notes), str))
    return TRUE;

  return FALSE;
}


/*   Public
 */

/**
 * r_personal_card_new
 *
 * create a new #RPersonalCard
 *
 * returns: a RPersonalCard*
 */
RPersonalCard*
r_personal_card_new(void)
{
  RPersonalCard* card;
  
  card = g_object_new(r_personal_card_get_type(), 
		      "card-type", "personal", NULL);
 
  return card;
}


/*  personal card is derived from card, 
    call card_free to free RPersonalCard
*/

/**
 * r_personal_card_set_contact
 * @card: a #RPersonalCard
 * @contact: a #RContact
 *
 * store contact's data in the card
 */
void 
r_personal_card_set_contact (RPersonalCard* card, RContact* contact)
{
  g_return_if_fail(IS_R_PERSONAL_CARD(card));
  g_return_if_fail(IS_R_CONTACT(contact));

  if (card->priv->contact)
    r_contact_free(R_CONTACT(card->priv->contact));

  card->priv->contact = contact;
}


/**
 * r_personal_card_set_work
 * @card: a #RPersonalCard
 * @work: a #RWork
 *
 * store contact's work data in card
 */
void 
r_personal_card_set_work (RPersonalCard* card, RWork* work)
{
  g_return_if_fail(IS_R_PERSONAL_CARD(card));
  g_return_if_fail(IS_R_WORK(work));

  if (card->priv->work)
    r_work_free(R_WORK(card->priv->work));

  card->priv->work = work;
}

/**
 * r_personal_card_set_notes
 * @card: a #RPersonalCard
 * @notes: a #RNotes
 *
 * store contact's notes in card
 */
void 
r_personal_card_set_notes (RPersonalCard* card, RNotes* notes)
{
  g_return_if_fail(IS_R_PERSONAL_CARD(card));
  g_return_if_fail(IS_R_NOTES(notes));

  if (card->priv->notes)
    r_notes_free(R_NOTES(card->priv->notes));

  card->priv->notes = notes;
}



/**
 * r_personal_card_get_contact
 * @card: a #RPersonalCard
 *
 * get the contact's personal data from card
 *
 * returns: a RContact* or NULL
 */
RContact*      
r_personal_card_get_contact (RPersonalCard* card)
{  
  g_return_val_if_fail(IS_R_PERSONAL_CARD(card), NULL);

  return card->priv->contact;
}


/**
 * r_personal_card_get_work
 * @card: a #RPersonalCard
 *
 * get the contact work's data from card
 *
 * returns: a RWork* or NULL
 */
RWork*       
r_personal_card_get_work (RPersonalCard* card)
{
  g_return_val_if_fail(IS_R_PERSONAL_CARD(card), NULL);

  return card->priv->work;
}


/**
 * r_personal_card_get_notes
 * @card: a #RPersonalCard
 *
 * get the contact's notes from card
 *
 * returns: a RNotes* or NULL
 */ 
RNotes*      
r_personal_card_get_notes (RPersonalCard* card)
{
  g_return_val_if_fail(IS_R_PERSONAL_CARD(card), NULL);

  return card->priv->notes;
}


/**
 * r_personal_card_copy
 * @card: a #RPersonalCard
 *
 * make a copy of given card
 *
 * returns: a new #RPersonalCard* or NULL
 */
RPersonalCard*  
r_personal_card_copy (RPersonalCard* card) 
{
  RPersonalCard* new;
  RContact* contact = NULL;
  RWork* work = NULL;
  RNotes* notes = NULL;
  
  g_return_val_if_fail(IS_R_PERSONAL_CARD(card), NULL);

  new = r_personal_card_new();

  contact = r_contact_copy(r_personal_card_get_contact(card));
  work    = r_work_copy(r_personal_card_get_work(card));
  notes   = r_notes_copy(r_personal_card_get_notes(card));
 
  r_personal_card_set_contact(new, contact);
  r_personal_card_set_work(new, work);
  r_personal_card_set_notes(new, notes);
  

  return new;
}







/**
 * r_personal_belongs_to_genre
 * @card: a #RPersonalCard
 * @genre: a string. male or female
 *
 * check if the card belongs to the given genre
 *
 * returns: %TRUE if card belongs to the given genre, %FALSE otherwise
 */
gboolean  
r_personal_card_belongs_to_genre (RPersonalCard* card, const gchar* genre)
{
  RContact* contact = NULL;
  gchar* contact_genre;

  g_return_val_if_fail(IS_R_PERSONAL_CARD(card), FALSE);
  g_return_val_if_fail(genre != NULL, FALSE);

  contact = r_personal_card_get_contact(card);

  if (!contact)
    return FALSE;
  
  g_object_get(G_OBJECT(contact), "genre", &contact_genre, NULL);
  
  if (contact_genre && (g_ascii_strcasecmp(genre, contact_genre) == 0))
    return TRUE;

  return FALSE;
}
