/*
*  file: plugin.c
*
*  
*  Copyright (C) 2005 Nicola Fragale <nicolafragale@libero.it>
*
*  This program is free software; you can redistribute it and/or modify
*  it under the terms of the GNU General Public License as published by
*  the Free Software Foundation; either version 3 of the License
*
*  This program is distributed in the hope that it will be useful,
*  but WITHOUT ANY WARRANTY; without even the implied warranty of
*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*  GNU General Public License for more details.
*
*  You should have received a copy of the GNU General Public License
*  along with this program; if not, write to the Free Software
*  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#include <glib.h>
#include <glib-object.h>
#include <glib/gi18n-lib.h>
#include <libintl.h>

#include "plugin.h"
#include "filter.h"
#include "types.h"




/*  property enumeration
*/
enum {
  PLUGIN_PROP_0,
  PLUGIN_NAME,            /* the name of the plugin                */
  PLUGIN_FILENAME,        /* shared object file                    */
  PLUGIN_LABEL,           /* short plugin's description            */
  PLUGIN_INFO,            /* plugin's description                  */
  PLUGIN_EXTENSIONS,      /* extensions that the plugin recognizes */
  PLUGIN_CONFIGURABLE,    /* is the plugin configurable?           */
};
   


struct _RPluginPrivate {      
  //  RPluginInit plugin_init;  // plugin init function
  //  RPluginInit plugin_fini;  // plugin termination function

  //  GModule* plugin;          // plugin module (shared object, the .so file)
  gchar*   filename;        // plugin's shared object file name (the .so file)
  gpointer engine;          // new allocated object in plugin shared object

  gchar*   name;            // plugin's name
  gchar*   label;           // plugin's infos
  gchar*   info;            // plugin's infos
  gchar*   extensions;      // plugin's extension (infos string)
  gboolean configurable;
  
  GList*   filters;
  GList*   actions;         // plugin's actions

  gboolean dispose_has_run;
};



static void r_plugin_class_init (RPluginClass* klass);
static void r_plugin_init       (RPlugin* obj);

static void r_plugin_dispose    (RPlugin* obj);
static void r_plugin_finalize   (RPlugin* obj);

static void r_plugin_set_property (GObject* obj, guint property_id,
				   const GValue* value, GParamSpec* spec);
static void r_plugin_get_property (GObject* obj, guint property_id,
				   GValue* value, GParamSpec* spec);

GType
r_plugin_get_type()
{
  static GType plugin_type = 0;
  
  if (!plugin_type)
    {
      static const GTypeInfo plugin_info =
	{
	  sizeof(RPluginClass),
	  NULL,
	  NULL,
	  (GClassInitFunc) r_plugin_class_init,
	  NULL,
	  NULL,
	  sizeof(RPlugin),
	  0,
	  (GInstanceInitFunc) r_plugin_init
	};
      
      plugin_type = g_type_register_static (G_TYPE_OBJECT, 
					    "RPlugin",
					    &plugin_info, 0);	
    }
  
  return plugin_type;
}


static void
r_plugin_class_init(RPluginClass* klass)
{
  GObjectClass *class;
  GParamSpec* pspec;
  
  klass->run_configure_gui = NULL;

  class = G_OBJECT_CLASS (klass);
  class->dispose  = (GObjectFinalizeFunc) r_plugin_dispose;
  class->finalize = (GObjectFinalizeFunc) r_plugin_finalize;

  class->set_property = r_plugin_set_property;
  class->get_property = r_plugin_get_property;

  /**
   * RPlugin:plugin-name
   *
   * the plungin's name
   */
  pspec = g_param_spec_string("plugin-name",
			      "the plungin's name",
			      NULL,
			      NULL,
			      G_PARAM_READWRITE);
  g_object_class_install_property(class, PLUGIN_NAME, pspec);  

  /**
   * RPlugin:plugin-filename
   *
   * the plungin's file name
   */
  pspec = g_param_spec_string("plugin-filename",
			      "the plungin's file name",
			      NULL,
			      NULL,
			      G_PARAM_READWRITE);
  g_object_class_install_property(class, PLUGIN_FILENAME, pspec);  


  /**
   * RPlugin:plugin-label
   *
   * a short plugin's description
   */
  pspec = g_param_spec_string("plugin-label",
			      "a short plugin's description",
			      NULL,
			      NULL,
			      G_PARAM_READWRITE);
  g_object_class_install_property(class, PLUGIN_LABEL, pspec);  


  /**
   * RPlugin:plugin-info
   *
   * a plugin's description
   */
  pspec = g_param_spec_string("plugin-info",
			      "a plugin's description",
			      NULL,
			      NULL,
			      G_PARAM_READWRITE);
  g_object_class_install_property(class, PLUGIN_INFO, pspec);  

  /**
   * RPlugin:plugin-extensions
   *
   * a short plugin's extensions description
   */
  pspec = g_param_spec_string("plugin-extensions",
			      "a short plugin's extensions description",
			      NULL,
			      NULL,
			      G_PARAM_READWRITE);
  g_object_class_install_property(class, PLUGIN_EXTENSIONS, pspec);


  /**
   * RPlugin:plugin-configurable
   *
   * is the plugin configurable?
   */
  pspec = g_param_spec_boolean("plugin-configurable",
			       "is the plugin configurable?",
			       NULL,
			       FALSE,
			       G_PARAM_READWRITE);
  g_object_class_install_property(class, PLUGIN_CONFIGURABLE, pspec);  
}


static void
r_plugin_init(RPlugin* plugin)
{
  plugin->priv = g_new(RPluginPrivate, 1);
  
  plugin->priv->filename     = NULL;
  plugin->priv->engine       = NULL;
  
  plugin->priv->name         = NULL;
  plugin->priv->label        = NULL;  
  plugin->priv->info         = NULL;
  plugin->priv->extensions   = NULL;
  plugin->priv->configurable = FALSE;
  
  plugin->priv->filters      = NULL;
  plugin->priv->actions      = NULL;
  
  plugin->priv->dispose_has_run = FALSE;
}


static void 
r_plugin_dispose (RPlugin* plugin)
{
  g_return_if_fail(IS_R_PLUGIN(plugin));
  
  if (plugin->priv->dispose_has_run)
    return;

  plugin->priv->dispose_has_run = TRUE;
}


static void 
r_plugin_finalize (RPlugin* plugin)
{
  g_return_if_fail(IS_R_PLUGIN(plugin));
  
  g_free(plugin->priv);
  plugin->priv = NULL;
}

static void 
r_plugin_set_property (GObject* obj, guint property_id,
		       const GValue* value, GParamSpec* spec)
{
  RPlugin* self = (RPlugin*) obj;

#ifdef ENABLE_DEBUG
  g_log(G_LOG_DOMAIN, G_LOG_LEVEL_DEBUG, " ");
  g_log(G_LOG_DOMAIN, G_LOG_LEVEL_DEBUG, "%s -- %s: %d", 
	__FILE__, __FUNCTION__, __LINE__);
  g_log(G_LOG_DOMAIN, G_LOG_LEVEL_DEBUG, "property: %d", property_id);  
  if (G_VALUE_HOLDS_BOOLEAN(value))
    g_log(G_LOG_DOMAIN, G_LOG_LEVEL_DEBUG, "value: %d", 
	  g_value_get_boolean (value));
  else
    g_log(G_LOG_DOMAIN, G_LOG_LEVEL_DEBUG, "value: %s", 
	  g_value_get_string(value));
  g_log(G_LOG_DOMAIN, G_LOG_LEVEL_DEBUG, " ");
#endif

  switch(property_id)
    {
    case PLUGIN_FILENAME:
      if (self->priv->filename)
	g_free(self->priv->filename);

      if (value)
	self->priv->filename = g_value_dup_string(value);
      break;

    case PLUGIN_NAME:
      if (self->priv->name)
	g_free(self->priv->name);

      if (value)
	self->priv->name = g_value_dup_string(value);
      break;

    case PLUGIN_LABEL:
      if (self->priv->label)
	g_free(self->priv->label);

      if (value)
	self->priv->label = g_value_dup_string(value);      
      break;

    case PLUGIN_INFO:
      if (self->priv->info)
	g_free(self->priv->info);

      if (value)
	self->priv->info = g_value_dup_string(value);      
      break;

    case PLUGIN_EXTENSIONS:
      if (self->priv->extensions)
	g_free(self->priv->extensions);
      
      self->priv->extensions = g_value_dup_string(value);
      break;

    case PLUGIN_CONFIGURABLE:
      self->priv->configurable = g_value_get_boolean(value);
      break;
      
    default:
      break;
    }
}

static void 
r_plugin_get_property (GObject* obj, guint property_id,
		       GValue* value, GParamSpec* spec)
{
  RPlugin* self = (RPlugin*) obj;

  switch(property_id)
    {
    case PLUGIN_NAME:
      g_value_set_string(value, self->priv->name);
      break;

    case PLUGIN_FILENAME:
      g_value_set_string(value, self->priv->filename);
      break;

    case PLUGIN_LABEL:
      g_value_set_string(value, self->priv->label);   
      break;

    case PLUGIN_INFO:
      g_value_set_string(value, self->priv->info);   
      break;

    case PLUGIN_EXTENSIONS:
      g_value_set_string(value, self->priv->extensions);   
      break;

    case PLUGIN_CONFIGURABLE:
      g_value_set_boolean(value, self->priv->configurable);
      break;
    
    default:
      break;
    }
}


/*    Public
*/

/**
 * r_plugin_new
 *
 * create a new #RPlugin
 *
 * Returns: a new #RPlugin
 */
RPlugin* 
r_plugin_new(void)
{
  RPlugin* plugin;

  plugin = g_object_new(r_plugin_get_type(), NULL);

  return plugin;
}


/**
 * r_plugin_free
 * @plugin: a #RPlugin
 *
 * free the object
 */
void
r_plugin_free(RPlugin* plugin)
{
  g_return_if_fail(IS_R_PLUGIN(plugin));

  g_object_unref(plugin);
}



/**
 * r_plugin_get_engine
 * @plugin: a #RPlugin
 *
 * Get the real plugin (a gobject)
 *
 * Currently are available those engines (plugins) with rubrica:
 * Rubrica        (manages the rubrica's file format)
 * VCard          (manages the vcard's file format)
 * CsvGmail       (manages the gmail's csv file format)
 * CsvThunderbird (managed the thunderbird's csv file format)
 *
 * For an updated engines's list look in #RAbook documentation
 * 
 * Returns: a gpointer
 */
gpointer 
r_plugin_get_engine (RPlugin* plugin)
{
  g_return_val_if_fail(IS_R_PLUGIN(plugin), NULL);
 
  return plugin->priv->engine;
}


/**
 * r_plugin_set_engine
 * @plugin: a #RPlugin
 * @engine: a gpointer
 *
 * use this function to set the real plugin. This function is 
 * called by the plugin (see rubrica, csv, vcard plugins source) 
 * during initialization. The engine is returned by a g_object_new call
 * and is the object that makes work.
 *
 * Currently are available those engines (plugins) with rubrica:
 * Rubrica        (manages the rubrica's file format)
 * VCard          (manages the vcard's file format)
 * CsvGmail       (manages the gmail's csv file format)
 * CsvThunderbird (managed the thunderbird's csv file format)
 *
 * For an updated engines's list look in #RAbook documentation
 * 
 */
void     
r_plugin_set_engine (RPlugin* plugin, gpointer engine)
{
   g_return_if_fail(IS_R_PLUGIN(plugin));
   g_return_if_fail(engine != NULL);
 
   plugin->priv->engine = engine;
}

/**
 * r_plugin_add_filter
 * @plugin: a #RPlugin
 * @filter: a #RFilter
 *
 * Add a #RFilter to the plugin 
 */
void     
r_plugin_add_filter (RPlugin* plugin, RFilter* filter)
{
  g_return_if_fail(IS_R_PLUGIN(plugin));
  g_return_if_fail(filter != NULL);

  plugin->priv->filters = g_list_append(plugin->priv->filters, filter);
}


/**
 * r_plugin_add_action
 * @plugin: a #RPlugin
 * @action: a #RPluginAction
 *
 * add a #RPluginAction to the plugin.
 */
void     
r_plugin_add_action (RPlugin* plugin, RPluginAction* action)
{
  g_return_if_fail(IS_R_PLUGIN(plugin));
  g_return_if_fail(action != NULL);

  plugin->priv->actions = g_list_append(plugin->priv->actions, action);
}


/**
 * r_plugin_get_filters
 * @plugin: a #RPlugin
 *
 * Get the plugin's file filters
 *
 * Returns: a #GList of #RFilter
 */
GList* 
r_plugin_get_filters (RPlugin* plugin)
{
  g_return_val_if_fail(IS_R_PLUGIN(plugin), NULL);
  
  return plugin->priv->filters;
}


/**
 * r_plugin_get_handle
 * @plugin: a #RPlugin
 *
 * Get the handle with the given name 
 *
 * Returns: a pointer to the requested function or %NULL if 
 * function was not found
 */
gpointer 
r_plugin_get_handle (RPlugin* plugin, gchar* name)
{
  GList* iter;

  g_return_val_if_fail(IS_R_PLUGIN(plugin), NULL);
  g_return_val_if_fail(name != NULL, NULL);
  
  for (iter = plugin->priv->actions; iter; iter = iter->next)
    {
      RPluginAction* action = iter->data;

      if (g_ascii_strcasecmp(action->name, name) == 0)
	return action->handle;
    }
  
  return NULL;
}


/**
 * r_plugin_get_name
 * @plugin: a #RPlugin
 *
 * Get the plugin's name
 *
 * Returns: a gchar*
 */
gchar*   
r_plugin_get_name (RPlugin* plugin)
{
  gchar* ret;
  g_return_val_if_fail(IS_R_PLUGIN(plugin), NULL);
  
  g_object_get(plugin, "plugin-name", &ret, NULL);
  return ret;
}


/**
 * r_plugin_get_file_name
 * @plugin: a #RPlugin
 *
 * Get the plugin's file name
 *
 * Returns: a gchar* 
 */
gchar*   
r_plugin_get_file_name (RPlugin* plugin)
{
  g_return_val_if_fail(IS_R_PLUGIN(plugin), NULL);

  return plugin->priv->filename;
}


/**
 * r_plugin_get_label
 * @plugin: a #RPlugin
 *
 * Get a short info about the plugin
 *
 * Returns: a gchar*
 */
gchar*   
r_plugin_get_label (RPlugin* plugin)
{
  gchar* ret;
  g_return_val_if_fail(IS_R_PLUGIN(plugin), NULL);
  
  g_object_get(plugin, "plugin-label", &ret, NULL);
  return ret;  
}


/**
 * r_plugin_get_info
 * @plugin: a #RPlugin
 *
 * Get info about the plugin
 *
 * Returns: a gchar*
 */
gchar*   
r_plugin_get_info (RPlugin* plugin)
{
  gchar* ret;
  g_return_val_if_fail(IS_R_PLUGIN(plugin), NULL);
  
  g_object_get(plugin, "plugin-info", &ret, NULL);
  return ret;  
}



/**
 * r_plugin_is_configurable
 * @plugin: a #RPlugin
 *
 * Check if the plugin is configurable
 *
 * Returns: %TRUE if plugin is configurable, %FALSE otherwise
 */
gboolean 
r_plugin_is_configurable (RPlugin* plugin)
{
  gboolean ret;
  g_return_val_if_fail(IS_R_PLUGIN(plugin), FALSE);

  g_object_get(plugin, "plugin-configurable", &ret, NULL);
  return ret;  
}


/**
 * r_plugin_run_configure_gui 
 * @plugin: a #RPlugin
 *
 * if plugin is configurable, user can run the configuration
 * gui that the plugin installs
 */
void 
r_plugin_run_configure_gui (RPlugin* plugin)
{
  RPluginClass* klass;  
  
  g_return_if_fail(IS_R_PLUGIN(plugin));
  
  klass = R_PLUGIN_GET_CLASS(plugin);
  if (R_PLUGIN_CLASS(klass)->run_configure_gui)
    klass->run_configure_gui(plugin->priv->engine);
}


