/*
*  RAL -- Rubrica Addressbook Library
*  file: ref.c
*  
*  Copyright (C) Nicola Fragale <nicolafragale@libero.it>
*
*  This program is free software; you can redistribute it and/or modify
*  it under the terms of the GNU General Public License as published by
*  the Free Software Foundation; either version 3 of the License
*
*  This program is distributed in the hope that it will be useful,
*  but WITHOUT ANY WARRANTY; without even the implied warranty of
*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*  GNU General Public License for more details.
*
*  You should have received a copy of the GNU General Public License
*  along with this program; if not, write to the Free Software
*  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#include <glib.h>
#include <glib-object.h>

#include "ref.h"
#include "utils.h"


enum { 
  R_REF_TO = 1,
  R_REF_FROM,
  R_REF_NAME
};


/* 
   to: id of the card referenced
   from: id of the card that reference this one   
 */
struct _RRefPrivate {
  glong    to;           /* card's id: reference  to card  */  
  glong    from;         /* card's id: reference from card */  
  gchar*   info;         /* card's info */ 

  gboolean dispose_has_run;
};


#define R_REF_GET_PRIVATE(o)     (G_TYPE_INSTANCE_GET_PRIVATE((o),  \
			          R_REF_TYPE, RRefPrivate))

static void r_ref_class_init (RRefClass* klass);
static void r_ref_init       (RRef* obj);

static void r_ref_dispose    (RRef* obj);
static void r_ref_finalize   (RRef* obj);


static void r_ref_set_property (GObject* obj, guint property_id,
				const GValue* value, GParamSpec* spec);
static void r_ref_get_property (GObject* obj, guint property_id,
				GValue* value, GParamSpec* spec);




GType
r_ref_get_type()
{
  static GType r_ref_type = 0;
  
  if (!r_ref_type)
    {
      static const GTypeInfo r_ref_info =
	{
	  sizeof(RRefClass),
	  NULL,
	  NULL,
	  (GClassInitFunc) r_ref_class_init,
	  NULL,
	  NULL,
	  sizeof(RRef),
	  0,
	  (GInstanceInitFunc) r_ref_init
	};

      r_ref_type = g_type_register_static (G_TYPE_OBJECT, 
					   "RRef", &r_ref_info, 0);
    }
  
  return r_ref_type;
}


static void
r_ref_class_init(RRefClass* klass)
{
  GObjectClass *parent_class;
  GParamSpec* pspec;
  
  parent_class  = G_OBJECT_CLASS (klass);

  parent_class->dispose      = (GObjectFinalizeFunc) r_ref_dispose;
  parent_class->finalize     = (GObjectFinalizeFunc) r_ref_finalize;

  parent_class->set_property = r_ref_set_property;
  parent_class->get_property = r_ref_get_property;

  g_type_class_add_private (klass, sizeof(RRefPrivate));


  /* class property 
   */
  /**
   * RRef:ref-to
   *
   * the id of the referenced card by this one
   */
  pspec = g_param_spec_long("ref-to", 
			    "ref card id", 
			    "the id of the referenced card by this one",  
			    G_MINLONG,
			    G_MAXLONG,
			    0,
			    G_PARAM_CONSTRUCT | G_PARAM_READWRITE);
  g_object_class_install_property(parent_class, R_REF_TO, pspec);  
  
  /**
   * RRef:ref-from:
   *
   * the id of the card that reference this one
   */
  pspec = g_param_spec_long("ref-from",
			    "ref card id", 
			    "the id of the card that reference this one",  
			    G_MINLONG,
			    G_MAXLONG,
			    0,
			    G_PARAM_CONSTRUCT | G_PARAM_READWRITE);
  g_object_class_install_property(parent_class, R_REF_FROM, pspec); 

  /**
   * RRef:ref-info
   *
   * the infos related to the card
   */
  pspec = g_param_spec_string("ref-info", 
			      "card infos", 
			      "the infos related to the card", 
			      NULL,  
			      G_PARAM_CONSTRUCT | G_PARAM_READWRITE);  
  g_object_class_install_property(parent_class, R_REF_NAME, pspec);  
}


static void
r_ref_init(RRef* self)
{
  self->private = R_REF_GET_PRIVATE(self);

  self->private->to   = 0L;
  self->private->from = 0L;
  self->private->info = NULL;

  self->private->dispose_has_run = FALSE;
}

 
static void 
r_ref_dispose (RRef* self)
{
  RRefPrivate* private;

  g_return_if_fail(IS_R_REF(self));
  
  private = R_REF_GET_PRIVATE(self);
  if (private->dispose_has_run)
    return;

  private->dispose_has_run = TRUE;
}


static void 
r_ref_finalize (RRef* self)
{
  RRefPrivate* private;

  g_return_if_fail(IS_R_REF(self));
  
  private = R_REF_GET_PRIVATE(self);
  r_utils_free_string(private->info);
  
  g_free(private);
  private = NULL;
}


static void 
r_ref_set_property (GObject* obj, guint property_id,
		    const GValue* value, GParamSpec* spec)
{
  RRef* self = (RRef*) obj;
  RRefPrivate* private = R_REF_GET_PRIVATE(self);

  switch (property_id) 
    {
    case R_REF_TO:
      private->to = g_value_get_long(value); 
      break; 
      
    case R_REF_FROM:
      private->from = g_value_get_long(value);
      break;

    case R_REF_NAME:
      g_free(private->info);
      private->info = g_value_dup_string(value); 
      break; 

    default: 
      break; 
    } 
} 
 

static void 
r_ref_get_property (GObject* obj, guint property_id,
		    GValue* value, GParamSpec* spec)
{
  RRef* self = (RRef*) obj;
  RRefPrivate* private = R_REF_GET_PRIVATE(self);

  switch (property_id) 
    {
    case R_REF_TO: 
      g_value_set_long(value, private->to); 
      break; 
      
    case R_REF_FROM:
      g_value_set_long(value, private->from);
      break;

    case R_REF_NAME:
      g_value_set_string(value, private->info); 
      break; 
          
    default:
      break;  
    }  
}




/*   Public
*/
/**
 * r_ref_new
 * @id: the id of the referenced card
 *
 * create a #RRef
 *
 * returns: a new #RRef*
 */
RRef*
r_ref_new(glong id)
{
  RRef* ref;

  ref = g_object_new(r_ref_get_type(), "ref-to", id, NULL);

  return ref;
}


/**
 * r_ref_free
 * @ref: a #RRef
 *
 * free the #RRef
 */
void
r_ref_free(RRef* ref)
{
  g_return_if_fail(IS_R_REF(ref));

  g_object_unref(ref);   
}



/**
 * r_ref_copy
 * @ref: a #RRef
 *
 * copy the given object
 *
 * returns: a new allocated #RRef* 
 */
RRef* 
r_ref_copy (RRef* ref)
{
  RRef* new;
  gulong to, from;
  gchar* info;

  g_return_val_if_fail(IS_R_REF(ref), NULL);

  g_object_get(G_OBJECT(ref), "ref-to", &to, "ref-from", &from,
	       "ref-info", &info, NULL);

  new = r_ref_new(to);
  g_object_set(G_OBJECT(new), "ref-from", from, "ref-info", info, NULL);
  
  return new;
}


/**
 * r_ref_check
 * @ref: a #RRef
 * @property: an #RRef's property
 * @value: the property's value (if set)
 *
 * check if the given property is set. 
 *
 * returns: %FALSE if the property is %NULL, otherwise it return %TRUE and
 * the content of the property is copied into value
 **/
gboolean 
r_ref_check (RRef* ref, const gchar* property, gchar** value)
{
  gchar* tmp;

  g_return_val_if_fail(IS_R_REF(ref), FALSE);
  
  g_object_get(ref, property, &tmp, NULL);
  
  if (tmp)
    {
      if (value)
	*value = tmp;

      return TRUE;
    }
  
  return FALSE;  
}
