/*
*  RAL -- Rubrica Addressbook Library
*  file: telephone.h
*  
*  Copyright (C) Nicola Fragale <nicolafragale@libero.it>
*
*  This program is free software; you can redistribute it and/or modify
*  it under the terms of the GNU General Public License as published by
*  the Free Software Foundation; either version 3 of the License
*
*  This program is distributed in the hope that it will be useful,
*  but WITHOUT ANY WARRANTY; without even the implied warranty of
*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*  GNU General Public License for more details.
*
*  You should have received a copy of the GNU General Public License
*  along with this program; if not, write to the Free Software
*  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/


#include <glib.h>
#include <glib-object.h>
#include <glib/gi18n-lib.h>

#include "telephone.h"
#include "lookup.h"
#include "utils.h"

/*   property enumetations
*/
enum {
  PHONE_PROP_0,
  PHONE_NUMBER,
  PHONE_TYPE
};


static RLookupTable table[] = {
  {"home",          N_("home"),          R_TELEPHONE_HOME          },
  {"work",          N_("work"),          R_TELEPHONE_WORK          },
  {"cellphone",     N_("cellphone"),     R_TELEPHONE_CELLPHONE     },
  {"fax",           N_("fax"),           R_TELEPHONE_FAX           },
  {"pager",         N_("pager"),         R_TELEPHONE_PAGER         },
  {"other",         N_("other"),         R_TELEPHONE_OTHER         },  
  {"operator",      N_("operator"),      R_TELEPHONE_OPERATOR      },
  {"green",         N_("green"),         R_TELEPHONE_GREEN         },
  {"customer care", N_("customer care"), R_TELEPHONE_CUSTOMER_CARE },
  {"unknown",       N_("unknown"),       R_TELEPHONE_UNKNOWN       },
  {NULL,            NULL,                R_TELEPHONE_INVALID       }
};


struct _RTelephonePrivate {
  gchar* number;
  gint   type;

  gboolean dispose_has_run;
};


#define R_TELEPHONE_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE((o), \
                                    R_TELEPHONE_TYPE, RTelephonePrivate))



static void r_telephone_class_init (RTelephoneClass* klass);
static void r_telephone_init       (RTelephone* obj);

static void r_telephone_dispose    (RTelephone* obj);
static void r_telephone_finalize   (RTelephone* obj);


static void r_telephone_set_property (GObject* obj, guint property_id,
				      const GValue* value, GParamSpec* spec);
static void r_telephone_get_property (GObject* obj, guint property_id,
				      GValue* value, GParamSpec* spec);

GType
r_telephone_get_type()
{
  static GType tel_num_type = 0;
  
  if (!tel_num_type)
    {
      static const GTypeInfo tel_num_info =
	{
	  sizeof(RTelephoneClass),
	  NULL,
	  NULL,
	  (GClassInitFunc) r_telephone_class_init,
	  NULL,
	  NULL,
	  sizeof(RTelephone),
	  0,
	  (GInstanceInitFunc) r_telephone_init
	};

      tel_num_type = g_type_register_static (G_TYPE_OBJECT, 
					     "RTelephone",
					     &tel_num_info, 0);
    }
  
  return tel_num_type;
}


static void
r_telephone_class_init(RTelephoneClass* klass)
{
  GObjectClass *class;
  GParamSpec* pspec;
  
  class = G_OBJECT_CLASS (klass);

  class->dispose  = (GObjectFinalizeFunc) r_telephone_dispose;
  class->finalize = (GObjectFinalizeFunc) r_telephone_finalize;

  class->set_property = r_telephone_set_property;
  class->get_property = r_telephone_get_property;

  g_type_class_add_private (klass, sizeof(RTelephonePrivate));

  /**
   * RTelephone:telephone-number:
   *
   * contact's telephone number
   */
  pspec = g_param_spec_string("telephone-number",
			      "telephone number",
			      "contact's telephone number",
			      NULL,
			      G_PARAM_CONSTRUCT | G_PARAM_READWRITE);
  g_object_class_install_property(class, PHONE_NUMBER, pspec);
  
  /**
   * RTelephone:telephone-type:
   *
   * telephone number's type. See #RTelephoneType for valid values
   */
  pspec = g_param_spec_int("telephone-type",
			   "telephone number's type",
			   "telephone number's type",
			   R_TELEPHONE_HOME,        // min
			   R_TELEPHONE_UNKNOWN,     // max
			   R_TELEPHONE_UNKNOWN,     // default
			   G_PARAM_CONSTRUCT | G_PARAM_READWRITE);
  g_object_class_install_property(class, PHONE_TYPE, pspec);
}


static void
r_telephone_init(RTelephone* self)
{
  RTelephonePrivate* priv = R_TELEPHONE_GET_PRIVATE(self);

  priv->number          = NULL;
  priv->dispose_has_run = FALSE;
}


static void 
r_telephone_dispose (RTelephone* self)
{
  RTelephonePrivate* priv;

  g_return_if_fail(IS_R_TELEPHONE(self));
  
  priv = R_TELEPHONE_GET_PRIVATE(self);
  if (priv->dispose_has_run)
    return;

  priv->dispose_has_run = TRUE;
}


static void 
r_telephone_finalize (RTelephone* self)
{
  RTelephonePrivate* priv;

  g_return_if_fail(IS_R_TELEPHONE(self));

  priv = R_TELEPHONE_GET_PRIVATE(self);
  r_utils_free_string(priv->number);
}



static void 
r_telephone_set_property (GObject* obj, guint property_id,
			  const GValue* value, GParamSpec* spec)
{
  RTelephone* self = (RTelephone*) obj;
  RTelephonePrivate* priv = R_TELEPHONE_GET_PRIVATE(self);
  
  switch (property_id)
    {
    case PHONE_NUMBER:
      g_free(priv->number);
      priv->number = g_value_dup_string(value);
      break;

    case PHONE_TYPE:
      priv->type = g_value_get_int(value);
      break;

    default:
      break;
    }
}


static void 
r_telephone_get_property (GObject* obj, guint property_id,
			  GValue* value, GParamSpec* spec)
{
  RTelephone* self = (RTelephone*) obj;
  RTelephonePrivate* priv = R_TELEPHONE_GET_PRIVATE(self);
  
  switch (property_id)
    {
    case PHONE_NUMBER:
      g_value_set_string(value, priv->number);
      break;

    case PHONE_TYPE:
      g_value_set_int(value, priv->type);
      break;

    default:
      break;
    }
}


/**
 * r_telephone_new
 * 
 * create a new #RTelephone
 *
 * returns: a #RTelephone*
 */
RTelephone*
r_telephone_new(void)
{
  RTelephone* telephone;

  telephone = g_object_new(r_telephone_get_type(), NULL);

  return telephone;
}


/**
 * r_telephone_free
 * @telephone: a #RTelephone
 *
 * free the #RTelephone*
 */
void
r_telephone_free(RTelephone* telephone)
{
  g_return_if_fail(IS_R_TELEPHONE(telephone));

  g_object_unref(telephone);   
}


gboolean       
r_telephone_search (RTelephone *telephone, const gchar* str)
{
  RTelephonePrivate* priv;

  g_return_val_if_fail(IS_R_TELEPHONE(telephone), FALSE);
  g_return_val_if_fail(str != NULL, FALSE);
    
  priv = R_TELEPHONE_GET_PRIVATE(telephone);
  if (g_strrstr(priv->number, str))
    return TRUE;

  return FALSE; 
}




/**
 * r_telephone_check
 * @telephone: a #RTelephone
 * @property: an #RTelephone's property
 * @value: the property's value (if set)
 *
 * check if the given property is set. 
 *
 * returns: %FALSE if the property is %NULL, otherwise it return %TRUE and
 * the content of the property is copied into value
 **/
gboolean 
r_telephone_check (RTelephone* telephone, const gchar* property, gchar** value)
{
  gchar* tmp;

  g_return_val_if_fail(IS_R_TELEPHONE(telephone), FALSE);
  
  g_object_get(telephone, property, &tmp, NULL);
  
  if (tmp)
    {
      if (value)
	*value = tmp;

      return TRUE;
    }
  
  return FALSE;  
}



/**
 * r_telephone_get_number
 * @telephone: a #RTelephone
 * 
 * copy the given telephone 
 *
 * returns: a new #RTelephone* 
 */
RTelephone* 
r_telephone_copy (RTelephone *telephone)
{
  RTelephone* new;
  RTelephoneType type;
  gchar *number;

  g_return_val_if_fail(IS_R_TELEPHONE(telephone), NULL);

  new = r_telephone_new();

  g_object_get(G_OBJECT(telephone), 
	       "telephone-number", &number, 
	       "telephone-type", &type, 
	       NULL); 
  g_object_set(G_OBJECT(new), 
	       "telephone-number", number, 
	       "telephone-type", type,
	       NULL);

  return new;
}



/**
 * r_telephone_lookup_str2enum
 * @type: a gchar*
 *
 * encode the given string 
 *
 * returns: a #RTelephoneType 
 */
RTelephoneType 
r_telephone_lookup_str2enum (gchar* str)
{
  return (RTelephoneType) r_lookup_table_str2enum(table, str);
}


/**
 * r_telephone_lookup_enum2str
 * @type: a #RTelephoneType
 *
 * decode the given type
 *
 * returns: a gchar*
 */
gchar* 
r_telephone_lookup_enum2str (RTelephoneType type)
{
  return r_lookup_table_enum2str(table, type);
}


gchar* 
r_telephone_lookup_enum2lbl (RTelephoneType type)
{
  return r_lookup_table_enum2lbl(table, type);
}


gchar* 
r_telephone_lookup_str2lbl  (gchar* str)
{
  return r_lookup_table_str2lbl(table, str);  
}
