/*
*  RAL -- Rubrica Addressbook Library
*  file: gmail_csv.c
* 
*  Copyright (C) Nicola Fragale <nicolafragale@libero.it>
*
*  This program is free software; you can redistribute it and/or modify
*  it under the terms of the GNU General Public License as published by
*  the Free Software Foundation; either version 3 of the License, or
*  (at your option) any later version.
*
*  This program is distributed in the hope that it will be useful,
*  but WITHOUT ANY WARRANTY; without even the implied warranty of
*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*  GNU General Public License for more details.
*
*  You should have received a copy of the GNU General Public License
*  along with this program; if not, write to the Free Software
*  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#include <stdio.h>
#include <string.h>
#include <glib.h>
#include <glib/gstdio.h>
#include <glib-object.h>
#include <glib/gi18n-lib.h>

#include "csv_gmail.h"
#include "csv_engine.h"
#include "abook.h"
#include "personal.h"
#include "company.h"
#include "contact.h"
#include "net.h"
#include "notes.h"
#include "plugin.h"
#include "error.h"



#define R_CSV_GMAIL_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE((o),      \
			            R_CSV_GMAIL_TYPE, RCsvGmailPrivate))


/* GMail csv fields (coded as utf-16):
   Name, Email, Notes
 */
typedef enum {
  R_CSV_GMAIL_NAME = 0,          
  R_CSV_GMAIL_EMAIL,         
  R_CSV_GMAIL_NOTES,     
} RCsvGmailField;


struct _RCsvGmailPrivate{ 
  gboolean dispose_has_run;
};


static void   r_csv_gmail_init       (RCsvGmail* obj);
static void   r_csv_gmail_class_init (RCsvGmailClass* klass);

static void   r_csv_gmail_dispose    (GObject* obj);
static void   r_csv_gmail_finalize   (GObject* obj);

/*    Private
*/
static void     fill_card        (RCsvEngine* engine, GList* records,
				  gpointer data);
static void     write_card       (RCard* card, FILE* fp);
static gboolean is_csv_gmail_doc (const gchar* filename);

static gboolean r_csv_gmail_open_file      (RAbook* abook, 
					    const gchar* filename);
static gboolean r_csv_gmail_save_file      (RAbook* abook, 
					    const gchar* filename);
static gboolean r_csv_gmail_overwrite_file (RAbook* abook);



GType
r_csv_gmail_get_type()
{
  static GType r_csv_gmail_type = 0;
  
  if (!r_csv_gmail_type)
    {
      static const GTypeInfo r_csv_gmail_info =
	{
	  sizeof(RCsvGmailClass),
	  NULL,
	  NULL,
	  (GClassInitFunc) r_csv_gmail_class_init,
	  NULL,
	  NULL,
	  sizeof(RCsvGmail),
	  0,
	  (GInstanceInitFunc) r_csv_gmail_init
	};

      r_csv_gmail_type = g_type_register_static (R_CSV_ENGINE_TYPE, 
						 "RCsvGmail",
						 &r_csv_gmail_info, 0);
    }
  
  return r_csv_gmail_type;
}


static void
r_csv_gmail_class_init(RCsvGmailClass* klass)
{
  GObjectClass *class;
  
  class = G_OBJECT_CLASS (klass);

  class->dispose  = (GObjectFinalizeFunc) r_csv_gmail_dispose;
  class->finalize = (GObjectFinalizeFunc) r_csv_gmail_finalize;

  g_type_class_add_private (klass, sizeof(RCsvGmailPrivate));
}


static void
r_csv_gmail_init(RCsvGmail* self)
{
  RCsvGmailPrivate* priv;

  g_return_if_fail(IS_R_CSV_GMAIL(self));
  
  priv = R_CSV_GMAIL_GET_PRIVATE(self);
  priv->dispose_has_run = FALSE;
}


static void 
r_csv_gmail_dispose (GObject* obj)
{
  RCsvGmail* self = R_CSV_GMAIL(obj);
  RCsvGmailPrivate* priv;

  g_return_if_fail(IS_R_CSV_GMAIL(obj));
  
  priv = R_CSV_GMAIL_GET_PRIVATE(self);

  if (priv->dispose_has_run)
    return;

  priv->dispose_has_run = TRUE;
}


static void 
r_csv_gmail_finalize (GObject* obj)
{
  RCsvGmail* self = R_CSV_GMAIL(obj);

  g_return_if_fail(IS_R_CSV_GMAIL(self));
  
  g_object_unref(self);
}


static void 
fill_card (RCsvEngine* engine, GList* records, gpointer data)
{
  RAbook* book = (RAbook*) data;
  RPersonalCard *card;
  RContact* contact;
  RNetAddress *net;
  RNotes *notes;
  gchar *name, *email, *note;

  card    = r_personal_card_new();
  contact = r_contact_new();
  net     = r_net_address_new();
  notes   = r_notes_new();

  name  = (gchar*) g_list_nth_data(records, R_CSV_GMAIL_NAME);
  email = (gchar*) g_list_nth_data(records, R_CSV_GMAIL_EMAIL);
  note  = (gchar*) g_list_nth_data(records, R_CSV_GMAIL_NOTES);

  g_object_set(card, "card-name", name, NULL);
  g_object_set(contact, "first-name", name, NULL);
  g_object_set(net, "url", email, "url-type", R_NET_ADDRESS_EMAIL, NULL);  
  g_object_set(notes, "other-notes", note, NULL);
  
  r_card_add_net_address(R_CARD(card), net);
  r_personal_card_set_contact(card, contact);
  r_personal_card_set_notes(card, notes);
	       
  r_abook_add_loaded_card (R_ABOOK(book), R_CARD(card));
  g_signal_emit_by_name(R_ABOOK(book), "card_read", card, G_TYPE_POINTER);
}



static gboolean 
is_csv_gmail_doc (const gchar* filename)
{
  GIOChannel* channel = NULL;
  GError* error = NULL;
  gchar* buffer;
  gchar** splits;
  gchar** alias;
  gint i = 0;

  channel = g_io_channel_new_file(filename, "r", &error);
  if (!channel)
    {
      g_warning("Channel: %s", error->message);
     
      g_io_channel_shutdown(channel, TRUE, NULL);
      g_io_channel_unref(channel);
      return FALSE;
    }

  if (g_io_channel_set_encoding(channel, "utf16", &error) == G_IO_STATUS_ERROR)
    {
      //      g_warning("Channel: %s", error->message);

      g_io_channel_shutdown(channel, TRUE, NULL);
      g_io_channel_unref(channel);
      return FALSE;
    }
  
  if (g_io_channel_read_line(channel, &buffer, NULL, NULL, &error) == 
      G_IO_STATUS_ERROR)
    {
      //      g_warning("Channel: %s", error->message);

      g_io_channel_shutdown(channel, TRUE, NULL);
      g_io_channel_unref(channel);
      return FALSE;
    }

  alias = splits = g_strsplit(buffer, ",", -1);
  if (!splits)
    {
      g_io_channel_shutdown(channel, TRUE, NULL);
      g_io_channel_unref(channel);
      g_free(buffer);
      return FALSE;
    }
  
  for(; *alias; alias++)
    i++;

  g_io_channel_shutdown(channel, TRUE, NULL);
  g_io_channel_unref(channel);
  g_strfreev(splits);
  g_free(buffer);

  if (i !=3)
    return FALSE;

  return TRUE;
}


/* *****************************  Public functions  ***********************
 */

/**
 * r_csv_gmail_new
 *
 * create a new #RCsvGmail
 *
 * returns: a #RCsvGmail*
 */
RCsvGmail* 
r_csv_gmail_new(void)
{
  return (RCsvGmail*) g_object_new(r_csv_gmail_get_type(), NULL);
}


/**
 * r_csv_gmail_free
 * @csv: a #RCsvGmail
 * 
 * free the #RCsvGmail object
 */
void  
r_csv_gmail_free(RCsvGmail *csv) 
{ 
  g_return_if_fail(IS_R_CSV_GMAIL(csv)); 

  g_object_unref(csv); 
} 


/**
 * r_csv_gmail_open_file
 * @csv: #RCsvGmail
 * @file: the file's name
 *
 * Read a Gmail's csv file. 
 * When a csv file is open a "file_opened" signal is emitted, 
 * with an error code on file status.
 * Each time a csv's record is decoded, a "card_decoded" signal is 
 * emitted and a #RPersonalCard will be sent to the user callback
 * function. The "addressbook_decoded" signal is emitted when csv file
 * if successfully read.
 *
 * returns: a gint, %FILE_OPEN_FAIL or %FILE_IMPORT_SUCCESS
 */
static gboolean          
r_csv_gmail_open_file (RAbook* abook, const gchar* filename)
{
  RCsvGmail* csv;

  g_return_val_if_fail(IS_R_ABOOK(abook), FALSE);
  
  if (!filename)
    {
      g_signal_emit_by_name(R_ABOOK(abook), "open_fail", 
			    NO_FILENAME, G_TYPE_INT);
      
      return FALSE;
    }
  
  if (!g_file_test(filename, G_FILE_TEST_EXISTS))
    {
      g_signal_emit_by_name(R_ABOOK(abook), "open_fail", 
			    FILE_NOT_EXIST, G_TYPE_INT);
      
      return FALSE;
    }
   
  csv = (RCsvGmail*) r_abook_get_engine(abook);

  if (!IS_R_CSV_GMAIL(csv))
    return FALSE;
  
  g_object_set(R_CSV_ENGINE(csv), "encode", "utf16", NULL);
  if (!is_csv_gmail_doc(filename))
    return FALSE;

  g_signal_connect(G_OBJECT(csv), "record_decoded", 
		   G_CALLBACK(fill_card), (gpointer) abook);
  
  if (r_csv_engine_read_file(R_CSV_ENGINE(csv), filename))
    {
      g_object_set(R_ABOOK(abook),
		   "addressbook-name", g_path_get_basename(filename),
		   "addressbook-path", g_path_get_dirname(filename),
		   NULL);
      
      g_signal_emit_by_name(R_ABOOK(abook), "addressbook_read", 
			    NULL, G_TYPE_NONE);
  
      return TRUE;
    }
  
  return FALSE;
}


static void
write_card (RCard* card, FILE* fp)
{
  RNotes      *notes;
  RNetAddress *net;

  gchar *type;
  gchar *name  = NULL;
  gchar *email = NULL;
  gchar *note  = NULL;

  g_object_get(card, "card-type", &type, "card-name", &name, NULL);
  if (g_ascii_strcasecmp(type, "personal") == 0)
    {
      notes = r_personal_card_get_notes(R_PERSONAL_CARD(card));
      
      g_object_get(notes, "other-notes", &note, NULL);      
    }
  else // company
    {
      g_object_get(R_COMPANY_CARD(card), "notes", &note, NULL);           
    }

  /* reset net iter */
  r_card_reset_net_address(card);
  
  /* find first net address */
  net = r_card_find_net_address(card, R_NET_ADDRESS_EMAIL);
  if (net) 
    g_object_get(net, "url", &email, NULL); 
  
  fprintf(fp, "\"%s\",\"%s\",\"%s\"\n", 
	  name ? name : "",  email ? email : "", note ? note : "");  
}


static gboolean 
r_csv_gmail_save_file (RAbook* abook, const gchar* filename)
{  
  FILE* fp;

  g_return_val_if_fail(IS_R_ABOOK(abook), FALSE);
  g_return_val_if_fail(filename != NULL, FALSE);

  if (!(fp = fopen(filename, "w")))
    {
      g_warning("\nCan't write file: %s", filename);
      
      return FALSE;
    }
  
  r_abook_foreach_card (abook, (RFunc) write_card, fp);
  fflush(fp);

  return TRUE;
}


static gboolean
r_csv_gmail_overwrite_file(RAbook* abook)
{
  gchar* file;
  gchar* path;
  gchar* filename;
  
  g_return_val_if_fail(IS_R_ABOOK(abook), FALSE);
  
  g_object_get(abook, "addressbook-path", &path, 
	       "addressbook-name", &file, NULL);
  filename = g_strdup_printf("%s%s%s", path, G_DIR_SEPARATOR_S, file);
  
  if (g_file_test(filename, G_FILE_TEST_EXISTS))
    g_remove(filename);

  if (!r_csv_gmail_save_file(abook, filename))
    {
      g_signal_emit_by_name(abook, "save_fail", OVERWRITING, G_TYPE_INT);
      
      g_free(filename);
      return FALSE;
    }
    
  g_free(filename);
  g_signal_emit_by_name(abook, "addressbook_saved", NULL, G_TYPE_NONE);  
  
  return TRUE;  
}




G_MODULE_EXPORT void 
plugin_init (RPlugin* plugin, gchar* file)
{
  RPluginAction* action;
  RFilter* filter;

  g_return_if_fail(plugin != NULL);
 
#ifdef ENABLE_DEBUG
  g_log(G_LOG_DOMAIN, G_LOG_LEVEL_DEBUG, "plugin_init");
  g_log(G_LOG_DOMAIN, G_LOG_LEVEL_DEBUG, "Initializing CsvGmail plugin");
#else
  g_log(G_LOG_DOMAIN, G_LOG_LEVEL_INFO, "Initializing CvsGmail plugin");
#endif

  r_plugin_set_engine(plugin, r_csv_gmail_new());
  g_object_set(plugin, 
	       "plugin-name", "gmail", 
	       "plugin-filename", file,
	       "plugin-label", "gmail's csv file format",
	       "plugin-info", "This plugin manages the gmail csv file format",
	       "plugin-extensions", "csv",
	       "plugin-configurable", FALSE, NULL);
  
  filter = r_filter_new();
  g_object_set(filter, 
	       "filter-name", "gmail", 
	       "filter-extension", "csv",
	       "filter-mime", "text/x-comma-separated-values", NULL);

  r_filter_add_pattern(filter, "csv");
  r_filter_add_pattern(filter, "gmail");

  r_plugin_add_filter (plugin, filter);

  action = g_malloc(sizeof(RPluginAction));
  action->name   = g_strdup("read");
  action->handle = (gpointer) r_csv_gmail_open_file;
  r_plugin_add_action(plugin, action);  
    
  action = g_malloc(sizeof(RPluginAction));
  action->name   = g_strdup("write");
  action->handle = (gpointer) r_csv_gmail_save_file;
  r_plugin_add_action(plugin, action);  

  action = g_malloc(sizeof(RPluginAction));
  action->name   = g_strdup("overwrite");
  action->handle = (gpointer) r_csv_gmail_overwrite_file;
  r_plugin_add_action(plugin, action);  
}


G_MODULE_EXPORT void     
plugin_fini (void)
{
  // TODO: liberare la memoria e rilasciare l'handler al plugin
}
