/*
*  RAL -- Rubrica Addressbook Library
*  file: io.c
*  
*  Copyright (C) Nicola Fragale <nicolafragale@libero.it>
*
*  This program is free software; you can redistribute it and/or modify
*  it under the terms of the GNU General Public License as published by
*  the Free Software Foundation; either version 3 of the License
*
*  This program is distributed in the hope that it will be useful,
*  but WITHOUT ANY WARRANTY; without even the implied warranty of
*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*  GNU General Public License for more details.
*
*  You should have received a copy of the GNU General Public License
*  along with this program; if not, write to the Free Software
*  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#include <stdlib.h>
#include <time.h>
#include <glib.h>
#include <libxml/tree.h>

#include "io.h"


void     
r_io_write_str (xmlNodePtr node, gchar *label, gchar* string)
{
  g_return_if_fail(node != NULL);
  g_return_if_fail(label != NULL);
  
  xmlNewProp(node, (xmlChar*) label, (xmlChar*) string);
}


void   
r_io_write_bool (xmlNodePtr node, gchar *label, gboolean bool)
{
  g_return_if_fail(node != NULL);
  g_return_if_fail(label != NULL);
  
  if (bool)
    xmlNewProp(node, (xmlChar*) label, (xmlChar*) "true");
  else
    xmlNewProp(node, (xmlChar*) label, (xmlChar*) "false");
}


void     
r_io_write_number (xmlNodePtr node, gchar *label, gulong val)
{
  xmlChar* xmlstr;
  gchar* str;
  
  g_return_if_fail(node != NULL);
  g_return_if_fail(label != NULL);
  
  str  = g_strdup_printf("%ld", val);
  xmlstr = xmlStrdup((xmlChar*) str);
  
  xmlNewProp(node, (xmlChar*) label, xmlstr);
  g_free(str);
}



void     
r_io_write_date (xmlNodePtr node, gboolean bool, time_t date)
{
  GDate* gdate;
  GDateDay day;
  GDateMonth month;
  GDateYear year;
  
  g_return_if_fail(node != NULL);
  
  r_io_write_bool(node, "known", bool);

  gdate = g_date_new();
  g_date_set_time(gdate, date);
  day   = g_date_get_day(gdate);
  month = g_date_get_month(gdate);
  year  = g_date_get_year(gdate);
  
  if (!bool || day == G_DATE_BAD_DAY)  
    r_io_write_str(node, "day", "BadDay");
  else
    r_io_write_number(node, "day", day);
  
  if (!bool || month == G_DATE_BAD_MONTH)
    r_io_write_str(node, "month", "BadMonth");
  else
    r_io_write_number(node, "month", month);
  
  if (!bool || year == G_DATE_BAD_YEAR)
    r_io_write_str(node, "year", "BadYear");
  else
    r_io_write_number(node, "year", year);
  
  g_date_free(gdate);
}


gchar* 
r_io_get(xmlNodePtr node, const gchar* name, RError *err)
{
  xmlNodePtr child;

  *err = UNREADABLE_NODE;
  g_return_val_if_fail(node != NULL, NULL);

  child = node->children;
  if (xmlIsBlankNode(child))
    child = child->next;   

  while (child)
    {
      gchar* ret;

      if (xmlStrcmp(child->name, (xmlChar*) name) == 0)
	{
	  ret = (gchar *) xmlNodeGetContent(child); // caller must free memory
	  
	  if (xmlStrcmp((xmlChar*) ret, (xmlChar*) "") == 0)
	    {
	      xmlFree(ret);

	      return NULL;
	    }

	  return ret;
	}
      
      child = child->next;
      if (xmlIsBlankNode(child))
	child = child->next;
    }

  *err = UNREADABLE_NODE;
  return NULL;
}
     

gchar* 
r_io_get_content(xmlNodePtr node, RError *err)
{
  xmlChar* tmp = NULL;

  *err = UNREADABLE_NODE;
  g_return_val_if_fail(node != NULL, NULL);

  tmp = xmlNodeGetContent(node);   // caller must free memory
  if (!tmp || (xmlStrcmp(tmp, (xmlChar*) "") == 0))
    {
      *err = UNREADABLE_NODE;
      
      return NULL;
    }
  
  *err = NO_ERROR;
  return (gchar *) tmp;  
}



gboolean
r_io_get_bool(xmlNodePtr node, gchar* key, RError *err)
{
  gchar* tmp = NULL;
  gboolean ret = FALSE;

  *err = UNREADABLE_PROPERTY;
  g_return_val_if_fail(node != NULL, FALSE);

  *err = KEY_NOT_VALID;
  g_return_val_if_fail(key != NULL, FALSE);

  tmp = r_io_get_prop(node, key, err);

  if (!tmp)
    {
      *err = UNREADABLE_PROPERTY;
      return FALSE;
    }
  
  *err = NO_ERROR;
  if(xmlStrcmp((xmlChar*) tmp, (xmlChar*) "true") == 0)    
    ret = TRUE;
  else
    ret = FALSE;

  g_free(tmp);
  return ret;
}



glong 
r_io_get_date (xmlNodePtr node, gchar* key, RError *err)
{
  gchar* tmp = NULL;
  glong ret;

  *err = UNREADABLE_NODE;
  g_return_val_if_fail(node != NULL, 0);

  *err = KEY_NOT_VALID;
  g_return_val_if_fail(key  != NULL, 0);

  tmp = r_io_get_prop(node, key, err);
  if (!tmp)
    {
      *err = INVALID_DATE;
      return 0;
    }

  *err = NO_ERROR;
  ret = atol(tmp);
  g_free(tmp);

  return ret;
}



time_t 
r_io_get_calendar(xmlNodePtr node, gchar** day, gchar** month, 
			gchar** year, RError *err)
{
  GDate *gdate;
  struct tm tm;
  gchar *_day, *_month, *_year;
  
  *err = UNREADABLE_NODE;
  g_return_val_if_fail(node != NULL, 0);

  _day   = r_io_get_prop(node, "day", err);
  _month = r_io_get_prop(node, "month", err);
  _year  = r_io_get_prop(node, "year", err);
  
  if (_day && _month && _year)
    {
      if ((g_ascii_strcasecmp(_day,   "BadDay") == 0) ||
	  (g_ascii_strcasecmp(_day,   "") == 0)       ||
	  (atoi(_day) < 1))
	goto exit;
      if ((g_ascii_strcasecmp(_month, "BadMonth") == 0) ||
	  (g_ascii_strcasecmp(_month, "") == 0)         ||
	  (atoi(_month) < 1))
	goto exit;	
      if ((g_ascii_strcasecmp(_year,  "BadYear")  == 0) ||
	  (g_ascii_strcasecmp(_year,  "") == 0)         ||
	  (atoi(_year) < 1))
	goto exit;
    
      if (day)
	*day   = g_strdup(_day);
      if (month)
	*month = g_strdup(_month);
      if (year)
	*year  = g_strdup(_year);

      gdate = g_date_new_dmy(atoi(_day), atoi(_month), atoi(_year));
      g_date_to_struct_tm(gdate, &tm);
      g_date_free(gdate);

      *err = NO_ERROR;
      return mktime(&tm);
    }

 exit:
  *err = UNREADABLE_CALENDAR;
  if (day)
    *day  = "";
  if (month)
    *month = "";
  if (year)
    *year = "";

  if (_day)   
    g_free(_day);
  if (_month) 
    g_free(_month);
  if (_year)  
    g_free(_year);

  return (time_t) -1;
}


gchar* 
r_io_get_prop(xmlNodePtr node, gchar* key, RError *err)
{
  gchar* ret = NULL;

  *err = UNREADABLE_NODE;
  g_return_val_if_fail(node != NULL, NULL);

  *err = KEY_NOT_VALID;
  g_return_val_if_fail(key  != NULL, NULL);

  if (xmlHasProp(node, (xmlChar*) key))
    {
      *err = NO_ERROR;
      ret  = (gchar*) xmlGetProp(node, (xmlChar*) key);  // caller must free memory
      if (g_ascii_strcasecmp(ret, "") == 0)
	return NULL;

      return ret;
    }
  
  *err = UNREADABLE_PROPERTY;
  return NULL;
}


gchar*    
r_io_get_prop_from (xmlNodePtr node, gchar* node_name,
		    gchar* key, RError* err) 
{
  xmlNodePtr child;
  
  *err = UNREADABLE_NODE;
  g_return_val_if_fail(node != NULL, NULL);
  
  child = node->children;
  if (xmlIsBlankNode(child))
    child = child->next;   
  
  while (child)
    {
      if (xmlStrcmp(child->name, (xmlChar*) node_name) == 0)
	return r_io_get_prop(child, key, err);
      
      child = child->next;
      if (xmlIsBlankNode(child))
	child = child->next;
    }
  
  *err = UNREADABLE_NODE;
  return NULL;  
}


gboolean  
r_io_get_bool_from (xmlNodePtr node, gchar* node_name,
			  gchar* key, RError* err)
{
  xmlNodePtr child;
  
  *err = UNREADABLE_NODE;
  g_return_val_if_fail(node != NULL, FALSE);
  
  child = node->children;
  if (xmlIsBlankNode(child))
    child = child->next;   
  
  while (child)
    {
      if (xmlStrcmp(child->name, (xmlChar*) node_name) == 0)
	return r_io_get_bool(child, key, err);
      
      child = child->next;
      if (xmlIsBlankNode(child))
	child = child->next;
    }
  
  *err = UNREADABLE_NODE;
  return FALSE;
}


time_t   
r_io_get_calendar_from (xmlNodePtr node, gchar* node_name,
			      gchar** day, gchar** month, gchar** year,
			      RError *err)
{
  xmlNodePtr child;
  
  *err = UNREADABLE_NODE;
  g_return_val_if_fail(node != NULL, -1);
  
  child = node->children;
  if (xmlIsBlankNode(child))
    child = child->next;   
  
  while (child)
    {
      if (xmlStrcmp(child->name, (xmlChar*) node_name) == 0)
	return r_io_get_calendar(child, day, month, year, err);
      
      child = child->next;
      if (xmlIsBlankNode(child))
	child = child->next;
    }
  
  if (day)
    *day   = "BadDay";
  if (month)
    *month = "BadMonth";
  if (year)
    *year  = "BadYear";
  *err  = UNREADABLE_NODE;

  return (time_t) -1;  
}



xmlNodePtr 
r_io_get_node(xmlNodePtr node, xmlChar* key)
{
  xmlNodePtr child;
  
  if(xmlIsBlankNode(node))
    node = node->next;

  if (xmlStrcmp(key, (xmlChar*) "Card") == 0) 
    {
      return node;
    }
  
  child = node->children;
  if(xmlIsBlankNode(child))
    child = child->next;
  
  while (child)
    {
      if (xmlStrcmp(child->name, key) == 0)
	return child;
      
      child = child->next;
      if (xmlIsBlankNode(child))
	child = child->next;
    }    
  
  return NULL;
}


xmlNodePtr 
r_io_get_child(xmlNodePtr node, xmlChar* key)
{
  xmlNodePtr child;

  if (!node)
    return NULL;
  
  if (xmlIsBlankNode(node))
    node = node->next;

  child = node->children;
  if(xmlIsBlankNode(child))
    child = child->next;
  
  while (child)
    {
      if (xmlStrcmp(child->name, key) == 0)
	return child;
      
      child = child->next;
      if (xmlIsBlankNode(child))
	child = child->next;
    }      

  return NULL;
}

xmlNodePtr
r_io_get_brother(xmlNodePtr brother, xmlChar* key)
{
  if (!brother)
    return NULL;
  
  if (xmlIsBlankNode(brother))
    brother = brother->next;
  
  if (xmlStrcmp(brother->name, key) == 0)
    return brother;
  
  return NULL;
}
