/*
*  RAL -- Rubrica Addressbook Library
*  file: save.c
*  
*  Copyright (C) Nicola Fragale <nicolafragale@libero.it>
*
*  This program is free software; you can redistribute it and/or modify
*  it under the terms of the GNU General Public License as published by
*  the Free Software Foundation; either version 3 of the License
*
*  This program is distributed in the hope that it will be useful,
*  but WITHOUT ANY WARRANTY; without even the implied warranty of
*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*  GNU General Public License for more details.
*
*  You should have received a copy of the GNU General Public License
*  along with this program; if not, write to the Free Software
*  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#include <glib.h>
#include <libxml/tree.h>

#include "save.h"
#include "io.h"
#include "card.h"
#include "personal.h"
#include "company.h"
#include "contact.h"
#include "address.h"
#include "group.h"
#include "net.h"
#include "notes.h"
#include "telephone.h"
#include "error.h"


/*
  RubricaDoc file structure:
     version:    rubrica's version that made the addressbook file
     fileformat: #4
     doctype:    addressbook

  <?xml version="1.0" encoding="UTF-8"?>
  <Rubrica version="" fileformat="" doctype="">        root
     <Card                                             root child 
            name=""                                    the card name
            locked=""                                  is card locked?
            deleted=""                                 was card deleted?
            rate=""                                    card rank (1..5)
            id="pcxxx"                                 card id   (es:  pc643)
	    created=""                                 creation date
            last_change=""                             last change date
	    type="">                                   card's type 
         <Groups>                                      card child 
             <Group owner="" pixmap="">....</Group>    groups children
	     ...
         </Groups>     
	 <Refs>
	     <Ref refid="card_id">references's description</Ref>
	     ...
         </Refs>  
         <Data>                                           card child
             <FirstName>..................</FirstName>    data child 
	     <MiddleName>.................</MiddleName>     "    "
	     <LastName>...................</LastName>                    
	     <Nickname>...................</NickName>                    
	     <Profession>.................</Profession>                  
 	     <NamePrefix>.................</NamePrefix>                  
	     <Title>......................</Title>                       
	     <Photo type="">..............</Photo>
	     <Birthday known="" day="" month="" year=""/>  
         </Data>
	 <Work>
	     <Assignment>.................</Assignment>
	     <Organization>...............</Organization>
             <Department>.................</Department>
	     <SubDepartment>..............</SubDepartment>
	     <ManagerName>................</ManagerName>
	     <ManagerPhone>...............</ManagerPhone>
	     <CollabratorName>............</CollabratorName>
	     <CollaboratorPhone>..........</CollabratorPhone>
         </Work>
	 <Addresses>                                   card child
             <Address type="">                         addresses children
                 <Street number="">.......</Street>    address child
	         <City zip="">............</City>
	         <Province>...............</Province>
	         <State>..................</State>
	         <Country>................</Country>
 	      </Address>
	     ...
         </Addresses>
	 <Net>
	     <Uri type="web">.............</Uri>
	     <Uri type="email">...........</Uri>
	     <Uri type="aim">.............</Uri>
	     <Uri type="msn">.............</Uri>
	     <Uri type="yahoo">...........</Uri>
	     <Uri type="icq">.............</Uri>
	     <Uri type="jabber">..........</Uri>	     
	     ...
	 </Net>
	 <TelephoneNumbers>
	     <Telephone type="home">......</Telephone>
	     <Telephone type="work">......</Telephone>
	     <Telephone type="fax">.......</Telephone>
	     ...
 	 </TelephoneNumbers>
	 <Notes married="">
             <SpouseName>.................</SpouseName>
	     <Child>......................</Child>
	     <OtherNotes>.................</OtherNotes>
	     <PublicKey>..................</PublicKey>
	     <SpouseBirthday known="" day="" month="" year=""/>
	     <Anniversary known="" day="" month="" year""/>
         </Notes>
     </Card>

     <Card                                              root child 
            name=""                                     the card name
            locked=""                                   is card locked?
            deleted=""                                  was card deleted?
            rank=""                                     card rank (1..5)
            id="ccxxx"                                  card id   (es: cc1039)
	    created=""                                  creation date
            last_change=""                              last change date
	    type="company">                             card's type 
         <Groups>                                       card child 
             <Group>......................</Group>      groups children
	     ...
         </Groups>       
	 <Refs>
	     <Ref refid="card_id">references's description</Ref>
	     ...
         </Refs>
         <Company>
             <CompanyName>.............</CompanyName>
	     <Pi>......................</Pi>
	     <Notes>...................</Notes>
         </Company>
	 <Addresses>                                    card child
             <Address type="">                          addresses children
                 <Street number="">.......</Street>     address child
	         <City zip="">............</City>
	         <Province>...............</Province>
	         <State>..................</State>
	         <Country>................</Country>
 	      </Address>
	     ...
         </Addresses>
	 <Net>
	     <Uri type="web">..........</Uri>
	     <Uri type="email">........</Uri> 
	     ...
	 </Net>
	 <TelephoneNumbers>
	     <Telephone type="green">......</Telephone>
	     <Telephone type="operator">...</Telephone>
	     <Telephone type="fax">........</Telephone>
	     ...
 	 </TelephoneNumbers>
     </Card>
     ...
  </Rubrica>


  Rubrica
     |
     | children
     |
     v   next         next         next 
   Card-------->Card-------->....------->Card
     |
     | children
     |
     v   next                
   Groups--->Data--->Addresses--->Net--->....Phone-->Work-->Notes
     |        \          |         |           |        |      |      
     | child  | child    | child   |           |        |      |     
     |        |          v         v           v        v      v     
     |        |         ...       ...         ...      ...     ...    ...   
     v        v
   Group    FirstName--->MiddleName--->...

   pcxxx => personal card xxx
   ccxxx => company  card xxx
*/


void r_write_personal_card(RCard *card, xmlNodePtr cardxml);
void r_write_company_card(RCard *card, xmlNodePtr cardxml);



/**
 * r_write_card
 * @card: a #RCard
 * @cardxml: a #xmlNodePtr
 *
 * Write the #RCard into file. This function calls the
 * virtual functions defined in #RPersonalCard and #RCompanyCard.
 */
void
r_write_card(RCard *card, xmlNodePtr cardxml)
{
  gchar* type;

  g_return_if_fail(IS_R_CARD(card));
  g_return_if_fail(cardxml != NULL);
  
  g_object_get(card, "card-type", &type, NULL);

  if (g_ascii_strcasecmp(type, "personal") == 0)
    r_write_personal_card(card, cardxml);
  else
    r_write_company_card(card, cardxml);
}


void r_write_personal_card(RCard *card, xmlNodePtr cardxml)
{
  g_return_if_fail(IS_R_CARD(card));
  g_return_if_fail(cardxml != NULL);

  r_write_infos(card, cardxml);
  r_write_contact(R_PERSONAL_CARD(card), cardxml);
  r_write_group(card, cardxml);
  r_write_refs(card, cardxml);
  r_write_work(R_PERSONAL_CARD(card), cardxml);
  r_write_addresses(card, cardxml);
  r_write_net(card, cardxml);
  r_write_telephone(card, cardxml);
  r_write_notes(R_PERSONAL_CARD(card), cardxml);    
}


void r_write_company_card(RCard *card, xmlNodePtr cardxml)
{
  g_return_if_fail(IS_R_CARD(card));
  g_return_if_fail(cardxml != NULL);

  r_write_infos(card, cardxml);
  r_write_company(R_COMPANY_CARD(card), cardxml);
  r_write_group(card, cardxml);
  r_write_refs(card, cardxml);
  r_write_addresses(card, cardxml);
  r_write_net(card, cardxml);
  r_write_telephone(card, cardxml);
}


/*  Card Infos 
*/
void 
r_write_infos(RCard *card, xmlNodePtr cardxml)
{
  glong id;
  gchar *name, *type;
  gboolean locked, deleted;
  gint created, changed;
  RRate rate;

  g_object_get(card, 
	       "card-id",      &id,      "card-type",      &type,
	       "card-name",    &name,    "card-locked",    &locked, 
	       "card-deleted", &deleted, "card-rate",      &rate, 
	       "card-created", &created, "card-changed",   &changed,
	       NULL);

  r_io_write_number (cardxml, "id",          id);
  r_io_write_str    (cardxml, "type",        type);
  r_io_write_str    (cardxml, "name",        name);
  r_io_write_bool   (cardxml, "locked",      locked);
  r_io_write_bool   (cardxml, "deleted",     deleted); 
  r_io_write_number (cardxml, "rate",        rate);
  r_io_write_time   (cardxml, "created",     created);
  r_io_write_time   (cardxml, "last_change", changed);
  
  g_free(name);
  g_free(type);
}


/* Groups
 */
void
r_write_group (RCard *card,  xmlNodePtr xmlnode)
{
  gpointer data;
  xmlNodePtr groupxml;    /* group node  */

  g_return_if_fail(IS_R_CARD(card));

  groupxml = xmlNewTextChild(xmlnode, NULL, (xmlChar*) "Groups", NULL);
  data = r_card_get_group(R_CARD(card));
  for (; data; data = r_card_get_next_group(R_CARD(card)))
    {
      xmlNodePtr grxml;
      xmlChar* name;
      xmlChar* owner;
      xmlChar* pixmap;

      g_object_get(R_GROUP(data), 
		   "group-name", &name, 
		   "group-owner", &owner, 
		   "group-pixmap", &pixmap, NULL);

      grxml = xmlNewTextChild(groupxml, NULL, (xmlChar*) "Group", name);
      r_io_write_str(grxml, "owner",  (gchar*) owner);
      r_io_write_str(grxml, "pixmap", (gchar*) pixmap);
    }
}	


void 
r_write_refs (RCard *card, xmlNodePtr xmlnode)
{
  gpointer data;
  xmlNodePtr refsxml;    /* refs node  */

  g_return_if_fail(IS_R_CARD(card));  

  refsxml = xmlNewTextChild(xmlnode, NULL, (xmlChar*) "Refs", NULL);  
  data = r_card_get_ref(R_CARD(card));
  for (; data; data = r_card_get_next_ref(R_CARD(card)))
    {
      xmlNodePtr xmlref;
      xmlChar *info;
      glong id;

      g_object_get(R_REF(data), "ref-to", &id, "ref-info", &info, NULL);

      xmlref = xmlNewTextChild(refsxml, NULL, (xmlChar*) "Ref", info);
      r_io_write_number(xmlref, "refto", id);      
    }  
}


/*    Addresses
 */
void
r_write_addresses (RCard *card, xmlNodePtr xmlnode)
{
  gpointer data;
  xmlNodePtr addressxml;  /* addresses node, address's father */ 

  g_return_if_fail(IS_R_CARD(card));

  addressxml = xmlNewTextChild(xmlnode, NULL, (xmlChar*) "Addresses", NULL);
  data = r_card_get_address(R_CARD(card));
  for (; data; data = r_card_get_next_address(R_CARD(card)))
    {
      if (IS_R_ADDRESS(data))
	{
	  xmlNodePtr adsxml;      /* address node, addresses's child  */
	  xmlNodePtr node;        /* address's child  */
	  RAddressType adtype = R_ADDRESS_UNKNOWN;
	  gchar *type, *street, *number, *city, *zip;
	  gchar *province, *state, *country;

	  g_object_get(R_ADDRESS(data), "address-type", &adtype,  
		       "street", &street, "street-number", &number,  
		       "city", &city, "zip", &zip, "province", &province, 
		       "state", &state, "country", &country, NULL);
	  
	  if ((adtype < R_ADDRESS_PREF) || 
	      (adtype > R_ADDRESS_INVALID))
	    adtype = R_ADDRESS_UNKNOWN;

	  type = r_address_lookup_enum2str(adtype);
	  
	  adsxml = xmlNewTextChild(addressxml, NULL, 
				   (xmlChar*) "Address", NULL);

	  r_io_write_str(adsxml, "type", type);

	  node = xmlNewTextChild(adsxml, NULL, 
				 (xmlChar*) "Street", (xmlChar*) street);
	  r_io_write_str(node, "number", number);

	  node = xmlNewTextChild(adsxml, NULL, 
				 (xmlChar*) "City", (xmlChar*) city);
	  r_io_write_str(node, "zip", zip);

	  xmlNewTextChild(adsxml, NULL, 
			  (xmlChar*) "Province", (xmlChar*) province);
	  xmlNewTextChild(adsxml, NULL, 
			  (xmlChar*) "State", (xmlChar*) state);
	  xmlNewTextChild(adsxml, NULL, 
			  (xmlChar*) "Country", (xmlChar*) country);
	}
    }
}


/*      NetAddresses
 */
void
r_write_net (RCard *card, xmlNodePtr xmlnode)
{
  gpointer data;
  xmlNodePtr netxml;

  g_return_if_fail(IS_R_CARD(card));

  netxml = xmlNewTextChild(xmlnode, NULL, (xmlChar*) "Net", NULL);
  data = r_card_get_net_address(R_CARD(card));
  for (; data; data = r_card_get_next_net_address(R_CARD(card)))
    if (IS_R_NET_ADDRESS(data))
      {
	RNetAddressType nettype;
	xmlNodePtr uri;
	gchar *url, *type; 

	g_object_get(R_NET_ADDRESS(data), "url", &url, 
		     "url-type", &nettype, NULL);

	uri = xmlNewTextChild(netxml, NULL, (xmlChar*) "Uri", (xmlChar*) url);
	type = r_net_address_decode_type(nettype);
	r_io_write_str(uri, "type", type);
      }
}  


/*      Telephone numbers
 */
void
r_write_telephone (RCard *card, xmlNodePtr xmlnode)
{
  gpointer data;
  xmlNodePtr telxml;

  g_return_if_fail(IS_R_CARD(card));

  telxml = xmlNewTextChild(xmlnode, NULL, (xmlChar*) "TelephoneNumbers", NULL);
  data = r_card_get_telephone(R_CARD(card));
  for (; data; data = r_card_get_next_telephone(R_CARD(card)))
    if (IS_R_TELEPHONE(data))
      {
	xmlNodePtr tel;
	gchar *number, *type; 
	RTelephoneType ttype;
	
	g_object_get(R_TELEPHONE(data), "telephone-number", &number, 
		     "telephone-type", &ttype, NULL);

	tel = xmlNewTextChild(telxml, NULL,
			      (xmlChar*) "Telephone", (xmlChar*) number);
	type = r_telephone_lookup_enum2str(ttype);

	r_io_write_str(tel, "type", type);
      }
}


/*    Personal data
 */
void
r_write_contact (RPersonalCard *card, xmlNodePtr xmlnode)
{
  gpointer data;

  g_return_if_fail(IS_R_CARD(card));

  data = r_personal_card_get_contact(R_PERSONAL_CARD(card));
  if (IS_R_CONTACT(data))
    {
      xmlNodePtr contact;
      xmlNodePtr node;  
      gboolean bool;
      xmlChar *first, *middle, *last, *nick;
      xmlChar *prof, *prefix, *genre, *title, *photo;
      gchar *day, *month, *year; 
      
      g_object_get(R_CONTACT(data), 
		   "first-name", &first, 
		   "middle-name", &middle, "last-name", &last, 
		   "nick-name", &nick, "profession", &prof, 
		   "prefix", &prefix, "genre", &genre, 
		   "title", &title, "photo", &photo, NULL);

      day   = r_contact_get_birth_day   (R_CONTACT(data));
      month = r_contact_get_birth_month (R_CONTACT(data));
      year  = r_contact_get_birth_year  (R_CONTACT(data));

      contact = xmlNewTextChild(xmlnode, NULL, (xmlChar*) "Data", NULL); 

      xmlNewTextChild(contact, NULL, (xmlChar*) "FirstName",  first);
      xmlNewTextChild(contact, NULL, (xmlChar*) "MiddleName", middle);
      xmlNewTextChild(contact, NULL, (xmlChar*) "LastName",   last);
      xmlNewTextChild(contact, NULL, (xmlChar*) "NickName",   nick);
      xmlNewTextChild(contact, NULL, (xmlChar*) "Genre",      genre);
      xmlNewTextChild(contact, NULL, (xmlChar*) "Profession", prof);
      xmlNewTextChild(contact, NULL, (xmlChar*) "NamePrefix", prefix);
      xmlNewTextChild(contact, NULL, (xmlChar*) "Title",      title);

      node = xmlNewTextChild(contact, NULL, (xmlChar*) "Photo", photo);
      r_io_write_str(node, "type", "url");
      
      node = xmlNewTextChild(contact, NULL, (xmlChar*) "Birthday", NULL);
      bool = (g_ascii_strcasecmp(day, "BadDay") != 0);
      r_io_write_bool(node, "known", bool);     
      r_io_write_str (node, "day",   day);
      r_io_write_str (node, "month", month);
      r_io_write_str (node, "year",  year); 
    }
}


/*    Work data
 */
void
r_write_work (RPersonalCard *card, xmlNodePtr xmlnode)
{
  gpointer data;

  g_return_if_fail(IS_R_CARD(card));

  data = r_personal_card_get_work(R_PERSONAL_CARD(card));
  if (IS_R_WORK(data))
    {     
      xmlNodePtr work;
      xmlChar *assignment, *org, *dep, *subdep;
      xmlChar *manager, *mphone, *collaborator, *cphone;

      g_object_get(R_WORK(data), 
		   "assignment",     &assignment, 
		   "organization",   &org, 
		   "department",     &dep, 
		   "sub-department", &subdep, 
		   "manager-name",   &manager, 
		   "manager-phone",  &mphone, 
		   "collaborator",   &collaborator, 
		   "collaborator-phone", &cphone, NULL);

      work = xmlNewTextChild(xmlnode, NULL, (xmlChar*) "Work", NULL);
       
      xmlNewTextChild(work, NULL, (xmlChar*) "Assignment",       assignment);
      xmlNewTextChild(work, NULL, (xmlChar*) "Organization",     org);
      xmlNewTextChild(work, NULL, (xmlChar*) "Department",       dep);
      xmlNewTextChild(work, NULL, (xmlChar*) "SubDepartment",    subdep);
      xmlNewTextChild(work, NULL, (xmlChar*) "ManagerName",      manager);
      xmlNewTextChild(work, NULL, (xmlChar*) "ManagerPhone",     mphone);
      xmlNewTextChild(work, NULL, (xmlChar*) "CollaboratorName", collaborator);
      xmlNewTextChild(work, NULL, (xmlChar*) "CollaboratorPhone", cphone);
    }
}

  
void
r_write_notes (RPersonalCard *card, xmlNodePtr xmlnode)
{
  gpointer data;

  g_return_if_fail(IS_R_CARD(card));

  data = r_personal_card_get_notes(R_PERSONAL_CARD(card));
  if (IS_R_NOTES(data))
    {
      xmlNodePtr notes;
      xmlNodePtr node;

      gboolean has_partner, birth, anniversary;
      xmlChar *partner, *other, *pkey;
      gchar *bday, *bmonth, *byear;
      gchar *aday, *amonth, *ayear;

      birth       = r_notes_know_birthday(R_NOTES(data));
      anniversary = r_notes_know_anniversary(R_NOTES(data));

      bday   = r_notes_get_birth_day(R_NOTES(data));
      bmonth = r_notes_get_anniversary_month(R_NOTES(data));
      byear  = r_notes_get_birth_year(R_NOTES(data));
 
      aday   = r_notes_get_anniversary_day (R_NOTES(data));
      amonth = r_notes_get_anniversary_month(R_NOTES(data));
      ayear  = r_notes_get_anniversary_year(R_NOTES(data));

      g_object_get(R_NOTES(data), 
		   "has-partner",  &has_partner, 
		   "partner-name", &partner, 
		   "other-notes",  &other, 
		   "pubkey",       &pkey, NULL);
		   
      notes = xmlNewTextChild(xmlnode, NULL, (xmlChar*) "Notes", NULL);

      r_io_write_bool(notes, "partner", has_partner);
      
      xmlNewTextChild(notes, NULL, (xmlChar*) "PartnerName", partner);
      xmlNewTextChild(notes, NULL, (xmlChar*) "OtherNotes",  other);
      xmlNewTextChild(notes, NULL, (xmlChar*) "PublicKey",   pkey);

      node = xmlNewTextChild(notes, NULL, (xmlChar*) "PartnerBirthday", NULL);
      r_io_write_bool(node, "known", birth);
      r_io_write_str (node, "day",   bday);
      r_io_write_str (node, "month", bmonth);
      r_io_write_str (node, "year",  byear);

      node = xmlNewTextChild(notes, NULL, (xmlChar*) "Anniversary", NULL);
      r_io_write_bool(node, "known", anniversary);
      r_io_write_str (node, "day",   aday);
      r_io_write_str (node, "month", amonth);
      r_io_write_str (node, "year",  ayear);
    }
}


void
r_write_company (RCompanyCard *card, xmlNodePtr xmlnode)
{
  xmlNodePtr company;
  xmlChar *name, *logo, *vat, *notes;
  
  g_object_get(R_COMPANY_CARD(card), "company-name", &name, 
	       "company-logo", &logo, "company-vat", &vat, 
	       "company-notes", &notes, NULL);
  
  company = xmlNewTextChild(xmlnode, NULL, (xmlChar*) "Company", NULL);
  xmlNewTextChild(company, NULL, (xmlChar*) "CompanyName", name);
  xmlNewTextChild(company, NULL, (xmlChar*) "Logo",        logo);
  xmlNewTextChild(company, NULL, (xmlChar*) "VAT",         vat);
  xmlNewTextChild(company, NULL, (xmlChar*) "Notes",       notes);
}




