/*
*  RAL -- Rubrica Addressbook Library
*  file: csv.c
*  
*  Copyright (C) Nicola Fragale <nicolafragale@libero.it>
*
*  This program is free software; you can redistribute it and/or modify
*  it under the terms of the GNU General Public License as published by
*  the Free Software Foundation; either version 3 of the License
*
*  This program is distributed in the hope that it will be useful,
*  but WITHOUT ANY WARRANTY; without even the implied warranty of
*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*  GNU General Public License for more details.
*
*  You should have received a copy of the GNU General Public License
*  along with this program; if not, write to the Free Software
*  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#include <stdio.h>
#include <glib.h>
#include <glib/gstdio.h>
#include <glib-object.h>
#include <glib/gi18n-lib.h>

#include "csv_thunderbird.h"
#include "abook.h"
#include "csv_engine.h"
#include "error.h"
#include "plugin.h"
#include "card.h"
#include "personal.h"
#include "company.h"
#include "contact.h"
#include "work.h"
#include "notes.h"
#include "address.h"
#include "net.h"
#include "telephone.h"
#include "lookup.h"



/* Thunderbird's csv fields:

  First name, Last name, Display name, Nickname,           (4)
  First email, Second email,                               (2)
  Work telephone, Home Telephone, Fax, Pager, Cellphone    (5)
  Home address, Home address 2, city, state, zip, country  (6)
  Work address, Work address 2, city, state, zip, country  (6)
  Title, department, organization,                         (3)
  web 1, web 2                                             (2)
  birth year, birth month, birth day,                      (3)
  user defined 1, user def 2, user def 3, user def 4,      (4)
  notes,                                                   (1)
                                                          -----
							   36

  Nome,Cognome,Nome visualizzato,Soprannome,
  Email primaria,Email secondaria,
  Telefono ufficio,Telefono casa,fax,Numero cercapersone,Numero cellulare,
  Indirizzo di casa,Indirizzo di casa 2,Città di residenza,
  Provincia di residenza,CAP di residenza,Nazione di residenza,
  Indirizzo di lavoro,Indirizzo di lavoro 2,Città di lavoro,
  Provincia di lavoro,CAP di lavoro,Nazione di lavoro,
  Qualifica,Dipartimento,Organizzazione,
  Pagina web 1,Pagina web 2,
  Anno di nascita,Mese di nascita,Giorno di nascita,
  Personalizzato  1,Personalizzato  2,Personalizzato  3,Personalizzato 4,
  Note,
*/

typedef enum {
  CSV_THUNDERBIRD_FIRST_NAME = 0,           // 1°
  CSV_THUNDERBIRD_LAST_NAME,                // 2
  CSV_THUNDERBIRD_DISPLAY_NAME,             // 3
  CSV_THUNDERBIRD_NICKNAME,                 // 4
  CSV_THUNDERBIRD_EMAIL1,                   // 5
  CSV_THUNDERBIRD_EMAIL2,                   // 6
  CSV_THUNDERBIRD_TEL_WORK,                 // 7
  CSV_THUNDERBIRD_TEL_HOME,                 // 8
  CSV_THUNDERBIRD_TEL_FAX,                  // 9
  CSV_THUNDERBIRD_TEL_PAGER,                // 10
  CSV_THUNDERBIRD_TEL_MOBILE,               // 11
  CSV_THUNDERBIRD_HOME_STREET,              // 12
  CSV_THUNDERBIRD_HOME_STREET_MORE,         // 13
  CSV_THUNDERBIRD_HOME_CITY,                // 14
  CSV_THUNDERBIRD_HOME_STATE,               // 15
  CSV_THUNDERBIRD_HOME_ZIP,                 // 16
  CSV_THUNDERBIRD_HOME_COUNTRY,             // 17
  CSV_THUNDERBIRD_WORK_STREET,              // 18
  CSV_THUNDERBIRD_WORK_STREET_MORE,         // 19
  CSV_THUNDERBIRD_WORK_CITY,                // 20
  CSV_THUNDERBIRD_WORK_STATE,               // 21
  CSV_THUNDERBIRD_WORK_ZIP,                 // 22
  CSV_THUNDERBIRD_WORK_COUNTRY,             // 23
  CSV_THUNDERBIRD_TITLE,                    // 24
  CSV_THUNDERBIRD_DEPARTMENT,               // 25
  CSV_THUNDERBIRD_ORGANIZATION,             // 26
  CSV_THUNDERBIRD_WORK_WEB,                 // 27
  CSV_THUNDERBIRD_WEB,                      // 28
  CSV_THUNDERBIRD_BIRTH_YEAR,               // 29
  CSV_THUNDERBIRD_BIRTH_MONTH,              // 30
  CSV_THUNDERBIRD_BIRTH_DAY,                // 31
  CSV_THUNDERBIRD_CUSTOM1,                  // 32
  CSV_THUNDERBIRD_CUSTOM2,                  // 33
  CSV_THUNDERBIRD_CUSTOM3,                  // 34
  CSV_THUNDERBIRD_CUSTOM4,                  // 35
  CSV_THUNDERBIRD_NOTE,                     // 36
  CSV_LAST
} RCsvThunderbirdField;


RLookupTable look [] ={
  {"first name",    N_("first name"),    CSV_THUNDERBIRD_FIRST_NAME       },
  {"last  name",    N_("last  name"),    CSV_THUNDERBIRD_LAST_NAME        },
  {"display name",  N_("display name"),  CSV_THUNDERBIRD_DISPLAY_NAME     },
  {"nickname",      N_("nickname"),      CSV_THUNDERBIRD_NICKNAME         },
  {"email 1",       N_("email 1"),       CSV_THUNDERBIRD_EMAIL1           },
  {"email 2",       N_("email 2"),       CSV_THUNDERBIRD_EMAIL2           },
  {"tel work",      N_("tel work"),      CSV_THUNDERBIRD_TEL_WORK         },
  {"tel home",      N_("tel home"),      CSV_THUNDERBIRD_TEL_HOME         },
  {"tel fax",       N_("tel fax"),       CSV_THUNDERBIRD_TEL_FAX          }, 
  {"tel pager",     N_("tel pager"),     CSV_THUNDERBIRD_TEL_PAGER        },
  {"tel mobile",    N_("tel mobile"),    CSV_THUNDERBIRD_TEL_MOBILE       },
  {"home street",   N_("home street"),   CSV_THUNDERBIRD_HOME_STREET      },
  {"home street 2", N_("home street 2"), CSV_THUNDERBIRD_HOME_STREET_MORE },
  {"home city",     N_("home city"),     CSV_THUNDERBIRD_HOME_CITY        },
  {"home state",    N_("home state"),    CSV_THUNDERBIRD_HOME_STATE       },
  {"home zip",      N_("home zip"),      CSV_THUNDERBIRD_HOME_ZIP         },
  {"home country",  N_("home country"),  CSV_THUNDERBIRD_HOME_COUNTRY     },
  {"work street",   N_("work street"),   CSV_THUNDERBIRD_WORK_STREET      },
  {"work street 2", N_("work street 2"), CSV_THUNDERBIRD_WORK_STREET_MORE },
  {"work city",     N_("work city"),     CSV_THUNDERBIRD_WORK_CITY        },
  {"work state",    N_("work state"),    CSV_THUNDERBIRD_WORK_STATE       },
  {"work zip",      N_("work zip"),      CSV_THUNDERBIRD_WORK_ZIP         },
  {"work country",  N_("work country"),  CSV_THUNDERBIRD_WORK_COUNTRY     },
  {"title",         N_("title"),         CSV_THUNDERBIRD_TITLE            },
  {"department",    N_("department"),    CSV_THUNDERBIRD_DEPARTMENT       },
  {"org",           N_("org"),           CSV_THUNDERBIRD_ORGANIZATION     },
  {"web work",      N_("web work"),      CSV_THUNDERBIRD_WORK_WEB         },
  {"web",           N_("web"),           CSV_THUNDERBIRD_WEB              },
  {"birth year",    N_("birth year"),    CSV_THUNDERBIRD_BIRTH_YEAR       },
  {"birth month",   N_("birth month"),   CSV_THUNDERBIRD_BIRTH_MONTH      },
  {"birth day",     N_("birth day"),     CSV_THUNDERBIRD_BIRTH_DAY        },
  {"custom 1",      N_("custom 1"),      CSV_THUNDERBIRD_CUSTOM1          },
  {"custom 2",      N_("custom 2"),      CSV_THUNDERBIRD_CUSTOM2          },
  {"custom 3",      N_("custom 3"),      CSV_THUNDERBIRD_CUSTOM3          },
  {"custom 4",      N_("custom 4"),      CSV_THUNDERBIRD_CUSTOM4          },
  {"note",          N_("note"),          CSV_THUNDERBIRD_NOTE             },
};


#define R_CSV_THUNDERBIRD_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE((o),  \
			                  R_CSV_THUNDERBIRD_TYPE,           \
                                          RCsvThunderbirdPrivate))


struct _RCsvThunderbirdPrivate{
  gboolean dispose_has_run;
};


static void   r_csv_thunderbird_init       (RCsvThunderbird* obj);
static void   r_csv_thunderbird_class_init (RCsvThunderbirdClass* klass);

static void   r_csv_thunderbird_dispose    (GObject* obj);
static void   r_csv_thunderbird_finalize   (GObject* obj);





/* private functions
 */

static void     fill_card          (RCsvEngine* engine, GList* records, 
				    gpointer data);
static void     write_card         (RCard* card, FILE* fp);
static gboolean is_csv_thunderbird_doc (const gchar* filename);


static gint     r_csv_thunderbird_open_file      (RAbook* abook, 
						  gchar* filename);
static gboolean r_csv_thunderbird_save_file      (RAbook* abook, 
						  gchar* filename);
static gboolean r_csv_thunderbird_overwrite_file (RAbook* abook);




GType
r_csv_thunderbird_get_type()
{
  static GType r_csv_thunderbird_type = 0;
  
  if (!r_csv_thunderbird_type)
    {
      static const GTypeInfo r_csv_thunderbird_info =
	{
	  sizeof(RCsvThunderbirdClass),
	  NULL,
	  NULL,
	  (GClassInitFunc) r_csv_thunderbird_class_init,
	  NULL,
	  NULL,
	  sizeof(RCsvThunderbird),
	  0,
	  (GInstanceInitFunc) r_csv_thunderbird_init
	};

      r_csv_thunderbird_type = g_type_register_static(R_CSV_ENGINE_TYPE, 
						      "RCsvThunderbird",
						      &r_csv_thunderbird_info,
						      0);
    }
  
  return r_csv_thunderbird_type;
}


static void
r_csv_thunderbird_class_init(RCsvThunderbirdClass* klass)
{
  GObjectClass *class;
  
  class = G_OBJECT_CLASS (klass);

  class->dispose  = (GObjectFinalizeFunc) r_csv_thunderbird_dispose;
  class->finalize = (GObjectFinalizeFunc) r_csv_thunderbird_finalize;

  g_type_class_add_private (klass, sizeof(RCsvThunderbirdPrivate));
}


static void
r_csv_thunderbird_init(RCsvThunderbird* self)
{
  RCsvThunderbirdPrivate* priv;

  g_return_if_fail(IS_R_CSV_THUNDERBIRD(self));
  
  priv = R_CSV_THUNDERBIRD_GET_PRIVATE(self);
  priv->dispose_has_run = FALSE;
}


static void 
r_csv_thunderbird_dispose (GObject* obj)
{
  RCsvThunderbird* self = R_CSV_THUNDERBIRD(obj);
  RCsvThunderbirdPrivate* priv;
 
  g_return_if_fail(IS_R_CSV_THUNDERBIRD(self));

  priv = R_CSV_THUNDERBIRD_GET_PRIVATE(self);
  if (priv->dispose_has_run)
    return;
  
  priv->dispose_has_run = TRUE;
}


static void 
r_csv_thunderbird_finalize (GObject* obj)
{
  RCsvThunderbird* self = R_CSV_THUNDERBIRD(obj);

  g_return_if_fail(IS_R_CSV_THUNDERBIRD(self));
  
  g_object_unref(obj);
}


static void 
fill_card (RCsvEngine* engine, GList* records, gpointer data)
{
  RAbook* book = (RAbook*) data;
  RPersonalCard* card    = NULL;
  RContact*      contact = NULL;
  RAddress*      address = NULL;
  RWork*         work    = NULL;
  RNotes*        notes   = NULL;
  RNetAddress*   net     = NULL;
  RTelephone*    phone   = NULL;
  gchar *card_name = NULL;
  gchar *fn = NULL, *ln = NULL, *nick = NULL;
  gchar *str = NULL, *smore = NULL, *city = NULL;
  gchar *zip = NULL, *state = NULL, *country = NULL;
  gchar *wstr = NULL, *wsmore = NULL, *wcity = NULL;
  gchar *wstate = NULL, *wzip = NULL, *wcountry = NULL;
  gchar *email1 = NULL, *email2 = NULL;
  gchar *web = NULL, *wweb = NULL;
  gchar *wtel = NULL, *htel = NULL, *fax = NULL;
  gchar *pager = NULL, *mobile = NULL;
  gchar *role = NULL, *dep = NULL, *org = NULL;
  gchar *note = NULL;
  gchar *year, *month, *day;
  gchar *custom1, *custom2, *custom3, *custom4;

  gchar* street;

  fn        = (gchar*) g_list_nth_data(records, CSV_THUNDERBIRD_FIRST_NAME);
  ln        = (gchar*) g_list_nth_data(records, CSV_THUNDERBIRD_LAST_NAME);
  card_name = (gchar*) g_list_nth_data(records, CSV_THUNDERBIRD_DISPLAY_NAME);
  nick      = (gchar*) g_list_nth_data(records, CSV_THUNDERBIRD_NICKNAME);
  email1    = (gchar*) g_list_nth_data(records, CSV_THUNDERBIRD_EMAIL1);
  email2    = (gchar*) g_list_nth_data(records, CSV_THUNDERBIRD_EMAIL2);
  wtel      = (gchar*) g_list_nth_data(records, CSV_THUNDERBIRD_TEL_WORK);
  htel      = (gchar*) g_list_nth_data(records, CSV_THUNDERBIRD_TEL_HOME);
  fax       = (gchar*) g_list_nth_data(records, CSV_THUNDERBIRD_TEL_FAX);
  pager     = (gchar*) g_list_nth_data(records, CSV_THUNDERBIRD_TEL_PAGER);
  mobile    = (gchar*) g_list_nth_data(records, CSV_THUNDERBIRD_TEL_MOBILE);
  str       = (gchar*) g_list_nth_data(records, CSV_THUNDERBIRD_HOME_STREET);
  smore     = (gchar*) g_list_nth_data(records, 
				       CSV_THUNDERBIRD_HOME_STREET_MORE);
  city      = (gchar*) g_list_nth_data(records, CSV_THUNDERBIRD_HOME_CITY);
  state     = (gchar*) g_list_nth_data(records, CSV_THUNDERBIRD_HOME_STATE);
  zip       = (gchar*) g_list_nth_data(records, CSV_THUNDERBIRD_HOME_ZIP);
  country   = (gchar*) g_list_nth_data(records, CSV_THUNDERBIRD_HOME_COUNTRY);
  wstr      = (gchar*) g_list_nth_data(records, CSV_THUNDERBIRD_WORK_STREET);
  wsmore    = (gchar*) g_list_nth_data(records, 
				       CSV_THUNDERBIRD_WORK_STREET_MORE);
  wcity     = (gchar*) g_list_nth_data(records, CSV_THUNDERBIRD_WORK_CITY);
  wstate    = (gchar*) g_list_nth_data(records, CSV_THUNDERBIRD_WORK_STATE);
  wzip      = (gchar*) g_list_nth_data(records, CSV_THUNDERBIRD_WORK_ZIP);
  wcountry  = (gchar*) g_list_nth_data(records, CSV_THUNDERBIRD_WORK_COUNTRY);
  role      = (gchar*) g_list_nth_data(records, CSV_THUNDERBIRD_TITLE);
  dep       = (gchar*) g_list_nth_data(records, CSV_THUNDERBIRD_DEPARTMENT);
  org       = (gchar*) g_list_nth_data(records, CSV_THUNDERBIRD_ORGANIZATION);
  wweb      = (gchar*) g_list_nth_data(records, CSV_THUNDERBIRD_WORK_WEB);
  web       = (gchar*) g_list_nth_data(records, CSV_THUNDERBIRD_WEB);
  year      = (gchar*) g_list_nth_data(records, CSV_THUNDERBIRD_BIRTH_YEAR);
  month     = (gchar*) g_list_nth_data(records, CSV_THUNDERBIRD_BIRTH_MONTH);
  day       = (gchar*) g_list_nth_data(records, CSV_THUNDERBIRD_BIRTH_DAY);
  custom1   = (gchar*) g_list_nth_data(records, CSV_THUNDERBIRD_CUSTOM1);
  custom2   = (gchar*) g_list_nth_data(records, CSV_THUNDERBIRD_CUSTOM2);
  custom3   = (gchar*) g_list_nth_data(records, CSV_THUNDERBIRD_CUSTOM3);
  custom4   = (gchar*) g_list_nth_data(records, CSV_THUNDERBIRD_CUSTOM4);
  note      = (gchar*) g_list_nth_data(records, CSV_THUNDERBIRD_NOTE);
  

  card      = r_personal_card_new();
  contact   = r_contact_new();
  work      = r_work_new();
  notes     = r_notes_new();

  /* card, personal data */
  g_object_set(card, "card-name", card_name, NULL);
  g_object_set(contact, "first-name", fn, "last-name", ln, "nick-name", nick, 
	       NULL);

  /* work */
  g_object_set(work, 
	       "assignment", role, "organization", org,
	       "department", dep, NULL);
  r_personal_card_set_work(R_PERSONAL_CARD(card), work);

  /* net (web, emails) */
  net = r_net_address_new();
  g_object_set(net, "url", web, "url-type", R_NET_ADDRESS_WEB, NULL);  
  r_card_add_net_address(R_CARD(card), net);

  net = r_net_address_new();
  g_object_set(net, "url", wweb, "url-type", R_NET_ADDRESS_WORK_WEB, NULL);  
  r_card_add_net_address(R_CARD(card), net);
  
  net = r_net_address_new();
  g_object_set(net, "url", email1, "url-type", R_NET_ADDRESS_EMAIL, NULL);  
  r_card_add_net_address(R_CARD(card), net);

  net = r_net_address_new();
  g_object_set(net, "url", email2, "url-type", R_NET_ADDRESS_EMAIL, NULL);  
  r_card_add_net_address(R_CARD(card), net);
 
  /* addresses (home, work) */
  address = r_address_new();
  street = g_strdup_printf("%s %s", str, smore);
  g_object_set(address, "address-type", R_ADDRESS_HOME,
	       "street", street, "city", city, "zip", zip, 
	       "province", state, "country", country, NULL);
  r_card_add_address(R_CARD(card), address);
  g_free(street);
	       
  address = r_address_new();
  street = g_strdup_printf("%s %s", wstr, wsmore);
  g_object_set(address, "address-type", R_ADDRESS_WORK,
	       "street", street, "city", wcity, "zip", wzip, 
	       "province", wstate, "country", wcountry, NULL);
  r_card_add_address(R_CARD(card), address);
 
  /* telephones */
  phone = r_telephone_new();
  g_object_set(phone, "telephone-number", htel,
	       "telephone-type", R_TELEPHONE_HOME, NULL);
  r_card_add_telephone(R_CARD(card), phone);

  phone = r_telephone_new();
  g_object_set(phone, "telephone-number", wtel,
	       "telephone-type", R_TELEPHONE_WORK, NULL);
  r_card_add_telephone(R_CARD(card), phone);

  phone = r_telephone_new();
  g_object_set(phone, "telephone-number", fax,
	       "telephone-type", R_TELEPHONE_FAX, NULL);
  r_card_add_telephone(R_CARD(card), phone);

  phone = r_telephone_new();
  g_object_set(phone, "telephone-number", mobile,
	       "telephone-type", R_TELEPHONE_CELLPHONE, NULL);
  r_card_add_telephone(R_CARD(card), phone);

  phone = r_telephone_new();
  g_object_set(phone, "telephone-number", pager,
	       "telephone-type", R_TELEPHONE_PAGER, NULL);
  r_card_add_telephone(R_CARD(card), phone);

  /* notes */
  g_object_set(notes, "other-notes", note, NULL);
  r_notes_set_birthday(notes, atoi(day), atoi(month), atoi(year));

  r_card_add_address(R_CARD(card), address);
  r_personal_card_set_contact(card, contact);
  r_personal_card_set_notes(card, notes);
	       
  r_abook_add_loaded_card (R_ABOOK(book), R_CARD(card));
  g_signal_emit_by_name(R_ABOOK(book), "card_read", card, G_TYPE_POINTER);
}



static gboolean 
is_csv_thunderbird_doc (const gchar* filename)
{
  GIOChannel* channel = NULL;
  GError* error = NULL;
  gchar* buffer;
  gchar** splits;
  gchar** alias;
  gint i = 0;

  channel = g_io_channel_new_file(filename, "r", &error);
  if (!channel)
    {
      g_warning("Channel: %s", error->message);
     
      g_io_channel_shutdown(channel, TRUE, NULL);
      g_io_channel_unref(channel);
      return FALSE;
    }

  if (g_io_channel_set_encoding(channel, "utf8", &error) == G_IO_STATUS_ERROR)
    {
      //      g_warning("Channel: %s", error->message);

      g_io_channel_shutdown(channel, TRUE, NULL);
      g_io_channel_unref(channel);
      return FALSE;
    }
  
  if (g_io_channel_read_line(channel, &buffer, NULL, NULL, &error) == 
      G_IO_STATUS_ERROR)
    {
      //      g_warning("Channel: %s", error->message);

      g_io_channel_shutdown(channel, TRUE, NULL);
      g_io_channel_unref(channel);
      g_free(buffer);
      return FALSE;
    }

  alias = splits = g_strsplit(buffer, ",", -1);
  if (!splits)
    {
      g_io_channel_shutdown(channel, TRUE, NULL);
      g_io_channel_unref(channel);
      g_free(buffer);
      return FALSE;
    }
  
  for(; *alias; alias++)
    i++;

  g_io_channel_shutdown(channel, TRUE, NULL);
  g_io_channel_unref(channel);
  g_strfreev(splits);
  g_free(buffer);
  
  if (i != CSV_LAST)
    return FALSE;
  
  return TRUE;
}

/* *****************************  Public functions  ***********************
 */

/**
 * r_csv_thunderbird_new
 *
 * create a new #RCsvThunderbird
 *
 * returns: a #RCsvThunderbird*
 */
RCsvThunderbird* 
r_csv_thunderbird_new(void)
{
  return (RCsvThunderbird*) g_object_new(r_csv_thunderbird_get_type(), NULL);
}


/**
 * r_csv_thunderbird_free
 * @csv: a #RCsvThunderbird
 * 
 * free the #RCsvThunderbird object
 */
void  
r_csv_thunderbird_free(RCsvThunderbird *csv) 
{ 
  g_return_if_fail(IS_R_CSV_THUNDERBIRD(csv)); 

  g_object_unref(csv); 
} 


/**
 * r_csv_thunderbird_open_file
 * @csv: #RCsvThunderbird
 * @file: the file's name
 *
 * Read a csv file. 
 * When a csv file is open a "file_opened" signal is emitted, 
 * with an error code on file status.
 * Each time a csv's record is decoded, a "card_decoded" signal is 
 * emitted and a #RPersonalCard will be sent to the user callback
 * function. The "addressbook_decoded" signal is emitted when csv file
 * if successfully read.
 *
 * returns: a gint, %FILE_OPEN_FAIL or %FILE_IMPORT_SUCCESS
 */
static gboolean          
r_csv_thunderbird_open_file (RAbook* abook, gchar* filename)
{
  RCsvThunderbird* csv;

  g_return_val_if_fail(IS_R_ABOOK(abook), FALSE);

  if (!filename)
    {
      g_signal_emit_by_name(R_ABOOK(abook), "open_fail", 
			    NO_FILENAME, G_TYPE_INT);
      
      return FALSE;
    }
  
  if (!g_file_test(filename, G_FILE_TEST_EXISTS))
    {
      g_signal_emit_by_name(R_ABOOK(abook), "open_fail", 
			    FILE_NOT_EXIST, G_TYPE_INT);
      
      return FALSE;
    }
  
  csv = (RCsvThunderbird*) r_abook_get_engine(abook);

  if (!IS_R_CSV_THUNDERBIRD(csv))
    return FALSE;
  
  if (!is_csv_thunderbird_doc(filename))
    return FALSE;

  g_object_set(R_CSV_ENGINE(csv), "encode", "utf8", NULL);

  g_signal_connect(G_OBJECT(csv), "record_decoded", 
		   G_CALLBACK(fill_card), (gpointer) abook);
  
  if (r_csv_engine_read_file(R_CSV_ENGINE(csv), filename))
    {
      g_object_set(R_ABOOK(abook),
		   "addressbook-name", g_path_get_basename(filename),
		   "addressbook-path", g_path_get_dirname(filename),
		   NULL);
      
      g_signal_emit_by_name(R_ABOOK(abook), "addressbook_read", 
			    NULL, G_TYPE_NONE);
  
      return TRUE;
    }
  
  return FALSE;
}


static void
write_card (RCard* card, FILE* fp)
{
  RNotes*   notes;
  RAddress* address;
  RNetAddress* net;
  RTelephone* tel;

  gchar *type, *dn;
  gchar *fn = NULL, *ln = NULL, *nick = NULL;
  gchar *str = NULL, *city = NULL, *state = NULL;
  gchar *zip = NULL, *country = NULL;
  gchar *wstr = NULL, *wcity = NULL, *wstate = NULL;
  gchar *wzip = NULL, *wcountry = NULL;
  gchar *email1 = NULL, *email2 = NULL;
  gchar *web = NULL, *wweb = NULL;
  gchar *wtel = NULL, *htel = NULL, *fax = NULL;
  gchar *pager = NULL, *mobile = NULL;
  gchar *role = NULL, *dep = NULL, *org = NULL;
  gchar *year = NULL, *month = NULL, *day = NULL;
  gchar *note = NULL;

  g_object_get(card, "card-type", &type, "card-name", &dn, NULL);
  if (g_ascii_strcasecmp(type, "personal") == 0)
    {
      RContact* contact;
      RWork* work;
  
      contact = r_personal_card_get_contact(R_PERSONAL_CARD(card));
      work    = r_personal_card_get_work(R_PERSONAL_CARD(card));
      notes   = r_personal_card_get_notes(R_PERSONAL_CARD(card));
      
      if (contact)
	g_object_get(contact, 
		     "first-name", &fn,
		     "last-name",  &ln, 
		     "nick-name",  &nick, NULL);
      if (work)
	g_object_get(work, 
		     "assignment", &role, 
		     "department", &dep, 
		     "organization", &org, NULL);
      if (notes)
	{
	  g_object_get(notes, "other-notes", &note, NULL);   
	  day   = r_notes_get_birth_day   (notes);
	  month = r_notes_get_birth_month (notes);
	  year  = r_notes_get_birth_year  (notes);
	}
    }
  else // company
    {
      g_object_get(R_COMPANY_CARD(card), "company-notes", &note, NULL);
    }
        
  /* reset iter */
  r_card_reset_address(card);
  
  /* find address of type: R_ADDRESS_HOME */
  address = r_card_find_address(card, R_ADDRESS_HOME);
  if (address)
    g_object_get(address, 
		 "street",  &str,
		 "city",    &city,
		 "state",   &state,
		 "zip",     &zip,
		 "country", &country,
		 NULL);
  
  /* reset iter */
  r_card_reset_address(card);
  
  /* find address of type: R_ADDRESS_WORK */
  address = r_card_find_address(card, R_ADDRESS_WORK);
  if (address)
    g_object_get(address, 
		 "street",  &wstr,
		 "city",    &wcity,
		 "state",   &wstate,
		 "zip",     &wzip,
		 "country", &wcountry,
		 NULL);
  
  /* reset net iter */
  r_card_reset_net_address(card);
  
  /* find first net address */
  net = r_card_find_net_address(card, R_NET_ADDRESS_EMAIL);
  if (net) 
    g_object_get(net, "url", &email1, NULL);
  
  /* find next next address */
  net = r_card_find_net_address(card, R_NET_ADDRESS_EMAIL);
  if (net) 
    g_object_get(net, "url", &email2, NULL);     
  
  r_card_reset_net_address(card);   
  net = r_card_find_net_address(card, R_NET_ADDRESS_WEB);
  if (net) 
    g_object_get(net, "url", &web, NULL);
  
  r_card_reset_net_address(card);        
  net = r_card_find_net_address(card, R_NET_ADDRESS_WORK_WEB);
  if (net) 
    g_object_get(net, "url", &wweb, NULL);
  
  r_card_reset_telephone(card);
  tel = r_card_find_telephone(card, R_TELEPHONE_HOME);
  if (tel)
    g_object_get(tel, "telephone-number", &htel, NULL);
  
  r_card_reset_telephone(card);
  tel = r_card_find_telephone(card, R_TELEPHONE_WORK);
  if (tel)
    g_object_get(tel, "telephone-number", &wtel, NULL);
  
  r_card_reset_telephone(card);
  tel = r_card_find_telephone(card, R_TELEPHONE_FAX);
  if (tel)
    g_object_get(tel, "telephone-number", &fax, NULL);
  
  r_card_reset_telephone(card);
  tel = r_card_find_telephone(card, R_TELEPHONE_CELLPHONE);
  if (tel)
    g_object_get(tel, "telephone-number", &mobile, NULL);
  
  r_card_reset_telephone(card);
  tel = r_card_find_telephone(card, R_TELEPHONE_PAGER);
  if (tel)
    g_object_get(tel, "telephone-number", &pager, NULL); 
  
  fprintf(fp,  
	  "%s,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s," 
	  "%s,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s\n",  
	  fn ? fn : "", ln ? ln : "", dn ? dn : "", nick ? nick : "",
	  email1 ? email1 : "", email2 ? email2 : "",
	  wtel ? wtel : "", htel ? htel : "", fax ? fax : "", 
	  pager ? pager : "", mobile ? pager : "",
	  str ? str : "", "", city ? city : "", state ? state : "",
	  zip ? zip : "", country ? country : "",
	  wstr ? wstr : "", "", wcity ? wcity :"", wstate ? wstate : "",
	  wzip ? wzip : "", wcountry ? wcountry : "",
	  role ? role : "", dep ? dep : "", org ? org : "",
	  wweb ? wweb : "", web ? web : "",
	  year ? year : "", month ? month : "", day ? day : "", 
	  "", "", "", "",
	  note ? note : "");  
}

static gboolean 
r_csv_thunderbird_save_file (RAbook* abook, gchar* filename)
{  
  FILE* fp;
  g_return_val_if_fail(IS_R_ABOOK(abook), FALSE);
  g_return_val_if_fail(filename != NULL, FALSE);

  if (!(fp = fopen(filename, "w")))
    {
      g_warning("\nCan't write file: %s", filename);
      
      return FALSE;
    }
  
  r_abook_foreach_card (abook, (RFunc) write_card, fp);
  fflush(fp);

  return TRUE;
}


static gboolean
r_csv_thunderbird_overwrite_file(RAbook* abook)
{
  gchar* file;
  gchar* path;
  gchar* filename;
  
  g_return_val_if_fail(IS_R_ABOOK(abook), FALSE);
  
  g_object_get(abook, "addressbook-path", &path, 
	       "addressbook-name", &file, NULL);
  filename = g_strdup_printf("%s%s%s", path, G_DIR_SEPARATOR_S, file);
  
  if (g_file_test(filename, G_FILE_TEST_EXISTS))
    g_remove(filename);

  if (!r_csv_thunderbird_save_file(abook, filename))
    {
      g_signal_emit_by_name(abook, "save_fail", OVERWRITING, G_TYPE_INT);
      
      g_free(filename);
      return FALSE;
    }
    
  g_free(filename);
  g_signal_emit_by_name(abook, "addressbook_saved", NULL, G_TYPE_NONE);  
  
  return TRUE;  
}


G_MODULE_EXPORT void 
plugin_init (RPlugin* plugin, gchar* file)
{
  RPluginAction* action;
  RFilter* filter;

  g_return_if_fail(plugin != NULL);
 
#ifdef ENABLE_DEBUG
  g_log(G_LOG_DOMAIN, G_LOG_LEVEL_DEBUG, "plugin_init");
  g_log(G_LOG_DOMAIN, G_LOG_LEVEL_DEBUG, "Initializing Thunderbird's csv plugin");
#else
  g_log(G_LOG_DOMAIN, G_LOG_LEVEL_INFO, "Initializing Thunderbird's csv plugin");
#endif 
  
  r_plugin_set_engine(plugin, r_csv_thunderbird_new());
  g_object_set(plugin, 
	       "plugin-name", "thunderbird", 
	       "plugin-filename", file,
	       "plugin-label", "thunderbird's csv file format",
	       "plugin-info", 
	       "This plugin manages the thunderbird's csv file format",
	       "plugin-configurable", FALSE, NULL);
  
  filter = r_filter_new();
  g_object_set(filter, 
	       "filter-name", "CSV", 
	       "filter-extension", "csv",
	       "filter-mime", "text/x-comma-separated-values", NULL);
  r_filter_add_pattern(filter, "csv");
  r_filter_add_pattern(filter, "*.csv");
  r_plugin_add_filter (plugin, filter);


  action = g_malloc(sizeof(RPluginAction));
  action->name   = g_strdup("read");
  action->handle = (gpointer) r_csv_thunderbird_open_file;
  r_plugin_add_action(plugin, action);  
    
  action = g_malloc(sizeof(RPluginAction));
  action->name   = g_strdup("write");
  action->handle = (gpointer) r_csv_thunderbird_save_file;
  r_plugin_add_action(plugin, action);  

  action = g_malloc(sizeof(RPluginAction));
  action->name   = g_strdup("overwrite");
  action->handle = (gpointer) r_csv_thunderbird_overwrite_file;
  r_plugin_add_action(plugin, action);  

#ifdef ENABLE_DEBUG
  g_log(G_LOG_DOMAIN, G_LOG_LEVEL_DEBUG, "Initializing csv plugin, done"); 
#endif
}


G_MODULE_EXPORT void     
plugin_fini (void)
{
  // TODO: liberare la memoria e rilasciare l'handler al plugin
}
