/*
*  LibRAL
*  file: vcard_engine.c
* 
*  RFC2426 (vcard file format specifications)
*
*  Copyright (C) Nicola Fragale <nicolafragale@libero.it>
*
*  This program is free software; you can redistribute it and/or modify
*  it under the terms of the GNU General Public License as published by
*  the Free Software Foundation; either version 3 of the License
*
*  This program is distributed in the hope that it will be useful,
*  but WITHOUT ANY WARRANTY; without even the implied warranty of
*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*  GNU General Public License for more details.
*
*  You should have received a copy of the GNU General Public License
*  along with this program; if not, write to the Free Software
*  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#include <libral.h>
#include <glib.h>
#include <glib/gstdio.h>
#include <glib-object.h>


#include <string.h>
#include <stdlib.h>
#include <time.h>

#include "vcard.h"


typedef enum {
  N = 0, 
  FN, 
  TZ, 
  END, 
  ADR, 
  TEL, 
  GEO, 
  ORG, 
  REV, 
  UID, 
  URL, 
  KEY, 
  BDAY, 
  ROLE, 
  LOGO, 
  NOTE, 
  BEGIN,
  PHOTO, 
  LABEL,
  EMAIL, 
  TITLE, 
  AGENT, 
  PROID, 
  SOUND, 
  CLASS,
  MAILER, 
  VERSION,
  NICKNAME, 
  CATEGORIES,                  // last vcard token

  KDE_EXT_Profession,          // KDE extensions
  KDE_EXT_Department,
  KDE_EXT_ManagersName,
  KDE_EXT_AssistantsName,
  KDE_EXT_Office,
  KDE_EXT_SpousesName,
  KDE_EXT_Anniversary,
  KDE_EXT_CUSTOM,

  RUBRICA_EXT_Card,            // Rubrica extensions
  RUBRICA_EXT_Group, 
  RUBRICA_EXT_Ref,
  RUBRICA_EXT_Url,
  RUBRICA_EXT_CompanyName,
  RUBRICA_EXT_Street,
  RUBRICA_EXT_StreetNumber,
  RUBRICA_EXT_ZipCode,
  RUBRICA_EXT_City,
  RUBRICA_EXT_Province,
  RUBRICA_EXT_Country,
  RUBRICA_EXT_Web,
  RUBRICA_EXT_Email,
  RUBRICA_EXT_Operator,
  RUBRICA_EXT_Fax,
  RUBRICA_EXT_Green,
  RUBRICA_EXT_CustomerCare,
  RUBRICA_EXT_Notes,
  RUBRICA_EXT_Department,
  RUBRICA_EXT_SubDepartment,
  RUBRICA_EXT_Manager,
  RUBRICA_EXT_ManagerTelephone,
  RUBRICA_EXT_Collaborator,
  RUBRICA_EXT_CollaboratorTelephone,
  RUBRICA_EXT_SpouseName,
  RUBRICA_EXT_Child,
  RUBRICA_EXT_Anniversary,
  RUBRICA_EXT_Hobbies,

  EVOLUTION_EXT_File_as,       // Evolution extensions
  EVOLUTION_EXT_Office,
  EVOLUTION_EXT_Manager,
  EVOLUTION_EXT_Assistant,
  EVOLUTION_EXT_Spouse,
  EVOLUTION_EXT_Anniversary,

  MOZILLA_EXT_Html,

  BAD_TOKEN,                   // this isn't a VCard token
} RVCardTokenType;



typedef enum {
  TYPE = 0, 
  VALUE, 
  ENCODING,

  RATE,                        // rubrica's param
  OWNER,

  BAD_PARAM,                   // this isn't a VCard token
} RVCardParamType;



typedef enum {
  HOME = 0,  
  WORK,
  CELL, 
  FAX,
  POSTAL, 
  PARCEL, 
  DOM, 
  INTL, 
  PREF,
  VOICE, 
  MSG, 
  PAGER, 
  BBS, 
  MODEM, 
  CAR, 
  ISDN, 
  VIDEO,  
  AOL, 
  INTERNET,
  APPLELINK, 
  ATTMAIL, 
  CIS, 
  EWORLD, 
  IBMMail, 
  MCIMAIL, 
  POWERSHARE, 
  PRODIGY, 
  TLX, 
  X400,
  VCARD,
  PGP, 
  X509, 
  B,
  BINARY,
  TEXT,
  URI,
  BASIC, 
  JPEG,
  PUBLIC, 
  PRIVATE, 
  CONFIDENTIAL,
  QUOTED_PRINTABLE,  

  RUBRICA_EXT_USER,
  RUBRICA_EXT_RUBRICA,
  RUBRICA_EXT_IRC,
  RUBRICA_EXT_IRC_AIM,
  RUBRICA_EXT_IRC_ICQ,
  RUBRICA_EXT_IRC_JABBER,
  RUBRICA_EXT_IRC_YAHOO,
  RUBRICA_EXT_IRC_MSN,
  RUBRICA_EXT_EKIGA,
  
  RUBRICA_EXT_UNKNOWN,
  BAD_VALUE,                 // this isn't a vcard token
} RVCardValueType;


RLookupTable tokens[] = {
  {"N",     "", N},          {"FN",     "", FN},      {"TZ",      "", TZ},
  {"END",   "", END},        {"ADR",    "", ADR},     {"TEL",     "", TEL},
  {"GEO",   "", GEO},        {"ORG",    "", ORG},     {"REV",     "", REV},
  {"UID",   "", UID},        {"URL",    "", URL},     {"KEY",     "", KEY},
  {"BDAY",  "", BDAY},       {"ROLE",   "", ROLE},    {"LOGO",    "", LOGO}, 
  {"NOTE",  "",  NOTE},      {"BEGIN",  "", BEGIN},   {"PHOTO",   "", PHOTO},  
  {"LABEL", "", LABEL},      {"EMAIL",  "", EMAIL},   {"TITLE",   "", TITLE}, 
  {"AGENT", "", AGENT},      {"PROID",  "", PROID},   {"SOUND",   "", SOUND},  
  {"CLASS", "", CLASS},      {"MAILER", "", MAILER},  {"VERSION", "", VERSION},
  {"NICKNAME", "", NICKNAME}, {"CATEGORIES", "", CATEGORIES},

  /* kde extension.
  */
  {"X-KADDRESSBOOK-X-Profession",     "", KDE_EXT_Profession},
  {"X-KADDRESSBOOK-X-Department",     "", KDE_EXT_Department},
  {"X-KADDRESSBOOK-X-ManagersName",   "", KDE_EXT_ManagersName},
  {"X-KADDRESSBOOK-X-AssistantsName", "", KDE_EXT_AssistantsName},
  {"X-KADDRESSBOOK-X-Office",         "", KDE_EXT_Office},
  {"X-KADDRESSBOOK-X-SpousesName",    "", KDE_EXT_SpousesName},
  {"X-KADDRESSBOOK-X-Anniversary",    "", KDE_EXT_Anniversary},
  {"X-KADDRESSBOOK-X-CUSTOM",         "", KDE_EXT_CUSTOM},
  
  /* rubrica extension. 
     So rubrica can import a previously 
     exported rubrica's addressbook to vcard format
  */
  {"X-RUBRICA-X-Card",                  "", RUBRICA_EXT_Card},
  {"X-RUBRICA-X-Group",                 "", RUBRICA_EXT_Group},
  {"X-RUBRICA-X-Ref",                   "", RUBRICA_EXT_Ref},
  {"X-RUBRICA-X-URL",                   "", RUBRICA_EXT_Url},
  {"X-RUBRICA-X-CompanyName",           "", RUBRICA_EXT_CompanyName},
  {"X-RUBRICA-X-Street",                "", RUBRICA_EXT_Street},
  {"X-RUBRICA-X-StreetNumber",          "", RUBRICA_EXT_StreetNumber},
  {"X-RUBRICA-X-ZipCode",               "", RUBRICA_EXT_ZipCode},
  {"X-RUBRICA-X-City",                  "", RUBRICA_EXT_City},
  {"X-RUBRICA-X-Province",              "", RUBRICA_EXT_Province},
  {"X-RUBRICA-X-Country",               "", RUBRICA_EXT_Country},
  {"X-RUBRICA-X-Web",                   "", RUBRICA_EXT_Web},
  {"X-RUBRICA-X-Email",                 "", RUBRICA_EXT_Email},
  {"X-RUBRICA-X-Operator",              "", RUBRICA_EXT_Operator},
  {"X-RUBRICA-X-Fax",                   "", RUBRICA_EXT_Fax},
  {"X-RUBRICA-X-Green",                 "", RUBRICA_EXT_Green},
  {"X-RUBRICA-X-CustomerCare",          "", RUBRICA_EXT_CustomerCare},
  {"X-RUBRICA-X-Notes",                 "", RUBRICA_EXT_Notes},
  {"X-RUBRICA-X-Department",            "", RUBRICA_EXT_Department},
  {"X-RUBRICA-X-SubDepartment",         "", RUBRICA_EXT_SubDepartment},
  {"X-RUBRICA-X-Manager",               "", RUBRICA_EXT_Manager},
  {"X-RUBRICA-X-ManagerTelephone",      "", RUBRICA_EXT_ManagerTelephone},
  {"X-RUBRICA-X-Collaborator",          "", RUBRICA_EXT_Collaborator},
  {"X-RUBRICA-X-CollaboratorTelephone", "", RUBRICA_EXT_CollaboratorTelephone},
  {"X-RUBRICA-X-SpouseName",            "", RUBRICA_EXT_SpouseName},
  {"X-RUBRICA-X-Child",                 "", RUBRICA_EXT_Child},
  {"X-RUBRICA-X-Anniversary",           "", RUBRICA_EXT_Anniversary},
  {"X-RUBRICA-X-Hobbies",               "", RUBRICA_EXT_Hobbies},

  /* evolution extensions
  */
  {"X-EVOLUTION-FILE-AS",               "", EVOLUTION_EXT_File_as},  
  {"X-EVOLUTION-OFFICE",                "", EVOLUTION_EXT_Office},
  {"X-EVOLUTION-MANAGER",               "", EVOLUTION_EXT_Manager},  
  {"X-EVOLUTION-ASSISTANT",             "", EVOLUTION_EXT_Assistant},
  {"X-EVOLUTION-SPOUSE",                "", EVOLUTION_EXT_Spouse},  
  {"X-EVOLUTION-ANNIVERSARY",           "", EVOLUTION_EXT_Anniversary},

  {"X-MOZILLA-HTML",                    "", MOZILLA_EXT_Html},

  /*     TODO: add evolution extensions
  */
  {NULL,     "", BAD_TOKEN}
};
	
	
RLookupTable params[] = {
  {"type",     "", TYPE},
  {"value",    "", VALUE},   
  {"encoding", "", ENCODING},
  
  {"rate",     "", RATE},      // rubrica's param
  {"owner",    "", OWNER},
  
  {NULL,       "", BAD_PARAM}
};		


static RLookupTable values[] = {
  {"home",     "", HOME},     {"work",       "", WORK},     
  {"cell",     "", CELL},     {"fax",        "", FAX},    
  {"postal",   "", POSTAL},   {"parcel",     "", PARCEL},  
  {"dom",      "", DOM},      {"intl",       "", INTL}, 
  {"pref",     "", PREF},     {"voice",      "", VOICE}, 
  {"msg",      "", MSG},      {"pager",      "", PAGER},   
  {"bbs",      "", BBS},      {"modem",      "", MODEM},   
  {"car",      "", CAR},      {"isdn",       "", ISDN}, 
  {"video",    "", VIDEO},    {"aol",        "", AOL},     
  {"internet", "", INTERNET}, {"applelink",  "", APPLELINK},
  {"attmail",  "", ATTMAIL},  {"cis",        "", CIS},    
  {"eworld",   "", EWORLD},   {"ibmmail",    "", IBMMail},  
  {"mcimail",  "", MCIMAIL},  {"powershare", "", POWERSHARE},
  {"prodigy",  "", PRODIGY},  {"tlx",        "", TLX},   
  {"x400",     "", X400},     {"vcard",      "", VCARD},
  {"pgp",      "", PGP},      {"x509",       "", X509},     
  {"b",        "", BINARY},   {"binary",     "", BINARY},  
  {"text",     "", TEXT},     {"uri",        "", URI},
  {"basic",    "", BASIC},    {"jpeg",       "", JPEG},    
  {"public",   "", PUBLIC},   {"private",    "", PRIVATE}, 
  {"confidential",     "", CONFIDENTIAL},  
  {"quoted-printable", "", QUOTED_PRINTABLE},  
  {"user",     "", RUBRICA_EXT_USER}, 
  {"rubrica",  "", RUBRICA_EXT_RUBRICA},
  {"irc",      "", RUBRICA_EXT_IRC},
  {"aim",      "", RUBRICA_EXT_IRC_AIM},
  {"icq",      "", RUBRICA_EXT_IRC_ICQ},
  {"jabber",   "", RUBRICA_EXT_IRC_JABBER},
  {"yahoo",    "", RUBRICA_EXT_IRC_YAHOO},
  {"msn",      "", RUBRICA_EXT_IRC_MSN},
  {"ekiga",    "", RUBRICA_EXT_EKIGA},

  {"other",    "", RUBRICA_EXT_UNKNOWN},
  {NULL,       "", BAD_VALUE}
};


 
static RPairTable address_pairs[] = {
  {"home",  HOME,      R_ADDRESS_HOME},
  {"work",  WORK,      R_ADDRESS_WORK},
  {NULL,    BAD_VALUE, R_ADDRESS_OTHER},
};


static RPairTable phone_pairs[] = {
  {"home",      HOME,      R_TELEPHONE_HOME     },
  {"work",      WORK,      R_TELEPHONE_WORK     },
  {"cell",      CELL,      R_TELEPHONE_CELLPHONE},
  {"fax",       FAX,       R_TELEPHONE_FAX      },
  {"cellphone", CELL,      R_TELEPHONE_CELLPHONE},
  {"pref",      PREF,      R_TELEPHONE_OTHER    },
  {"voice",     VOICE,     R_TELEPHONE_OTHER    },
  {"msg",       MSG,       R_TELEPHONE_OTHER    },
  {"pager",     PAGER,     R_TELEPHONE_OTHER    },
  {"bbs",       BBS,       R_TELEPHONE_OTHER    },
  {"modem",     MODEM,     R_TELEPHONE_OTHER    },
  {"car",       CAR,       R_TELEPHONE_OTHER    },
  {"isdn",      ISDN,      R_TELEPHONE_OTHER    },
  {"video",     VIDEO,     R_TELEPHONE_OTHER    },
  {NULL,        BAD_VALUE, R_TELEPHONE_OTHER    },
};


typedef struct _Params{
  RVCardParamType param;
  RVCardValueType value;
} Params;


typedef struct _Token {  
  RVCardTokenType token;
  RVCardTokenType previous;
	
  gboolean more;	
  GList* params;
  GList* data;
  
  gchar* buffer;
} Token;


struct _RVCardPrivate {
  gboolean have_company;
  gboolean have_company_address;

  Token* token;
  gboolean check;

  gboolean dispose_has_run;
};



/*  property enumeration
 */
enum {
  VCARD_PROP_0,
  VCARD_FILENAME,  
  VCARD_RABOOK,
  VCARD_CHECK,
};

/*    signals enumeration 
 */
enum {
  CARD_DECODED,
  ADDRESSBOOK_SAVED,
  LAST_SIGNAL 
};

static guint r_vcard_signals[LAST_SIGNAL] = {0};

static void r_vcard_init         (RVCard* self);
static void r_vcard_class_init   (RVCardClass* klass);

static void r_vcard_set_property (GObject* obj, guint property_id,
				  const GValue* value, GParamSpec* spec);
static void r_vcard_get_property (GObject* obj, guint property_id,
				  GValue* value, GParamSpec* spec);

static void r_vcard_dispose      (RVCard* obj);
static void r_vcard_finalize     (RVCard* obj);




/* private functions
*/
static RCompanyCard* r_vcard_company_new        (RVCard *vcard, RCard* card);
static RAddress*     r_vcard_company_address_new(RVCard *vcard);


// static void     decode_vcard_token (gchar* line, Token* token);
static void     split_token        (Token* token, gchar* str);
static void     split_data         (Token* token, gchar* str);
static void     split_params       (Token* token, gchar* str);
static void     delete_data        (GList* data);
static void     free_token_data    (Token* token);
static gboolean is_vcard_doc       (RVCard* vcard, const gchar* filename);


#define R_VCARD_GET_PRIVATE(o)     (G_TYPE_INSTANCE_GET_PRIVATE((o),   \
				    R_VCARD_TYPE, RVCardPrivate))



RPersonalCard *card              = NULL;
RCompanyCard  *company           = NULL;
RContact      *contact           = NULL;
RWork         *work              = NULL;
RNotes        *notes             = NULL;
RAddress      *address           = NULL;
RTelephone    *tel               = NULL;
RNetAddress   *net               = NULL;
RAddress      *company_address   = NULL;
RTelephone    *company_telephone = NULL;
RNetAddress   *company_net       = NULL;


gchar** categories = NULL;


GType
r_vcard_get_type()
{
  static GType r_vcard_type = 0;
  
  if (!r_vcard_type)
    {
      static const GTypeInfo r_vcard_info =
	{
	  sizeof(RVCardClass),
	  NULL,
	  NULL,
	  (GClassInitFunc) r_vcard_class_init,
	  NULL,
	  NULL,
	  sizeof(RVCard),
	  0,
	  (GInstanceInitFunc) r_vcard_init
	};

      r_vcard_type = g_type_register_static (G_TYPE_OBJECT,
					     "RVCard", &r_vcard_info, 0);
    }
  
  return r_vcard_type;
}



static void
r_vcard_class_init(RVCardClass* klass)
{
  GObjectClass *class;
  GParamSpec* pspec;
  
  class = G_OBJECT_CLASS (klass);

  class->dispose      = (GObjectFinalizeFunc) r_vcard_dispose;
  class->finalize     = (GObjectFinalizeFunc) r_vcard_finalize;
  
  class->set_property = (gpointer) r_vcard_set_property;
  class->get_property = (gpointer) r_vcard_get_property;

  g_type_class_add_private (klass, sizeof(RVCardPrivate));


  /**
   * RVCard::addressbook-saved:
   * @vcard: 
   *
   * The "addressbook_decoded" signal emitted if the vcard addressbook
   * is successfully decoded
   */
  r_vcard_signals[ADDRESSBOOK_SAVED] =
    g_signal_new("addressbook-saved", 
		 R_VCARD_TYPE,
		 G_SIGNAL_RUN_LAST,
		 G_STRUCT_OFFSET(RVCardClass, addressbook_saved),
		 NULL,
		 NULL,
		 g_cclosure_marshal_VOID__VOID,
		 G_TYPE_NONE,
		 0,
		 G_TYPE_NONE);

  /**
   * VCard:filename
   *
   * vcard address book's name
   */
  pspec = g_param_spec_string("filename",
			      "addressbook's name",
			      "the name (filename) of the addressbook",
			      NULL,
			      G_PARAM_READWRITE);
  g_object_class_install_property(class, VCARD_FILENAME, pspec);

  /**
   * VCard:rabook
   *
   * rabook addressbook
   */
  pspec = g_param_spec_pointer("addressbook",
			       "ral addressbook",
			       "an RAbook",
			       G_PARAM_READWRITE);
  g_object_class_install_property(class, VCARD_RABOOK, pspec);   
}



static void
r_vcard_init(RVCard* self)
{
  g_return_if_fail(R_VCARD(self) != NULL);

  self->filename = NULL;
  self->book     = NULL;
  self->priv     = R_VCARD_GET_PRIVATE(self);
  
  self->priv->have_company         = FALSE;
  self->priv->have_company_address = FALSE;
  self->priv->check                = FALSE;

  self->priv->token = g_new0(Token, 1);
  
  self->priv->dispose_has_run = FALSE;
}



static void 
r_vcard_dispose (RVCard* vcard)
{
  g_return_if_fail(IS_R_VCARD(vcard));

  if (vcard->priv->dispose_has_run)
    return;

  vcard->priv->dispose_has_run = TRUE;
}


static void 
r_vcard_finalize (RVCard* vcard)
{
  g_return_if_fail(IS_R_VCARD(vcard));

  free_token_data(vcard->priv->token);
}


static void 
r_vcard_set_property (GObject* obj, guint property_id,
		      const GValue* value, GParamSpec* spec)
{
  RVCard* self = (RVCard*) obj;

  switch(property_id)
    {
    case VCARD_FILENAME:
      if (self->filename)
	g_free(self->filename);
      
      self->filename = g_value_dup_string(value);
      break;

    case VCARD_RABOOK:
      self->book = g_value_get_pointer(value);
      break;

    default:
      break;
    }
}

static void 
r_vcard_get_property (GObject* obj, guint property_id,
		      GValue* value, GParamSpec* spec)
{
  RVCard* self = (RVCard*) obj;

  switch(property_id)
    {
    case VCARD_FILENAME:
      g_value_set_string(value, self->filename);
      break;
 
    case VCARD_RABOOK:
      g_value_set_pointer(value, self->book);
      break;
     
    default:
      break;
    }
}



static RCompanyCard* 
r_vcard_company_new(RVCard *vcard, RCard* card)
{
  RRef* ref;
  RCompanyCard* company = NULL;
  gulong id;
  gchar* info;
      
  vcard->priv->have_company = TRUE;

  g_object_get(R_CARD(card), "card-id", &id, "card-name", &info, NULL);

  ref = r_ref_new(id);
  g_object_set(ref, "ref-info", info, NULL);

  company = r_company_card_new();
  r_card_add_ref(R_CARD(company), ref);	  
  
  return company;
}


static RAddress*
r_vcard_company_address_new(RVCard *vcard)
{
  g_return_val_if_fail(IS_R_VCARD(vcard), NULL);
  
  vcard->priv->have_company_address = TRUE;

  return r_address_new();
}


static void 
build_card(RVCard *vcard)
{
  RVCardPrivate* priv;
  Token* token;
  gint i=0;

  g_return_if_fail(IS_R_VCARD(vcard));
	
  priv = R_VCARD_GET_PRIVATE(vcard);
  token = priv->token;

  if (token->token == BAD_TOKEN)
    {
      token->token = token->previous;
      token->more  = TRUE;
    }
  else
    token->more = FALSE;
	
  switch(token->token)
    {
    case BEGIN:
      card    = r_personal_card_new();
      contact = r_contact_new();
      work    = r_work_new();
      notes   = r_notes_new();
      break;
      
    case END:
      r_personal_card_set_notes(card, notes);
      r_personal_card_set_work(card, work);     

      r_abook_add_loaded_card(R_ABOOK(vcard->book), R_CARD(card));
      g_signal_emit_by_name(R_ABOOK(vcard->book), 
			    "card_read", card, G_TYPE_POINTER);

      if (priv->have_company)
	{
	  r_card_add_address(R_CARD(company), company_address);
	  priv->have_company = FALSE;
	  priv->have_company_address = FALSE;

	  r_abook_add_loaded_card(R_ABOOK(vcard->book), R_CARD(company));
	  g_signal_emit_by_name(R_ABOOK(vcard->book), 
				"card_read", company, G_TYPE_POINTER);
	}
      break;

    case FN:
      /* card name */
      if (token->data && token->data->data)
	g_object_set(R_CARD(card), "card-name", 
		     (gchar*)token->data->data, NULL);
      else
	g_object_set(R_CARD(card), "card-name", "", NULL);	
      break;
      
    case N:
      /* 
	 (RFC2426)
	 The structured type value corresponds, in sequence, to:
	 the Family Name,               (pos 0)
	 Given Name,                    (pos 1)
	 Additional Names,              (pos 2)
	 Honorific Prefixes,            (pos 3)
	 Honorific Suffixes.            (pos 4)
      */
      if (token->data)
	{
 	  gchar *last, *name, *middle, *prefix, *title; 

	  last   = (gchar*) g_list_nth_data(token->data, 0);
	  name   = (gchar*) g_list_nth_data(token->data, 1);
	  middle = (gchar*) g_list_nth_data(token->data, 2);
	  prefix = (gchar*) g_list_nth_data(token->data, 3);
	  title  = (gchar*) g_list_nth_data(token->data, 4);

	  g_object_set(contact, 
		       "last-name",   last,
		       "first-name",  name, 
		       "middle-name", middle,
		       "prefix",      prefix,
		       "title",       title, NULL);	
  
	  r_personal_card_set_contact(card, contact);
	}
      break;

    case TZ:
      /* ignore */
      break;

    case REV:
      /* ignore */
      break;

    case ADR:
      if (token->params && token->data)
	{
	  /* ADR Type Definition: 
	     
	     order:
	     post office box;                       // ignored   (pos: 0)
	     the extended address;                  // ignored   (pos: 1)
	     the street address;                    (pos 2)
	     the locality (e.g., city);             (pos 3)
	     the region (e.g., state or province);  (pos 4)
	     the postal code;                       (pos 5)
	     the country name                       (pos 6)
	  */
	  GList* iter = token->data;
	  gchar *street, *city, *state, *zip, *country; 
	  
	  street   = (gchar*) g_list_nth_data(iter, 2);
	  city     = (gchar*) g_list_nth_data(iter, 3);
	  state    = (gchar*) g_list_nth_data(iter, 4);
	  zip      = (gchar*) g_list_nth_data(iter, 5);
	  country  = (gchar*) g_list_nth_data(iter, 6);

	  for (iter = token->params; iter; iter = iter->next)
	    {
	      Params* param = (Params*) iter->data;
	      
	      address = r_address_new();

	      g_object_set(address, 
			   "address-type", param->value,
			   "street",       street, 
			   "city",         city,
			   "state",        state,	       
			   "zip",          zip, 
			   "country",      country, NULL);
	
	      r_card_add_address(R_CARD(card), address);	      
	    }
	}
      break;
      
    case TEL:
      if (token->params && token->data)
	{
	  gchar* number;
	  GList* iter = token->params;

	  number = (gchar*) token->data->data;
	  
	  for (; iter; iter = g_list_next(iter))
	    {
	      Params* param = (Params*) iter->data;

	      tel = r_telephone_new();
	      
	      g_object_set(tel, 
			   "telephone-number", number, 
			   "telephone-type", param->value, 
			   NULL);	      
	      
	      r_card_add_telephone(R_CARD(card), tel);	  
	    }
	}
      break;
      
    case EMAIL:  
      net = r_net_address_new();
      
      g_object_set(net, 
		   "url",      (gchar*) token->data->data, 
		   "url-type", R_NET_ADDRESS_EMAIL, 
		   NULL);
      r_card_add_net_address(R_CARD(card), net);
      break;

    case URL:      
      if (token->data && token->data->data)
	{
	  net = r_net_address_new();
	  
	  g_object_set(net, 
		       "url",      (gchar*) token->data->data, 
		       "url-type", R_NET_ADDRESS_WEB,
		       NULL);
	  r_card_add_net_address(R_CARD(card), net);
	}
     break;

    case GEO:
      /* ignore */
      break;

    case BDAY:
      if (token->data)
	{
	  // BDAY:2001-10-21T15:53:29 ==> drop time T15:53:29 
	  gint year, month, day;
	  
	  day  = atoi(g_list_nth_data(token->data, 0));
	  month = atoi(g_list_nth_data(token->data, 1));
	  year   = atoi(g_list_nth_data(token->data, 2));
	    
	  r_contact_set_birthday(contact, day, month, year);
	}
      break;
      
    case TITLE:
    case KDE_EXT_Profession:
      if (token->data && token->data->data)
	g_object_set(contact, "profession", token->data->data, NULL);
      break;

    case ORG:
      if (token->data && token->data->data)
	g_object_set(work, "organization", token->data->data, NULL);
      break;

    case CATEGORIES:
      if (token->data && token->data->data)
	{
	  categories = g_strsplit(token->data->data, ",", 0);
	  for (i = 0; categories[i]; i++)
	    {
	      RGroup* group;
	      
	      group = r_group_new();
	      g_object_set(group, 
			   "group-name", categories[i], 
			   "group-owner", "user",
			   NULL);
	      
	      r_card_add_group(R_CARD(card), group);
	    }
	  g_strfreev(categories); 
	}
      break;

    case RUBRICA_EXT_Group:
      if (token->params && token->data)
	{
	  RGroup* group;
	  GList* iter = token->data;
	  Params* param = (Params*) token->params->data;;
	  gchar* group_name, *pixmap, *owner;

	  group_name = iter->data;
	  iter   = g_list_next(iter);
	  pixmap = iter->data;
	  owner  = r_lookup_table_enum2str(values, param->value);

	  group = r_group_new();
	  g_object_set(group, 
		       "group-name",   group_name, 
		       "group-pixmap", pixmap,
		       "group-owner",  owner,
		       NULL);
	  
	  r_card_add_group(R_CARD(card), group);
	}
      break;
     
    case NOTE:
      if (token->more)
	{
	  gchar* tmp;
	  gchar* buf;

	  g_object_get(notes, "other-notes", &tmp, NULL);
	  buf = g_strdup_printf("%s\n%s", tmp, (gchar*) token->buffer);
	  
	  g_object_set(notes, "other-notes", buf, NULL);
	  g_free(buf);
	}
      else
	g_object_set(notes, "other-notes", token->data->data, NULL);
      break;
      
    case KEY: 
      if (token->more)
	{
	  gchar* tmp;
	  gchar* buf;

	  g_object_get(notes, "pubkey", &tmp, NULL);
	  buf = g_strdup_printf("%s\n%s", tmp, (gchar*) token->buffer);
	  
	  g_object_set(notes, "pubkey", buf, NULL);
	  g_free(buf);
	}
      else      
	g_object_set(notes, "pubkey", token->data->data, NULL);
      break;

    case ROLE:
      g_object_set(work, "assignment", token->data->data, NULL);
      break;

      /*   TODO -- recuperare campi condivisi
       */
    case UID:
      /* ignore */
      break;

    case LOGO:
      /* TODO */
      break;

    case PHOTO:
      /* TODO */
      break;

    case LABEL:
      /* ignore */
      break;

    case AGENT:
      /* ignore */
      break;
      
    case PROID:
      /* ignore */
      break;
      
    case SOUND:
      /* ignore */
      break;
      
    case CLASS:
      /* ignore */
      break;
      
    case MAILER:
      /* ignore */
      break;
      
    case VERSION:
      /* ignore */
      break;
      
    case NICKNAME:
      if (token->data && token->data->data)
	g_object_set(contact, "nick-name", token->data->data, NULL);
      break;
      
    case RUBRICA_EXT_Card:      
      if (token->params && token->data)
	{
	  GList* iter = token->data;
	  Params* param;
	  gboolean locked = TRUE, deleted = FALSE;
	  gchar *lock, *del;
	  
	  lock = iter->data;
	  iter = g_list_next(iter);
	  del  = iter->data;
	  
	  if (token->params)
	    param = (Params*) token->params->data;
	  
	  if (g_ascii_strcasecmp(g_ascii_strdown(lock, -1), "true") == 0)
	    locked = TRUE;
	  
	  if (g_ascii_strcasecmp(g_ascii_strdown(del, -1), "true") == 0)
	    deleted = TRUE;

	  g_object_set(card, 
		       "card-rate", param->value, 
		       "card-locked",  locked,
		       "card-deleted", deleted,
		       NULL);
	}
      break;

    case RUBRICA_EXT_Ref:
      // ref id
      break;

    case RUBRICA_EXT_Url:
      // other url
      break;
      
    case RUBRICA_EXT_Department:
    case KDE_EXT_Department:
      if (token->data && token->data->data)
	g_object_set(work, "department", token->data->data, NULL);
      break;
      
    case RUBRICA_EXT_SubDepartment:
      if (token->data && token->data->data)
	g_object_set(work, "sub-department", token->data->data, NULL);
      break;

    case RUBRICA_EXT_Collaborator:
    case KDE_EXT_AssistantsName:
    case EVOLUTION_EXT_Assistant:
      if (token->data && token->data->data)
	g_object_set(work, "collaborator", token->data->data, NULL);
      break;
      
    case RUBRICA_EXT_CollaboratorTelephone:
      if (token->data && token->data->data)
	g_object_set(work, "collaborator-phone", token->data->data, NULL);
      break;
      
    case RUBRICA_EXT_SpouseName:
    case KDE_EXT_SpousesName:
    case EVOLUTION_EXT_Spouse:
      if (token->data && token->data->data)
	g_object_set(notes, "has-partner", TRUE, 
		     "partner-name", token->data->data, NULL);
      break;
      
    case RUBRICA_EXT_Child:
      if (token->data && token->data->data)
	g_object_set(notes, "children", token->data->data, NULL);
      break;

    case RUBRICA_EXT_Anniversary:
    case KDE_EXT_Anniversary:
    case EVOLUTION_EXT_Anniversary:
      if (token->data)
	{
	  gint year, month, day;

	  day  = atoi(g_list_nth_data(token->data, 0));
	  month = atoi(g_list_nth_data(token->data, 1));
	  year   = atoi(g_list_nth_data(token->data, 2));
  
	  r_notes_set_know_anniversary(notes, TRUE);
	  r_notes_set_anniversary(notes, day, month, year);
	}
    break;
      
    case RUBRICA_EXT_Manager:
    case KDE_EXT_ManagersName:
    case EVOLUTION_EXT_Manager:
      if (token->data && token->data->data)
	g_object_set(work, "manager-name", token->data->data, NULL);
      break;

    case RUBRICA_EXT_ManagerTelephone:
      if (token->data && token->data->data)
	g_object_set(work, "manager-phone", token->data->data, NULL);
      break;

    case KDE_EXT_Office:
    case EVOLUTION_EXT_Office:
      /* ignore */
      break;

    case KDE_EXT_CUSTOM:  
      /* ignore */
      break;
      
      /* Company tokens 
       */
    case RUBRICA_EXT_CompanyName:
      if (token->data && token->data->data)
	{
	  if (!priv->have_company) 
	    company = r_vcard_company_new(vcard, R_CARD(card));
	  
	  g_object_set(company, "company-name", token->data->data, NULL); 
	  g_object_set(R_CARD(company), "card-name", token->data->data, NULL);
	}
      break;

    case RUBRICA_EXT_Street:
      if (token->data && token->data->data)
	{
	  if (!priv->have_company)
	    company = r_vcard_company_new(vcard, R_CARD(card));
	  
	  if (!priv->have_company_address)
	    company_address = r_vcard_company_address_new(vcard);
	  
	  g_object_set(company_address, "street", token->data->data, NULL);
	}
      break;
      
    case RUBRICA_EXT_StreetNumber:
      if (token->data && token->data->data)
	{
	  if (!priv->have_company)
	    company = r_vcard_company_new(vcard, R_CARD(card));
       
	  if (!priv->have_company_address)
	    company_address = r_vcard_company_address_new(vcard);
	  
	  g_object_set(company_address, "street-number", 
		       token->data->data, NULL);
	}
      break;
      
    case RUBRICA_EXT_ZipCode:
      if (token->data && token->data->data)
	{
	  if (!priv->have_company) 
	    company = r_vcard_company_new(vcard, R_CARD(card));
	  
	  if (!priv->have_company_address)
	    company_address = r_vcard_company_address_new(vcard);
	  
	  g_object_set(company_address, "zip", token->data->data, NULL);
	}
      break;
      
    case RUBRICA_EXT_City:
      if (token->data && token->data->data)
	{
	  if (!priv->have_company) 
	    company = r_vcard_company_new(vcard, R_CARD(card));
	  
	  if (!priv->have_company_address)
	    company_address = r_vcard_company_address_new(vcard);
	  
	  g_object_set(company_address, "city", token->data->data, NULL);
	}
      break;
      
    case RUBRICA_EXT_Province:
      if (token->data && token->data->data)
	{
	  if (!priv->have_company)
	    company = r_vcard_company_new(vcard, R_CARD(card));
	  
	  if (!priv->have_company_address)
	    company_address = r_vcard_company_address_new(vcard);
	  
	  g_object_set(company_address, "province", token->data->data, NULL);
	}
      break;
      
    case RUBRICA_EXT_Country:
      if (token->data && token->data->data)
	{
	  if (!priv->have_company) 
	    company = r_vcard_company_new(vcard, R_CARD(card));
	  
	  if (!priv->have_company_address)
	    company_address = r_vcard_company_address_new(vcard);
	  
	  g_object_set(company_address, "country", token->data->data, NULL);
	}
      break;
      
    case RUBRICA_EXT_Web:
      if (token->data && token->data->data)
	{
	  if (!priv->have_company) 
	    company = r_vcard_company_new(vcard, R_CARD(card));
	  
	  company_net = r_net_address_new();
	  g_object_set(company_net, 
		       "url", token->data->data, 
		       "url-type", R_NET_ADDRESS_WEB, NULL);
	  r_card_add_net_address(R_CARD(company), company_net);
	}
      break;
      
    case RUBRICA_EXT_Email:
      if (token->data && token->data->data)
	{
	  if (!priv->have_company) 
	    company = r_vcard_company_new(vcard, R_CARD(card));
	  
	  company_net = r_net_address_new();
	  g_object_set(company_net, 
		       "url", token->data->data, 
		       "url-type", R_NET_ADDRESS_EMAIL, NULL);
	  r_card_add_net_address(R_CARD(company), company_net);
	}
      break;
      
    case RUBRICA_EXT_Operator:
      if (token->data && token->data->data)
	{
	  if (!priv->have_company) 
	    company = r_vcard_company_new(vcard, R_CARD(card));
	  
	  company_telephone = r_telephone_new();
	  g_object_set(company_telephone, 
		       "telephone-number", token->data->data, 
		       "telephone-type", R_TELEPHONE_OPERATOR, NULL);
	  r_card_add_telephone(R_CARD(company), company_telephone);
	}
      break;

    case RUBRICA_EXT_Fax:
      if (token->data && token->data->data)
	{
	  if (!priv->have_company) 
	    company = r_vcard_company_new(vcard, R_CARD(card));
	  
	  company_telephone = r_telephone_new();
	  g_object_set(company_telephone, 
		       "telephone-number", token->data->data, 
		       "telephone-type", R_TELEPHONE_FAX, NULL);
	  r_card_add_telephone(R_CARD(company), company_telephone);
	}
      break;
      
    case RUBRICA_EXT_Green:
      if (token->data && token->data->data)
	{
	  if (!priv->have_company) 
	    company = r_vcard_company_new(vcard, R_CARD(card));
	  
	  company_telephone = r_telephone_new();
	  g_object_set(company_telephone, 
		       "telephone-number", token->data->data, 
		       "telephone-type", R_TELEPHONE_GREEN, NULL);      
	  r_card_add_telephone(R_CARD(company), company_telephone);   
	}
      break;
      
    case RUBRICA_EXT_CustomerCare:
      if (token->data && token->data->data)
	{
	  if (!priv->have_company) 
	    company = r_vcard_company_new(vcard, R_CARD(card));
	  
	  company_telephone = r_telephone_new();
	  g_object_set(company_telephone, 
		       "telephone-number", token->data->data, 
		       "telephone-type", R_TELEPHONE_CUSTOMER_CARE, NULL);
	  r_card_add_telephone(R_CARD(company), company_telephone);
	}
      break;

    case EVOLUTION_EXT_File_as:
      /* ignore */
      break;
		
    default:   
      break;
    }

  token->previous = token->token;			
  free_token_data(token);
}


/*
 N:Verger;Gèrard Verger;;;
 X-RUBRICA-X-Group;OWNER=rubrica:relatives;go-home
 TEL;TYPE=work:verger@spid.com
 EMAIL;INTERNET:djangelo@gmail.com
 URL:http://rubrica.berlios.de
*/
static void
decode_vcard_buffer(RVCard* vcard, gchar* line)
{
  RVCardPrivate* priv;
  gchar *tok, *data; 
  Token* token;

  g_return_if_fail(IS_R_VCARD(vcard));
  g_return_if_fail(line != NULL);
	
  priv = R_VCARD_GET_PRIVATE(vcard);	
		
  token = priv->token;	
  token->buffer = g_strdup(line);

  tok  = strtok(line, ":");
  data = strtok(NULL, "\n");
  split_token(token, tok);

  if (token->token != BAD_TOKEN)
    split_data(token, data);
}


/*
  N
  X-RUBRICA-X-Group;OWNER=rubrica
  TEL;TYPE=work
  EMAIL;INTERNET
*/
static void 
split_token(Token* token, gchar* str)
{
  gchar* tok;

  g_return_if_fail(str != NULL);

  tok = strtok(str, ";");
  token->token = r_lookup_table_str2enum(tokens, tok);

  while ((tok = strtok(NULL, ";\n")))
    split_params(token, tok);
}


/*
  Fragale;Nicola Fragale;;;
  relatives;go-home
  Traduttori;/home/nicola/.rubrica2/groups/traduttori.png
  djangelo@gmail.com
*/ 
static void 
split_data(Token* token, gchar* str)
{
  gchar** splits;
  gchar** alias;

  g_return_if_fail(str != NULL);

  if (token->token != BDAY)
    alias = splits = g_strsplit(str, ";", -1);
  else
    {
      alias = splits = g_strsplit(str, "-", -1);
      
      if (splits[2][2] == 'T')
	// BDAY:2001-10-21T15:53:29 ==> 21T15:53:29 
	// drop time 
	splits[2][2] = '\0';
    }
    
  for(; *alias; alias++)
    token->data = g_list_append(token->data, g_strdup(*alias));
  
  g_strfreev (splits);    
}

/*  
  TYPE=work;TYPE=home
  INTERNET
  RATE=2
*/
static void 
split_params(Token* token, gchar* str)
{
  gchar* tok;
  Params *param;

  g_return_if_fail(str != NULL);

  param = g_malloc0(sizeof(Params));

  tok = strtok(str, ";=");
  if (tok)
    {
      if (g_ascii_strcasecmp(tok, "INTERNET") == 0)
	{
	  param->param = TYPE;
	  param->value = INTERNET;	    
	}
      else if (token->token == TEL)
	{	  
	  RVCardValueType value;

	  param->param = TYPE;
	  tok   = strtok(NULL, "=\n");
	  value = r_telephone_lookup_str2enum(tok);
	  if (value > R_TELEPHONE_UNKNOWN)
	    param->value = R_TELEPHONE_UNKNOWN;
	  else
	    param->value = value;
	}
      else
	param->param = r_lookup_table_str2enum(params, tok);
    }

  tok = strtok(NULL, "=\n");
  if (tok)
    {
      if (param->param == RATE)
	param->value = atoi(tok);
      else
	param->value = r_lookup_table_str2enum(values, tok);
    }

  token->params = g_list_append(token->params, param);
}



static void 
delete_data(GList* data)
{
  if (data->next)
    delete_data(data->next);
  
  g_free(data->data);
}


static void
free_token_data(Token* token)
{
  g_return_if_fail(token != NULL);

  if (token->params)
    {
      g_list_free(token->params);	
      token->params = NULL;
    }

  if (token->data)
    {
      delete_data(token->data);
      token->data   = NULL;  
    }

  if (token->buffer)
    {
      g_free(token->buffer);
      token->buffer = NULL;
    }
}



static void
write_card (RCard* card, FILE* fp)
{
  gpointer data;
  gchar *id, *name, *cardtype, *tmp, *type;
  gboolean locked, deleted;
  gint created, changed;
  RRate rate;
  gchar *day, *month, *year;
  gchar *first, *middle, *last, *nick, *prefix, *prof, *title, *photo;

  fprintf(fp, "\n");
  fprintf(fp, "BEGIN:VCARD\n");
  fprintf(fp, "VERSION:3.0\n");

  g_object_get(card, 
	       "card-id",      &id,      "card-type",    &cardtype,
	       "card-name",    &name,    "card-locked",  &locked, 
	       "card-deleted", &deleted, "card-rate",    &rate, 
	       "card-created", &created, "card-changed", &changed,
	       NULL);
  
  if (name)
    fprintf(fp, "FN:%s\n", name);

  /*      Card's data (rubrica's extensions)
  */
  tmp = g_strdup_printf("%d", rate);
  fprintf(fp, "X-RUBRICA-X-Card;RATE=%s:%s;%s\n", tmp,
	  locked  ? "true" : "false", deleted ? "true" : "false");
  g_free(tmp);
  
  /*      Groups (rubrica's extensions)
  */
  data = r_card_get_group(R_CARD(card));
  for (; data; data = r_card_get_next_group(R_CARD(card)))
    {
      gchar* owner;
      gchar* pixmap = NULL;

      g_object_get(R_GROUP(data), 
		   "group-name",   &name, 
		   "group-owner",  &owner, 
		   "group-pixmap", &pixmap, NULL);
      
      fprintf(fp, "X-RUBRICA-X-Group;OWNER=%s:%s;%s\n", 
	      owner, 
	      name, 
	      pixmap);
    }


  /*      References (rubrica's extensions)
  */
  data = r_card_get_ref(R_CARD(card));
  for (; data; data = r_card_get_next_ref(R_CARD(card)))
    {
      glong id;
      gchar *info, *tmp;

      g_object_get(R_REF(data), "ref-to", &id, "ref-info", &info, NULL);
      tmp = g_strdup_printf("%ld", id);
      
      fprintf(fp, "X-RUBRICA-X-Ref;ID=%s:%s\n", tmp, info);
      g_free(tmp);
    }  

  if (g_ascii_strcasecmp(cardtype, "personal") == 0)
    {
      /*
	The structured type value corresponds, in sequence, to:
	the Family Name,               (pos 0)
	Given Name,                    (pos 1)
	Additional Names,              (pos 2)
	Honorific Prefixes,            (pos 3)
	Honorific Suffixes.            (pos 4)
	The text components are separated by the SEMI-COLON character
	Individual text components can include multiple text values 
	(e.g., multiple Additional Names) separated by the COMMA character 
      */
      /*      Personal data
       */
      data = r_personal_card_get_contact(R_PERSONAL_CARD(card));
      g_object_get(R_CONTACT(data), "first-name", &first, 
		   "middle-name", &middle, "last-name", &last, 
		   "nick-name", &nick, "profession", &prof, 
		   "prefix", &prefix, "title", &title, "photo", &photo, NULL);

      if (first || middle || last || prefix || title)
	fprintf(fp, "N:%s;%s;%s;%s;%s\n", 
		last   ? last   : "", 
		first  ? first  : "",
		middle ? middle : "",
		prefix ? prefix : "",
		title  ? title  : "");
    
      // BDAY:year-month-day
      day   = r_contact_get_birth_day   (R_CONTACT(data));
      month = r_contact_get_birth_month (R_CONTACT(data));
      year  = r_contact_get_birth_year  (R_CONTACT(data)); 
      if ((day   && (g_ascii_strcasecmp(day, "BadDay")   != 0)) && 
	  (month && (g_ascii_strcasecmp(day, "BadMonth") != 0)) &&
	  (year  && (g_ascii_strcasecmp(day, "BadYear")  != 0)))	
	fprintf(fp, "BDAY:%s-%s-%s\n", year, month, day);      
    }
  else
    {
      /*    TODO ***** Company  ******
	    
      fprintf(fp, ":%s\n", );
      
      
      
      */
    } 
  
  /* 
     ADR Type Definition: (RFC2426)
     
     order:
     post office box;                       // ignored   (pos: 0)
     the extended address;                  // ignored   (pos: 1)
     the street address;                    (pos 2)
     the locality (e.g., city);             (pos 3)
     the region (e.g., state or province);  (pos 4)
     the postal code;                       (pos 5)
     the country name                       (pos 6)
  */
  /*      Addresses
   */
  data = r_card_get_address(R_CARD(card));
  for (; data; data = r_card_get_next_address(R_CARD(card)))
    {
      if (IS_R_ADDRESS(data))
	{
	  RAddressType adtype = R_ADDRESS_UNKNOWN;
	  gchar *street, *city, *zip;
	  gchar *state, *country;

	  g_object_get(R_ADDRESS(data), 
		       "address-type", &adtype, 
		       "street", &street, "city", &city, "zip", &zip, 
		       "state", &state, "country", &country, NULL);
	  
	  type = r_address_lookup_enum2str(adtype);
	  if (street || city || state || zip || country)
	    fprintf(fp, "ADR;TYPE=%s:%s;%s;%s;%s;%s;%s;%s\n", type, 
		    "", "", // Post office; extended address (ignored)
		    street  ? street  : "", 
		    city    ? city    : "", 
		    state   ? state   : "", 
		    zip     ? zip     : "", 
		    country ? country : "");
	}
    }
  
  
  /*      Net (emails, web urls)
  */
  data = r_card_get_net_address(R_CARD(card));
  for (; data; data = r_card_get_next_net_address(R_CARD(card)))
    if (IS_R_NET_ADDRESS(data))
      {
	RNetAddressType nettype;
	gchar *url;
	
	g_object_get(R_NET_ADDRESS(data), 
		     "url", &url, "url-type", &nettype, NULL);
	type = r_net_address_decode_type(nettype);
	
	switch (nettype)
	  {
	  case R_NET_ADDRESS_EMAIL: 
	    fprintf(fp, "EMAIL;INTERNET:%s\n", url);
	    break;

	  case R_NET_ADDRESS_WEB:
	    fprintf(fp, "URL:%s\n", url);
	    break;

	  case R_NET_ADDRESS_EKIGA:
	  case R_NET_ADDRESS_IRC:
	  case R_NET_ADDRESS_IRC_AIM:
	  case R_NET_ADDRESS_IRC_ICQ:
	  case R_NET_ADDRESS_IRC_JABBER:
	  case R_NET_ADDRESS_IRC_YAHOO:
	  case R_NET_ADDRESS_IRC_MSN:
	  case R_NET_ADDRESS_WORK_WEB:
	  case R_NET_ADDRESS_WORK_EMAIL:
	  case R_NET_ADDRESS_UNKNOWN:
	    fprintf(fp, "X-RUBRICA-X-URL;TYPE=%s:%s\n", type, url);
	    break;
	    
	  default:
	    break;	      
	  }
      }
     
  
  /*     Telephone
  */
  data = r_card_get_telephone(R_CARD(card));
  for (; data; data = r_card_get_next_telephone(R_CARD(card)))
    if (IS_R_TELEPHONE(data))
      {
	gchar *number, *type; 
	RTelephoneType ttype;
	
	g_object_get(R_TELEPHONE(data), 
		     "telephone-number", &number, 
		     "telephone-type", &ttype, NULL);
	type = r_telephone_lookup_enum2str(ttype);
	
	fprintf(fp, "TEL;TYPE=%s:%s\n", type, number);
      }


  /*      Work
  */
  if (g_ascii_strcasecmp(cardtype, "personal") == 0)
    {
      data = r_personal_card_get_work(R_PERSONAL_CARD(card));
      if (IS_R_WORK(data))
	{     
	  gchar *assignment, *org, *dep, *subdep;
	  gchar *manager, *mphone, *collaborator, *cphone;
	  
	  g_object_get(R_WORK(data), 
		       "assignment",     &assignment, 
		       "organization",   &org, 
		       "department",     &dep, 
		       "sub-department", &subdep, 
		       "manager-name",   &manager, 
		       "manager-phone",  &mphone, 
		       "collaborator",   &collaborator, 
		       "collaborator-phone", &cphone, NULL);
	  if (assignment)
	    fprintf(fp, "ROLE:%s\n", assignment);
	  if (org)
	    fprintf(fp, "ORG:%s\n", org);
	  if (dep)
	    fprintf(fp, "X-RUBRICA-Department:%s\n", dep);
	  if (subdep)
	    fprintf(fp, "X-RUBRICA-X-SubDepartment:%s\n", subdep);
	  if (manager)
	    fprintf(fp, "X-RUBRICA-X-Manager:%s\n", manager);
	  if (mphone)
	    fprintf(fp, "X-RUBRICA-X-ManagerTelephone:%s\n", mphone);
	  if (collaborator)
	    fprintf(fp, "X-RUBRICA-X-Collaborator:%s\n", collaborator);
	  if (cphone)
	    fprintf(fp, "X-RUBRICA-X-CollaboratorTelephone:%s\n", cphone);
	} 
    }  
  
  /*      Notes
  */
  if (g_ascii_strcasecmp(cardtype, "personal") == 0)
    {
      data = r_personal_card_get_notes(R_PERSONAL_CARD(card));
      if (IS_R_NOTES(data))
	{
	  gchar *other, *pkey;
	  
	  g_object_get(R_NOTES(data), 
		       "other-notes", &other, 
		       "pubkey", &pkey, NULL);
	  
	  if (other)
	    fprintf(fp, "NOTE:%s\n", other);
	  
	  if (pkey)
	    fprintf(fp, "KEY:%s\n", pkey);  
	}
    }

  fprintf(fp, "END:VCARD\n\n");
}

static gboolean 
is_vcard_doc (RVCard* vcard, const gchar* filename)
{
  gchar* buffer;
  GIOChannel* channel = NULL;
  GError* error = NULL;
  Token* token;
  gboolean done = FALSE;
  gboolean ret = FALSE;

  channel = g_io_channel_new_file(filename, "r", &error);
  if (!channel)
    g_error("channel: %s", error->message);
  
  while (!done)
    {
      g_io_channel_read_line(channel, &buffer, NULL, NULL, &error);
      
      if (*buffer != '\n')
	{
	  gchar *data = NULL;
	  gint len = g_utf8_strlen(buffer, -1);
	  
	  if (buffer[len-1] == '\n')
	    {
	      if (buffer[len-2] == '\r')
		buffer[len-2] = '\0';
	      else
		buffer[len-1] = '\0';
	    }
	    
	  decode_vcard_buffer(vcard, buffer);
 	  done  = TRUE;
	  token = vcard->priv->token;
	  if (token->data)
	    data  = (gchar*) token->data->data;
	  
	  if (token->token == BEGIN && (g_ascii_strcasecmp(data,"VCARD") == 0))
	    ret = TRUE;
	}   
    }
  
  g_io_channel_shutdown(channel, TRUE, NULL);
  g_io_channel_unref(channel);
  g_free(buffer);
  return ret;
}


/* Public functions
 */

RVCard* 
r_vcard_new(void)
{
  RVCard *vcard;

  vcard = g_object_new(r_vcard_get_type(), NULL);

  return (RVCard*) vcard;
}


void 
r_vcard_free (RVCard *vcard)
{
  g_return_if_fail(IS_R_VCARD(vcard));

  g_object_unref(vcard);
}


static gboolean 
r_vcard_write_file (RAbook* abook, gchar* filename, gint crate)
{
  FILE* fp;

  g_return_val_if_fail(IS_R_ABOOK(abook), FALSE);
  g_return_val_if_fail(filename != NULL, FALSE);
  
  if (!(fp = fopen(filename, "w")))
    {
      g_warning("\nCan't write file: %s", filename);
      g_signal_emit_by_name(abook, "save_fail", WRITING_FILE, G_TYPE_INT);
      
      return FALSE;
    }
  
  r_abook_foreach_card (abook, (RFunc) write_card, fp);
  fflush(fp);

  g_signal_emit_by_name(abook, "addressbook_saved", NULL, G_TYPE_NONE);

  return TRUE;
}


static gboolean 
r_vcard_overwrite_file(RAbook* abook, gint compression_rate)
{
  gchar* file;
  gchar* path;
  gchar* filename;
  
  g_return_val_if_fail(IS_R_ABOOK(abook), FALSE);
  
  g_object_get(abook, "addressbook-path", &path, 
	       "addressbook-name", &file, NULL);
  filename = g_strdup_printf("%s%s%s", path, G_DIR_SEPARATOR_S, file);
  
  if (g_file_test(filename, G_FILE_TEST_EXISTS))
    g_remove(filename);

  if (!r_vcard_write_file(abook, filename, compression_rate)) 
    {
      g_signal_emit_by_name(abook, "save_fail", OVERWRITING, G_TYPE_INT);
      
      g_free(filename);
      return FALSE;
    }
    
  g_free(filename);
  
  return TRUE;
}



static gboolean 
r_vcard_open_file (RAbook* abook, gchar* filename)
{
  RVCard* vcard;
  gchar* buffer;
  GIOChannel* channel = NULL;
  GError* error = NULL;
  
  g_return_val_if_fail(IS_R_ABOOK(abook), FALSE);
  
  if (!filename)
    {
      g_signal_emit_by_name(R_ABOOK(abook), "open_fail", 
			    NO_FILENAME, G_TYPE_INT);

      return FALSE;
    }

  if (!g_file_test(filename, G_FILE_TEST_EXISTS))
    {
      g_signal_emit_by_name(R_ABOOK(abook), "open_fail", 
			    FILE_NOT_EXIST, G_TYPE_INT);
      
      return FALSE;
    }
  
  vcard = (RVCard*) r_abook_get_engine(abook);

   if (!IS_R_VCARD(vcard))
     return FALSE;
     
  if (!is_vcard_doc(vcard, filename))
    return FALSE;
  
  g_object_set(G_OBJECT(vcard), 
	       "filename", filename, "addressbook", abook, NULL);

  channel = g_io_channel_new_file(vcard->filename, "r", &error);
  if (!channel)
    g_error("channel: %s", error->message);
  
  /* g_print non stampa gli unicode in modo corretto, usare printf
  */
  while (g_io_channel_read_line(channel, &buffer, NULL, NULL, &error) != 
	 G_IO_STATUS_EOF)
    {      
      if (*buffer != '\n')
	{
	  gint len = g_utf8_strlen(buffer, -1);
	  
	  if (buffer[len-1] == '\n')
	    {
	      if (buffer[len-2] == '\r')
		buffer[len-2] = '\0';
	      else
		buffer[len-1] = '\0';
	    }
	  
	  decode_vcard_buffer(vcard, buffer);
	  build_card(vcard);  
	}
    }   
  
  g_object_set(R_ABOOK(abook),
	       "addressbook-name", g_path_get_basename(filename),
	       "addressbook-path", g_path_get_dirname(filename), NULL);

  g_free(buffer);
  return TRUE;
}



G_MODULE_EXPORT void
plugin_init (RPlugin* plugin, gchar* file)
{
  RFilter* filter;
  RPluginAction* action;

  g_return_if_fail(plugin != NULL);

#ifdef ENABLE_DEBUG
  g_log(G_LOG_DOMAIN, G_LOG_LEVEL_DEBUG, "plugin_init");
  g_log(G_LOG_DOMAIN, G_LOG_LEVEL_DEBUG, "Initializing Vcard plugin");
#else
  g_log(G_LOG_DOMAIN, G_LOG_LEVEL_INFO, "Initializing Vcard plugin");
#endif

  r_plugin_set_engine(plugin, r_vcard_new());
  g_object_set(plugin, 
	       "plugin-name", "vcard", 
	       "plugin-filename", file,
	       "plugin-label", "vcard's file format",
	       "plugin-info", "This plugin manages the vcard file format",
	       "plugin-extensions", "vcard, vcr, vcd",
	       "plugin-configurable", FALSE, NULL);
  
  filter = r_filter_new();
  g_object_set(filter, 
	       "filter-name", "vcard", 
	       "filter-extension", "vcard",
	       "filter-mime", "text/x-vcard", NULL);
  r_filter_add_pattern(filter, "vcard");
  r_filter_add_pattern(filter, "vcr");
  r_filter_add_pattern(filter, "vcd");
  r_filter_add_pattern(filter, "*.vcr");
  r_filter_add_pattern(filter, "*.vcd");
  r_filter_add_pattern(filter, "*.vcard");

  r_plugin_add_filter (plugin, filter);
  
  action = g_malloc(sizeof(RPluginAction));
  action->name   = g_strdup("read");
  action->handle = (gpointer) r_vcard_open_file;
  r_plugin_add_action(plugin, action);  
    
  action = g_malloc(sizeof(RPluginAction));
  action->name   = g_strdup("write");
  action->handle = (gpointer) r_vcard_write_file;;
  r_plugin_add_action(plugin, action);  

  action = g_malloc(sizeof(RPluginAction));
  action->name   = g_strdup("overwrite");
  action->handle = (gpointer) r_vcard_overwrite_file;
  r_plugin_add_action(plugin, action);  

  g_log(G_LOG_DOMAIN, G_LOG_LEVEL_INFO, "Done"); 
  g_log(G_LOG_DOMAIN, G_LOG_LEVEL_INFO, " ");
}


G_MODULE_EXPORT void     
plugin_fini (void)
{
}

























