/*
*  program: Rubrica
*  file: app.c
*
*  
*  Copyright (C) 2000-2005 Nicola Fragale <nicolafragale@libero.it>
*
*  This program is free software; you can redistribute it and/or modify
*  it under the terms of the GNU General Public License as published by
*  the Free Software Foundation; either version  3 of the License
*
*  This program is distributed in the hope that it will be useful,
*  but WITHOUT ANY WARRANTY; without even the implied warranty of
*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*  GNU General Public License for more details.
*
*  You should have received a copy of the GNU General Public License
*  along with this program; if not, write to the Free Software
*  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/


#include <gtk/gtk.h>
#include <glade/glade.h>
#include <gconf/gconf-client.h>
#include <gdk/gdkkeysyms.h>
#include <glib/gi18n-lib.h>
#include <errno.h>

#include <libnotify/notify.h>
#include <libnotify/notification.h>

#include "types.h"

#include "app.h"
#include "models.h"
#include "cards.h"
#include "callbacks.h"
#include "preferences.h"
#include "groups_cb.h"
#include "popmenu.h"
#include "browse.h"
#include "print.h"
#include "cards_view.h"
#include "trash_view.h"
#include "search_view.h"
#include "data_view.h"
#include "groups_view.h"
#include "search_dialog.h"
#include "statusbar.h" 
#include "tab.h"

#include "utils.h"



/*  pixmap,                     name,           label   */
static RubricaDefaultCategories defcats[] = {
  {"all",                      "all groups",    N_("all groups")   },

  {"flame",                    "hot contacts",  N_("hot contacts") },
  {"system-users",             "friends",       N_("friends")      },
  {"emblem-favorite",          "heart",         N_("heart")        },
  {"go-home",                  "relatives",     N_("relatives")    },
  
  {"chat",                     "chat",          N_("chat")         },
  
  {"emblem-photos",            "holidays",      N_("holidays")     },
  {"applications-multimedia",  "hobbies",       N_("hobbies")      },
  {"sport",                    "sport",         N_("sport")        },

  {"bussines",                 "bussines",      N_("bussines")     },
  {"system-file-manager",      "company",       N_("company")      },
  {"application-certificate",  "legal",         N_("legal")        },
  {"applications-office",      "office",        N_("office")       },
  {"applications-development", "work",          N_("work")         },
  {"customer",                 "customers",     N_("customers")    },
  {"strategies",               "strategies",    N_("strategies")   },

  {"gnu",                      "free software", N_("free software")},
  {"tux",                      "programmers",   N_("programmers")  },
  {"freebsd",                  "BSD friends",   N_("BSD friends")  },

  {"university",               "university",    N_("university")   },
  {NULL, NULL}
};
 
 
typedef struct _RubricaMenu {
  GtkWidget* menubar;
  GtkWidget* new;
  GtkWidget* open;
  GtkWidget* save;
  GtkWidget* saveas;
  GtkWidget* quit;

  GtkWidget* cut;
  GtkWidget* copy;
  GtkWidget* paste;

  GtkWidget* personal;  
  GtkWidget* company;
  GtkWidget* modify;
  GtkWidget* delete;

  GtkWidget* prefer;

  GtkWidget* about;
} RubricaMenu;



typedef struct _RubricaToolbar {
  GtkWidget* handle;
  GtkWidget* toolbar;

  GtkWidget* new;
  GtkWidget* open;
  GtkWidget* save;
  GtkWidget* print;
  GtkWidget* close;

  GtkWidget* add;
  GtkWidget* choose;
  GtkWidget* delete;
  GtkWidget* modify;

  GtkWidget* next;
  GtkWidget* previous;
  GtkWidget* first;
  GtkWidget* last;
  GtkWidget* trash;
  GtkWidget* search;
  GtkWidget* quit;
} RubricaToolbar;



struct _RubricaAppPrivate {
  RGroupBox      *groups;         // rubrica's groups

  RubricaMenu    *menu;
  RubricaToolbar *toolbar;
  GtkWidget      *statusbar;

  GtkWidget      *notebook;       // contains addressbooks  (CardsView)

  GtkWidget      *groups_view;
  GtkWidget      *trash_view;     // trash notebook's page  (TrashView)
  GtkWidget      *search_view;    // search notebook's page (SearchView)
  GtkWidget      *data_view;      // dispay card's data     (DataView)

  GtkWidget      *paned;
  GtkWidget      *group_paned;

  GList          *cutted;         // stack of cutted cards
     
  NotifyNotification* notify;
  gchar          *active_plugin;       
  RPluginManager *manager;

  GtkPrintSettings* print_settings;

  gboolean show_card;

  gboolean dispose_has_run;
};



/*    signals enumeration 
 */

static void rubrica_app_class_init  (RubricaAppClass* klass);
static void rubrica_app_init        (RubricaApp* obj);

static void rubrica_app_dispose     (RubricaApp* obj);
static void rubrica_app_finalize    (RubricaApp* obj);


/*   Private
*/
static void            init_group_box      (RGroupBox* box);
static void            add_card_groups2set (RubricaApp* app, RCard* card);

static RubricaMenu*    build_menu          (GladeXML* gui, RubricaApp* app);
static RubricaToolbar* build_toolbar       (GladeXML* gui, RubricaApp* app);

static GtkWidget*      find_addressbook    (RubricaApp* app, RAbook* book);
static void            load_addressbooks   (RubricaApp* app);
static void            save_addressbook    (RubricaApp* app, RAbook* book);
static void            display_cards_list  (RubricaApp* app, GList *ids);

static GtkWidget*      get_current_tab            (RubricaApp* app);
static GtkWidget*      get_current_tab_label      (RubricaApp* app);
static gchar*          get_current_tab_label_text (RubricaApp* app);


/*   Private callbacks
*/
static gboolean on_cards_event_occurred  (GtkTreeView* tree, GdkEvent *event,
					  gpointer data);
static void     on_trash_event_occurred  (GtkTreeView* tree, GdkEvent *event, 
					  gpointer data);
static void     on_search_event_occurred (GtkTreeView* tree, GdkEvent *event, 
					  gpointer data);
static gboolean on_key_press_event_cb    (GtkWidget* w, GdkEventKey* event,
					  gpointer data);

static void hide_trash_on_click  (GtkWidget* button, gpointer data);
static void hide_search_on_click (GtkWidget* button, gpointer data);

static void on_card_added        (RAbook* book, gpointer card, gpointer data);
static void on_card_destroyed    (RAbook* book, gchar* arg, gpointer data);
static void on_card_deleted      (RAbook* book, gpointer id, gpointer data);
static void on_card_recovered    (RAbook* book, gpointer card, gpointer data);
static void on_card_modified     (RAbook* book, gpointer card, gpointer data);

static void on_show_menu_cb      (GtkMenuToolButton *mtbutton,  gpointer data);
static void on_trash_toggled_cb  (GtkToggleToolButton *toggle,  gpointer data);

static void on_addressbook_changed   (RAbook* book, gpointer data);
static void on_addressbook_saved     (RAbook* book, gpointer data);
static void on_addressbook_save_fail (RAbook* book, gint err, gpointer data);

static void on_autosave_changed      (GConfClient *client, guint cnxn_id, 
				      GConfEntry *entry, gpointer data);

static void on_launch_cb             (GtkWidget* data_view, gpointer link, 
				      gpointer data);

static void on_app_view_changed      (GtkWidget* view, gchar* message, 
				      gpointer data);
static void on_cards_view_changed    (GtkWidget* view, gchar* message, 
				      gpointer data);
static void on_card_marked           (RubricaCardsView* view, glong card_id, 
				      RubricaCardsViewMark mark, 
				      gpointer data);



/*  Private
*/
static void
init_group_box(RGroupBox* box)
{
  RGroup* group;
  RubricaDefaultCategories* cat = defcats;

  for(; cat->label; cat++) 
    {
      group = r_group_new();
      
      g_object_set(group, 
		   "group-name",  cat->label, 
		   "group-label", _(cat->label), 
		   "group-owner", "rubrica",
		   "group-pixmap", cat->pixmap,
		   "enabled", FALSE, NULL); 
      
      r_group_box_add_group(box, group);
    }    
}


static void 
add_card_groups2set(RubricaApp* app, RCard* card)
{
  RGroup* group;
  
  r_card_reset_group(card);
  group = r_card_get_group(card);
  for(; group; group = r_card_get_next_group(card))
    r_group_box_add_group(app->priv->groups, group);
}


static void 
load_addressbooks (RubricaApp* app)
{
  GConfClient* client;
  GSList* files;
  
  client = gconf_client_get_default();
  files  = gconf_client_get_list(client, RUBRICA_GCONF_FILES, 
				 GCONF_VALUE_STRING, NULL);
  
  //  rubrica_app_set_plugin_name(app, "autodetect");  
  for (; files; files = files->next) 
    rubrica_app_load_file(app, (gchar*) files->data);
      
  g_object_unref(client);  
}


static void
save_addressbook(RubricaApp* app, RAbook* book)
{
  GConfClient* client;
  gboolean backup;
  gint rate;
  gchar* filename = NULL;
  
  if (!book)
    book = rubrica_app_get_active_addressbook(app);

  client = gconf_client_get_default();
  
  backup = gconf_client_get_bool(client, RUBRICA_GCONF_BACKUP, NULL);
  rate   = gconf_client_get_int (client, RUBRICA_GCONF_RATE, NULL);
 
  g_object_get(book, "addressbook-name", &filename, NULL);

  if (!filename || (g_ascii_strcasecmp(_("no name"), filename) == 0))
    on_saveas_cb(NULL, app);
  else
    r_abook_overwrite_file(book, backup, rate);  

  g_object_unref(client);
}


static GtkWidget*
get_current_tab(RubricaApp* app)
{
  GtkWidget* child;
  GtkWidget* tab;
  gint page;
  
  page  = gtk_notebook_get_current_page(GTK_NOTEBOOK(app->priv->notebook));
  child = gtk_notebook_get_nth_page (GTK_NOTEBOOK(app->priv->notebook), page);
  g_object_get(child, "view-brother", &tab, NULL);  
 
  return tab;
}


static GtkWidget*
get_current_tab_label(RubricaApp* app)
{
  GtkWidget* tab;
  GtkWidget* label = NULL;
  
  tab   = get_current_tab(app);
  label = rubrica_tab_get_label(RUBRICA_TAB(tab));

  return label;
}


static gchar* 
get_current_tab_label_text(RubricaApp* app)
{
  GtkWidget* tab;
  gchar* text;

  tab = get_current_tab(app);
  g_object_get(tab, "tab-label", &text, NULL);
  
  return text;
}


static GtkWidget*
find_addressbook (RubricaApp* app, RAbook* book)
{
  gint n;
  
  g_return_val_if_fail(IS_RUBRICA_APP(app), NULL);
  g_return_val_if_fail(IS_R_ABOOK(book), NULL);
  
  n = gtk_notebook_get_n_pages(GTK_NOTEBOOK(app->priv->notebook));
  for (; n; n--)
    {
      GtkWidget* view;
      RAbook* adb;
      
      view = gtk_notebook_get_nth_page(GTK_NOTEBOOK(app->priv->notebook), n);
      adb = rubrica_cards_view_get_addressbook(RUBRICA_CARDS_VIEW(view));
      if (adb == book)
	return view;
    }
  
  return NULL;
}



static void
display_cards_list(RubricaApp* app, GList *ids)
{
  GtkWidget* view;
  gint cards; 
  gchar* message;
  
  view  = rubrica_app_get_current_view(app);
  cards = g_list_length(ids);

  rubrica_cards_view_display_results(RUBRICA_CARDS_VIEW(view), ids);

  if (cards > 1)
    message = g_strdup_printf(_("%d cards found"), cards);
  else
    message = g_strdup_printf(_("%d card found"), cards);
  rubrica_app_notify(app, message);
  
  g_free(message);
}



/*  Callbacks
*/
static void 
on_app_view_changed(GtkWidget* view, gchar* message, gpointer data)
{
  RubricaStatusbar* statusbar = RUBRICA_STATUSBAR(data);
  
  rubrica_statusbar_enable_extra (statusbar, FALSE);
  rubrica_statusbar_enable_images(statusbar, FALSE);
  rubrica_statusbar_push_message(statusbar, message);
}


static void 
on_cards_view_changed(GtkWidget* view, gchar* message, gpointer data)
{
  RubricaStatusbar* statusbar = RUBRICA_STATUSBAR(data);
  
  rubrica_statusbar_enable_extra (statusbar, TRUE);
  rubrica_statusbar_enable_images(statusbar, TRUE);
  rubrica_statusbar_extra_set_text(statusbar, message); 
}


static void 
on_card_added (RAbook* book, gpointer the_card, gpointer data)
{
  RubricaApp* app = (RubricaApp*) data;
  RCard* card = (RCard*) the_card;
  
  rubrica_app_add_card(app, card);
}


static void 
on_card_destroyed (RAbook* book, gchar* arg, gpointer data)
{
  gchar* message;
  GtkWidget* view;

  view    = find_addressbook(RUBRICA_APP(data), book);
  message = g_strdup_printf(_("Card %s successfully destroyed"), arg);
  rubrica_app_show_message(RUBRICA_APP(data), message);
  rubrica_cards_view_update_infos (RUBRICA_CARDS_VIEW(view));

  g_free(message);
  
}



static void 
on_card_deleted  (RAbook* book, gpointer card, gpointer data)
{
  RubricaApp* app = (RubricaApp*) data;
  GtkWidget* view;
  GtkWidget* trash;
  
  view  = rubrica_app_get_current_view(app);
  trash = rubrica_app_get_trash(app);
  
  rubrica_trash_view_add_card(RUBRICA_TRASH_VIEW(trash), book, card);
  rubrica_cards_view_delete_card(RUBRICA_CARDS_VIEW(view), card);
}


static void
on_card_recovered (RAbook* book, gpointer card, gpointer data)
{
  RubricaApp* app = (RubricaApp*) data;
  GtkWidget* view = NULL;
  
  view = find_addressbook(app, book);
  if (view)
    rubrica_cards_view_add_card(RUBRICA_CARDS_VIEW(view), R_CARD(card));
}


static void 
on_card_modified (RAbook* book, gpointer newcard, gpointer data)
{
  RubricaApp *app = (RubricaApp*) data;    
  GtkWidget  *view = NULL;

  view = rubrica_app_get_current_view(app);
  rubrica_cards_view_modify_card(RUBRICA_CARDS_VIEW(view), newcard);
  rubrica_app_display_card_data(app, R_CARD(newcard));
}



static void 
on_show_menu_cb (GtkMenuToolButton *menutoolbutton, gpointer data)
{
  GtkMenu* menu = data;

  gtk_menu_popup (menu, NULL, NULL, NULL, NULL, 1,
		  gtk_get_current_event_time());    

  gtk_widget_show_all(GTK_WIDGET(menu));  
}


static void 
on_trash_toggled_cb (GtkToggleToolButton *toggle, gpointer data)
{
  RubricaApp* app = (RubricaApp*) data;

  if (gtk_toggle_tool_button_get_active(GTK_TOGGLE_TOOL_BUTTON(toggle)))
    gtk_widget_show(app->priv->trash_view);
  else
    gtk_widget_hide(app->priv->trash_view); 
}


static void 
hide_trash_on_click (GtkWidget* button, gpointer data)
{
  RubricaApp* app = (RubricaApp*) data;
  GtkWidget* toggle;

  gtk_widget_hide(GTK_WIDGET(app->priv->trash_view));
  toggle = app->priv->toolbar->trash;
  if (gtk_toggle_tool_button_get_active(GTK_TOGGLE_TOOL_BUTTON(toggle)))
    gtk_toggle_tool_button_set_active (GTK_TOGGLE_TOOL_BUTTON(toggle), FALSE);
}


static void 
hide_search_on_click (GtkWidget* button, gpointer data)
{
  gtk_widget_hide(GTK_WIDGET(data));
}


static void
on_addressbook_changed(RAbook* book, gpointer data)
{
  RubricaApp* app = (RubricaApp*) data;
  GConfClient* client;
  GtkWidget* view = NULL;
  GtkWidget* label;
  const gchar *text = NULL;
  gchar *markup;
 
  client = gconf_client_get_default();
  view = find_addressbook(app, book);

  if (gconf_client_get_bool(client, RUBRICA_GCONF_AUTOSAVE, NULL))
    save_addressbook(app, book);         
  else
    {
      label  = get_current_tab_label(app);
      //      text   = gtk_label_get_text(GTK_LABEL(text));
      text   = get_current_tab_label_text(app);
      markup = g_markup_printf_escaped ("<span style=\"italic\">%s</span>", 
					text);

      g_object_set(view, "has-changed", TRUE, NULL);
      gtk_label_set_markup (GTK_LABEL (label), markup);
      g_free (markup);              
  }

  rubrica_cards_view_update_infos(RUBRICA_CARDS_VIEW(view));
  g_object_unref(client);
}


static void 
on_addressbook_saved (RAbook* book, gpointer data)
{
  RubricaApp* app = (RubricaApp*) data;
  GtkWidget* view = NULL;
  GtkWidget* tab;
  gchar *file, *report, *markup, *type = NULL;
  
  view = rubrica_app_get_current_view(app);

  g_object_get(view, "child-type", &type, NULL);
  if (!type || (g_ascii_strcasecmp(type, "Addressbook") != 0))
    view = find_addressbook(app, book);

  file = rubrica_cards_view_get_addressbook_name(RUBRICA_CARDS_VIEW(view));
  rubrica_cards_view_update_infos(RUBRICA_CARDS_VIEW(view));
  g_object_get(view, "view-brother", &tab, "view-message", &report, NULL);
  
  rubrica_statusbar_extra_set_text(RUBRICA_STATUSBAR(app->priv->statusbar), 
				   report);
  markup = g_markup_printf_escaped ("<span style=\"normal\">%s</span>", file);
  g_object_set(tab, "tab-label", markup, NULL);

  g_free (markup);    
}


static void 
on_addressbook_save_fail (RAbook* book, gint err, gpointer data)
{
  RubricaApp* app = (RubricaApp*) data;
  gchar* file;
  gchar* message = NULL;
  
  g_object_get(book, "addressbook-name", &file, NULL);

  switch (err)
    {
    case WRITING_FILE:
    case WRITING_XML:
      message = g_strdup_printf(_("Can't save the file %s"), file);
      break;

      break;

    case OVERWRITING:
      message = g_strdup_printf(_("Can't overwrite the file %s"), file);      
      break;

    default:
      break;
    }
  
  rubrica_app_show_message(app, message);
  g_free(message);
}


static void 
on_autosave_changed(GConfClient *client, guint cnxn_id, 
		    GConfEntry *entry, gpointer data)
{
  GtkWidget* button = (GtkWidget* ) data;
  gboolean autosave;
  
  autosave = gconf_client_get_bool(client, RUBRICA_GCONF_AUTOSAVE, NULL);
  gtk_widget_set_sensitive(button, !autosave);
}


static void 
on_launch_cb(GtkWidget* data_view, gpointer arg1, gpointer data)
{
  RubricaApp* app = (RubricaApp*) data;
  RubricaLink* link = (RubricaLink*) arg1;
  GConfClient* client;
  GPid pid;
  GError *error = NULL;
  gchar* argv[4];
  gchar* manager = NULL;
  
  
  if (link && link->uri)
    {
      gchar* arg = NULL;
      gchar* notify;
      gchar* message;

      client  = gconf_client_get_default();
      
      switch ((RubricaLinkType) link->type)
	{
	case WEB_LINK:
	  manager = gconf_client_get_string(client, RUBRICA_GCONF_BROWSER,
					    NULL);
	  if (g_ascii_strcasecmp(manager, "other") == 0)
	    manager = gconf_client_get_string(client, RUBRICA_GCONF_BROWSER_UD,
					      NULL); 
	  arg = g_strdup_printf("%s", link->uri);
	  break;

	case EMAIL_LINK:
	  manager = gconf_client_get_string(client, RUBRICA_GCONF_EMAILS,
					    NULL);
	  if (g_ascii_strcasecmp(manager, "other") == 0)
	    manager = gconf_client_get_string(client, RUBRICA_GCONF_EMAILS_UD,
					      NULL);	  
	  if (g_ascii_strcasecmp(manager, "balsa") == 0)
	    {	      
	      arg = g_strdup_printf("--compose=%s", link->uri);
	      break;
	    }

	  if (g_ascii_strcasecmp(manager, "seamonkey") == 0)
	    {	      
	      arg = g_strdup_printf("mailto:%s", link->uri);
	      break;
	    }

	    arg = g_strdup_printf("mailto:%s", link->uri);
	  break;

	case IRC_LINK:
	  manager = gconf_client_get_string(client, RUBRICA_GCONF_IRC, NULL);
	  
	  if (g_ascii_strcasecmp(manager, "other") == 0)
	    manager = gconf_client_get_string(client, RUBRICA_GCONF_IRC_UD, 
					      NULL);

	  arg = g_strdup_printf("-l %s", link->uri); 
	  break;

	case TELEPHONE_LINK:
	  manager = g_strdup("ekiga");
	  arg = g_strdup_printf("-c %s", link->uri); 
	  break;

	default:
	  break;
	}
      
      if(!g_find_program_in_path(manager))
	{
	  message = g_strdup_printf(_("Can't find %s in your path"), 
				    manager);
	  
	  rubrica_app_show_message (app, message);
	  return;
	}

      argv[0] = manager;
      argv[1] = arg;
      argv[2] = NULL;
      
      error = NULL;
      notify  = g_strdup_printf(_("Launching %s. Please wait..."), 
				manager);  
      message =  g_strdup_printf(_("Launching %s. Please wait..."), 
				 manager); 
      
      rubrica_app_notify (app, notify);
      notify_notification_update(app->priv->notify, _("Rubrica Addressbook"),
				 message, "info");
      if(!notify_notification_show (app->priv->notify, &error))
	{
	  g_warning("notification error: %s", error->message);
	}
      
      if (g_spawn_async ("/usr/bin", 
			 argv,
			 NULL,  // inherit parent's environment
			 G_SPAWN_CHILD_INHERITS_STDIN,
			 NULL,  // function to run in the child before exec
			 NULL,  // user data for child_setup
			 &pid,
			 &error));
      g_free(notify);
      g_free(message);
    }
}



static gboolean
on_cards_event_occurred (GtkTreeView* tree, GdkEvent *event, gpointer data)
{
  RubricaApp       *app = (RubricaApp*) data;
  GdkEventButton   *ev  = (GdkEventButton *)event;
  GtkTreeModel     *model;
  GtkTreeSelection *selection; 
  GtkTreePath      *path = NULL;
  GtkTreeIter       iter;
  RAbook           *book;
  glong id;

  book      = rubrica_app_get_active_addressbook (app);
  model     = gtk_tree_view_get_model(GTK_TREE_VIEW(tree));
  selection = gtk_tree_view_get_selection(tree);

  gtk_tree_selection_unselect_all(selection);

  /* if a card is selected then get card from database
   */
  if (gtk_tree_view_get_path_at_pos(tree, ev->x, ev->y, &path, 
				    NULL, NULL, NULL))
    {     
      gpointer card;	    
      
      /* Get the selected card 
       */
      gtk_tree_selection_select_path(selection, path);
      gtk_tree_model_get_iter(model, &iter, path);
      
      gtk_tree_model_get(model, &iter, ID_COLUMN, &id, -1);
      card = r_abook_get_card_by_id(R_ABOOK(book), id);
            
      /* Do something with the card 
       */      
      /* if rigth button was pressed  ==> show popup menu 
       */
      if ((ev->button == 3) && (ev->type == GDK_BUTTON_PRESS))
	{
	  GtkMenu* menu;

	  menu = (GtkMenu*) rubrica_menu_do_card_popup(RUBRICA_APP(app), 
						       tree, event);
	  
	  gtk_widget_show_all(GTK_WIDGET(menu));
	  gtk_menu_popup (menu, NULL, NULL, NULL, NULL, 
			  ev->button, ev->time);
	}
      /* else if the left button was pressed
       */
      else
	{
	  /* update application's window                  
	   */
	  /* check if next/previous cards are available  */
	  rubrica_app_enable_previouses(app, TRUE);	  
	  rubrica_app_enable_nexts(app, TRUE);
	  
	  if (!gtk_tree_path_prev(path))
	    rubrica_app_enable_previouses(app, FALSE);
	  if (!gtk_tree_model_iter_next(model, &iter))
	    rubrica_app_enable_nexts(app, FALSE);
	  
	  gtk_tree_path_free(path);
	  
	  if (ev->button == 1)
	    {
	      switch(ev->type)
		{
		case GDK_BUTTON_PRESS: 
		  /* left button clicked once ==> card selected, show it 
		   */
		  rubrica_app_display_card_data(RUBRICA_APP(app), card);
		  break;

		case GDK_2BUTTON_PRESS:
		  /* left button clicked twice  ==> modify the selected card 
		   */		  
		  cards_modify_card(RUBRICA_APP(app), book, card);
		  break;

		default:
		  break;
		}
	    }
	}
    } // end: if a card is selected
  else  /* rigth button pressed and no cards selected ==> show popup menu */
    if ((ev->type == GDK_BUTTON_PRESS) && (ev->button == 3))
      {
	GtkMenu* menu;  
	
	menu = (GtkMenu*) rubrica_menu_do_card_popup(RUBRICA_APP(app), 
						     tree, event);
	
	gtk_widget_show_all(GTK_WIDGET(menu));
	gtk_menu_popup (menu, NULL, NULL, NULL, NULL, ev->button, ev->time);
      }

  // returning FALSE it doesn't block signal for other callbacks
  return FALSE;
}


static void 
on_trash_event_occurred (GtkTreeView* tree, GdkEvent *event, gpointer data)
{
  RubricaApp*       app = (RubricaApp*) data;
  GtkTreeIter       iter;
  GtkTreeModel*     model;
  GtkTreePath*      path = NULL;
  GtkTreeSelection* selection; 
  GdkEventButton*   ev = (GdkEventButton *)event;

  model     = gtk_tree_view_get_model(GTK_TREE_VIEW(tree));
  selection = gtk_tree_view_get_selection(tree);
  
  /* if a card has been selected */
  if (gtk_tree_view_get_path_at_pos(tree, ev->x, ev->y, &path, 
				    NULL, NULL, NULL))
    {
      gtk_tree_selection_select_path(selection, path);

      /* left button pressed  ==> show card */
      if(ev->button == 1) 
	{
	  RAbook* book  = NULL;    
	  gpointer card = NULL;	    
	  glong id = 0;	       
	  
	  gtk_tree_model_get_iter(model, &iter, path);
	  gtk_tree_path_free(path);
	  
	  gtk_tree_model_get(model, &iter, 
			     TRASH_ID, &id,
			     TRASH_ADDRESSBOOK, &book,	 
			     -1);
	  	  
	  card = r_abook_get_card_by_id(R_ABOOK(book), id);	      
	  rubrica_app_display_card_data(RUBRICA_APP(app), card);
	}
 
      /* rigth button pressed  ==> show popup menu */
      if (ev->button == 3)
	{
	  GtkMenu* menu;  
	  
	  menu = (GtkMenu*) rubrica_menu_do_trash_popup(RUBRICA_APP(app), 
							tree, event);
	  
	  gtk_widget_show_all(GTK_WIDGET(menu));
	  gtk_menu_popup (menu, NULL, NULL, NULL, NULL, ev->button, ev->time);
	}
    }
}


static void 
on_search_event_occurred (GtkTreeView* tree, GdkEvent *event, gpointer data)
{
  RubricaApp*       app = (RubricaApp*) data;
  GtkTreeIter       iter;
  GtkTreeModel*     model;
  GtkTreePath*      path = NULL;
  GtkTreeSelection* selection; 
  GdkEventButton*   ev = (GdkEventButton *)event;

  model     = gtk_tree_view_get_model(GTK_TREE_VIEW(tree));
  selection = gtk_tree_view_get_selection(tree);
  
  /* if a card has been selected */
  if (gtk_tree_view_get_path_at_pos(tree, ev->x, ev->y, &path, 
				    NULL, NULL, NULL))
    {
      gtk_tree_selection_select_path(selection, path);

      /* left button pressed  ==> show card */
      if(ev->button == 1) 
	{
	  RAbook* book  = NULL;    
	  gpointer card = NULL;	    
	  glong id = 0;	       
	  
	  gtk_tree_model_get_iter(model, &iter, path);
	  gtk_tree_path_free(path);
	  
	  gtk_tree_model_get(model, &iter, 
			     SEARCH_ID_COLUMN,   &id,
			     SEARCH_BOOK_COLUMN, &book,	 
			     -1);
	  	  
	  card = r_abook_get_card_by_id(R_ABOOK(book), id);	      
	  rubrica_app_display_card_data(RUBRICA_APP(app), card);
	}
 
      /* rigth button pressed  ==> show popup menu */
      if (ev->button == 3)	
	{
	  /*
	  GtkMenu* menu;  
	  
	  menu = (GtkMenu*) rubrica_menu_do_trash_popup(RUBRICA_APP(app), 
							tree, event);
	  
	  gtk_widget_show_all(GTK_WIDGET(menu));
	  gtk_menu_popup (menu, NULL, NULL, NULL, NULL, ev->button, ev->time);
	  */
	}
    }
}



static gboolean
on_key_press_event_cb (GtkWidget* widget, GdkEventKey* event, gpointer data)
{
  RubricaApp* app = (RubricaApp*) data;

  switch (event->keyval) 
    { 
    case GDK_KP_Delete: 
    case GDK_Delete: 
      on_delete_card_cb (NULL, app);
      break; 
 
    case GDK_Return:

      break;

    case GDK_Home:
      rubrica_browse_first_card(app);
     break; 

    case GDK_End:
      rubrica_browse_last_card(app);
      break; 

    case GDK_Left:
      break;
 
    case GDK_Right:
      break;
 
    case GDK_Up:
      rubrica_browse_previous_card(app);
      break;

    case GDK_Down:
      rubrica_browse_next_card(app);
      break; 
      
    default  : 
      break; 
    }

  return FALSE;
}


static void 
on_card_marked (RubricaCardsView* view, glong card_id, 
		RubricaCardsViewMark mark, gpointer data)
{
  RubricaApp* app = (RubricaApp*) data;
  RAbook* book;
  RCard* card;
  gint marked = 0;

  book = rubrica_app_get_active_addressbook(app);
  card = r_abook_get_card_by_id(book, card_id);
  g_object_get(book, "printable-cards", &marked, NULL);
  
  switch(mark)
    {
    case CARD_MARKED_PRINT:
      marked++;
      g_object_set(card, "card-printable", TRUE, NULL);     
      break;

    case CARD_UNMARKED_PRINT:      
      marked--;
      g_object_set(card, "card-printable", FALSE, NULL);
      break;

    default:
      break;
    }
  
  g_object_set(book, "printable-cards", marked, NULL);
}



RubricaMenu*
build_menu(GladeXML* gui, RubricaApp* app)
{

  RubricaMenu* menu = NULL;
  GConfClient* client;
  gboolean autosave;

  client = gconf_client_get_default();
  menu   = g_malloc0(sizeof(RubricaMenu));
  if (!menu)
    g_error("\nCan't allocate memory");

  menu->menubar  = glade_xml_get_widget(gui, "menubar");

  menu->new      = glade_xml_get_widget(gui, "new");
  menu->open     = glade_xml_get_widget(gui, "open");
  menu->save     = glade_xml_get_widget(gui, "save");
  menu->saveas   = glade_xml_get_widget(gui, "saveas");
  menu->quit     = glade_xml_get_widget(gui, "quit");

  menu->cut      = glade_xml_get_widget(gui, "cut");
  menu->copy     = glade_xml_get_widget(gui, "copy");
  menu->paste    = glade_xml_get_widget(gui, "paste");
  menu->personal = glade_xml_get_widget(gui, "personal");
  menu->company  = glade_xml_get_widget(gui, "company");
  menu->delete   = glade_xml_get_widget(gui, "delete");
  menu->modify   = glade_xml_get_widget(gui, "modify");
  menu->prefer   = glade_xml_get_widget(gui, "prefer");

  menu->about    = glade_xml_get_widget(gui, "about"); 

  /*     menu handler
  */ 
  g_signal_connect(G_OBJECT(menu->new), "activate",   
		   G_CALLBACK(on_new_cb), app);   
 
  g_signal_connect(G_OBJECT(menu->open), "activate",  
		   G_CALLBACK(on_open_cb), app);

  g_signal_connect(G_OBJECT(menu->save), "activate", 
		   G_CALLBACK(on_save_cb), app);

  g_signal_connect(G_OBJECT(menu->saveas), "activate", 
		   G_CALLBACK(on_saveas_cb), app);

  g_signal_connect(G_OBJECT(menu->cut), "activate", 
		   G_CALLBACK(on_cut_cb), NULL);

  g_signal_connect(G_OBJECT(menu->copy), "activate", 
		   G_CALLBACK(on_copy_cb), NULL);

  g_signal_connect(G_OBJECT(menu->paste), "activate", 
		   G_CALLBACK(on_paste_cb), NULL);

  g_signal_connect(G_OBJECT(menu->personal), "activate", 
		   G_CALLBACK(on_add_personal_cb), app);

  g_signal_connect(G_OBJECT(menu->company), "activate", 
		   G_CALLBACK(on_add_company_cb), app);

  g_signal_connect(G_OBJECT(menu->delete), "activate", 
		   G_CALLBACK(on_delete_card_cb), app);

  g_signal_connect(G_OBJECT(menu->modify), "activate", 
		   G_CALLBACK(on_modify_card_cb), app);

  g_signal_connect(G_OBJECT(menu->prefer), "activate", 
		   G_CALLBACK(on_prefenences_cb), app);

  g_signal_connect(G_OBJECT(menu->about), "activate", 
		   G_CALLBACK(on_info_cb), NULL);

  g_signal_connect(G_OBJECT(menu->quit), "activate", 
		   G_CALLBACK(on_quit_cb), app);
  
  autosave = gconf_client_get_bool(client, RUBRICA_GCONF_AUTOSAVE, NULL);
  if (autosave)
    gtk_widget_set_sensitive(menu->save, FALSE);

  gconf_client_notify_add(client, RUBRICA_GCONF_AUTOSAVE,
			  on_autosave_changed, menu->save, NULL, NULL);
  g_object_unref(client);

  return menu;
}



RubricaToolbar*
build_toolbar(GladeXML* gui, RubricaApp* app)
{
  GConfClient* client;
  RubricaToolbar* toolbar = NULL;
  GtkMenu* menu = NULL;
  GtkWidget *personal;
  GtkWidget *company; 
  gboolean autosave;

  client  = gconf_client_get_default();
  toolbar = g_malloc0(sizeof(RubricaToolbar));  
  if (!toolbar)
    g_error("\nCan't allocate memory");

  toolbar->handle   = glade_xml_get_widget(gui, "handle");
  toolbar->toolbar  = glade_xml_get_widget(gui, "toolbar");
  toolbar->new      = glade_xml_get_widget(gui, "new_bt");
  toolbar->open     = glade_xml_get_widget(gui, "open_bt");
  toolbar->save     = glade_xml_get_widget(gui, "save_bt");
  toolbar->print    = glade_xml_get_widget(gui, "print_bt");

  toolbar->add      = glade_xml_get_widget(gui, "add_bt");
  toolbar->delete   = glade_xml_get_widget(gui, "delete_bt");
  toolbar->modify   = glade_xml_get_widget(gui, "modify_bt");

  toolbar->previous = glade_xml_get_widget(gui, "previous");
  toolbar->next     = glade_xml_get_widget(gui, "next");
  toolbar->trash    = glade_xml_get_widget(gui, "trash");

  toolbar->first    = glade_xml_get_widget(gui, "first_bt");
  toolbar->last     = glade_xml_get_widget(gui, "last_bt");
  toolbar->search   = glade_xml_get_widget(gui, "search_bt");
  toolbar->quit     = glade_xml_get_widget(gui, "quit_bt");  

  menu = (GtkMenu*) gtk_menu_new();
  
  personal = gtk_menu_item_new_with_mnemonic (_("_Personal card"));
  gtk_widget_show (personal);
  gtk_container_add (GTK_CONTAINER (menu), personal);
  
  company = gtk_menu_item_new_with_mnemonic (_("_Company card"));
  gtk_widget_show (company);
  gtk_container_add (GTK_CONTAINER (menu), company);
  
  gtk_menu_tool_button_set_menu(GTK_MENU_TOOL_BUTTON(toolbar->add), 
				(GtkWidget*) menu);

  g_signal_connect (G_OBJECT(personal), "activate",
		    G_CALLBACK (on_add_personal_cb), app);
  
  g_signal_connect (G_OBJECT(company), "activate",
		    G_CALLBACK (on_add_company_cb), app);   
  
  g_signal_connect(G_OBJECT(toolbar->new), "clicked", 
		   G_CALLBACK(on_new_cb), app);

  g_signal_connect(G_OBJECT(toolbar->open), "clicked",
		   G_CALLBACK(on_open_cb), app);

  g_signal_connect(G_OBJECT(toolbar->save), "clicked",
		   G_CALLBACK(on_save_cb), app);
  
  g_signal_connect(G_OBJECT(toolbar->print), "clicked", 
		   G_CALLBACK(on_print_cb), app);

  g_signal_connect(G_OBJECT(toolbar->add), "clicked",
		   G_CALLBACK(on_add_personal_cb), app);
  
  g_signal_connect(G_OBJECT(toolbar->add), "show-menu",
		   G_CALLBACK(on_show_menu_cb), menu);
  
  g_signal_connect(G_OBJECT(toolbar->delete), "clicked", 
		   G_CALLBACK(on_delete_card_cb), app);

  g_signal_connect(G_OBJECT(toolbar->modify), "clicked", 
		   G_CALLBACK(on_modify_card_cb), app);

  g_signal_connect(G_OBJECT(toolbar->previous), "clicked", 
		   G_CALLBACK(on_prev_cb), app);
  
  g_signal_connect(G_OBJECT(toolbar->next), "clicked", 
		   G_CALLBACK(on_next_cb), app);

  g_signal_connect(G_OBJECT(toolbar->first), "clicked", 
		   G_CALLBACK(on_first_cb), app);

  g_signal_connect(G_OBJECT(toolbar->last), "clicked", 
		   G_CALLBACK(on_last_cb), app);

  g_signal_connect(G_OBJECT(toolbar->trash), "toggled", 
		   G_CALLBACK(on_trash_toggled_cb), app);  

  g_signal_connect(G_OBJECT(toolbar->quit), "clicked", 
		   G_CALLBACK(on_quit_cb), app);

  autosave = gconf_client_get_bool(client, RUBRICA_GCONF_AUTOSAVE, NULL);
  
  if (autosave)
    gtk_widget_set_sensitive(toolbar->save, FALSE);

  gconf_client_notify_add(client, RUBRICA_GCONF_AUTOSAVE,
			  on_autosave_changed, toolbar->save, NULL, NULL);
  g_object_unref(client);  

  return toolbar;
}



GType
rubrica_app_get_type()
{
  static GType app_type = 0;
  
  if (!app_type)
    {
      static const GTypeInfo app_info =
	{
	  sizeof(RubricaAppClass),
	  NULL,
	  NULL,
	  (GClassInitFunc) rubrica_app_class_init,
	  NULL,
	  NULL,
	  sizeof(RubricaApp),
	  0,
	  (GInstanceInitFunc) rubrica_app_init
	};

      app_type = g_type_register_static (G_TYPE_OBJECT, "RubricaApp",
					 &app_info, 0);
    }
  
  return app_type;
}



static void
rubrica_app_class_init(RubricaAppClass* klass)
{
  GObjectClass *class; 

  class           = G_OBJECT_CLASS (klass);
  class->dispose  = (GObjectFinalizeFunc) rubrica_app_dispose;
  class->finalize = (GObjectFinalizeFunc) rubrica_app_finalize;

  g_type_class_add_private (klass, sizeof(RubricaAppPrivate));
}


static void
rubrica_app_init(RubricaApp* self)
{
  GladeXML* gui;

  GtkWidget* groups_box;
  GtkWidget* notebook_box;
  GtkWidget* data_box;
  GtkWidget* statusbar_box;
  GtkWidget* tab;
  GtkWidget* button;
  GtkWidget* tree;

  gui = glade_xml_new (RUBRICA_GUI_DIR"/MainWindow.glade", NULL, NULL);
  if (!gui)
    g_error("Can't load gui");

  self->priv = g_new0(RubricaAppPrivate, 1);

  //  self->priv->groups = r_group_box_new();
  self->priv->groups = r_lib_get_group_box();
  init_group_box(self->priv->groups);

  self->window               = glade_xml_get_widget(gui, "RubricaMainWindow");
  groups_box                 = glade_xml_get_widget(gui, "groups_box");
  notebook_box               = glade_xml_get_widget(gui, "notebook_box");
  data_box                   = glade_xml_get_widget(gui, "data_box");
  statusbar_box              = glade_xml_get_widget(gui, "statusbar_box");

  self->priv->paned          = glade_xml_get_widget(gui, "data_paned");
  self->priv->group_paned    = glade_xml_get_widget(gui, "group_paned");

  self->priv->menu           = build_menu(gui, self);
  self->priv->toolbar        = build_toolbar(gui, self);

  self->priv->notebook       = gtk_notebook_new();
  self->priv->groups_view    = rubrica_groups_view_new(self->priv->groups,
						    APP_GROUPS);
  self->priv->trash_view     = rubrica_trash_view_new();
  self->priv->search_view    = rubrica_search_view_new();
  self->priv->data_view      = rubrica_data_view_new();
  self->priv->statusbar      = rubrica_statusbar_new();
  
  self->priv->cutted         = NULL;
  self->priv->notify         = notify_notification_new("Rubrica Addressbook",
						      NULL, "info", NULL);
  self->priv->active_plugin  = g_strdup("autodetect");
  self->priv->manager        = r_lib_get_manager();
  self->priv->print_settings = NULL;
  self->priv->show_card      = TRUE;

  g_object_set(self->priv->notebook, "homogeneous", FALSE, NULL);
    
  gtk_box_pack_start(GTK_BOX(groups_box), self->priv->groups_view,
		     TRUE, TRUE, 0);
  gtk_box_pack_start(GTK_BOX(notebook_box), self->priv->notebook,
		     TRUE, TRUE, 0);
  gtk_box_pack_start(GTK_BOX(data_box), self->priv->data_view, 
		     TRUE, TRUE, 0);
  gtk_box_pack_start(GTK_BOX(statusbar_box), self->priv->statusbar,
		     TRUE, TRUE, 0);
  
  gtk_notebook_set_scrollable(GTK_NOTEBOOK(self->priv->notebook), TRUE);
  
  /*    Append Trash page in notebook and initialize
  */
  tab = rubrica_tab_new_with_label(self->priv->trash_view, _("Trash"));
  gtk_notebook_append_page(GTK_NOTEBOOK(self->priv->notebook), 
			   self->priv->trash_view, tab);

  tree   = rubrica_view_get_tree(RUBRICA_VIEW(self->priv->trash_view));
  button = rubrica_tab_get_button(RUBRICA_TAB(tab));

  g_signal_connect(GTK_OBJECT (tree), "button_press_event",
		   G_CALLBACK (on_trash_event_occurred), self);
  g_signal_connect(button, "clicked", G_CALLBACK(hide_trash_on_click), self);
  
  g_signal_connect(self->priv->trash_view, "view-changed", 
		   G_CALLBACK (on_app_view_changed), self->priv->statusbar);
  

  /*    Append Search page in notebook
   */
  tab = rubrica_tab_new_with_label(self->priv->search_view, 
				   _("Search results"));
  gtk_notebook_append_page(GTK_NOTEBOOK(self->priv->notebook), 
			   self->priv->search_view, tab);    

  tree   = rubrica_view_get_tree(RUBRICA_VIEW(self->priv->search_view));
  button = rubrica_tab_get_button(RUBRICA_TAB(tab));
  
  g_signal_connect(GTK_OBJECT (tree), "button_press_event",
		   G_CALLBACK (on_search_event_occurred), self); 
  g_signal_connect(button, "clicked", G_CALLBACK(hide_search_on_click), 
		   self->priv->search_view);
    
  g_signal_connect(self->priv->search_view, "view-changed", 
		   G_CALLBACK (on_app_view_changed), self->priv->statusbar);

  /*    Groups view
  */
  tree = rubrica_view_get_tree(RUBRICA_VIEW(self->priv->groups_view));
  g_signal_connect(GTK_OBJECT (tree), "button_press_event",
		   G_CALLBACK (on_app_group_view_event_occurred), self);

  g_signal_connect (GTK_OBJECT (tree), "key_press_event",
		    G_CALLBACK (on_key_press_event_cb), self);
  
  /*    Other Handle callbacks
   */
  g_signal_connect(G_OBJECT(self->window), "delete_event", 
		   G_CALLBACK(on_delete_event_cb), self);    
  
  g_signal_connect(G_OBJECT(self->priv->data_view), "launch", 
		   G_CALLBACK(on_launch_cb), self);  
  
  gtk_widget_show(groups_box);
  gtk_widget_show(notebook_box);
  gtk_widget_show(data_box);
  gtk_widget_show(statusbar_box);

  self->priv->dispose_has_run = FALSE;
}



static void 
rubrica_app_dispose (RubricaApp* self)
{
  g_return_if_fail(IS_RUBRICA_APP(self));
  
  if (self->priv->dispose_has_run)
    return;
  
  g_object_unref(self->window);
  g_free(self->priv->menu);
  g_free(self->priv->toolbar);
    
  self->priv->dispose_has_run = TRUE;
}


static void 
rubrica_app_finalize (RubricaApp* self)
{
  g_return_if_fail(IS_RUBRICA_APP(self));
  
  g_free(self->priv);
  self->priv = NULL;
}




/*    Public
*/

/**
 * rubrica_app_new
 *
 * create a new #RubricaApp
 *
 * returns: a new allocated #RubricaApp
 */
RubricaApp* 
rubrica_app_new (void)
{
  RubricaApp* app;
  GConfClient* client;
  gboolean have_files;
  
#ifdef ENABLE_DEBUG
  g_log(G_LOG_DOMAIN, G_LOG_LEVEL_DEBUG, "rubrica_app_new");
  g_log(G_LOG_DOMAIN, G_LOG_LEVEL_DEBUG, "Building gui...");
#endif

  app = g_object_new(rubrica_app_get_type(), NULL);
  
  client     = gconf_client_get_default();  
  have_files = gconf_client_get_bool(client, RUBRICA_GCONF_LOAD_FILES, NULL);
  if (have_files)
    load_addressbooks(app);
  else 
    rubrica_app_append_addressbook(app); 
  //    rubrica_app_append_empty_addressbook(app); 

  g_object_unref(client);
  
  return app;
}


/**
 * rubrica_app_free
 * @set: a #RubricaApp
 *
 * free memory owned by a #RubricaApp
 */
void
rubrica_app_free(RubricaApp* app)
{
  g_return_if_fail(IS_RUBRICA_APP(app));

  g_object_unref(app);   
}


/*   Get
*/
GtkWidget*
rubrica_app_get_search (RubricaApp* app)
{
  g_return_val_if_fail(IS_RUBRICA_APP(app), NULL);

  return(GtkWidget*) app->priv->search_view;
}

GtkWidget*  
rubrica_app_get_trash (RubricaApp* app)
{
  g_return_val_if_fail(IS_RUBRICA_APP(app), NULL);

  return (GtkWidget*) app->priv->trash_view;
}


GtkWidget* 
rubrica_app_get_notebook (RubricaApp* app)
{
  g_return_val_if_fail(IS_RUBRICA_APP(app), NULL);

  return (GtkWidget*) app->priv->notebook;  
}



/**
 * rubrica_app_get_current_view
 * @app: a #RubricaApp
 *
 * get the current RubricaCardView widget
 *
 * returns: a *GtkWidget
 */
GtkWidget*  
rubrica_app_get_current_view (RubricaApp* app)
{
  GtkWidget* child;
  gint page;

  g_return_val_if_fail(IS_RUBRICA_APP(app), NULL);
  
  page  = gtk_notebook_get_current_page(GTK_NOTEBOOK(app->priv->notebook));
  child = gtk_notebook_get_nth_page (GTK_NOTEBOOK(app->priv->notebook), page);

  return child;
}


RAbook* 
rubrica_app_get_active_addressbook (RubricaApp* app)
{
  GtkWidget* child;

  g_return_val_if_fail(IS_RUBRICA_APP(app), NULL);
  
  child = rubrica_app_get_current_view(app);
  
  return rubrica_cards_view_get_addressbook(RUBRICA_CARDS_VIEW(child));
}


RGroupBox* 
rubrica_app_get_groups (RubricaApp* app)
{
  g_return_val_if_fail(IS_RUBRICA_APP(app), NULL);
  
  return app->priv->groups;
}


GtkWidget*  
rubrica_app_get_dataview (RubricaApp* app)
{
  g_return_val_if_fail(IS_RUBRICA_APP(app), NULL);

  return app->priv->data_view;
}

GtkWidget* 
rubrica_app_get_groups_view(RubricaApp* app)
{
  g_return_val_if_fail(IS_RUBRICA_APP(app), NULL);

  return app->priv->groups_view;  
}


GtkWidget*
rubrica_app_get_statusbar (RubricaApp* app)
{
  g_return_val_if_fail(IS_RUBRICA_APP(app), NULL);
  
  return app->priv->statusbar;
}

GtkWidget* 
rubrica_app_get_paned (RubricaApp* app)
{
  g_return_val_if_fail(IS_RUBRICA_APP(app), NULL);
  
  return app->priv->paned;  
}


GtkWidget* 
rubrica_app_get_group_paned (RubricaApp* app)
{
  g_return_val_if_fail(IS_RUBRICA_APP(app), NULL);
  
  return app->priv->group_paned;
}


gchar* 
rubrica_app_get_view_report (RubricaApp* app)
{
  GtkWidget* view;
  gchar* report;

  g_return_val_if_fail(IS_RUBRICA_APP(app), NULL);
  
  view = rubrica_app_get_current_view(app);
  g_object_get(view, "report", &report, NULL);

  return report;
}


gchar* 
rubrica_app_get_plugin_name (RubricaApp* app)
{
  g_return_val_if_fail(IS_RUBRICA_APP(app), NULL);
  
  return app->priv->active_plugin;
}


RPluginManager* 
rubrica_app_get_plugin_manager  (RubricaApp* app)
{
 g_return_val_if_fail(IS_RUBRICA_APP(app), NULL);

 return app->priv->manager;
}


RCard*
rubrica_app_get_selected_card (RubricaApp* app)
{
  RAbook* abook;
  
  g_return_val_if_fail(IS_RUBRICA_APP(app), NULL);
  
  abook = rubrica_app_get_active_addressbook(app);
  
  return r_abook_get_selected_card(R_ABOOK(abook));
}



void
rubrica_app_run(RubricaApp* app)
{
  GConfClient* client;
  gint width, height, position, group_pos, style;

  g_return_if_fail(IS_RUBRICA_APP(app));

#ifdef ENABLE_DEBUG
  g_log(G_LOG_DOMAIN, G_LOG_LEVEL_DEBUG, "rubrica_app_run");
  g_log(G_LOG_DOMAIN, G_LOG_LEVEL_DEBUG, "Reading preferences...");
#endif

  client = gconf_client_get_default();
  gconf_client_add_dir(client, KEY_MISC,    GCONF_CLIENT_PRELOAD_NONE, NULL);
  gconf_client_add_dir(client, KEY_FILE,    GCONF_CLIENT_PRELOAD_NONE, NULL);
  gconf_client_add_dir(client, KEY_DISPLAY, GCONF_CLIENT_PRELOAD_NONE, NULL);
  gconf_client_add_dir(client, KEY_SEARCH,  GCONF_CLIENT_PRELOAD_NONE, NULL);
  gconf_client_add_dir(client, KEY_URLS,    GCONF_CLIENT_PRELOAD_NONE, NULL);
  gconf_client_add_dir(client, KEY_APPEARANCE, 
		       GCONF_CLIENT_PRELOAD_NONE, NULL);
 
  width     = gconf_client_get_int(client, RUBRICA_GCONF_WIDTH, NULL);
  height    = gconf_client_get_int(client, RUBRICA_GCONF_HEIGHT, NULL);  
  position  = gconf_client_get_int(client, RUBRICA_GCONF_PANED_POSITION, NULL);
  group_pos = gconf_client_get_int(client, RUBRICA_GCONF_GRP_PANED_POSITION, 
				   NULL);
  style     = rubrica_preferences_get_toolbar_style();
 
  rubrica_app_set_toolbar_style(app, style);
  gtk_window_resize(GTK_WINDOW(app->window), width, height);
  gtk_paned_set_position(GTK_PANED(app->priv->paned), position);
  gtk_paned_set_position(GTK_PANED(app->priv->group_paned), group_pos);

  gtk_widget_show_all(app->window);
  gtk_widget_hide(app->priv->search_view);
  gtk_widget_hide(app->priv->trash_view);

  rubrica_app_enable_previouses(app, FALSE);
  rubrica_app_enable_nexts(app, FALSE);

  g_object_unref(client);  
}

/*
void
rubrica_app_append_empty_addressbook(RubricaApp* app)
{
  rubrica_app_append_addressbook(app, NULL);
}
*/

GtkWidget* 
rubrica_app_append_addressbook(RubricaApp* app)
{
  GtkWidget *view = NULL;  
  RAbook    *book;    
  GtkWidget *tab;
  GtkWidget *button;
  GtkWidget *tree;
  gint page;

  view   = rubrica_cards_view_new(NULL);  
  tab    = rubrica_tab_new_with_label(view, _("no name"));
  button = rubrica_tab_get_button(RUBRICA_TAB(tab));
  book   = rubrica_cards_view_get_addressbook(RUBRICA_CARDS_VIEW(view));
  tree   = rubrica_view_get_tree(RUBRICA_VIEW(view));
  g_object_set(view, "view-brother", tab, NULL);

  
  gtk_widget_show(view); // mostra sempre il widget prima di inserirlo
                         // nel notebook, se non vuoi diventare pazzo 
                         // a capire perchè non switcha 

  // set the new addressbook as active and switch notebook to display it  
  page = gtk_notebook_append_page(GTK_NOTEBOOK(app->priv->notebook),
				  view, tab); 
  gtk_notebook_set_current_page(GTK_NOTEBOOK(app->priv->notebook), page);
       
  g_signal_connect(GTK_OBJECT (button), "clicked",
		   G_CALLBACK (on_close_cb), app);		   

  g_signal_connect(app->priv->notebook, "switch-page",
		   G_CALLBACK (on_notebook_switch_page), app);

  g_signal_connect(view, "view-changed", G_CALLBACK (on_cards_view_changed), 
		   app->priv->statusbar);
  
  g_signal_connect(view, "mark", G_CALLBACK(on_card_marked), app);
  //  g_signal_connect(view, "unmark", G_CALLBACK(on_card_unmarked), app);
  
  g_signal_connect(GTK_WIDGET(tree), "key-press-event",  
		   G_CALLBACK(on_key_press_event_cb), app);
  
  g_signal_connect(GTK_OBJECT (tree), "button_press_event",
		   G_CALLBACK (on_cards_event_occurred), app);   
  
  g_signal_connect(G_OBJECT(book), "card_added",  
		   G_CALLBACK(on_card_added), app);

  g_signal_connect(G_OBJECT(book), "card_deleted",
		   G_CALLBACK(on_card_deleted), app);

  g_signal_connect(G_OBJECT(book), "card_recovered",
		   G_CALLBACK(on_card_recovered), app);

  g_signal_connect(G_OBJECT(book), "card_replaced",
		   G_CALLBACK(on_card_modified), app);
  
  g_signal_connect(G_OBJECT(book), "card_destroyed",
		   G_CALLBACK(on_card_destroyed), app);

  g_signal_connect(G_OBJECT(book), "addressbook_changed",
		   G_CALLBACK(on_addressbook_changed), app);  

  g_signal_connect(G_OBJECT(book), "addressbook_saved",
		   G_CALLBACK(on_addressbook_saved), app);

  g_signal_connect(G_OBJECT(book), "save_fail",
		   G_CALLBACK(on_addressbook_save_fail), app); 

  return view;
}


gboolean 
rubrica_app_load_file(RubricaApp* app, gchar* file/*, const gchar* filter*/)
{  
  GtkWidget *view = NULL;
  GtkWidget *tab;
  RAbook    *book;    
  gchar     *message;

  g_return_val_if_fail(IS_RUBRICA_APP(app), FALSE);
  g_return_val_if_fail(file != NULL, FALSE);
 
  view = rubrica_app_append_addressbook(app);
  book = rubrica_app_get_active_addressbook(app);
  g_object_set(book, "file-filter", /*filter*/app->priv->active_plugin, NULL);

  if (r_abook_open_file(book, file))
    {
      RCard* card;
      RubricaStatusbar* bar = RUBRICA_STATUSBAR(app->priv->statusbar);
      
      app->priv->show_card = FALSE;
      
      r_abook_reset_book(book);
      card = r_abook_get_card(book);
      for (; card; card = r_abook_get_next_card(book))
	rubrica_app_add_card(app, card);
      
      g_object_get(view, "view-brother", &tab, "view-message", &message, NULL);
      g_object_set(tab, "tab-label", g_path_get_basename(file), NULL);  
      
      rubrica_statusbar_enable_extra(bar, TRUE);
      rubrica_statusbar_enable_images(bar, TRUE);
      rubrica_statusbar_extra_set_text(bar, message);
      
      app->priv->show_card = TRUE;
   }
  else
    {
      GtkWidget* notebook;
      gint page;
      
      notebook = rubrica_app_get_notebook(app);
      page = gtk_notebook_page_num(GTK_NOTEBOOK(notebook), view);  
      gtk_notebook_remove_page(GTK_NOTEBOOK(notebook), page);  
      
      return FALSE;
    }

  rubrica_browse_nth_card(app, "0", FALSE);
  return TRUE;
}


void 
rubrica_app_add_card(RubricaApp* app, RCard* card)
{
  GtkWidget *view = NULL;
  gboolean deleted;
    
  g_return_if_fail(IS_RUBRICA_APP(app));
  g_return_if_fail(IS_R_CARD(card));
  
  view = rubrica_app_get_current_view(app);

  g_object_get(card, "card-deleted", &deleted, NULL);

  add_card_groups2set(app, card);
  
  if (deleted)
    rubrica_app_move_card_to_trash(app, card);
  else
    {
      rubrica_cards_view_add_card(RUBRICA_CARDS_VIEW(view), card); 
      if (app->priv->show_card)
	rubrica_app_display_card_data(RUBRICA_APP(app), card);
   }
}



 
/*  if a loaded card is marked deleted, it is moved to trash
    TRASH_ADDRESSBOK - pointer to the original addressbook (card came from)
    TRASH_ADB        - pointer to the adb (card came from)
 */
void 
rubrica_app_move_card_to_trash (RubricaApp* app, RCard* card)
{
  RAbook* book;
  GtkWidget* trash;

  g_return_if_fail(IS_R_CARD(card));
  
  trash = rubrica_app_get_trash(app);
  book  = rubrica_app_get_active_addressbook(app);

  rubrica_trash_view_add_card(RUBRICA_TRASH_VIEW(trash), book, card);
}


void       
rubrica_app_display_card_data (RubricaApp* app, RCard* card)
{
  RRate rate;
  gboolean locked;
  GtkWidget* view;
  RubricaPixmap lock;
  RubricaStatusbar* bar = RUBRICA_STATUSBAR(app->priv->statusbar);

  view = rubrica_app_get_current_view(app);
  g_object_set(view, "view-selected", TRUE, NULL);
  
  g_object_get(card, "card-locked", &locked, "card-rate", &rate, NULL);
  lock = PIXMAP_UNLOCK + (gint) locked;
  
  rubrica_statusbar_enable_extra(bar, TRUE);  
  rubrica_statusbar_enable_images(bar, TRUE);
  rubrica_statusbar_extra_set_rate(bar, rate);
  rubrica_statusbar_extra_set_lock(bar, lock);

  rubrica_data_view_show_card(RUBRICA_DATA_VIEW(app->priv->data_view), card);
}

void
rubrica_app_set_plugin_name (RubricaApp* app, const gchar* plugin_name)
{
  RAbook* abook;

  g_return_if_fail(IS_RUBRICA_APP(app));
  
  abook = rubrica_app_get_active_addressbook(app);

  g_free(app->priv->active_plugin);
  if (!plugin_name || (g_ascii_strcasecmp(plugin_name, "") == 0))
    app->priv->active_plugin = NULL;
  else
    app->priv->active_plugin = g_strdup(plugin_name);

  r_abook_load_plugin(abook, app->priv->active_plugin);
}


void 
rubrica_app_set_toolbar_style (RubricaApp* app, gint style)
{
  GtkToolbarStyle gnome_default;

  g_return_if_fail(IS_RUBRICA_APP(app));
  
  switch(style)
    {
    case RUB_ICONS:
      gtk_toolbar_set_style(GTK_TOOLBAR(app->priv->toolbar->toolbar), 
			    GTK_TOOLBAR_ICONS);
      break;
      
    case RUB_TEXT:
      gtk_toolbar_set_style(GTK_TOOLBAR(app->priv->toolbar->toolbar),
			    GTK_TOOLBAR_TEXT);
      break;
     
    case RUB_BOTH:
      gtk_toolbar_set_style(GTK_TOOLBAR(app->priv->toolbar->toolbar),
			    GTK_TOOLBAR_BOTH);
      break;

    case RUB_GNOME:
      gnome_default = rubrica_preferences_get_gnome_default(); 
      gtk_toolbar_set_style(GTK_TOOLBAR(app->priv->toolbar->toolbar),
			    (GtkToolbarStyle) gnome_default);
      break;
      
    default:
      break;
    }
}

void      
rubrica_app_set_font (RubricaApp* app, gchar* font)
{
  g_return_if_fail(IS_RUBRICA_APP(app));

  g_object_set(app->priv->data_view, "data-font", font, NULL);
}


void             
rubrica_app_show_page (RubricaApp* app, gint page)
{
  GtkWidget* notebook;

  g_return_if_fail(IS_RUBRICA_APP(app));

  notebook = app->priv->notebook;
  gtk_notebook_set_current_page(GTK_NOTEBOOK(notebook), page);   
}


gint             
rubrica_app_get_stack_len (RubricaApp* app)
{
  g_return_val_if_fail(IS_RUBRICA_APP(app), 0);
 
  return g_list_length(app->priv->cutted);
}


void             
rubrica_app_push_card (RubricaApp* app, gpointer data)
{
  g_return_if_fail(IS_RUBRICA_APP(app));
  
  app->priv->cutted = g_list_append(app->priv->cutted, data);
}


gpointer         
rubrica_app_pop_card (RubricaApp* app)
{
  GList* link = NULL;
  gpointer data = NULL;

  g_return_val_if_fail(IS_RUBRICA_APP(app), NULL);
  
  link = g_list_last(app->priv->cutted);
  
  if (link)
    {
      data = link->data;
      app->priv->cutted = g_list_remove_link(app->priv->cutted, link);
      
      g_list_free_1(link);
    }
  
  return data;
}


void             
rubrica_app_notify (RubricaApp* app, gchar* msg)
{
  g_return_if_fail(IS_RUBRICA_APP(app));

  rubrica_statusbar_push_message(RUBRICA_STATUSBAR(app->priv->statusbar), msg);
}


void 
rubrica_app_show_message(RubricaApp* app, gchar* msg)
{
  GtkWidget* dialog;

  g_return_if_fail(IS_RUBRICA_APP(app));
  g_return_if_fail(msg != NULL);
  
  dialog = gtk_message_dialog_new (GTK_WINDOW(app->window),
				   GTK_DIALOG_DESTROY_WITH_PARENT,
				   GTK_MESSAGE_WARNING,
				   GTK_BUTTONS_OK, 
				   msg);

  gtk_dialog_run (GTK_DIALOG (dialog));
  gtk_widget_destroy (dialog);
}


void        
rubrica_app_update_view (RubricaApp* app)
{
  GtkWidget* view;
  GtkTreeView  *tree;  
  GtkTreeSelection *selection;

  g_return_if_fail(IS_RUBRICA_APP(app));

  view = rubrica_app_get_current_view(app);
  tree = GTK_TREE_VIEW(rubrica_view_get_tree(RUBRICA_VIEW(view)));
  selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(tree));  

  gtk_tree_selection_unselect_all(selection); 
}


void       
rubrica_app_display_search_results (RubricaApp* app, GList* results)
{
  RAbook* book;
  RubricaSearchView* search;

  g_return_if_fail(IS_RUBRICA_APP(app));

  gtk_widget_show(app->priv->search_view);
  book   = rubrica_app_get_active_addressbook(app);
  search = RUBRICA_SEARCH_VIEW(app->priv->search_view);
  gtk_notebook_set_current_page (GTK_NOTEBOOK(app->priv->notebook), 1);

  rubrica_search_view_set_addressbook(search, book);
  rubrica_search_view_display_results(search, results);
}


void
rubrica_app_display_cards_by_group(RubricaApp* app, gchar* group)
{
  RAbook* book;
  GList *ids;  // id list of cards that belongs to the given group
  
  g_return_if_fail(IS_RUBRICA_APP(app));

  book = rubrica_app_get_active_addressbook(app);
  ids  = r_abook_find_cards_by_group(book, group);
  
  display_cards_list(app, ids);

  g_list_free(ids);    
}

void 
rubrica_app_display_cards_by_type(RubricaApp* app, gchar* type)
{
  RAbook* book;
  GList *ids;  // id list of cards that belongs to the given type
  
  g_return_if_fail(IS_RUBRICA_APP(app));

  book = rubrica_app_get_active_addressbook(app);
  ids  = r_abook_find_cards_by_type (book, type); 

  display_cards_list(app, ids);

  g_list_free(ids);    
}


void
rubrica_app_display_cards_by_genre(RubricaApp* app, gchar* genre)
{
  RAbook* book;
  GList *ids;  // id list of cards that belongs to the given genre

  g_return_if_fail(IS_RUBRICA_APP(app));

  book = rubrica_app_get_active_addressbook(app);
  ids = r_abook_find_cards_by_genre(book, genre); 
  
  display_cards_list(app, ids);

  g_list_free(ids);      
}

void 
rubrica_app_display_cards_by_rate(RubricaApp* app, RRate rate)
{
  RAbook* book;
  GList *ids;  // id list of cards that belongs to the given rate

  g_return_if_fail(IS_RUBRICA_APP(app));

  book = rubrica_app_get_active_addressbook(app);
  ids  = r_abook_find_cards_by_rate(book, rate); 
  
  display_cards_list(app, ids);

  g_list_free(ids);    
}


void  
rubrica_app_enable_previouses (RubricaApp* app, gboolean bool)
{
  g_return_if_fail(IS_RUBRICA_APP(app));
  
  gtk_widget_set_sensitive(app->priv->toolbar->previous, bool);
  gtk_widget_set_sensitive(app->priv->toolbar->first, bool);
}


void 
rubrica_app_enable_nexts (RubricaApp* app, gboolean bool)
{
  g_return_if_fail(IS_RUBRICA_APP(app));

  gtk_widget_set_sensitive(app->priv->toolbar->next, bool);
  gtk_widget_set_sensitive(app->priv->toolbar->last, bool);
}


void 
rubrica_app_print(RubricaApp* app)
{
  RubricaPrint* print;
  GtkWidget* dialog;
  GtkWidget* view;
  RAbook* book;
  GList* cards = NULL;
  GError* error = NULL;
  gchar* font;
  gint res;
  
  print = rubrica_print_new();  

  view  = rubrica_app_get_current_view(app);
  book  = rubrica_app_get_active_addressbook(app);
  font  = g_strdup("Liberation Sans"); // gconf_get_font
  cards = rubrica_cards_view_get_selected_cards(RUBRICA_CARDS_VIEW(view));

  g_object_set(print, 
	       "printing-addressbook", book, 
	       "printing-cards", cards, 
	       "printing-font", font, 
	       "printing-font-size", 10, NULL);	       
	       
  if (app->priv->print_settings != NULL)
    gtk_print_operation_set_print_settings(print->operation, 
					   app->priv->print_settings);  

  res = gtk_print_operation_run(print->operation, 
				GTK_PRINT_OPERATION_ACTION_PRINT_DIALOG,
				GTK_WINDOW(app->window), &error);
  
  if (res == GTK_PRINT_OPERATION_RESULT_APPLY)
    {
      GtkPrintSettings* settings;

      if (app->priv->print_settings != NULL)
	g_object_unref(app->priv->print_settings);
      
      settings = gtk_print_operation_get_print_settings(print->operation);
      app->priv->print_settings = settings;
      g_object_ref(settings);
    }
  else if (error)
    {
      dialog = gtk_message_dialog_new(GTK_WINDOW(app->window),
				      GTK_DIALOG_DESTROY_WITH_PARENT,
				      GTK_MESSAGE_ERROR, 
				      GTK_BUTTONS_CLOSE,
				      error->message);
      g_error_free(error);
      gtk_dialog_run(GTK_DIALOG(dialog));
      gtk_widget_destroy(dialog);
    }

  g_object_unref(print->operation);    
}
