/*
*  Program: Rubrica
*  file: callbacks.c
*
*  
*  Copyright (C) 2000-2005 Nicola Fragale <nicolafragale@libero.it>
*
*  This program is free software; you can redistribute it and/or modify
*  it under the terms of the GNU General Public License as published by
*  the Free Software Foundation; either version 3 of the License
*
*  This program is distributed in the hope that it will be useful,
*  but WITHOUT ANY WARRANTY; without even the implied warranty of
*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*  GNU General Public License for more details.
*
*  You should have received a copy of the GNU General Public License
*  along with this program; if not, write to the Free Software
*  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#include <gtk/gtk.h>
#include <glib.h>
#include <glib/gstdio.h>
#include <glade/glade.h>
#include <gconf/gconf-client.h>
#include <glib/gi18n-lib.h>
#include "libral.h"

#include "types.h"
#include "app.h"
#include "callbacks.h"
#include "cards.h"
#include "view.h"
#include "data_view.h"
#include "cards_view.h"
#include "utils.h"
#include "browse.h"
#include "personal_gui.h"
#include "filechooser_extra.h"
#include "preferences.h"
#include "groups_cb.h"
#include "popmenu.h"
#include "statusbar.h"
#include "tab.h"






void     confirm_overwrite (RubricaApp* app, RAbook* book);
void     save_session      (RubricaApp* app);
gboolean want_to_quit      (GtkWindow* window); 
gboolean want_to_save      (RubricaApp* app, gchar* filename);





/* Callbacks utils
 */
void 
confirm_overwrite (RubricaApp* app, RAbook* book)
{
  GtkWidget* message;
  gint response;
  GConfClient* client;
  gboolean backup;
  gint rate;
  
  client = gconf_client_get_default();

  message = gtk_message_dialog_new(GTK_WINDOW(app->window), 
				   GTK_DIALOG_MODAL, 
				   GTK_MESSAGE_WARNING,
				   GTK_BUTTONS_YES_NO,
				   "%s", _("Confirm overwrite"));
  gtk_message_dialog_format_secondary_text(GTK_MESSAGE_DIALOG(message), "%s", 
					   _("This addressbook exist."
					     "Do you want to overwrite?"));
  response = gtk_dialog_run(GTK_DIALOG(message));
  switch(response)
    {
    case GTK_RESPONSE_YES:
      backup = gconf_client_get_bool(client, RUBRICA_GCONF_BACKUP, NULL);
      rate   = gconf_client_get_int (client, RUBRICA_GCONF_RATE, NULL);
      
      if (r_abook_overwrite_file(book, backup, rate))
	{
	  gchar* file;
	  gchar* message;
	  
	  g_object_get(book, "addressbook-name", &file, NULL);
	  
	  message = g_strdup_printf(_("File %s was successfully saved."), 
				    file);
	  g_log(G_LOG_DOMAIN, G_LOG_LEVEL_INFO, 
		"file %s replaced successfully", file);	  
	  rubrica_app_notify(app, message);
	  
	  g_free(message);
	}
      break;
      
    case GTK_RESPONSE_NO:
      break;
      
    default:
      break;
    }
  
  gtk_widget_destroy(message);
  g_object_unref(client);
}


void 
save_session(RubricaApp* app)
{
  GtkWidget* paned;
  GtkWidget* grp_paned;
  GConfClient* client;
  gint width, height, position, grp_position;
 
  client = gconf_client_get_default();

  paned        = rubrica_app_get_paned(app);
  grp_paned    = rubrica_app_get_group_paned(app);
  position     = gtk_paned_get_position(GTK_PANED(paned));
  grp_position = gtk_paned_get_position(GTK_PANED(grp_paned));
  
  gtk_window_get_size(GTK_WINDOW(app->window), &width, &height);
  gconf_client_set_int(client, RUBRICA_GCONF_WIDTH, width, NULL);
  gconf_client_set_int(client, RUBRICA_GCONF_HEIGHT, height, NULL);
  gconf_client_set_int(client,RUBRICA_GCONF_PANED_POSITION, position, NULL);
  gconf_client_set_int(client,RUBRICA_GCONF_GRP_PANED_POSITION, 
		       grp_position, NULL);
  
  g_object_unref(client);
}


gboolean
want_to_quit(GtkWindow* window)
{
  GtkWidget* dialog;
  gint response;
  
  dialog = gtk_message_dialog_new(window, GTK_DIALOG_MODAL,
				  GTK_MESSAGE_QUESTION,
				  GTK_BUTTONS_YES_NO,
				  _("Do you want exit from rubrica?"));
  
  response = gtk_dialog_run(GTK_DIALOG(dialog));
  
  gtk_widget_destroy(dialog);
  if (response == GTK_RESPONSE_NO)
    return FALSE;
  
  return TRUE;
}


gboolean
want_to_save(RubricaApp* app, gchar* filename)
{
  GtkWidget* dialog;
  gint response;
  gchar* message;
  
  message = g_strdup_printf(_("Unsaved data will be lost. "
			      "Do you want to save \"%s\"?"), 
			    filename);
  dialog = gtk_message_dialog_new(GTK_WINDOW(app->window),
				  GTK_DIALOG_MODAL, GTK_MESSAGE_QUESTION,
				  GTK_BUTTONS_YES_NO, message);
  
  response = gtk_dialog_run(GTK_DIALOG(dialog));
  
  gtk_widget_destroy(dialog);
  if (response == GTK_RESPONSE_NO)
    return FALSE;
  
  return TRUE;
}



/* Toolbar - Menu callbacks 
*/
void 
on_new_cb (GtkWidget* button, gpointer data)
{
  rubrica_app_append_addressbook((RubricaApp*) data);
}


void 
on_open_cb (GtkWidget* button, gpointer data)
{
  RubricaApp* app = (RubricaApp*) data;  
  GtkWidget* chooser;
  GtkWidget* extra;
  GConfClient* client;
  gchar* folder = NULL;
  
  client  = gconf_client_get_default();
  folder  = gconf_client_get_string(client, RUBRICA_GCONF_FOLDER, NULL);
  
  chooser = gtk_file_chooser_dialog_new(_("Open an addressbook"), 
					GTK_WINDOW(app->window),
					GTK_FILE_CHOOSER_ACTION_OPEN,
					GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
					GTK_STOCK_OPEN, GTK_RESPONSE_ACCEPT,
					NULL);
  extra = rubrica_chooser_extra_new(app, RUBRICA_CHOOSER_OPEN_FILE);
  gtk_file_chooser_set_extra_widget (GTK_FILE_CHOOSER (chooser), extra);

  rubrica_utils_add_filters(chooser);
  rubrica_utils_enable_filter (GTK_FILE_CHOOSER (chooser), _("All files"));
  if (folder)
    gtk_file_chooser_set_current_folder (GTK_FILE_CHOOSER(chooser), folder);

  if (gtk_dialog_run(GTK_DIALOG(chooser)) == GTK_RESPONSE_ACCEPT)
    {
      gchar *file;

      file = gtk_file_chooser_get_filename (GTK_FILE_CHOOSER (chooser));

      if (!rubrica_app_load_file(app, file))
	{
	  GtkWidget* dialog;
	  gchar* message;
	  	  
	  g_log(G_LOG_DOMAIN, G_LOG_LEVEL_INFO, "Can't open file %s", file);
	  message = g_strdup_printf(_("Can't open file %s"), file);

	  rubrica_app_notify(app, message);	
	  dialog = gtk_message_dialog_new (NULL,
					   GTK_DIALOG_DESTROY_WITH_PARENT,
					   GTK_MESSAGE_ERROR,
					   GTK_BUTTONS_CLOSE,
					   message);
	  
	  gtk_dialog_run (GTK_DIALOG (dialog));
	  gtk_widget_destroy (dialog);  
	  
	  g_free(message);	  
	}
      
      g_free (file);
    }
  
  gtk_widget_destroy(chooser);
}


void 
on_save_cb (GtkWidget* button, gpointer data)
{
  RubricaApp* app = (RubricaApp*) data;
  RAbook* abook;
  GConfClient* client;
  gchar* name = NULL;
  gchar* path = NULL;
  gchar* file = NULL;
  gint rate;

  client = gconf_client_get_default();
  rate   = gconf_client_get_int (client, RUBRICA_GCONF_RATE, NULL);
  abook  = rubrica_app_get_active_addressbook(app);

  g_object_get(abook, 
	       "addressbook-name", &name, 
	       "addressbook-path", &path, NULL);

  if ((!name) || (g_ascii_strcasecmp(name, _("no name")) == 0))
    on_saveas_cb(NULL, app);
  else
    {
      file = g_strdup_printf("%s%s%s", path, G_DIR_SEPARATOR_S, name);
  
      if (g_file_test(file, G_FILE_TEST_EXISTS))
	confirm_overwrite(app, abook);
      else
	r_abook_save_file(abook, file, rate);  
      
      g_free(file);
    }
  
  g_free(path);
  g_free(name);
}


void 
on_saveas_cb (GtkWidget* button, gpointer data)
{
  RubricaApp* app = (RubricaApp*) data;
  RAbook* abook;
  GtkWidget* chooser;
  GtkWidget* extra;
  GConfClient* client;
  gchar* folder = NULL;
  gchar* file;
  gchar* oldfile = NULL;  
  gchar* plugin;

  client = gconf_client_get_default();  
  folder = gconf_client_get_string(client, RUBRICA_GCONF_FOLDER, NULL);
  abook  = rubrica_app_get_active_addressbook(app);
  plugin = r_abook_get_plugin_name(abook);
  
  chooser = gtk_file_chooser_dialog_new(_("Save addressbook as..."), 
					GTK_WINDOW(app->window),
					GTK_FILE_CHOOSER_ACTION_SAVE,
					GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
					GTK_STOCK_SAVE, GTK_RESPONSE_ACCEPT,
					NULL);
  extra = rubrica_chooser_extra_new(app, RUBRICA_CHOOSER_SAVE_FILE);
  rubrica_chooser_extra_enable_plugin(RUBRICA_CHOOSER_EXTRA(extra), plugin);

  gtk_file_chooser_set_extra_widget (GTK_FILE_CHOOSER (chooser), extra);
  
  rubrica_utils_add_filters(chooser);
  rubrica_utils_enable_filter (GTK_FILE_CHOOSER (chooser), _("All files"));

  if (folder)
    gtk_file_chooser_set_current_folder (GTK_FILE_CHOOSER(chooser), folder);

  g_object_get(abook, "addressbook-name", &oldfile, NULL);
  if (oldfile)
    {
      gchar* file;
      
      file = g_strdup_printf("%s%s%s", folder, G_DIR_SEPARATOR_S, oldfile);

      gtk_file_chooser_select_filename(GTK_FILE_CHOOSER(chooser), file);
      gtk_file_chooser_set_current_name(GTK_FILE_CHOOSER(chooser), oldfile);
      g_free(file);
    }
  else
    gtk_file_chooser_set_current_name(GTK_FILE_CHOOSER(chooser), _("no name"));
    
  if (gtk_dialog_run(GTK_DIALOG(chooser)) == GTK_RESPONSE_ACCEPT)
    {
      gint rate;
      gboolean backup;

      file   = gtk_file_chooser_get_filename (GTK_FILE_CHOOSER (chooser));
      backup = gconf_client_get_bool(client, RUBRICA_GCONF_BACKUP, NULL);
      rate   = gconf_client_get_int (client, RUBRICA_GCONF_RATE, NULL);
      
      if (g_ascii_strcasecmp(oldfile, _("no name")) == 0)
	{
	  //	  g_free(oldfile);
	  oldfile = g_strdup(file);
	}
      
      if (!r_abook_is_empty(abook))
	{
	  gchar* tmp;

	  if (oldfile && backup && (g_ascii_strcasecmp(file, oldfile) == 0) &&
	      g_file_test(oldfile, G_FILE_TEST_EXISTS))
	    {
	      tmp = g_strdup_printf("%s~", oldfile);

	      g_rename(oldfile, tmp);
	      g_remove(oldfile);

	      g_free(tmp);
	      g_free(oldfile);
	    }

	  if(r_abook_save_file(abook, file, rate))
	    {
	      gchar* message;
	      
	      g_log(G_LOG_DOMAIN, G_LOG_LEVEL_INFO, 
		    "file %s saved successfully", file);
	      message = g_strdup_printf(_("File %s was successfully saved."), 
					file);
	      rubrica_app_notify(app, message);	
	      
	      g_free(message);
	    } 
	}
    }
  
  gtk_widget_destroy(chooser);
  g_object_unref(client);
}



void 
on_cut_cb (GtkWidget* button, gpointer data)
{
  RubricaApp* app = (RubricaApp*) data;
  RAbook* book;
  gpointer card = NULL;

  book = rubrica_app_get_active_addressbook(app);
  card = r_abook_get_selected_card(R_ABOOK(book));
  
  if (card)
    {
      rubrica_app_push_card(app, card);
      rubrica_app_notify(app, _("Card cutted"));
      g_signal_emit_by_name(G_OBJECT(book), "card-cutted", card, NULL);
    }
}


void 
on_copy_cb (GtkWidget* button, gpointer data)
{
  RubricaApp* app = (RubricaApp*) data;
  RAbook* book;
  gpointer card, copy = NULL;

  book = rubrica_app_get_active_addressbook(app);
  card = r_abook_get_selected_card(R_ABOOK(book));
  copy = r_card_copy(R_CARD(card));
  
  if (copy)
    {
      rubrica_app_push_card(app, copy);
      rubrica_app_notify(app, _("Card copied"));
    }
}


void 
on_paste_cb (GtkWidget* button, gpointer data)
{
  RubricaApp* app = (RubricaApp*) data;
  RAbook* book;
  gpointer card = NULL;

  book = rubrica_app_get_active_addressbook(app);
  card = rubrica_app_pop_card(app);
  if (card)
    {
      r_abook_paste_card(R_ABOOK(book), R_CARD(card));
      rubrica_app_notify(app, _("Card pasted"));     
    }
}



void 
on_add_personal_cb (GtkWidget* button, gpointer data)
{  
  RGroupBox* box;

  box = r_lib_get_group_box();
  r_group_box_disable_all(box);
  cards_append_personal_card(RUBRICA_APP(data));
}


void 
on_add_company_cb (GtkWidget* button, gpointer data)
{
  RGroupBox* box;

  box = r_lib_get_group_box();
  r_group_box_disable_all(box);
  cards_append_company_card(RUBRICA_APP(data));
}


void 
on_delete_card_cb(GtkWidget* button, gpointer data)
{
  RubricaApp* app = (RubricaApp*) data;
  RAbook* book;
  gpointer card = NULL;

  book = rubrica_app_get_active_addressbook(app);
  card = r_abook_get_selected_card(R_ABOOK(book));
  
  if (card)
    {
      gboolean locked = FALSE;

      g_object_get(G_OBJECT(card), "card-locked", &locked, NULL);
      
      if (!locked)  // card is not locked, it is deletable
	{
	  if (r_card_has_refs(R_CARD(card)))
	    {  
	      /* TODO:
		messaggio chiedere se cancellare la scheda 
		o scheda e riferimenti 
		o scheda e alcuni riferimenti (creare lista)
		
		chiamare un foreach sulla lista delle schede
		da cancellare
	      
		nella foreach si chiamano
		
		r_abook_delete_card(R_ABOOK(book), card);
		rubrica_app_update_cards_view_on_delete(app);
		rubrica_app_move_card_to_trash(app, card);   	       
	      */
	    }

	  rubrica_app_push_card(app, card);

	  /* "delete" the card, a "card_removed" 
	     signal is emitted 
	  */
	  r_abook_delete_card(R_ABOOK(book), card);

	  rubrica_app_notify(app, _("Card moved to trash"));
	}
      else  // card is locked, it is undeletable
	{
	  rubrica_app_notify(app, _("This card is locked. "
				    "Modify card's property and retry."));
	  rubrica_app_show_message(app, 
				   _("This card is locked. "
				     "Modify card's property and retry."));
	  g_log(G_LOG_DOMAIN, G_LOG_LEVEL_INFO, 
		"Can't delete this card because it is locked. "
		"Modify card's property and retry.");

	  rubrica_app_update_view(app);
	  r_abook_unselect_cards(R_ABOOK(book));
	}      
    }
  else
    {
      gchar* message;

      message = g_strdup(_("You haven't selected any card. "
			   "Please select the card that you want to delete"));

      rubrica_app_notify(app, message);
      rubrica_app_show_message(app, message);
      
      g_free(message);
    }
}

void 
on_modify_card_cb (GtkWidget* button, gpointer data)
{ 
  RubricaApp* app = (RubricaApp*) data;
  RAbook* book;
  RCard* oldcard;
  RGroupBox* box;

  box = r_lib_get_group_box();
  r_group_box_disable_all(box);
  book = rubrica_app_get_active_addressbook(app);

  if (r_abook_is_empty(book))
    {
      rubrica_app_show_message(app, _("Warning. This addressbook is empty"));
      
      return;
    }

  oldcard = r_abook_get_selected_card (R_ABOOK(book));
  
  if (oldcard)
    cards_modify_card(app, book, oldcard);
  else
    rubrica_app_show_message(app, _("Warning. You must select a card first"));
}


void 
on_prefenences_cb(GtkWidget* button, gpointer data)
{
  rubrica_preferences_run_gui(NULL, data);
}


void 
on_info_cb (GtkWidget* button, gpointer data)
{
  GtkWidget* about;  
  gchar* authors[] ={
    "Nicola Fragale", 
    NULL}; 
  gchar* documenters[] = {NULL};
  GtkWidget* image;
  GdkPixbuf* logo;
  gchar* translators = 
    "Italian Nicola Fragale <nicolafragale@gmail.com>\n"
    "Ukrainian Iryna Rtveliashvili <irina.rtv@mail.ru>\n"
    "Deutsch Christian Faulhammer <christian@faulhammer.org>\n"
    "French Julien Lesouef <julien.lesouef@free.fr>\n"
    "Russian Dmitry Koroban <dimka.rs@gmail.com>";


  const gchar* license = 
    "   Copyright (C) Nicola Fragale <nicolafragale@gmail.com>\n\n"
    ""
    "   This program is free software; you can redistribute\n"
    "   it and/or modify it under the terms of the\n"
    "   GNU General Public License as published by\n"
    "   the Free Software Foundation; version 3 of the License.\n\n"
    "   This program is distributed in the hope that\n"
    "   it will be useful, but WITHOUT ANY WARRANTY;\n"
    "   without even the implied warranty of\n"
    "   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.\n"
    "   See the GNU General Public License for more details.\n\n"
    "   You should have received a copy of the\n"
    "   GNU General Public License along with this program;\n"
    "   if not, write to the\n"
    "   Free Software Foundation, Inc.\n"
    "   59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.\n";
 
  image = gtk_image_new_from_file(RUBRICA_GUI_DIR"/logo.png");
  logo  = gtk_image_get_pixbuf(GTK_IMAGE(image));

  about = gtk_about_dialog_new();  
  g_object_set(about, 
	       "authors",       authors,
	       "comments",      _("Rubrica, an addressbook for "
				  "GNOME environment"),
	       "copyright",     _("Copyright Nicola Fragale "
				  "<nicolafragale@gmail.com>"),
	       "documenters",   documenters,
	       "license",       license,
	       "translator-credits", translators,
	       "logo",           logo,
	       "version",        RUBRICA_VERSION, 
	       "website",       "http://rubrica.berlios.de",
	       "website-label", _("Rubrica's home page"), 
	       "wrap-license",  FALSE,
	       NULL);
  
  gdk_pixbuf_unref (logo);
       
  gtk_dialog_run (GTK_DIALOG(about));
  gtk_widget_destroy (about);
}



void on_print_cb(GtkWidget* button, gpointer data)
{  
  RubricaApp* app = (RubricaApp*) data;
  GtkWidget* view;
  GtkTreeModel *model;
  GtkTreeSelection *selection; 
  GtkTreeIter iter;

  view  = rubrica_app_get_current_view(app);
  model = rubrica_view_get_model(RUBRICA_VIEW(view));  
  selection = rubrica_view_get_selection(RUBRICA_VIEW(view));  

  if (gtk_tree_selection_get_selected(selection, &model, &iter))
    {
      gtk_list_store_set(GTK_LIST_STORE(model), &iter, PRINT_COLUMN, TRUE, -1);
      rubrica_app_print(RUBRICA_APP(data));
    }
  else
    rubrica_app_show_message(app, _("Warning. You must select a card first"));

  gtk_list_store_set(GTK_LIST_STORE(model), &iter, PRINT_COLUMN, FALSE, -1);  
}


void on_close_cb (GtkWidget* button, gpointer data)
{
  RubricaApp* app = (RubricaApp*) data;
  GtkWidget* notebook;
  GtkWidget* brother = NULL;
  GtkWidget* tab;
  gboolean changed = FALSE;
  gchar* name;
  gint page;
  
  notebook = rubrica_app_get_notebook(app);
  tab      = gtk_widget_get_ancestor (button, RUBRICA_TAB_TYPE);
  
  g_object_get(tab, "tab-brother", &brother, NULL);
  g_object_get(brother, "has-changed", &changed, NULL);
  name = rubrica_cards_view_get_addressbook_name(RUBRICA_CARDS_VIEW(brother));
  page = gtk_notebook_page_num(GTK_NOTEBOOK(notebook), brother);  
  gtk_notebook_set_current_page(GTK_NOTEBOOK(notebook), page-1);

  if (changed && want_to_save(app, name))
    on_save_cb(NULL, app);
  
  gtk_notebook_remove_page(GTK_NOTEBOOK(notebook), page);  
}



void on_first_cb(GtkWidget* button, gpointer data)
{
  rubrica_browse_first_card(RUBRICA_APP(data));
}


void on_last_cb(GtkWidget* button, gpointer data)
{
  rubrica_browse_last_card(RUBRICA_APP(data));
}



void 
on_prev_cb(GtkWidget* button, gpointer data)
{
  rubrica_browse_previous_card(RUBRICA_APP(data));
}


void 
on_next_cb (GtkWidget* button, gpointer data)
{
  rubrica_browse_next_card(RUBRICA_APP(data));
}


void 
on_quit_cb(GtkWidget* widget, gpointer data)
{
  RubricaApp* app = (RubricaApp*) data;
  GConfClient* client;
  gboolean confirm = FALSE;
  
  client = gconf_client_get_default();
  confirm = gconf_client_get_bool(client, RUBRICA_GCONF_CONFIRM_EXIT, NULL);
  g_object_unref(client);
    
  if (confirm)
    {
      if (!want_to_quit(GTK_WINDOW(app->window)))
	return;
    }
  
  save_session(app);  
  gtk_main_quit(); 
}


/* manages the delete event (x button was clicked in window) */
gboolean
on_delete_event_cb(GtkWidget* widget, GdkEvent  *event, gpointer data)
{
  RubricaApp* app = (RubricaApp*) data;
  GConfClient* client;
  gboolean confirm = FALSE;
  
  client  = gconf_client_get_default();
  confirm = gconf_client_get_bool(client, RUBRICA_GCONF_CONFIRM_EXIT, NULL);
  g_object_unref(client);
    
  if (confirm)
    {
      if (!want_to_quit(GTK_WINDOW(app->window)))
	return TRUE;
    }
  
  save_session(app);  
  
  gtk_main_quit(); 
  return FALSE;
}


void
on_notebook_switch_page(GtkNotebook *notebook, GtkNotebookPage *page,
			guint page_num,	gpointer data) 
{
  RubricaApp* app = (RubricaApp*) data;  
  GtkWidget* view;
  GtkWidget* statusbar;
  gchar* message = NULL;
  gboolean selected;

  view      = gtk_notebook_get_nth_page(GTK_NOTEBOOK(notebook), page_num);
  statusbar = rubrica_app_get_statusbar(app);
  g_object_get(view, "view-message", &message, "view-selected", &selected,
	       NULL);

  if (page_num >= 2)
    {
      rubrica_app_enable_previouses(app, selected);
      rubrica_app_enable_nexts(app, selected);
      
      rubrica_statusbar_enable_extra (RUBRICA_STATUSBAR(statusbar), TRUE);
      rubrica_statusbar_enable_images(RUBRICA_STATUSBAR(statusbar), TRUE);   
      rubrica_statusbar_extra_set_text (RUBRICA_STATUSBAR(statusbar), message);
    }
  else
    {
      rubrica_statusbar_enable_extra (RUBRICA_STATUSBAR(statusbar), FALSE);
      rubrica_statusbar_enable_images(RUBRICA_STATUSBAR(statusbar), FALSE);   
      rubrica_statusbar_push_message (RUBRICA_STATUSBAR(statusbar), message);
    }  
}

