/*
 *  Program: Rubrica
 *  file: ref.c
 *  
 *  Copyright (C) Nicola Fragale <nicolafragale@libero.it>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 3 of the License
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <gtk/gtk.h>
#include <glib/gi18n-lib.h>

#include "libral.h"
#include "app.h"
#include "filechooser_extra.h"


enum {
  FILEFORMAT_COLUMN,
  INFOS_COLUMN,
  EXTENSIONS_COLUMN,
  LAST_COLUMN
};


enum {
  PROP_0,
  APPLICATION,
  CHOOSER_FLAG,
};
  

struct _RubricaChooserExtraPrivate {
  RubricaApp* app;
  RubricaChooserFlag flag;
  RPluginManager*  manager;
  
  GtkWidget* expander;
  GtkWidget* tree;
  
  gboolean dispose_has_run;
};


#define RUBRICA_CHOOSER_EXTRA_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE((o),\
                                              RUBRICA_CHOOSER_EXTRA_TYPE,     \
                                              RubricaChooserExtraPrivate))


static GObjectClass *parent_class = NULL;


static void rubrica_chooser_extra_class_init (RubricaChooserExtraClass* klass);
static void rubrica_chooser_extra_init       (RubricaChooserExtra* obj);

static void rubrica_chooser_extra_finalize   (RubricaChooserExtra* self);
static void rubrica_chooser_extra_dispose    (RubricaChooserExtra* self);


static void rubrica_chooser_extra_set_property (GObject* obj, 
						guint property_id,
						const GValue* value, 
						GParamSpec* spec);
static void rubrica_chooser_extra_get_property (GObject* obj,
						guint property_id,
						GValue* value, 
						GParamSpec* spec);


static void     rubrica_chooser_extra_populate_tree (GtkTreeView* tree,
						     RPluginManager* manager,
						     gint flag);
static void     on_chooser_tree_event_occurred      (GtkTreeView* tree,
						     GdkEvent *event, 
						     gpointer data);

GType
rubrica_chooser_extra_get_type()
{
  static GType extra_type = 0;
  
  if (!extra_type)
    {
      static const GTypeInfo extra_info =
	{
	  sizeof(RubricaChooserExtraClass),
	  NULL,
	  NULL,
	  (GClassInitFunc) rubrica_chooser_extra_class_init,
	  NULL,
	  NULL,
	  sizeof(RubricaChooserExtra),
	  0,
	  (GInstanceInitFunc) rubrica_chooser_extra_init
	};

      extra_type = g_type_register_static (GTK_TYPE_VBOX, 
					   "RubricaChooserExtra", 
					   &extra_info, 0);
    }
  
  return extra_type;
}


static void 
rubrica_chooser_extra_dispose (RubricaChooserExtra* self)
{
  RubricaChooserExtraPrivate* priv;
  
  priv = RUBRICA_CHOOSER_EXTRA_GET_PRIVATE(self);
  if (priv->dispose_has_run)
    return;

  priv->dispose_has_run = TRUE;  
}


static void 
rubrica_chooser_extra_finalize (RubricaChooserExtra* self)
{
  RubricaChooserExtraPrivate* priv;

  priv = RUBRICA_CHOOSER_EXTRA_GET_PRIVATE(self);
  
  g_object_unref(priv->tree);
  g_object_unref(priv->expander);  
}


static void
rubrica_chooser_extra_class_init(RubricaChooserExtraClass* klass)
{
  GObjectClass *obj_class = G_OBJECT_CLASS (klass);
  GParamSpec* pspec;
  
  parent_class = g_type_class_peek_parent (klass);
  
  obj_class->finalize = (GObjectFinalizeFunc) rubrica_chooser_extra_finalize;
  obj_class->dispose  = (GObjectFinalizeFunc) rubrica_chooser_extra_dispose;

  obj_class->set_property = (gpointer) rubrica_chooser_extra_set_property;
  obj_class->get_property = (gpointer) rubrica_chooser_extra_get_property;

  g_type_class_add_private (klass, sizeof(RubricaChooserExtraPrivate));

 
  /**
   * RubricaChooserExtra:application
   *
   * a message to display in statusbar
   */
  pspec = g_param_spec_pointer("application", 
			       "pointer to main window's application",
			       "pointer to main window's application",
			       G_PARAM_CONSTRUCT | G_PARAM_READWRITE);
  g_object_class_install_property(obj_class, APPLICATION, pspec);

  /**
   * RubricaChooserExtra:chooser-flag
   *
   * the chooser's flag
   */
  pspec = g_param_spec_int("chooser-flag",  
			   "child-type",  
			   "the type of chooser_extra child",  
			   RUBRICA_CHOOSER_OPEN_FILE,
			   RUBRICA_CHOOSER_UNKNOW,
			   RUBRICA_CHOOSER_UNKNOW,
			   G_PARAM_CONSTRUCT | G_PARAM_READWRITE);
  g_object_class_install_property(obj_class, CHOOSER_FLAG, pspec);
}


static void
rubrica_chooser_extra_init(RubricaChooserExtra* self)
{
  RubricaChooserExtraPrivate* priv;
  GtkWidget* vbox;
  GtkWidget* scroll;
  GtkTreeSelection *selection;

  gtk_box_set_homogeneous(GTK_BOX(self), FALSE);
  gtk_box_set_spacing(GTK_BOX(self), 6);


  priv           = RUBRICA_CHOOSER_EXTRA_GET_PRIVATE(self); 
  priv->expander = gtk_expander_new(_("_Select the file's type"));
  priv->tree     = gtk_tree_view_new();
  vbox           = gtk_vbox_new(TRUE, 6);
  scroll         = gtk_scrolled_window_new (NULL, NULL);
  selection      = gtk_tree_view_get_selection(GTK_TREE_VIEW(priv->tree));

  gtk_expander_set_use_underline(GTK_EXPANDER(priv->expander), TRUE);
  gtk_expander_set_expanded(GTK_EXPANDER(priv->expander), FALSE);
  
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scroll),
				 GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
  gtk_scrolled_window_set_shadow_type(GTK_SCROLLED_WINDOW(scroll),
				      GTK_SHADOW_IN);  
  
  gtk_tree_selection_set_mode (selection, GTK_SELECTION_SINGLE);
  
  gtk_box_pack_start(GTK_BOX(vbox), scroll, TRUE, TRUE, 6);
  gtk_container_add(GTK_CONTAINER(scroll), priv->tree);
  gtk_container_add(GTK_CONTAINER(priv->expander), vbox);
  gtk_box_pack_start(GTK_BOX(self), priv->expander, TRUE, TRUE, 6);

  gtk_widget_show_all(GTK_WIDGET(self));

  priv->dispose_has_run = FALSE;
}


static void 
rubrica_chooser_extra_set_property (GObject* obj, guint property_id,
				    const GValue* value, GParamSpec* spec)
{
  RubricaChooserExtra* self = (RubricaChooserExtra*) obj;
  RubricaChooserExtraPrivate* priv = RUBRICA_CHOOSER_EXTRA_GET_PRIVATE(self);

  switch (property_id) 
    {
    case APPLICATION:
      priv->app = g_value_get_pointer(value);
      break;

    case CHOOSER_FLAG:
      priv->flag = g_value_get_int(value);
      break;

    default: 
      break; 
    } 
} 
 

static void 
rubrica_chooser_extra_get_property (GObject* obj, guint property_id,
				    GValue* value, GParamSpec* spec)
{
  RubricaChooserExtra* self = (RubricaChooserExtra*) obj;
  RubricaChooserExtraPrivate* priv = RUBRICA_CHOOSER_EXTRA_GET_PRIVATE(self);

  switch (property_id) 
    {    
    case APPLICATION:
      g_value_set_pointer(value, priv->app);
      break;      
      
    case CHOOSER_FLAG:
      g_value_set_int(value, priv->flag); 
      break;   

    default:
      break;  
    }  
}



static void 
on_chooser_tree_event_occurred (GtkTreeView* tree, GdkEvent *event, 
				gpointer data)
{
  RubricaChooserExtraPrivate* priv = RUBRICA_CHOOSER_EXTRA_GET_PRIVATE(data);  
  GtkTreeIter       iter;
  GtkTreeModel*     model;
  GtkTreePath*      path = NULL;
  GtkTreeSelection* selection; 
  GdkEventButton*   ev = (GdkEventButton *)event;
  gchar* label = NULL; 

  model     = gtk_tree_view_get_model(GTK_TREE_VIEW(tree));
  selection = gtk_tree_view_get_selection(tree);
  
  if (gtk_tree_view_get_path_at_pos(tree, ev->x, ev->y, &path, 
				    NULL, NULL, NULL))
    {
      gtk_tree_selection_select_path(selection, path);

      if(ev->button == 1) 
	{
	  const gchar *plugin_name;	       
	  
	  gtk_tree_model_get_iter(model, &iter, path);
	  gtk_tree_path_free(path);
	  
	  gtk_tree_model_get(model, &iter, 
			     FILEFORMAT_COLUMN, &plugin_name, -1);

	  if (g_ascii_strcasecmp(plugin_name, _("Autodetect")) == 0)
	    {
	      gchar* label; 
	      rubrica_app_set_plugin_name(priv->app, "autodetect");
	      label = g_strdup(_("_Select the file's type (try to "
				 "autodetect the file format)"));
	    }
	  else
	    {	      
	      rubrica_app_set_plugin_name(priv->app, plugin_name);
	      label = g_strdup_printf(_("_Select the file's type "
					"(now is active the %s plugin)"), 
				      plugin_name);
	    }
	  
	  gtk_expander_set_label(GTK_EXPANDER(priv->expander), label); 
	  g_free(label);	  
	}
    }  
}




static void 
rubrica_chooser_extra_populate_tree (GtkTreeView* tree,
				     RPluginManager* manager, gint flag)
{
  GtkTreeModel* model;
  GtkCellRenderer *renderer;
  GtkTreeIter iter;
  RPlugin* plugin;
  gint n, i;
  
  model = (GtkTreeModel*) gtk_list_store_new(LAST_COLUMN,
					     G_TYPE_STRING,
					     G_TYPE_STRING,
					     G_TYPE_STRING);
  gtk_tree_view_set_model(tree, model);
  
  renderer = gtk_cell_renderer_text_new();
  gtk_tree_view_insert_column_with_attributes(tree, -1, _("File's type"), 
					      renderer, "text", 
					      FILEFORMAT_COLUMN, NULL);  
  renderer = gtk_cell_renderer_text_new();
  gtk_tree_view_insert_column_with_attributes(tree, -1, _("Description"), 
					      renderer, "text", 
					      INFOS_COLUMN, NULL);   
  renderer = gtk_cell_renderer_text_new();
  gtk_tree_view_insert_column_with_attributes(tree, -1, _("Extensions"), 
					      renderer, "text", 
					      EXTENSIONS_COLUMN, NULL);   

  if (flag == RUBRICA_CHOOSER_OPEN_FILE)
    {
      gtk_list_store_append(GTK_LIST_STORE(model), &iter);  
      gtk_list_store_set(GTK_LIST_STORE(model), &iter,
			 FILEFORMAT_COLUMN, _("Autodetect"),
			 INFOS_COLUMN, _("Try to autodetect the file format"),
			 EXTENSIONS_COLUMN, "",
			 -1);   
    }

  n = r_plugin_manager_how_many(manager);  
  for (i = 0; i < n; i++)
    {
      gchar *name, *info, *ext;
      
      plugin = r_plugin_manager_get_nth_plugin(manager, i);
      
      g_object_get(plugin, 
		   "plugin-name", &name, 
		   "plugin-info", &info,
		   "plugin-extensions", &ext, NULL);

      gtk_list_store_append(GTK_LIST_STORE(model), &iter);  
      gtk_list_store_set(GTK_LIST_STORE(model), &iter,
			 FILEFORMAT_COLUMN, name,
			 INFOS_COLUMN,      info,
			 EXTENSIONS_COLUMN, ext,
			 -1);
    }
}




/*    Public
*/
GtkWidget* 
rubrica_chooser_extra_new(RubricaApp* app, RubricaChooserFlag flag)
{
  GtkWidget* extra;
  RubricaChooserExtraPrivate* priv;

  extra = GTK_WIDGET(g_object_new(RUBRICA_CHOOSER_EXTRA_TYPE, 
				  "application", app, 
				  "chooser-flag", flag, NULL));

  priv = RUBRICA_CHOOSER_EXTRA_GET_PRIVATE(extra);
  priv->manager = rubrica_app_get_plugin_manager(priv->app);
  rubrica_chooser_extra_populate_tree(GTK_TREE_VIEW(priv->tree),
				      priv->manager, priv->flag);
  
  g_signal_connect(GTK_OBJECT (priv->tree), "button_press_event",
		   G_CALLBACK (on_chooser_tree_event_occurred), extra);
  
  return extra;
}


void 
rubrica_chooser_extra_enable_plugin (RubricaChooserExtra* extra, 
				     gchar* plugin)
{
  RubricaChooserExtraPrivate* priv;
  GtkTreeModel *model;
  GtkTreeSelection *select;
  GtkTreeIter iter;
  gchar* label; 

  g_return_if_fail(IS_RUBRICA_CHOOSER_EXTRA(extra));
  g_return_if_fail(plugin != NULL);

  priv = RUBRICA_CHOOSER_EXTRA_GET_PRIVATE(extra);
  label = g_strdup_printf(_("_Select the file's type "
			    "(now is active the %s plugin)"), plugin);
  gtk_expander_set_label(GTK_EXPANDER(priv->expander), label); 
  g_free(label);
  
  model  = gtk_tree_view_get_model(GTK_TREE_VIEW(priv->tree));
  select = gtk_tree_view_get_selection(GTK_TREE_VIEW(priv->tree));
  if (gtk_tree_model_get_iter_first(GTK_TREE_MODEL(model), &iter))
    {
      gboolean done = FALSE;
      GtkTreePath *path;

      while(!done)
	{
	  gchar* fileformat;
	  
	  gtk_tree_model_get(model, &iter, 
			     FILEFORMAT_COLUMN, &fileformat, -1);
	  if (g_ascii_strcasecmp(plugin, fileformat) == 0)
	    {
	      path  = gtk_tree_model_get_path(model, &iter);	  
	      gtk_tree_selection_select_iter(select, &iter);
	      gtk_tree_view_scroll_to_cell(GTK_TREE_VIEW(priv->tree), path, 
					   NULL, FALSE, 0.5, 0.0);
	      done = TRUE;
	    }
	  else
	    if (!gtk_tree_model_iter_next(model, &iter))
	      break;
	}
    }
}

