/*
 *   file: form_address.c
 *  
 *  Copyright (C) Nicola Fragale <nicolafragale@libero.it>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <gtk/gtk.h>
#include <glib/gi18n-lib.h>
#include <glade/glade.h>

#include "libral.h"

#include "app.h"
#include "form.h"
#include "form_address.h"
#include "groups_cb.h"
#include "models.h"
#include "types.h"
#include "utils.h"


struct _RubricaFormAddressPrivate {
  GtkWidget* st;
  GtkWidget* num;
  GtkWidget* zip;
  GtkWidget* city;
  GtkWidget* prov;
  GtkWidget* state;
  GtkWidget* country;
  GtkWidget* add_type;
  GtkWidget* add_tree;
  GtkWidget* add_add;
  GtkWidget* add_del;
  GtkWidget* add_mod;
  
  gboolean dispose_has_run;
};


static void rubrica_form_address_class_init (RubricaFormAddressClass* klass);
static void rubrica_form_address_init       (RubricaFormAddress* obj);

static void rubrica_form_address_dispose    (RubricaFormAddress* obj);
static void rubrica_form_address_finalize   (RubricaFormAddress* obj);


static gboolean extract_addresses         (GtkTreeModel *model,
					   GtkTreePath *path,
					   GtkTreeIter *iter, gpointer data); 

static void     form_address_extract_data (RubricaFormAddress* form, 
					   RCard *card);
static void     form_address_display_data (RubricaFormAddress* form, 
					   RCard *card);
static void     form_address_clean_form   (RubricaFormAddress* form);


static RubricaFormClass* parent_class;


GType
rubrica_form_address_get_type()
{
  static GType form_address_type = 0;
  
  if (!form_address_type)
    {
      static const GTypeInfo form_address_info =
	{
	  sizeof(RubricaFormAddressClass),
	  NULL,
	  NULL,
	  (GClassInitFunc) rubrica_form_address_class_init,
	  NULL,
	  NULL,
	  sizeof(RubricaFormAddress),
	  0,
	  (GInstanceInitFunc) rubrica_form_address_init
	};

      form_address_type = g_type_register_static (RUBRICA_FORM_TYPE, 
						  "RubricaFormAddress",
						  &form_address_info, 0);
    }
  
  return form_address_type;
}


static void
rubrica_form_address_class_init(RubricaFormAddressClass* klass)
{
  GObjectClass *class;
 
  class  = G_OBJECT_CLASS (klass);
  class->dispose  = (GObjectFinalizeFunc) rubrica_form_address_dispose;
  class->finalize = (GObjectFinalizeFunc) rubrica_form_address_finalize;

  parent_class = RUBRICA_FORM_CLASS(klass);
  parent_class->extract_data = (gpointer) form_address_extract_data;
  parent_class->display_data = (gpointer) form_address_display_data;
  parent_class->clean_form   = (gpointer) form_address_clean_form;
}


static void
rubrica_form_address_init(RubricaFormAddress* self)
{
  self->priv = g_new(RubricaFormAddressPrivate, 1);
  
  self->priv->st       = NULL;
  self->priv->num      = NULL;
  self->priv->zip      = NULL;
  self->priv->city     = NULL;
  self->priv->prov     = NULL;
  self->priv->state    = NULL;
  self->priv->country  = NULL;
  self->priv->add_type = NULL;
  self->priv->add_tree = NULL;
  self->priv->add_add  = NULL;
  self->priv->add_del  = NULL;
  self->priv->add_mod  = NULL;

  self->priv->dispose_has_run = FALSE;
}

 

static void 
rubrica_form_address_dispose (RubricaFormAddress* self)
{
  g_return_if_fail(IS_RUBRICA_FORM(self));
  
  if (self->priv->dispose_has_run)
    return;
  
  g_object_unref(self->priv->st);
  g_object_unref(self->priv->num);
  g_object_unref(self->priv->zip);
  g_object_unref(self->priv->city);
  g_object_unref(self->priv->prov);
  g_object_unref(self->priv->state);
  g_object_unref(self->priv->country);
  g_object_unref(self->priv->add_type);
  g_object_unref(self->priv->add_tree);
  g_object_unref(self->priv->add_add);
  g_object_unref(self->priv->add_del);
  g_object_unref(self->priv->add_mod);
  
  self->priv->dispose_has_run = TRUE;
}


static void 
rubrica_form_address_finalize (RubricaFormAddress* self)
{
  g_return_if_fail(IS_RUBRICA_FORM(self));
  
  g_free(self->priv);
  self->priv = NULL;
}


gboolean 
extract_addresses (GtkTreeModel *model, GtkTreePath *path,
		   GtkTreeIter *iter, gpointer data)
{
  RCard* card = (RCard*) data;
  RAddress* address = NULL;
  RAddressType adtype;
  gchar *type, *street, *number, *city;
  gchar *zip, *province, *state, *country;

  gtk_tree_model_get(model, iter, 
		     ADDRESS_STREET_COLUMN,   &street,
		     ADDRESS_NUMBER_COLUMN,   &number,
		     ADDRESS_CITY_COLUMN,     &city,
		     ADDRESS_TYPE_COLUMN,     &type,
		     ADDRESS_ZIP_COLUMN,      &zip,
		     ADDRESS_PROVINCE_COLUMN, &province,
		     ADDRESS_STATE_COLUMN,    &state,
		     ADDRESS_COUNTRY_COLUMN,  &country,
		     -1);
  adtype = r_address_lookup_str2enum(type);

  address = r_address_new();
  g_object_set(address, "address-type", adtype, "street", street, 
	       "street-number", number, "city", city, "zip", zip, 
	       "province", province, "state", state, 
	       "country", country, NULL);
  r_card_add_address(R_CARD(card), address);

  return FALSE;
}


static void 
form_address_extract_data (RubricaFormAddress* form, RCard *card)
{
  GtkTreeModel* model = NULL;

  g_return_if_fail(IS_R_CARD(card));

  model = gtk_tree_view_get_model(GTK_TREE_VIEW(form->priv->add_tree));
  if (model)
    {
      gtk_tree_model_foreach(GTK_TREE_MODEL(model), extract_addresses, card);
      g_object_unref(model);
    }
}


static void 
form_address_display_data (RubricaFormAddress* form, RCard *card)
{
  GtkTreeView* tree;
  GtkTreeModel *model;
  GtkTreeIter iter;
  gpointer data;

  gtk_combo_box_set_active(GTK_COMBO_BOX(form->priv->add_type),
			   R_ADDRESS_PREF);    

  tree  = (GtkTreeView*) form->priv->add_tree;
  model = gtk_tree_view_get_model(tree);

  data = r_card_get_address(R_CARD(card));
  for (; data; data = r_card_get_next_address(R_CARD(card)))
    {
      RAddressType adtype;
      gchar *street, *number, *city, *label, *type;
      gchar *zip, *province, *state, *country;
      
      g_object_get(R_ADDRESS(data), "address-type", &adtype, 
		   "street", &street, "street-number", &number, 
		   "city", &city, "zip", &zip, "province", &province, 
		   "state", &state, "country", &country, NULL);
      label = r_address_lookup_enum2lbl(adtype);
      type  = r_address_lookup_enum2str(adtype);

      gtk_list_store_append(GTK_LIST_STORE(model), &iter);
      gtk_list_store_set(GTK_LIST_STORE(model), &iter, 
			 ADDRESS_STREET_COLUMN,   street,
			 ADDRESS_NUMBER_COLUMN,   number,
			 ADDRESS_CITY_COLUMN,     city,
			 ADDRESS_LABEL_COLUMN,    _(label),
			 ADDRESS_TYPE_COLUMN,     type,
			 ADDRESS_ZIP_COLUMN,      zip,
			 ADDRESS_PROVINCE_COLUMN, province,
			 ADDRESS_STATE_COLUMN,    state,
			 ADDRESS_COUNTRY_COLUMN,  country,
			 -1);  
    }
}


static void 
form_address_clean_form (RubricaFormAddress* form)
{
  GtkTreeModel *model;

  gtk_entry_set_text(GTK_ENTRY(form->priv->st),      "");
  gtk_entry_set_text(GTK_ENTRY(form->priv->num),     "");
  gtk_entry_set_text(GTK_ENTRY(form->priv->zip),     "");
  gtk_entry_set_text(GTK_ENTRY(form->priv->city),    "");
  gtk_entry_set_text(GTK_ENTRY(form->priv->prov),    "");
  gtk_entry_set_text(GTK_ENTRY(form->priv->state),   "");
  gtk_entry_set_text(GTK_ENTRY(form->priv->country), "");  
  
  gtk_combo_box_set_active(GTK_COMBO_BOX(form->priv->add_type), 
			   R_ADDRESS_PREF);

  model = gtk_tree_view_get_model(GTK_TREE_VIEW(form->priv->add_tree));
  gtk_list_store_clear(GTK_LIST_STORE(model));  
}


/*  Callbacks
*/
void 
form_addresses_list_activated(GtkTreeSelection *selection, gpointer data)
{
  RubricaFormAddress* form = (RubricaFormAddress*) data;
  GtkTreeModel *model;
  GtkTreeIter iter;

  model = gtk_tree_view_get_model(GTK_TREE_VIEW(form->priv->add_tree));

  if (gtk_tree_selection_get_selected(selection, &model, &iter))
    {
      RAddressType adtype;
      gchar *street, *number, *city, *zip;
      gchar *province, *state, *country, *type;
      
      gtk_tree_model_get(model, &iter, 
			 ADDRESS_STREET_COLUMN,   &street,
			 ADDRESS_NUMBER_COLUMN,   &number,
			 ADDRESS_CITY_COLUMN,     &city,
			 ADDRESS_TYPE_COLUMN,     &type,
			 ADDRESS_ZIP_COLUMN,      &zip,
			 ADDRESS_PROVINCE_COLUMN, &province,
			 ADDRESS_STATE_COLUMN,    &state,
			 ADDRESS_COUNTRY_COLUMN,  &country,
			 -1);
      adtype = r_address_lookup_str2enum(type);

      gtk_combo_box_set_active(GTK_COMBO_BOX(form->priv->add_type), adtype);
      gtk_entry_set_text(GTK_ENTRY(form->priv->st),      street ? street : "");
      gtk_entry_set_text(GTK_ENTRY(form->priv->num),     number ? number : "");
      gtk_entry_set_text(GTK_ENTRY(form->priv->zip),     zip  ? zip : "");
      gtk_entry_set_text(GTK_ENTRY(form->priv->city),    city ? city : "");
      gtk_entry_set_text(GTK_ENTRY(form->priv->prov),
			 province ? province : "");
      gtk_entry_set_text(GTK_ENTRY(form->priv->state),   state ? state : "");
      gtk_entry_set_text(GTK_ENTRY(form->priv->country), 
			 country ? country : "");
      
      gtk_widget_set_sensitive(form->priv->add_mod, TRUE);
      gtk_widget_set_sensitive(form->priv->add_del, TRUE);
    }
}


void
form_address_add_clicked (GtkButton *button, gpointer data)
{
  GtkTreeModel *model;
  GtkTreeIter iter;
  RubricaFormAddress *form = (RubricaFormAddress*) data;
  RAddress* address;
  RAddressType addtype;
  gchar *street, *number, *city, *zip, *type;
  gchar *label, *prov, *state, *country;
  
  gtk_widget_set_sensitive(GTK_WIDGET(form->priv->add_mod), FALSE);
  gtk_widget_set_sensitive(GTK_WIDGET(form->priv->add_del), FALSE);

  model = gtk_tree_view_get_model(GTK_TREE_VIEW(form->priv->add_tree));
  g_return_if_fail(GTK_IS_TREE_MODEL(model));
  
  address = r_address_new();
  
  addtype = gtk_combo_box_get_active(GTK_COMBO_BOX(form->priv->add_type));
  street  = g_strdup(gtk_entry_get_text(GTK_ENTRY(form->priv->st)));
  number  = g_strdup(gtk_entry_get_text(GTK_ENTRY(form->priv->num)));
  city    = g_strdup(gtk_entry_get_text(GTK_ENTRY(form->priv->city)));
  zip     = g_strdup(gtk_entry_get_text(GTK_ENTRY(form->priv->zip)));
  prov    = g_strdup(gtk_entry_get_text(GTK_ENTRY(form->priv->prov)));
  state   = g_strdup(gtk_entry_get_text(GTK_ENTRY(form->priv->state)));
  country = g_strdup(gtk_entry_get_text(GTK_ENTRY(form->priv->country)));
  
  g_object_set(address, "address-type", addtype, "street", street, 
	       "street-number", number, "city", city, "zip", zip, 
	       "province", prov, "state", state, "country", country, NULL);

  type  = r_address_lookup_enum2str(addtype);
  label = _(r_address_lookup_enum2lbl(addtype));

  gtk_list_store_append(GTK_LIST_STORE(model), &iter);
  gtk_list_store_set(GTK_LIST_STORE(model), &iter, 
		     ADDRESS_STREET_COLUMN,   street,
		     ADDRESS_NUMBER_COLUMN,   number,
		     ADDRESS_LABEL_COLUMN,    label,
		     ADDRESS_TYPE_COLUMN,     type,
		     ADDRESS_CITY_COLUMN,     city,
		     ADDRESS_ZIP_COLUMN,      zip,
		     ADDRESS_PROVINCE_COLUMN, prov,
		     ADDRESS_STATE_COLUMN,    state,
		     ADDRESS_COUNTRY_COLUMN,  country,		     
		     -1);

  gtk_combo_box_set_active(GTK_COMBO_BOX(form->priv->add_type),
			   R_ADDRESS_PREF);
  gtk_entry_set_text(GTK_ENTRY(form->priv->st),     "");
  gtk_entry_set_text(GTK_ENTRY(form->priv->num),    "");
  gtk_entry_set_text(GTK_ENTRY(form->priv->zip),    "");
  gtk_entry_set_text(GTK_ENTRY(form->priv->city),   "");
  gtk_entry_set_text(GTK_ENTRY(form->priv->prov),   "");
  gtk_entry_set_text(GTK_ENTRY(form->priv->state),  "");
  gtk_entry_set_text(GTK_ENTRY(form->priv->country),"");
}


void
form_address_mod_clicked (GtkButton *button, gpointer data)
{
  GtkTreeModel *model;
  GtkTreeIter iter;
  GtkTreeSelection *sel;
  RAddress* address;
  RubricaFormAddress* form = (RubricaFormAddress*) data;
  
  model = gtk_tree_view_get_model(GTK_TREE_VIEW(form->priv->add_tree));
  g_return_if_fail(GTK_IS_TREE_MODEL(model));

  sel = gtk_tree_view_get_selection(GTK_TREE_VIEW(form->priv->add_tree));

  if (gtk_tree_selection_get_selected(sel, NULL, &iter))
  {
    gint i;
    GtkTreePath *path;
    gchar *street, *number, *city, *zip, *type;
    gchar *prov, *state, *country, *label;
    RAddressType addtype;

    /* create a new address and fill with form's data */
    address = r_address_new();    

    addtype = gtk_combo_box_get_active(GTK_COMBO_BOX(form->priv->add_type));
    street  = g_strdup(gtk_entry_get_text(GTK_ENTRY(form->priv->st)));
    number  = g_strdup(gtk_entry_get_text(GTK_ENTRY(form->priv->num)));
    city    = g_strdup(gtk_entry_get_text(GTK_ENTRY(form->priv->city)));
    zip     = g_strdup(gtk_entry_get_text(GTK_ENTRY(form->priv->zip)));
    prov    = g_strdup(gtk_entry_get_text(GTK_ENTRY(form->priv->prov)));
    state   = g_strdup(gtk_entry_get_text(GTK_ENTRY(form->priv->state)));
    country = g_strdup(gtk_entry_get_text(GTK_ENTRY(form->priv->country)));

    g_object_set(address, "address-type", addtype, "street", street, 
		 "street-number", number, "city", city, "zip", zip, 
		 "province", prov, "state", state, "country", country, NULL);

    type  = r_address_lookup_enum2str(addtype);
    label = _(r_address_lookup_enum2lbl(addtype));

    /* remove old address from tree list 
     */    
    path = gtk_tree_model_get_path(model, &iter);
    i = gtk_tree_path_get_indices(path)[0];
    
    gtk_list_store_remove(GTK_LIST_STORE(model), &iter);
    gtk_tree_path_free(path);
    
    /* insert new address into the tree list 
     */
    gtk_list_store_insert(GTK_LIST_STORE(model), &iter, i);
    gtk_list_store_set(GTK_LIST_STORE(model), &iter,
		       ADDRESS_STREET_COLUMN,   street,
		       ADDRESS_NUMBER_COLUMN,   number,
		       ADDRESS_LABEL_COLUMN,    label,
		       ADDRESS_TYPE_COLUMN,     type,
		       ADDRESS_CITY_COLUMN,     city,
		       ADDRESS_ZIP_COLUMN,      zip,
		       ADDRESS_PROVINCE_COLUMN, prov,
		       ADDRESS_STATE_COLUMN,    state,
		       ADDRESS_COUNTRY_COLUMN,  country,	       
		       -1);

    /* clean form 
     */
    gtk_combo_box_set_active(GTK_COMBO_BOX(form->priv->add_type),
			     R_ADDRESS_PREF);
    gtk_entry_set_text(GTK_ENTRY(form->priv->st),      "");
    gtk_entry_set_text(GTK_ENTRY(form->priv->num),     "");
    gtk_entry_set_text(GTK_ENTRY(form->priv->zip),     "");
    gtk_entry_set_text(GTK_ENTRY(form->priv->city),    "");
    gtk_entry_set_text(GTK_ENTRY(form->priv->prov),    "");
    gtk_entry_set_text(GTK_ENTRY(form->priv->state),   "");
    gtk_entry_set_text(GTK_ENTRY(form->priv->country), "");		       
    
    gtk_widget_set_sensitive(GTK_WIDGET(form->priv->add_mod), FALSE);
    gtk_widget_set_sensitive(GTK_WIDGET(form->priv->add_del), FALSE);   
  }  
}


void
form_address_del_clicked (GtkButton *button, gpointer data)
{
  GtkTreeModel *model;
  GtkTreeIter iter;
  GtkTreeSelection *sel;
  RubricaFormAddress* form = (RubricaFormAddress*) data;
  
  model = gtk_tree_view_get_model(GTK_TREE_VIEW(form->priv->add_tree));
  sel = gtk_tree_view_get_selection(GTK_TREE_VIEW(form->priv->add_tree));
  
  if (gtk_tree_selection_get_selected(sel, NULL, &iter))
  {
    gtk_list_store_remove(GTK_LIST_STORE(model), &iter);
    
    gtk_combo_box_set_active(GTK_COMBO_BOX(form->priv->add_type),
			     R_ADDRESS_PREF);
    gtk_entry_set_text(GTK_ENTRY(form->priv->st),      "");
    gtk_entry_set_text(GTK_ENTRY(form->priv->num),     "");
    gtk_entry_set_text(GTK_ENTRY(form->priv->zip),     "");
    gtk_entry_set_text(GTK_ENTRY(form->priv->city),    "");
    gtk_entry_set_text(GTK_ENTRY(form->priv->prov),    "");
    gtk_entry_set_text(GTK_ENTRY(form->priv->state),   "");
    gtk_entry_set_text(GTK_ENTRY(form->priv->country), "");

    gtk_widget_set_sensitive(GTK_WIDGET(form->priv->add_mod), FALSE);
    gtk_widget_set_sensitive(GTK_WIDGET(form->priv->add_del), FALSE);   
  }  
}





/*   ***************************** Public *****************************
*/


/**
 * rubrica_form_address_new
 *
 * create a #RubricaFormAddress
 *
 * Returns: a new #RubricaFormAddress
 */
RubricaFormAddress*
rubrica_form_address_new(RubricaApp* app, GladeXML *gui)
{
  RubricaFormAddress* form;
  GtkTreeView *tree;
  GtkTreeModel *model;
  GtkTreeSelection *selection;
  GtkWidget* combo_box;
  GtkWidget* add_label;
  gint i;

  form = g_object_new(rubrica_form_address_get_type(), NULL);
  
  form->priv->st       = glade_xml_get_widget(gui, "st");
  form->priv->num      = glade_xml_get_widget(gui, "num");
  form->priv->zip      = glade_xml_get_widget(gui, "zip");
  form->priv->city     = glade_xml_get_widget(gui, "city");
  form->priv->prov     = glade_xml_get_widget(gui, "prov");
  form->priv->state    = glade_xml_get_widget(gui, "state");
  form->priv->country  = glade_xml_get_widget(gui, "country");
  form->priv->add_tree = glade_xml_get_widget(gui, "addresses_tree");
  form->priv->add_add  = glade_xml_get_widget(gui, "add_add");
  form->priv->add_del  = glade_xml_get_widget(gui, "add_del");
  form->priv->add_mod  = glade_xml_get_widget(gui, "add_mod");
  
  add_label = glade_xml_get_widget(gui, "add_label");  
  combo_box = glade_xml_get_widget(gui, "add_combo_box");

  form->priv->add_type = gtk_combo_box_new_text ();
  gtk_box_pack_start(GTK_BOX(combo_box), form->priv->add_type, TRUE, TRUE, 0);
  gtk_widget_show(form->priv->add_type);
  for (i = R_ADDRESS_PREF; i <= R_ADDRESS_UNKNOWN; i++)
    gtk_combo_box_append_text(GTK_COMBO_BOX(form->priv->add_type), 
			      _(r_address_lookup_enum2lbl(i)));	

  gtk_label_set_mnemonic_widget(GTK_LABEL(add_label), form->priv->add_type);
  gtk_combo_box_set_active(GTK_COMBO_BOX(form->priv->add_type), 
			   R_ADDRESS_PREF);  

  gtk_widget_set_sensitive(GTK_WIDGET(form->priv->add_mod), FALSE);
  gtk_widget_set_sensitive(GTK_WIDGET(form->priv->add_del), FALSE);
  
  /*   address model/view 
   */
  tree = (GtkTreeView *) form->priv->add_tree;
  model = rubrica_model_new(ADDRESS_MODEL);
  gtk_tree_view_set_model(tree, GTK_TREE_MODEL(model));
  selection = gtk_tree_view_get_selection(tree);
  gtk_tree_selection_set_mode(selection, GTK_SELECTION_SINGLE);

  rubrica_model_add_columns(tree, ADDRESS_COLUMNS);

  g_signal_connect(G_OBJECT(selection), "changed", 
		   G_CALLBACK(form_addresses_list_activated), form);  
  g_object_unref(model);
  
  /*    address buttons
  */  
  g_signal_connect (G_OBJECT (form->priv->add_add), "clicked", 
		    G_CALLBACK (form_address_add_clicked), form);
  g_signal_connect (G_OBJECT (form->priv->add_mod), "clicked", 
		    G_CALLBACK (form_address_mod_clicked), form); 
  g_signal_connect (G_OBJECT (form->priv->add_del), "clicked", 
		    G_CALLBACK (form_address_del_clicked), form);
  
  gtk_widget_grab_focus (form->priv->st);
  return form;
}


/**
 * rubrica_form_address_free
 * @form: a #RubricaFormAddress
 * 
 * free the #RubricaFormAddress
 */
void
rubrica_form_address_free(RubricaFormAddress* form)
{
  g_return_if_fail(IS_RUBRICA_FORM(form));

  g_object_unref(form);   
}
