/*
 *   file: form_card.c
 *  
 *  Copyright (C) Nicola Fragale <nicolafragale@libero.it>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <glib/gi18n-lib.h>
#include <gtk/gtk.h>
#include <glade/glade.h>

#include "libral.h"

#include "app.h"
#include "form.h"
#include "form_card.h"
#include "app.h"
#include "view.h"
#include "groups_view.h"
#include "groups_cb.h"
#include "models.h"
#include "types.h"
#include "themes.h"


struct _RubricaFormCardPrivate {
  RGroupBox* groups;

  GtkWidget* card;
  GtkWidget* groups_view;
  GtkWidget* lock;
  GtkWidget* rate;  
  
  gboolean dispose_has_run;
};


RubricaThemedItem faces[] = {
  {N_("None"),       "rate1"},     // R_RATE_NONE
  {N_("Few"),        "rate2"},     // R_RATE_FEW
  {N_("Normal"),     "rate3"},     // R_RATE_NORMAL
  {N_("High"),       "rate4"},     // R_RATE_HIGH
  {N_("Very high"),  "rate5"},     // R_RATE_VERY_HIGH
  {NULL}
};


static void rubrica_form_card_class_init (RubricaFormCardClass* klass);
static void rubrica_form_card_init       (RubricaFormCard* obj);

static void rubrica_form_card_dispose    (RubricaFormCard* obj);
static void rubrica_form_card_finalize   (RubricaFormCard* obj);

static gboolean extract_groups (GtkTreeModel *model, GtkTreePath *path,
				GtkTreeIter *iter, gpointer data); 

static void form_card_extract_data (RubricaFormCard* form, RCard *card);
static void form_card_display_data (RubricaFormCard* form, RCard *card);
static void form_card_clean_form   (RubricaFormCard* form);


static RubricaFormClass* parent_class;


GType
rubrica_form_card_get_type()
{
  static GType form_card_type = 0;
  
  if (!form_card_type)
    {
      static const GTypeInfo form_card_info =
	{
	  sizeof(RubricaFormCardClass),
	  NULL,
	  NULL,
	  (GClassInitFunc) rubrica_form_card_class_init,
	  NULL,
	  NULL,
	  sizeof(RubricaFormCard),
	  0,
	  (GInstanceInitFunc) rubrica_form_card_init
	};

      form_card_type = g_type_register_static (RUBRICA_FORM_TYPE, 
					       "RubricaFormCard",
					       &form_card_info, 0);
    }
  
  return form_card_type;
}


static void
rubrica_form_card_class_init(RubricaFormCardClass* klass)
{
  GObjectClass *class;
 
  class  = G_OBJECT_CLASS (klass);
  class->dispose  = (GObjectFinalizeFunc) rubrica_form_card_dispose;
  class->finalize = (GObjectFinalizeFunc) rubrica_form_card_finalize;

  parent_class = RUBRICA_FORM_CLASS(klass);
  parent_class->extract_data = (gpointer) form_card_extract_data;
  parent_class->display_data = (gpointer) form_card_display_data;
  parent_class->clean_form   = (gpointer) form_card_clean_form;
}


static void
rubrica_form_card_init(RubricaFormCard* self)
{
  self->priv = g_new(RubricaFormCardPrivate, 1);
  
  self->priv->groups     = NULL;
  self->priv->card       = NULL;
  self->priv->lock       = NULL;
  self->priv->rate       = NULL;

  self->priv->dispose_has_run = FALSE;
}

 

static void 
rubrica_form_card_dispose (RubricaFormCard* self)
{
  g_return_if_fail(IS_RUBRICA_FORM(self));
  
  if (self->priv->dispose_has_run)
    return;
  
  g_object_unref(self->priv->card);
  g_object_unref(self->priv->lock);
  g_object_unref(self->priv->rate);
  
  self->priv->dispose_has_run = TRUE;
}


static void 
rubrica_form_card_finalize (RubricaFormCard* self)
{
  g_return_if_fail(IS_RUBRICA_FORM(self));
  
  g_free(self->priv);
  self->priv = NULL;
}

static gboolean 
extract_groups (GtkTreeModel *model, GtkTreePath *path,
		GtkTreeIter *iter, gpointer data)
{
  RCard* card  = (RCard*) data;
  gchar* name  = NULL;
  gchar* label = NULL;
  gchar* owner = NULL;
  gchar* uri   = NULL;
  gboolean belong = FALSE;

  gtk_tree_model_get(model, iter, 
		     GROUP_BELONG_COLUMN, &belong,
		     GROUP_NAME_COLUMN,   &name,
		     GROUP_LABEL_COLUMN,  &label,
		     GROUP_OWNER_COLUMN,  &owner,
		     GROUP_PIXMAP_PATH,   &uri,
		     -1);
  if (belong) 
    {
      RGroup* group;

      group = r_group_new();
      g_object_set(group, 
		   "group-name",   name,
		   "group-label",  label,
		   "group-owner",  owner,
		   "group-pixmap", uri, 
		   "enabled",      TRUE, 
		   NULL);

      r_card_add_group(R_CARD(card), group);

      belong = FALSE;
    }
  
  return FALSE;  
}


static void 
form_card_extract_data (RubricaFormCard* form, RCard *card)
{
  GtkWidget  *tree;
  GtkTreeModel *model = NULL;
  const gchar *name;
  gboolean lock;
  gint rate;
  

  name = gtk_entry_get_text(GTK_ENTRY(form->priv->card));
  lock = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(form->priv->lock));
  rate = gtk_combo_box_get_active(GTK_COMBO_BOX(form->priv->rate));

  g_object_set(R_CARD(card), 
	       "card-locked", lock,  "card-deleted", FALSE,
	       "card-marked", FALSE, "card-name",    name, 
	       "card-rate",   rate,  NULL);  
  
  /* extract groups's data */
  tree  = rubrica_view_get_tree(RUBRICA_VIEW(form->priv->groups_view));
  model = gtk_tree_view_get_model(GTK_TREE_VIEW(tree));
  if (model)
    {
      gtk_tree_model_foreach(GTK_TREE_MODEL(model), extract_groups, card);
      g_object_unref(model);
    }  
}


static void 
form_card_display_data (RubricaFormCard* form, RCard *card)
{
  GtkWidget* tree;
  gpointer data;
  gchar* name;
  gboolean locked;
  RRate rate;
  
  tree  = rubrica_view_get_tree(RUBRICA_VIEW(form->priv->groups_view));
  
  g_object_get(R_CARD(card), 
	       "card-name",   &name, 
	       "card-locked", &locked, 
	       "card-rate",   &rate,
	       NULL);
  
  gtk_entry_set_text(GTK_ENTRY(form->priv->card), name);  
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(form->priv->lock), locked);
  gtk_combo_box_set_active(GTK_COMBO_BOX(form->priv->rate), rate); 
  
  r_group_box_disable_all (form->priv->groups);
  data = r_card_get_group(R_CARD(card));
  for (; data; data = r_card_get_next_group(R_CARD(card)))
    r_group_box_enable_group(form->priv->groups, data);

  rubrica_model_redisplay_groups_view(GTK_TREE_VIEW(tree), 
				      form->priv->groups);
}


static void 
form_card_clean_form (RubricaFormCard* form)
{
  GtkWidget* tree;
  
  tree  = rubrica_view_get_tree(RUBRICA_VIEW(form->priv->groups_view));

  gtk_entry_set_text(GTK_ENTRY(form->priv->card), "");  
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(form->priv->lock), FALSE);
  gtk_combo_box_set_active(GTK_COMBO_BOX(form->priv->rate), R_RATE_NORMAL);
  
  r_group_box_disable_all (form->priv->groups);
  rubrica_model_redisplay_groups_view(GTK_TREE_VIEW(tree), 
				      form->priv->groups);
}


/*   ***************************** Public *****************************
*/


/**
 * rubrica_form_card_new
 *
 * create a #RubricaFormCard
 *
 * Returns: a new #RubricaFormCard
 */
RubricaFormCard*
rubrica_form_card_new(RubricaApp* app, GladeXML *gui)
{
  RubricaFormCard *form;
  GtkWidget* tree;
  GtkWidget* rate_box;
  GtkWidget* group_box;
  GtkWidget* rate_label;
  GtkTreeModel* model;
  
  model = rubrica_model_combo_new();
  form  = g_object_new(rubrica_form_card_get_type(), NULL);
  
  form->priv->card        = glade_xml_get_widget(gui, "card");
  form->priv->lock        = glade_xml_get_widget(gui, "lock");
  form->priv->rate        = gtk_combo_box_new_with_model(model);  
  form->priv->groups      = rubrica_app_get_groups(app);
  form->priv->groups_view = rubrica_groups_view_new(form->priv->groups,
						    FORM_GROUPS);
  
  group_box  = glade_xml_get_widget(gui, "group_box");
  rate_box   = glade_xml_get_widget(gui, "rate_box");
  rate_label = glade_xml_get_widget(gui, "rate_label");

  gtk_box_pack_start(GTK_BOX(group_box), form->priv->groups_view, 
		     TRUE, TRUE, 6);
  gtk_box_pack_start(GTK_BOX(rate_box), form->priv->rate, TRUE, TRUE, 6);

  gtk_label_set_mnemonic_widget(GTK_LABEL(rate_label), form->priv->rate);

  rubrica_themes_create_themed_list(GTK_COMBO_BOX(form->priv->rate), faces);
  gtk_combo_box_set_active(GTK_COMBO_BOX(form->priv->rate), R_RATE_NORMAL);
  gtk_widget_show(form->priv->rate);
  
  tree = rubrica_view_get_tree(RUBRICA_VIEW(form->priv->groups_view));

  g_signal_connect(GTK_OBJECT (tree), "button_press_event",
  		   G_CALLBACK (on_form_group_view_event_occurred), 
		   form->priv->groups_view);
  
  gtk_widget_grab_focus (form->priv->card);
  gtk_widget_show(group_box);
  gtk_widget_show(form->priv->groups_view);

  return form;
}


/**
 * rubrica_form_card_free
 * @form: a #RubricaFormCard
 * 
 * free the #RubricaFormCard
 */
void
rubrica_form_card_free(RubricaFormCard* form)
{
  g_return_if_fail(IS_RUBRICA_FORM(form));

  g_object_unref(form);   
}
