/*
 *   file: form_contact.c
 *  
 *  Copyright (C) Nicola Fragale <nicolafragale@libero.it>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; version 3 of the License
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <gtk/gtk.h>
#include <glib/gi18n-lib.h>
#include <glade/glade.h>

#include "libral.h"

#include "app.h"
#include "form.h"
#include "form_contact.h"
#include "app.h"
#include "groups_cb.h"
#include "models.h"
#include "types.h"
#include "utils.h"
#include "themes.h"
#include "calendar.h"

enum {
  MALE   = 0,
  FEMALE
};

struct _RubricaFormContactPrivate {

  GtkWidget* fn;
  GtkWidget* sn;
  GtkWidget* ln;
  GtkWidget* nick;
  GtkWidget* prof;
  GtkWidget* title;
  GtkWidget* prefix;
  GtkWidget* genre;
  GtkWidget* birth;
  GtkWidget* birth_box;
  GtkWidget* photo_bt; 
  GtkWidget* photo;
  
  gchar* photo_uri;

  gboolean dispose_has_run;
};


static void rubrica_form_contact_class_init (RubricaFormContactClass* klass);
static void rubrica_form_contact_init       (RubricaFormContact* obj);

static void rubrica_form_contact_dispose    (RubricaFormContact* obj);
static void rubrica_form_contact_finalize   (RubricaFormContact* obj);



static void form_contact_extract_data (RubricaFormContact* form, RCard *card);
static void form_contact_display_data (RubricaFormContact* form, RCard *card);
static void form_contact_clean_form   (RubricaFormContact* form, RCard *card);


static RubricaFormClass* parent_class;


GType
rubrica_form_contact_get_type()
{
  static GType form_contact_type = 0;
  
  if (!form_contact_type)
    {
      static const GTypeInfo form_contact_info =
	{
	  sizeof(RubricaFormContactClass),
	  NULL,
	  NULL,
	  (GClassInitFunc) rubrica_form_contact_class_init,
	  NULL,
	  NULL,
	  sizeof(RubricaFormContact),
	  0,
	  (GInstanceInitFunc) rubrica_form_contact_init
	};

      form_contact_type = g_type_register_static (RUBRICA_FORM_TYPE, 
						  "RubricaFormContact",
						  &form_contact_info, 0);
    }
  
  return form_contact_type;
}


static void
rubrica_form_contact_class_init(RubricaFormContactClass* klass)
{
  GObjectClass *class;
 
  class  = G_OBJECT_CLASS (klass);
  class->dispose  = (GObjectFinalizeFunc) rubrica_form_contact_dispose;
  class->finalize = (GObjectFinalizeFunc) rubrica_form_contact_finalize;

  parent_class = RUBRICA_FORM_CLASS(klass);
  parent_class->extract_data = (gpointer) form_contact_extract_data;
  parent_class->display_data = (gpointer) form_contact_display_data;
  parent_class->clean_form   = (gpointer) form_contact_clean_form;
}


static void
rubrica_form_contact_init(RubricaFormContact* self)
{
  self->priv = g_new(RubricaFormContactPrivate, 1);

  self->priv->fn        = NULL;
  self->priv->sn        = NULL;
  self->priv->ln        = NULL;
  self->priv->nick      = NULL;
  self->priv->prof      = NULL;
  self->priv->title     = NULL;
  self->priv->prefix    = NULL;
  self->priv->genre     = NULL;
  self->priv->birth     = NULL;
  self->priv->photo_bt  = NULL;
  self->priv->photo     = NULL;
  self->priv->photo_uri = NULL;

  self->priv->dispose_has_run = FALSE;
}

 

static void 
rubrica_form_contact_dispose (RubricaFormContact* self)
{
  g_return_if_fail(IS_RUBRICA_FORM(self));
  
  if (self->priv->dispose_has_run)
    return;
  
  g_object_unref(self->priv->fn);
  g_object_unref(self->priv->sn);
  g_object_unref(self->priv->ln);
  g_object_unref(self->priv->nick);
  g_object_unref(self->priv->prof);
  g_object_unref(self->priv->prefix);
  g_object_unref(self->priv->title);
  g_object_unref(self->priv->genre);
  g_object_unref(self->priv->birth);
  g_object_unref(self->priv->photo);
  g_object_unref(self->priv->photo_bt);
  if (self->priv->photo_uri)
    g_free(self->priv->photo_uri);

  self->priv->dispose_has_run = TRUE;
}


static void 
rubrica_form_contact_finalize (RubricaFormContact* self)
{
  g_return_if_fail(IS_RUBRICA_FORM(self));
  
  g_free(self->priv);
  self->priv = NULL;
}



static void 
form_contact_extract_data (RubricaFormContact* form, RCard *card)
{
  RContact* contact;
  gint day = 0, month = 0, year = 0, genre;
  
  g_return_if_fail(IS_R_CARD(card));

  contact = r_contact_new();
  
  g_object_set(contact,
	       "first-name", gtk_entry_get_text(GTK_ENTRY(form->priv->fn)),
	       "middle-name",gtk_entry_get_text(GTK_ENTRY(form->priv->sn)),
	       "last-name",  gtk_entry_get_text(GTK_ENTRY(form->priv->ln)),
	       "nick-name",  gtk_entry_get_text(GTK_ENTRY(form->priv->nick)),
	       "profession", gtk_entry_get_text(GTK_ENTRY(form->priv->prof)),
	       "title",      gtk_entry_get_text(GTK_ENTRY(form->priv->title)),
	       "prefix",     gtk_entry_get_text(GTK_ENTRY(form->priv->prefix)),
	       "photo",      form->priv->photo_uri,
	       NULL);

  g_object_get(form->priv->birth, "calendar-day", &day, 
	       "calendar-month", &month, "calendar-year", &year, NULL);
  r_contact_set_birthday(contact, day, month, year);

  genre = gtk_combo_box_get_active (GTK_COMBO_BOX(form->priv->genre));
  genre == MALE ?  
    g_object_set(contact, "genre", "male", NULL) :
    g_object_set(contact, "genre", "female", NULL);
  
  r_personal_card_set_contact(R_PERSONAL_CARD(card), contact);
}


static void 
form_contact_display_data (RubricaFormContact* form, RCard *card)
{
  RContact* contact;
  RDate* birth;
  GtkImage  *image;
  GdkPixbuf *pixbuf;
 
  gchar *first, *middle, *last, *nick, *prof, *pref, *genre;
  gchar *title, *photo;
  gint day, month, year;

  contact = r_personal_card_get_contact (R_PERSONAL_CARD(card));
  birth   = r_contact_get_birthday(contact);

  g_object_get(contact, 
	       "first-name", &first, "middle-name", &middle, 
	       "last-name",  &last,  "nick-name",   &nick, 
	       "profession", &prof,  "prefix",      &pref, 
	       "genre",      &genre, "title",       &title, 
	       "photo",      &photo, NULL);

  g_object_get(birth, "day", &day, "month", &month, "year", &year, NULL);
  g_object_set(form->priv->birth, "calendar-day", day, 
	       "calendar-month", month, "calendar-year", year, NULL);

  if (first)  gtk_entry_set_text(GTK_ENTRY(form->priv->fn), first);
  if (middle) gtk_entry_set_text(GTK_ENTRY(form->priv->sn), middle);
  if (last)   gtk_entry_set_text(GTK_ENTRY(form->priv->ln), last);
  if (nick)   gtk_entry_set_text(GTK_ENTRY(form->priv->nick), nick);
  if (prof)   gtk_entry_set_text(GTK_ENTRY(form->priv->prof), prof);
  if (title)  gtk_entry_set_text(GTK_ENTRY(form->priv->title), title);
  if (pref)   gtk_entry_set_text(GTK_ENTRY(form->priv->prefix), pref);
  if (photo)  form->priv->photo_uri = g_strdup(photo);
  if (!genre || (g_ascii_strcasecmp(genre, "male") == 0))
    gtk_combo_box_set_active (GTK_COMBO_BOX(form->priv->genre), MALE);
  else
    gtk_combo_box_set_active (GTK_COMBO_BOX(form->priv->genre), FEMALE);
   
  if (photo)
    {
      image  = (GtkImage*) form->priv->photo;
      pixbuf = gdk_pixbuf_new_from_file_at_size (photo, 124, 124, NULL);
      gtk_image_set_from_pixbuf (GTK_IMAGE (image), pixbuf);
    }
}


static void 
form_contact_clean_form (RubricaFormContact* form, RCard *card)
{
  GtkIconTheme* theme;
  GtkImage  *image;
  GdkPixbuf *pixbuf;

  gtk_entry_set_text(GTK_ENTRY(form->priv->fn),     "");
  gtk_entry_set_text(GTK_ENTRY(form->priv->sn),     "");
  gtk_entry_set_text(GTK_ENTRY(form->priv->ln),     "");
  gtk_entry_set_text(GTK_ENTRY(form->priv->nick),   "");
  gtk_entry_set_text(GTK_ENTRY(form->priv->prof),   "");
  gtk_entry_set_text(GTK_ENTRY(form->priv->title),  "");
  gtk_entry_set_text(GTK_ENTRY(form->priv->prefix), "");
  
  rubrica_calendar_clean(RUBRICA_CALENDAR(form->priv->birth));  
  gtk_combo_box_set_active (GTK_COMBO_BOX(form->priv->genre), MALE);
  
  theme  = gtk_icon_theme_get_default();       
  image  = (GtkImage*) form->priv->photo;
  pixbuf = gtk_icon_theme_load_icon (theme, "photo", 48, 0, NULL);  
  gtk_image_set_from_pixbuf (GTK_IMAGE (image), pixbuf);
 
  g_free(form->priv->photo_uri);
  form->priv->photo_uri = NULL;
}



/*    Callbacks
*/
void
form_add_photo(GtkWidget* button, gpointer data)
{
  RubricaFormContact* form = (RubricaFormContact*) data;
  GtkImage* image;
  GdkPixbuf *pixbuf;
  GtkIconTheme* theme;
  gchar* uri = NULL;

  theme = gtk_icon_theme_get_default();       
  image = (GtkImage*) form->priv->photo;
  uri   = rubrica_themes_load_pixmap(button, image);
 
  if (uri)
    {
      pixbuf = gdk_pixbuf_new_from_file_at_size (uri, 124, 124, NULL);
      form->priv->photo_uri = g_strdup(uri);      
    }
  else
    if (form->priv->photo_uri)
      pixbuf = gdk_pixbuf_new_from_file_at_size (form->priv->photo_uri, 
						 124, 124, NULL);
    else
      {
	pixbuf = gtk_icon_theme_load_icon (theme, "photo", 48, 0, NULL);

	g_free(form->priv->photo_uri);
	form->priv->photo_uri = NULL;
      }
  
  gtk_image_set_from_pixbuf (GTK_IMAGE (image), pixbuf);  
}

/*   ***************************** Public *****************************
*/


/**
 * rubrica_form_contact_new
 *
 * create a #RubricaFormContact
 *
 * Returns: a new #RubricaFormContact
 */
RubricaFormContact*
rubrica_form_contact_new(RubricaApp* app, GladeXML *gui)
{
  RubricaFormContact* form;
  GtkWidget* birth_box;
  GtkWidget* genre_box;
  GtkWidget* genre_label;
  GtkWidget* birth_label;
  GtkWidget* button;

  form = g_object_new(rubrica_form_contact_get_type(), NULL);
  
  form->priv->fn       = glade_xml_get_widget(gui, "fn");
  form->priv->nick     = glade_xml_get_widget(gui, "nick");
  form->priv->sn       = glade_xml_get_widget(gui, "sn");
  form->priv->ln       = glade_xml_get_widget(gui, "ln");
  form->priv->prof     = glade_xml_get_widget(gui, "prof");
  form->priv->title    = glade_xml_get_widget(gui, "title");
  form->priv->prefix   = glade_xml_get_widget(gui, "prefix");
  form->priv->photo_bt = glade_xml_get_widget(gui, "photo_bt");
  form->priv->photo    = glade_xml_get_widget(gui, "img");
  
  genre_box   = glade_xml_get_widget(gui, "genre_box");
  genre_label = glade_xml_get_widget(gui, "genre_label");
  birth_label = glade_xml_get_widget(gui, "birth_label");
  birth_box   = glade_xml_get_widget(gui, "birthday_box");

  form->priv->birth = rubrica_calendar_new();
  button = rubrica_calendar_get_button(RUBRICA_CALENDAR(form->priv->birth));
  gtk_box_pack_start(GTK_BOX(birth_box), form->priv->birth, TRUE, TRUE, 0);
  gtk_label_set_mnemonic_widget(GTK_LABEL(birth_label), button);
  gtk_widget_show(form->priv->birth);
  
  form->priv->genre = gtk_combo_box_new_text();
  gtk_box_pack_start(GTK_BOX(genre_box), form->priv->genre, TRUE, TRUE, 0);
  gtk_label_set_mnemonic_widget(GTK_LABEL(genre_label), form->priv->genre);
  gtk_widget_show(form->priv->genre);
  
  gtk_combo_box_append_text(GTK_COMBO_BOX(form->priv->genre), _("male"));
  gtk_combo_box_append_text(GTK_COMBO_BOX(form->priv->genre), _("female"));
  gtk_combo_box_set_active (GTK_COMBO_BOX(form->priv->genre), MALE);
  
  g_signal_connect (G_OBJECT (form->priv->photo_bt), "clicked",
		    G_CALLBACK(form_add_photo), form);
  
  return form;
}


/**
 * rubrica_form_contact_free
 * @form: a #RubricaFormContact
 * 
 * free the #RubricaFormContact
 */
void
rubrica_form_contact_free(RubricaFormContact* form)
{
  g_return_if_fail(IS_RUBRICA_FORM(form));

  g_object_unref(form);   
}
