/*
 *  Program: Rubrica
 *  file: form_net.c
 *  
 *  Copyright (C) Nicola Fragale <nicolafragale@libero.it>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 3 of the License
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <gtk/gtk.h>
#include <glade/glade.h>

#include "libral.h"

#include "app.h"
#include "form.h"
#include "form_net.h"
#include "app.h"
#include "groups_cb.h"
#include "models.h"
#include "types.h"
#include "utils.h"


struct _RubricaFormNetPrivate {
  FormNetType type;

  GtkWidget* net;
  GtkWidget* tree;
  GtkWidget* add;
  GtkWidget* del;
  GtkWidget* mod;

  GtkWidget* web;
  GtkWidget* email;
  GtkWidget* vconf;
  GtkWidget* irc;
  GtkWidget* aim;
  GtkWidget* msn;
  GtkWidget* icq;
  GtkWidget* yahoo;
  GtkWidget* jabber;
  
  gboolean dispose_has_run;
};


static void rubrica_form_net_class_init (RubricaFormNetClass* klass);
static void rubrica_form_net_init       (RubricaFormNet* obj);

static void rubrica_form_net_dispose    (RubricaFormNet* obj);
static void rubrica_form_net_finalize   (RubricaFormNet* obj);

static gboolean extract_net       (GtkTreeModel *model,GtkTreePath *path,
				   GtkTreeIter *iter, gpointer data); 

static void form_net_extract_data (RubricaFormNet* form, RCard *card);
static void form_net_display_data (RubricaFormNet* form, RCard *card);
static void form_net_clean_form   (RubricaFormNet* form, RCard *card);


static RubricaFormClass* parent_class;


GType
rubrica_form_net_get_type()
{
  static GType form_net_type = 0;
  
  if (!form_net_type)
    {
      static const GTypeInfo form_net_info =
	{
	  sizeof(RubricaFormNetClass),
	  NULL,
	  NULL,
	  (GClassInitFunc) rubrica_form_net_class_init,
	  NULL,
	  NULL,
	  sizeof(RubricaFormNet),
	  0,
	  (GInstanceInitFunc) rubrica_form_net_init
	};

      form_net_type = g_type_register_static (RUBRICA_FORM_TYPE, 
					      "RubricaFormNet",
					      &form_net_info, 0);
    }
  
  return form_net_type;
}


static void
rubrica_form_net_class_init(RubricaFormNetClass* klass)
{
  GObjectClass *class;
 
  class  = G_OBJECT_CLASS (klass);
  class->dispose  = (GObjectFinalizeFunc) rubrica_form_net_dispose;
  class->finalize = (GObjectFinalizeFunc) rubrica_form_net_finalize;

  parent_class = RUBRICA_FORM_CLASS(klass);
  parent_class->extract_data = (gpointer) form_net_extract_data;
  parent_class->display_data = (gpointer) form_net_display_data;
  parent_class->clean_form   = (gpointer) form_net_clean_form;
}


static void
rubrica_form_net_init(RubricaFormNet* self)
{
  self->priv = g_new(RubricaFormNetPrivate, 1);
  
  self->priv->type   = FORM_NET_PERSONAL;
  self->priv->net    = NULL;
  self->priv->tree   = NULL;
  self->priv->add    = NULL;
  self->priv->mod    = NULL;
  self->priv->del    = NULL;
  self->priv->web    = NULL;
  self->priv->email  = NULL;
  self->priv->vconf  = NULL;
  self->priv->irc    = NULL;
  self->priv->aim    = NULL;
  self->priv->msn    = NULL;
  self->priv->icq    = NULL;
  self->priv->yahoo  = NULL;
  self->priv->jabber = NULL;

  self->priv->dispose_has_run = FALSE;
}

 

static void 
rubrica_form_net_dispose (RubricaFormNet* self)
{
  g_return_if_fail(IS_RUBRICA_FORM(self));
  
  if (self->priv->dispose_has_run)
    return;
  
  g_object_unref(self->priv->net);
  g_object_unref(self->priv->tree);
  g_object_unref(self->priv->add);
  g_object_unref(self->priv->del);
  g_object_unref(self->priv->mod);
  g_object_unref(self->priv->web);
  g_object_unref(self->priv->email);

  if (self->priv->type == FORM_NET_PERSONAL)
    {
      g_object_unref(self->priv->vconf);  
      g_object_unref(self->priv->irc);
      g_object_unref(self->priv->aim);
      g_object_unref(self->priv->msn);
      g_object_unref(self->priv->icq);
      g_object_unref(self->priv->yahoo);
      g_object_unref(self->priv->jabber);
    }

  self->priv->dispose_has_run = TRUE;
}


static void 
rubrica_form_net_finalize (RubricaFormNet* self)
{
  g_return_if_fail(IS_RUBRICA_FORM(self));
  
  g_free(self->priv);
  self->priv = NULL;
}


static gboolean 
extract_net (GtkTreeModel *model, GtkTreePath *path, GtkTreeIter *iter, 
	     gpointer data)
{
  RCard*       card = (RCard*) data;
  RNetAddress* net  = NULL;
  RNetAddressType nettype;
  gchar *url, *type;

  gtk_tree_model_get(model, iter,
		     URI_COLUMN,      &url, 
		     URI_TYPE_COLUMN, &type,
		     -1);
  nettype = r_net_address_encode_type(type);

  net = r_net_address_new();
  g_object_set(net, "url", url, "url-type", nettype, NULL);
  r_card_add_net_address(R_CARD(card), net);
  
  return FALSE;
}


static void 
form_net_extract_data (RubricaFormNet* form, RCard *card)
{
  GtkTreeModel* model = NULL;

  g_return_if_fail(IS_R_CARD(card));

  /* extract net's data */
  model = gtk_tree_view_get_model(GTK_TREE_VIEW(form->priv->tree));
  if (model)
    {
      gtk_tree_model_foreach(GTK_TREE_MODEL(model), extract_net, card);
      g_object_unref(model);
    }    
}


static void 
form_net_display_data (RubricaFormNet* form, RCard *card)
{
  GtkTreeView* tree;
  GtkTreeModel *model;
  GtkTreeIter iter;
  gpointer data;

  tree  = (GtkTreeView*) form->priv->tree;
  model = gtk_tree_view_get_model(tree);

  data = r_card_get_net_address(R_CARD(card));
  for (; data; data = r_card_get_next_net_address(R_CARD(card)))
    {
      RNetAddressType nettype;
      gchar *url, *type;
      
      g_object_get(R_NET_ADDRESS(data), "url", &url, 
		   "url-type", &nettype, NULL);
      type = r_net_address_decode_type(nettype);
      
      gtk_list_store_append(GTK_LIST_STORE(model), &iter);
      gtk_list_store_set(GTK_LIST_STORE(model), &iter, 
			 URI_COLUMN,      url, 
			 URI_TYPE_COLUMN, type,
			 -1);     
    } 
}


static void 
form_net_clean_form (RubricaFormNet* form, RCard *card)
{
  GtkTreeModel *model;

  gtk_entry_set_text(GTK_ENTRY(form->priv->net), "");
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(form->priv->web), TRUE);

  model = gtk_tree_view_get_model(GTK_TREE_VIEW(form->priv->tree));
  gtk_list_store_clear(GTK_LIST_STORE(model));  
}



/*   Callbacks
*/
void 
form_net_list_activated(GtkTreeSelection *selection, gpointer data)
{
  GtkTreeModel *model;
  GtkTreeIter iter;
  RubricaFormNet *form = (RubricaFormNet *) data;
  gchar *str, *type_str;
  RNetAddressType type;

  model = gtk_tree_view_get_model(GTK_TREE_VIEW(form->priv->tree));
  
  if (gtk_tree_selection_get_selected(selection, &model, &iter))
    {
      gtk_tree_model_get(model, &iter, 
			 URI_COLUMN, &str,
			 URI_TYPE_COLUMN, &type_str,
			 -1);
      
      gtk_entry_set_text(GTK_ENTRY(form->priv->net), str);
      type = r_net_address_encode_type(type_str);

      switch(type)
	{
	case R_NET_ADDRESS_WEB:
	  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(form->priv->web), 
				       TRUE);
	  break;

	case R_NET_ADDRESS_EMAIL:
	  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(form->priv->email),
				       TRUE);
	  break;

	case R_NET_ADDRESS_EKIGA:
	  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(form->priv->vconf),
				       TRUE);
	  break;

	case R_NET_ADDRESS_IRC:
	  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(form->priv->irc), 
				       TRUE);
	  break;

	case R_NET_ADDRESS_IRC_AIM:
	  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(form->priv->aim), 
				       TRUE);
	  break;

	case R_NET_ADDRESS_IRC_MSN:
	  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(form->priv->msn),
				       TRUE);
	  break;

	case R_NET_ADDRESS_IRC_ICQ:
	  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(form->priv->icq), 
				       TRUE);
	  break;

	case R_NET_ADDRESS_IRC_YAHOO:
	  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(form->priv->yahoo), 
				       TRUE);
	  break;

	case R_NET_ADDRESS_IRC_JABBER:
	  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(form->priv->jabber),
				       TRUE);
	  break;

	default:
	  break;
	}

      gtk_widget_set_sensitive(form->priv->mod, TRUE);
      gtk_widget_set_sensitive(form->priv->del, TRUE);
      g_free(str);
      g_free(type_str);
    }
}


void
form_net_add_clicked (GtkButton *button, gpointer data)
{
  GtkTreeModel *model;
  GtkTreeIter iter;
  RNetAddress* net;
  RNetAddressType type;   
  RubricaFormNet *form = (RubricaFormNet *) data;  
  gchar *url, *str;

  gtk_widget_set_sensitive(GTK_WIDGET(form->priv->mod), FALSE);
  gtk_widget_set_sensitive(GTK_WIDGET(form->priv->del), FALSE);

  model = gtk_tree_view_get_model(GTK_TREE_VIEW(form->priv->tree));

  net = r_net_address_new();
  url = g_strdup(gtk_entry_get_text(GTK_ENTRY(form->priv->net)));
  
  if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(form->priv->web)))
    type = R_NET_ADDRESS_WEB;
  if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(form->priv->email)))
    type = R_NET_ADDRESS_EMAIL;
  if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(form->priv->vconf)))
    type = R_NET_ADDRESS_EKIGA;
  if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(form->priv->irc)))
    type = R_NET_ADDRESS_IRC;
  if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(form->priv->aim)))
    type = R_NET_ADDRESS_IRC_AIM;
  if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(form->priv->msn)))
    type = R_NET_ADDRESS_IRC_MSN;
  if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(form->priv->icq)))
    type = R_NET_ADDRESS_IRC_ICQ;
  if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(form->priv->yahoo)))
    type = R_NET_ADDRESS_IRC_YAHOO;
  if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(form->priv->jabber)))
    type = R_NET_ADDRESS_IRC_JABBER;

  str = r_net_address_decode_type(type);
  g_object_set(net, "url", url, "url-type", type, NULL);
  
  gtk_list_store_append(GTK_LIST_STORE(model), &iter);
  gtk_list_store_set(GTK_LIST_STORE(model), &iter, 
		     URI_COLUMN,      url, 
		     URI_TYPE_COLUMN, str,
		     -1);

  gtk_entry_set_text(GTK_ENTRY(form->priv->net), "");
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(form->priv->web), TRUE);
  gtk_widget_set_sensitive(GTK_WIDGET(form->priv->mod), FALSE);
  gtk_widget_set_sensitive(GTK_WIDGET(form->priv->del), FALSE);  
}


void
form_net_mod_clicked (GtkButton *button, gpointer data)
{
  GtkTreeModel *model;
  GtkTreeIter iter;
  GtkTreeSelection *selection;
  RNetAddress* net;
  RubricaFormNet* form = (RubricaFormNet *) data;
  RNetAddressType type;   
  gchar* url;

  model     = gtk_tree_view_get_model(GTK_TREE_VIEW(form->priv->tree));
  selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(form->priv->tree));

  net = r_net_address_new();
  url = g_strdup(gtk_entry_get_text(GTK_ENTRY(form->priv->net)));

  if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(form->priv->web)))
    type = R_NET_ADDRESS_WEB;
  if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(form->priv->email)))
    type = R_NET_ADDRESS_EMAIL;
  if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(form->priv->vconf)))
    type = R_NET_ADDRESS_EKIGA;
  if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(form->priv->irc)))
    type = R_NET_ADDRESS_IRC;
  if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(form->priv->aim)))
    type = R_NET_ADDRESS_IRC_AIM;
  if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(form->priv->msn)))
    type = R_NET_ADDRESS_IRC_MSN;
  if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(form->priv->icq)))
    type = R_NET_ADDRESS_IRC_ICQ;
  if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(form->priv->yahoo)))
    type = R_NET_ADDRESS_IRC_YAHOO;
  if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(form->priv->jabber)))
    type = R_NET_ADDRESS_IRC_JABBER;

  g_object_set(net, "url", url, "url-type", type, NULL);

  if (gtk_tree_selection_get_selected(selection, NULL, &iter))
  {
    gint i;
    gchar *url_type;
    GtkTreePath *path;
    
    url_type = r_net_address_decode_type(type);
    
    path = gtk_tree_model_get_path(model, &iter);
    i    = gtk_tree_path_get_indices(path)[0];
    
    gtk_list_store_remove(GTK_LIST_STORE(model), &iter);
    gtk_tree_path_free(path);
    
    gtk_list_store_insert(GTK_LIST_STORE(model), &iter, i);
    gtk_list_store_set(GTK_LIST_STORE(model), &iter, 
		       URI_COLUMN,      url, 
		       URI_TYPE_COLUMN, url_type,
		       -1);

    gtk_entry_set_text(GTK_ENTRY(form->priv->net), "");
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(form->priv->web), TRUE);

    gtk_widget_set_sensitive(GTK_WIDGET(form->priv->mod), FALSE);
    gtk_widget_set_sensitive(GTK_WIDGET(form->priv->del), FALSE);   
  }  
}


void
form_net_del_clicked (GtkButton *button, gpointer data)
{
  GtkTreeModel *model;
  GtkTreeIter iter;
  GtkTreeSelection *selection;
  RubricaFormNet* form = (RubricaFormNet *) data;
  
  model     = gtk_tree_view_get_model(GTK_TREE_VIEW(form->priv->tree));
  selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(form->priv->tree));
  
  if (gtk_tree_selection_get_selected(selection, NULL, &iter))
  {
    gtk_list_store_remove(GTK_LIST_STORE(model), &iter);
   
    gtk_entry_set_text(GTK_ENTRY(form->priv->net), "");

    gtk_widget_set_sensitive(GTK_WIDGET(form->priv->mod), FALSE);
    gtk_widget_set_sensitive(GTK_WIDGET(form->priv->del), FALSE);   
  }  
}





/*   ***************************** Public *****************************
*/


/**
 * rubrica_form_net_new
 *
 * create a #RubricaFormNet
 *
 * Returns: a new #RubricaFormNet
 */
RubricaFormNet*
rubrica_form_net_new(RubricaApp* app, GladeXML *gui, FormNetType type)
{
  RubricaFormNet   *form;
  GtkTreeView      *tree;
  GtkTreeModel     *model;
  GtkTreeSelection *selection;

  form = g_object_new(rubrica_form_net_get_type(), NULL);

  form->priv->type   = type;
  form->priv->net    = glade_xml_get_widget(gui, "net");
  form->priv->tree   = glade_xml_get_widget(gui, "net_tree"); 
  form->priv->add    = glade_xml_get_widget(gui, "net_add");
  form->priv->del    = glade_xml_get_widget(gui, "net_del");
  form->priv->mod    = glade_xml_get_widget(gui, "net_mod");

  form->priv->web    = glade_xml_get_widget(gui, "web");
  form->priv->email  = glade_xml_get_widget(gui, "email");
  
  if (type == FORM_NET_PERSONAL)
    {
      form->priv->vconf  = glade_xml_get_widget(gui, "vconf");
      form->priv->irc    = glade_xml_get_widget(gui, "irc");
      form->priv->aim    = glade_xml_get_widget(gui, "aim");
      form->priv->msn    = glade_xml_get_widget(gui, "msn");
      form->priv->icq    = glade_xml_get_widget(gui, "icq");
      form->priv->yahoo  = glade_xml_get_widget(gui, "yahoo");
      form->priv->jabber = glade_xml_get_widget(gui, "jabber");
    }
  
  gtk_widget_set_sensitive(GTK_WIDGET(form->priv->mod), FALSE);
  gtk_widget_set_sensitive(GTK_WIDGET(form->priv->del), FALSE);

  /*   net urls model/view 
   */
  model = rubrica_model_new(NET_MODEL);
  tree  = GTK_TREE_VIEW(form->priv->tree);

  gtk_tree_view_set_model(tree, GTK_TREE_MODEL(model));
  selection = gtk_tree_view_get_selection(tree);
  gtk_tree_selection_set_mode(selection, GTK_SELECTION_SINGLE);

  rubrica_model_add_columns(tree, URI_COLUMNS);

  g_signal_connect(G_OBJECT(selection), "changed", 
		   G_CALLBACK(form_net_list_activated), form);
  g_object_unref(model);

  /*    net buttons 
  */
  g_signal_connect (G_OBJECT (form->priv->add), "clicked", 
		    G_CALLBACK (form_net_add_clicked), form); 
  g_signal_connect (G_OBJECT (form->priv->mod), "clicked", 
		    G_CALLBACK(form_net_mod_clicked), form); 
  g_signal_connect (G_OBJECT (form->priv->del), "clicked", 
		    G_CALLBACK (form_net_del_clicked), form); 

  gtk_widget_grab_focus (form->priv->net);
  return form;
}


/**
 * rubrica_form_net_free
 * @form: a #RubricaFormNet
 * 
 * free the #RubricaFormNet
 */
void
rubrica_form_net_free(RubricaFormNet* form)
{
  g_return_if_fail(IS_RUBRICA_FORM(form));

  g_object_unref(form);   
}
