/*
 *  Program: Rubrica
 *  file: form_notes.c
 *  
 *  Copyright (C) Nicola Fragale <nicolafragale@libero.it>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 3 of the License
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <gtk/gtk.h>
#include <glade/glade.h>

#include "libral.h"

#include "app.h"
#include "form.h"
#include "form_notes.h"
#include "app.h"
#include "groups_cb.h"
#include "models.h"
#include "types.h"
#include "utils.h"
#include "calendar.h"

struct _RubricaFormNotesPrivate {
  GtkWidget* has_partner;      /* */
  GtkWidget* table;   
  GtkWidget* partner;
  GtkWidget* birthday;
  GtkWidget* anniversary;
  GtkWidget* notes;
  GtkWidget* pubkey;
  
  gboolean dispose_has_run;
};


static void rubrica_form_notes_class_init (RubricaFormNotesClass* klass);
static void rubrica_form_notes_init       (RubricaFormNotes* obj);

static void rubrica_form_notes_dispose    (RubricaFormNotes* obj);
static void rubrica_form_notes_finalize   (RubricaFormNotes* obj);


static gchar* extract_text            (GtkWidget *text);

static void   form_notes_extract_data (RubricaFormNotes* form, 
				       RCard *card);
static void   form_notes_display_data (RubricaFormNotes* form,
				       RCard *card);
static void   form_notes_clean_form   (RubricaFormNotes* form);


static RubricaFormClass* parent_class;


GType
rubrica_form_notes_get_type()
{
  static GType form_notes_type = 0;
  
  if (!form_notes_type)
    {
      static const GTypeInfo form_notes_info =
	{
	  sizeof(RubricaFormNotesClass),
	  NULL,
	  NULL,
	  (GClassInitFunc) rubrica_form_notes_class_init,
	  NULL,
	  NULL,
	  sizeof(RubricaFormNotes),
	  0,
	  (GInstanceInitFunc) rubrica_form_notes_init
	};

      form_notes_type = g_type_register_static (RUBRICA_FORM_TYPE, 
					       "RubricaFormNotes",
					       &form_notes_info, 0);
    }
  
  return form_notes_type;
}


static void
rubrica_form_notes_class_init(RubricaFormNotesClass* klass)
{
  GObjectClass *class;
 
  class  = G_OBJECT_CLASS (klass);
  class->dispose  = (GObjectFinalizeFunc) rubrica_form_notes_dispose;
  class->finalize = (GObjectFinalizeFunc) rubrica_form_notes_finalize;

  parent_class = RUBRICA_FORM_CLASS(klass);
  parent_class->extract_data = (gpointer) form_notes_extract_data;
  parent_class->display_data = (gpointer) form_notes_display_data;
  parent_class->clean_form   = (gpointer) form_notes_clean_form;
}


static void
rubrica_form_notes_init(RubricaFormNotes* self)
{
  self->priv = g_new(RubricaFormNotesPrivate, 1);

  self->priv->has_partner = NULL;
  self->priv->table       = NULL;   // just a container
  self->priv->partner     = NULL;
  self->priv->birthday    = NULL;
  self->priv->anniversary = NULL;
  self->priv->notes       = NULL;
  self->priv->pubkey      = NULL;

  self->priv->dispose_has_run = FALSE;
}

 

static void 
rubrica_form_notes_dispose (RubricaFormNotes* self)
{
  g_return_if_fail(IS_RUBRICA_FORM(self));
  
  if (self->priv->dispose_has_run)
    return;

  g_object_unref(self->priv->has_partner);
  g_object_unref(self->priv->table);
  g_object_unref(self->priv->partner);
  g_object_unref(self->priv->birthday);
  g_object_unref(self->priv->anniversary);
  g_object_unref(self->priv->notes);
  g_object_unref(self->priv->pubkey);
  
  self->priv->dispose_has_run = TRUE;
}


static void 
rubrica_form_notes_finalize (RubricaFormNotes* self)
{
  g_return_if_fail(IS_RUBRICA_FORM(self));
  
  g_free(self->priv);
  self->priv = NULL;
}


/** 
 * extract_text
 * @text: the text widget pointer 
 *  
 * extract text stored into the text widget.  
 * 
 * return: an allocated UTF-8 string  
 */ 
gchar*
extract_text (GtkWidget *text) 
{
  GtkTextIter start, end;
  GtkTextBuffer *text_buffer;
  gchar *ret_val = NULL;         /* an allocated UTF-8 string */
  
  text_buffer = gtk_text_view_get_buffer(GTK_TEXT_VIEW (text));
  
  gtk_text_buffer_get_bounds(text_buffer, &start, &end);
  ret_val = gtk_text_buffer_get_text(text_buffer, &start, &end, FALSE);

  return ret_val;
}



static void 
form_notes_extract_data (RubricaFormNotes* form, RCard *card)
{
  RNotes* notes;
  gboolean has_partner = FALSE;
  const gchar *partner;
  gchar *pubkey = NULL;
  gchar *other  = NULL;
  gboolean write_notes = FALSE;

  g_return_if_fail(IS_R_CARD(card));

  notes = r_notes_new();
  
  other = extract_text(form->priv->notes);
  if (other && (g_ascii_strcasecmp(other, "") != 0))
    {
      g_object_set(notes, "other-notes",  other, NULL);  
      write_notes = TRUE;
    } 

  pubkey = extract_text(form->priv->pubkey);
  if (pubkey && (g_ascii_strcasecmp(pubkey, "") != 0))
    {
      g_object_set(notes, "pubkey", pubkey, NULL);  
      write_notes = TRUE;
    }
  
  has_partner = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON
					     (form->priv->has_partner));
  if (has_partner)
    {
      partner = gtk_entry_get_text(GTK_ENTRY(form->priv->partner));
      
      if (partner && (g_ascii_strcasecmp(partner, "") != 0))
	g_object_set(notes, "has-partner", TRUE, "partner-name", partner,
		     NULL);  

      if (rubrica_calendar_has_date(RUBRICA_CALENDAR(form->priv->birthday)))
	{
	  gint day, month, year;

	  g_object_get(form->priv->birthday, 
		       "calendar-day", &day, 
		       "calendar-month", &month, 
		       "calendar-year", &year, NULL);
	  
	  r_notes_set_know_birthday(notes, TRUE); 
	  r_notes_set_birthday(notes, day, month, year);
	}
      
      if (rubrica_calendar_has_date(RUBRICA_CALENDAR(form->priv->anniversary)))
	{
	  gint day, month, year;
	  
	  g_object_get(form->priv->anniversary, 
		       "calendar-day", &day, 
		       "calendar-month", &month, 
		       "calendar-year", &year, NULL);

	  r_notes_set_know_anniversary(notes, TRUE);
	  r_notes_set_anniversary(notes, day, month, year);
	}

      write_notes = TRUE;    
    }
  
  if (write_notes)
    r_personal_card_set_notes(R_PERSONAL_CARD(card), notes);
  else
    r_notes_free(notes);
}


static void 
form_notes_display_data (RubricaFormNotes* form, RCard *card)
{
  RNotes* notes = NULL;
  GtkTextView* view;
  GtkTextIter start, end;
  GtkTextBuffer *buffer;
  gchar *pubkey, *other, *partner;
  gboolean has_partner;

  notes = r_personal_card_get_notes (R_PERSONAL_CARD(card));

  if (!notes)
    return;

  g_object_get(notes, "pubkey", &pubkey, "other-notes", &other, 
	       "has-partner", &has_partner, "partner-name", &partner, NULL);
  
  /*    Public Key
  */
  if (pubkey)
    {
      view   = GTK_TEXT_VIEW(form->priv->pubkey);
      buffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW (view));
      gtk_text_buffer_get_bounds(buffer, &start, &end);
      gtk_text_buffer_delete(buffer, &start, &end);
      gtk_text_buffer_get_iter_at_offset(buffer, &start, 0);  
      gtk_text_buffer_insert (buffer, &start, pubkey, -1);
    }
  
  /*    Other notes
  */
  if (other)
    {
      view   = GTK_TEXT_VIEW(form->priv->notes);
      buffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW (view));
      gtk_text_buffer_get_bounds(buffer, &start, &end);
      gtk_text_buffer_delete(buffer, &start, &end);
      gtk_text_buffer_get_iter_at_offset(buffer, &start, 0);  
      gtk_text_buffer_insert (buffer, &start, other, -1);
    }

  if (has_partner)
    {
      RDate* rdate;
      gint day, month, year;
      
      gtk_widget_set_sensitive(GTK_WIDGET(form->priv->table), TRUE); 
      gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(form->priv->has_partner),
				   TRUE);      
      gtk_entry_set_text(GTK_ENTRY(form->priv->partner), partner);
      
      if (r_notes_know_birthday(notes))
	{
	  rdate = r_notes_get_birthday(notes);
	  
	  g_object_get(rdate, "day", &day, "month", &month, 
		       "year", &year, NULL);
	  
	  g_object_set(form->priv->birthday, "calendar-day", day, 
		       "calendar-month", month, "calendar-year", year, NULL);
	}
      
      if (r_notes_know_anniversary(notes))
	{
	  
	  rdate = r_notes_get_anniversary(notes);
	  
	  g_object_get(rdate, "day", &day, "month", &month, 
		       "year", &year, NULL);
	  
	  g_object_set(form->priv->anniversary, "calendar-day", day, 
		       "calendar-month", month, "calendar-year", year, NULL);
	}        
    }
}


static void 
form_notes_clean_form (RubricaFormNotes* form)
{
  GtkTextView* view;
  GtkTextBuffer *buffer;
  GtkTextIter start, end;

  gtk_entry_set_text(GTK_ENTRY(form->priv->partner), "");
  rubrica_calendar_clean(RUBRICA_CALENDAR(form->priv->birthday));
  rubrica_calendar_clean(RUBRICA_CALENDAR(form->priv->anniversary));
  gtk_widget_set_sensitive(GTK_WIDGET(form->priv->table), FALSE); 
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(form->priv->has_partner),
			       FALSE);

  view   = GTK_TEXT_VIEW(form->priv->pubkey);
  buffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW (view));
  gtk_text_buffer_get_bounds(buffer, &start, &end);
  gtk_text_buffer_delete(buffer, &start, &end);

  view   = GTK_TEXT_VIEW(form->priv->notes);
  buffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW (view));
  gtk_text_buffer_get_bounds(buffer, &start, &end);
  gtk_text_buffer_delete(buffer, &start, &end);
}





/*    Callbacks
*/

void 
rubrica_form_married_check_toggled(GtkToggleButton *togglebutton, 
				   gpointer data)
{
  GtkWidget* table = (GtkWidget*) data;
  gboolean married;
  
  married = RUBRICA_GET_CHECK(togglebutton);
  gtk_widget_set_sensitive(GTK_WIDGET(table), married);
}



/*   ***************************** Public *****************************
*/


/**
 * rubrica_form_notes_new
 *
 * create a #RubricaFormNotes
 *
 * Returns: a new #RubricaFormNotes
 */
RubricaFormNotes*
rubrica_form_notes_new(RubricaApp* app, GladeXML *gui)
{
  RubricaFormNotes* form;
  GtkWidget* birth_box;
  GtkWidget* ann_box;
  GtkWidget* note_birth;
  GtkWidget* note_ann;
  GtkWidget* bt;
  
  form = g_object_new(rubrica_form_notes_get_type(), NULL);
  
  form->priv->has_partner = glade_xml_get_widget(gui, "spouse_check");
  form->priv->table       = glade_xml_get_widget(gui, "marriage_table");
  form->priv->partner     = glade_xml_get_widget(gui, "spouse");
  form->priv->notes       = glade_xml_get_widget(gui, "notes");
  form->priv->pubkey      = glade_xml_get_widget(gui, "pubkey");

  note_birth = glade_xml_get_widget(gui, "note_birth");
  note_ann   = glade_xml_get_widget(gui, "note_ann");
  birth_box  = glade_xml_get_widget(gui, "birth_box");
  ann_box    = glade_xml_get_widget(gui, "ann_box");

  form->priv->birthday    = rubrica_calendar_new();
  form->priv->anniversary = rubrica_calendar_new();

  gtk_box_pack_start(GTK_BOX(birth_box), form->priv->birthday, TRUE, TRUE, 0);
  gtk_box_pack_start(GTK_BOX(ann_box), form->priv->anniversary, TRUE, TRUE, 0);
  gtk_widget_show(form->priv->birthday);
  gtk_widget_show(form->priv->anniversary);

  bt = rubrica_calendar_get_button(RUBRICA_CALENDAR(form->priv->birthday));
  gtk_label_set_mnemonic_widget(GTK_LABEL(note_birth), bt);

  bt = rubrica_calendar_get_button(RUBRICA_CALENDAR(form->priv->anniversary));
  gtk_label_set_mnemonic_widget(GTK_LABEL(note_ann), bt);

  gtk_widget_set_sensitive(GTK_WIDGET(form->priv->table), FALSE);

  /*     notes checkbutton 
   */
  g_signal_connect(G_OBJECT(form->priv->has_partner), "toggled", 
		   G_CALLBACK(rubrica_form_married_check_toggled), 
		   form->priv->table);

  gtk_widget_grab_focus (form->priv->has_partner);
  return form;
}


/**
 * rubrica_form_notes_free
 * @form: a #RubricaFormNotes
 * 
 * free the #RubricaFormNotes
 */
void
rubrica_form_notes_free(RubricaFormNotes* form)
{
  g_return_if_fail(IS_RUBRICA_FORM(form));

  g_object_unref(form);   
}
