/*
 *  program: Rubrica
 *  file: form_work.c
 *  
 *  Copyright (C) Nicola Fragale <nicolafragale@libero.it>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 3 of the License
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <gtk/gtk.h>
#include <glade/glade.h>

#include "libral.h"

#include "app.h"
#include "form.h"
#include "form_work.h"
#include "app.h"
#include "groups_cb.h"
#include "models.h"
#include "types.h"
#include "utils.h"


struct _RubricaFormWorkPrivate {
  GtkWidget* role;
  GtkWidget* org;
  GtkWidget* dep;
  GtkWidget* subdep;
  GtkWidget* manager;
  GtkWidget* mphone;
  GtkWidget* helper;
  GtkWidget* cphone;
  
  gboolean dispose_has_run;
};


static void rubrica_form_work_class_init (RubricaFormWorkClass* klass);
static void rubrica_form_work_init       (RubricaFormWork* obj);

static void rubrica_form_work_dispose    (RubricaFormWork* obj);
static void rubrica_form_work_finalize   (RubricaFormWork* obj);


static void form_work_extract_data (RubricaFormWork* form, RCard *card);
static void form_work_display_data (RubricaFormWork* form, RCard *card);
static void form_work_clean_form   (RubricaFormWork* form);


static RubricaFormClass* parent_class;


GType
rubrica_form_work_get_type()
{
  static GType form_work_type = 0;
  
  if (!form_work_type)
    {
      static const GTypeInfo form_work_info =
	{
	  sizeof(RubricaFormWorkClass),
	  NULL,
	  NULL,
	  (GClassInitFunc) rubrica_form_work_class_init,
	  NULL,
	  NULL,
	  sizeof(RubricaFormWork),
	  0,
	  (GInstanceInitFunc) rubrica_form_work_init
	};

      form_work_type = g_type_register_static (RUBRICA_FORM_TYPE, 
					       "RubricaFormWork",
					       &form_work_info, 0);
    }
  
  return form_work_type;
}


static void
rubrica_form_work_class_init(RubricaFormWorkClass* klass)
{
  GObjectClass *class;
 
  class  = G_OBJECT_CLASS (klass);
  class->dispose  = (GObjectFinalizeFunc) rubrica_form_work_dispose;
  class->finalize = (GObjectFinalizeFunc) rubrica_form_work_finalize;

  parent_class = RUBRICA_FORM_CLASS(klass);
  parent_class->extract_data = (gpointer) form_work_extract_data;
  parent_class->display_data = (gpointer) form_work_display_data;
  parent_class->clean_form   = (gpointer) form_work_clean_form;
}


static void
rubrica_form_work_init(RubricaFormWork* self)
{
  self->priv = g_new(RubricaFormWorkPrivate, 1);  

  self->priv->role    = NULL;
  self->priv->org     = NULL;
  self->priv->dep     = NULL;
  self->priv->subdep  = NULL;
  self->priv->manager = NULL;
  self->priv->mphone  = NULL;
  self->priv->helper  = NULL;
  self->priv->cphone  = NULL;

  self->priv->dispose_has_run = FALSE;
}

 

static void 
rubrica_form_work_dispose (RubricaFormWork* self)
{
  g_return_if_fail(IS_RUBRICA_FORM_WORK(self));
  
  if (self->priv->dispose_has_run)
    return;
  
  g_object_unref(self->priv->role);
  g_object_unref(self->priv->org);
  g_object_unref(self->priv->dep);
  g_object_unref(self->priv->subdep);
  g_object_unref(self->priv->manager);
  g_object_unref(self->priv->mphone);
  g_object_unref(self->priv->helper);
  g_object_unref(self->priv->cphone);
  
  self->priv->dispose_has_run = TRUE;
}


static void 
rubrica_form_work_finalize (RubricaFormWork* self)
{
  g_return_if_fail(IS_RUBRICA_FORM(self));
  
  g_free(self->priv);
  self->priv = NULL;
}



static void 
form_work_extract_data (RubricaFormWork* form, RCard *card)
{
  RWork* work;  
  const gchar *role   = NULL, *org     = NULL, *dep    = NULL;
  const gchar *subdep = NULL, *manager = NULL, *mphone = NULL;
  const gchar *helper = NULL, *cphone  = NULL;
  gboolean write_card = FALSE;

  g_return_if_fail(IS_R_CARD(card));
    
  role    = gtk_entry_get_text(GTK_ENTRY(form->priv->role));
  org     = gtk_entry_get_text(GTK_ENTRY(form->priv->org));
  dep     = gtk_entry_get_text(GTK_ENTRY(form->priv->dep));
  subdep  = gtk_entry_get_text(GTK_ENTRY(form->priv->subdep));
  manager = gtk_entry_get_text(GTK_ENTRY(form->priv->manager));
  mphone  = gtk_entry_get_text(GTK_ENTRY(form->priv->mphone));
  helper  = gtk_entry_get_text(GTK_ENTRY(form->priv->helper));
  cphone  = gtk_entry_get_text(GTK_ENTRY(form->priv->cphone));

  work = r_work_new();      

  if (role && (g_ascii_strcasecmp(role, "") != 0))
    {
      g_object_set(work, "assignment", role, NULL);
      write_card = TRUE;
    }
  
  if (org && (g_ascii_strcasecmp(org, "") != 0))
    {
      g_object_set(work, "organization", org, NULL);      

      write_card = TRUE;
    }

  if (dep && (g_ascii_strcasecmp(dep, "") != 0))
    {
      g_object_set(work, "department", dep, NULL);       

      write_card = TRUE;
    }
  
  if (subdep && (g_ascii_strcasecmp(subdep, "") != 0))
    {
      g_object_set(work, "sub-department", subdep, NULL);      
      
      write_card = TRUE;
    }

  if (manager && (g_ascii_strcasecmp(manager, "") != 0))
    {
      g_object_set(work, "manager-name", manager, NULL);      
      
      write_card = TRUE;
    }

  if (mphone && (g_ascii_strcasecmp(mphone, "") != 0))
    {
      g_object_set(work, "manager-phone", mphone, NULL);      
      
      write_card = TRUE;
    }

  if (helper && (g_ascii_strcasecmp(helper, "") != 0))
    {
      g_object_set(work, "collaborator", helper, NULL);      
      
      write_card = TRUE;
    }

  if (cphone && (g_ascii_strcasecmp(cphone, "") != 0))
    {
      g_object_set(work, "collaborator-phone", cphone, NULL);      

      write_card = TRUE;
    }

  if (write_card)
    r_personal_card_set_work(R_PERSONAL_CARD(card), work);    
  else
    r_work_free(work);
}


static void 
form_work_display_data (RubricaFormWork* form, RCard *card)
{
  RWork* work = NULL;
  gchar *role, *org, *dep, *subdep, *manager, *mphone, *helper, *cphone;

  work = r_personal_card_get_work (R_PERSONAL_CARD(card));

  if (!IS_R_WORK(work))
    return;

  g_object_get(work, "assignment", &role, "organization", &org,
	       "department", &dep, "sub-department", &subdep,
	       "manager-name", &manager, "manager-phone", &mphone, 
	       "collaborator", &helper, "collaborator-phone", &cphone, NULL);
  
  if (role)    gtk_entry_set_text(GTK_ENTRY(form->priv->role),    role);
  if (org)     gtk_entry_set_text(GTK_ENTRY(form->priv->org),     org);
  if (dep)     gtk_entry_set_text(GTK_ENTRY(form->priv->dep),     dep);
  if (subdep)  gtk_entry_set_text(GTK_ENTRY(form->priv->subdep),  subdep);
  if (manager) gtk_entry_set_text(GTK_ENTRY(form->priv->manager), manager);
  if (mphone)  gtk_entry_set_text(GTK_ENTRY(form->priv->mphone),  mphone);
  if (helper)  gtk_entry_set_text(GTK_ENTRY(form->priv->helper),  helper);
  if (cphone)  gtk_entry_set_text(GTK_ENTRY(form->priv->cphone),  cphone);  
}


static void 
form_work_clean_form (RubricaFormWork* form)
{
  gtk_entry_set_text(GTK_ENTRY(form->priv->role),    "");
  gtk_entry_set_text(GTK_ENTRY(form->priv->org),     "");
  gtk_entry_set_text(GTK_ENTRY(form->priv->dep),     "");
  gtk_entry_set_text(GTK_ENTRY(form->priv->subdep),  "");
  gtk_entry_set_text(GTK_ENTRY(form->priv->manager), "");
  gtk_entry_set_text(GTK_ENTRY(form->priv->mphone),  "");
  gtk_entry_set_text(GTK_ENTRY(form->priv->helper),  "");
  gtk_entry_set_text(GTK_ENTRY(form->priv->cphone),  "");
}


/*   ***************************** Public *****************************
*/


/**
 * rubrica_form_work_new
 *
 * create a #RubricaFormWork
 *
 * Returns: a new #RubricaFormWork
 */
RubricaFormWork*
rubrica_form_work_new(RubricaApp* app, GladeXML *gui)
{
  RubricaFormWork* form;

  form = g_object_new(rubrica_form_work_get_type(), NULL);
  
  form->priv->role    = glade_xml_get_widget(gui, "role");
  form->priv->org     = glade_xml_get_widget(gui, "org");
  form->priv->dep     = glade_xml_get_widget(gui, "dep");
  form->priv->subdep  = glade_xml_get_widget(gui, "subdep");
  form->priv->manager = glade_xml_get_widget(gui, "manager");
  form->priv->mphone  = glade_xml_get_widget(gui, "mphone");
  form->priv->helper  = glade_xml_get_widget(gui, "helper");
  form->priv->cphone  = glade_xml_get_widget(gui, "cphone");  
  
  gtk_widget_grab_focus (form->priv->role);
  return form;
}


/**
 * rubrica_form_work_free
 * @form: a #RubricaFormWork
 * 
 * free the #RubricaFormWork
 */
void
rubrica_form_work_free(RubricaFormWork* form)
{
  g_return_if_fail(IS_RUBRICA_FORM(form));

  g_object_unref(form);   
}
