/*
*  program: Rubrica
*  file: groups_view.c
*
*  
*  Copyright (C) Nicola Fragale <nicolafragale@libero.it>
*
*  This program is free software; you can redistribute it and/or modify
*  it under the terms of the GNU General Public License as published by
*  the Free Software Foundation; either version 3 of the License
*
*  This program is distributed in the hope that it will be useful,
*  but WITHOUT ANY WARRANTY; without even the implied warranty of
*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*  GNU General Public License for more details.
*
*  You should have received a copy of the GNU General Public License
*  along with this program; if not, write to the Free Software
*  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#include <gtk/gtk.h>
#include <glib/gi18n-lib.h>

#include "groups_view.h"
#include "types.h"

enum {
  PROP_0,
  GROUP_BOX
};




struct _RubricaGroupsViewPrivate {
  RGroupBox *box;

  gboolean dispose_has_run;
};



#define RUBRICA_GROUPS_VIEW_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE((o), \
                                            RUBRICA_GROUPS_VIEW_TYPE,        \
                                            RubricaGroupsViewPrivate))


static GObjectClass *parent_class = NULL;


static void rubrica_groups_view_class_init   (RubricaGroupsViewClass* klass);
static void rubrica_groups_view_init         (RubricaGroupsView* obj);

static void rubrica_groups_view_finalize     (RubricaGroupsView* self);
static void rubrica_groups_view_dispose      (RubricaGroupsView* self);


static void rubrica_groups_view_set_property (GObject* obj, guint property_id,
					      GValue* value,  
					      GParamSpec* spec);
static void rubrica_groups_view_get_property (GObject* obj, guint property_id,
					      GValue* value, 
					      GParamSpec* spec);


/* Callbacks
*/
static void model_toggled                (GtkCellRendererToggle *cell, 
					  gchar *path,  gpointer data);

static void group_added_cb    (RGroupBox *box, gpointer group, gpointer data);
static void group_removed_cb  (RGroupBox *box, gint group_id, gpointer data);
static void group_modifyed_cb (RGroupBox *box, gpointer group,gpointer data);


/* Private
 */  
static void     groups_view_populate_view (RubricaGroupsView* view, 
					   RGroupBox* box, 
					   RubricaGroupViewType type);
static void     theme_changed             (GtkIconTheme* theme, gpointer data);

static gboolean remove_group              (GtkTreeModel *model, 
					   GtkTreePath *path,
					   GtkTreeIter *iter, gpointer data);



static gboolean 
remove_group (GtkTreeModel *model, GtkTreePath *path, GtkTreeIter *iter, 
	      gpointer data)
{
  gint id;
  gint group_id = GPOINTER_TO_INT(data);

  gtk_tree_model_get(model, iter, GROUP_ID, &id, -1);
  if (id == group_id)
    {
      gtk_list_store_remove(GTK_LIST_STORE(model), iter);

      return TRUE;
    }

  return FALSE;
}



static void 
model_toggled(GtkCellRendererToggle *cell, gchar *path_str, gpointer data)
{
  GtkTreeModel *model;  
  GtkTreeIter iter;  
  GtkTreePath *path;  
  gboolean status;       

  model = (GtkTreeModel *) data; 
  path  = gtk_tree_path_new_from_string(path_str);  

  gtk_tree_model_get_iter (model, &iter, path);  
  gtk_tree_model_get (model, &iter, GROUP_BELONG_COLUMN, &status, -1);  

  status ^= 1;  
  gtk_list_store_set(GTK_LIST_STORE(model), &iter, 
		     GROUP_BELONG_COLUMN, status, -1);    
}



static void 
group_added_cb (RGroupBox *box, gpointer group, gpointer data)
{
  GtkTreeView* tree;
  GtkTreeModel* model;   
  GtkTreeIter iter;
  GdkPixbuf* pixbuf = NULL;
  gchar *uri, *name, *label;
  gboolean enabled;
  gint id;
  
  tree = GTK_TREE_VIEW(rubrica_view_get_tree(RUBRICA_VIEW(data)));

  g_object_get(R_GROUP(group), 
	       "group-name", &name, "id", &id, 
	       "group-label", &label, "group-pixmap", &uri, 
	       "enabled", &enabled, NULL);
  
  if (uri)
    pixbuf = gdk_pixbuf_new_from_file_at_size (uri, 20, 20, NULL);   

  model = gtk_tree_view_get_model(GTK_TREE_VIEW(tree));
  gtk_list_store_append(GTK_LIST_STORE(model), &iter);
  gtk_list_store_set(GTK_LIST_STORE(model), &iter, 
		     GROUP_BELONG_COLUMN,   enabled, 
		     GROUP_PIXMAP_COLUMN,   pixbuf, 
		     GROUP_NAME_COLUMN,     name,
		     GROUP_LABEL_COLUMN,    label,
		     GROUP_OWNER_COLUMN,    "user",
		     GROUP_PIXMAP_PATH,     uri,
		     GROUP_ID,              id,
		     -1);  
}


static void 
group_removed_cb (RGroupBox *box, gint id,gpointer data)
{
  GtkTreeView* tree;
  GtkTreeModel* model;   
  
  tree  = GTK_TREE_VIEW(rubrica_view_get_tree(RUBRICA_VIEW(data)));
  model = gtk_tree_view_get_model(tree);
  if (model)
    gtk_tree_model_foreach(GTK_TREE_MODEL(model), remove_group, 
			   GINT_TO_POINTER(id));  
}


static void 
group_modifyed_cb (RGroupBox *box, gpointer group, gpointer data)
{

  GtkTreeView* tree;
  GtkTreeModel* model;   
  GtkTreeIter iter;
  GdkPixbuf* pixbuf = NULL;
  gchar *uri, *name, *label;
  gboolean enabled;
  gint id, group_id;
  GError *error = NULL;
  
  tree   = GTK_TREE_VIEW(rubrica_view_get_tree(RUBRICA_VIEW(data)));
  model  = gtk_tree_view_get_model(tree);

  g_object_get(R_GROUP(group), 
	       "group-name", &name, "id", &group_id, 
	       "group-label", &label, "group-pixmap", &uri, 
	       "enabled", &enabled, NULL);
  
  if (uri)
    pixbuf = gdk_pixbuf_new_from_file_at_size (uri, 20, 20, &error);
  
  if (!pixbuf)
    g_warning("\npixbuf: %s", error->message);

#ifdef ENABLE_DEBUG
  g_log(G_LOG_DOMAIN, G_LOG_LEVEL_DEBUG, "uri: %s", uri);
  g_log(G_LOG_DOMAIN, G_LOG_LEVEL_DEBUG, "pixbuf: %p", pixbuf);  
#endif

  if (gtk_tree_model_get_iter_first(model, &iter))
  {
    do 
      {
	gtk_tree_model_get(model, &iter, GROUP_ID, &id, -1);

	if (id == group_id)
	  gtk_list_store_set(GTK_LIST_STORE(model), &iter, 
			     GROUP_BELONG_COLUMN,   enabled, 
			     GROUP_PIXMAP_COLUMN,   pixbuf, 
			     GROUP_NAME_COLUMN,     name,
			     GROUP_LABEL_COLUMN,    label,
			     GROUP_OWNER_COLUMN,    "user",
			     GROUP_PIXMAP_PATH,     uri,
			     GROUP_ID,              id,
			     -1); 
      }	
    while (gtk_tree_model_iter_next(model, &iter));
  }
}


static void 
theme_changed (GtkIconTheme* theme, gpointer data)
{
  RubricaGroupsView* view = (RubricaGroupsView*) data;  
  GtkTreeView* tree;
  GtkTreeModel* model;
  GtkTreeIter iter;  
  RGroup* group;

  theme = gtk_icon_theme_get_default(); 
  tree  = GTK_TREE_VIEW(rubrica_view_get_tree(RUBRICA_VIEW(view)));
  model = gtk_tree_view_get_model(tree);  
  
  gtk_list_store_clear(GTK_LIST_STORE(model));

  r_group_box_reset(view->priv->box); 
  group = r_group_box_get_group(view->priv->box);
  for(; group; group = r_group_box_get_next_group(view->priv->box))
    {
      GError *error = NULL;
      gboolean enabled = FALSE;
      gchar* name   = NULL;
      gchar* label  = NULL;
      gchar* owner  = NULL;
      gchar* pixmap = NULL;
      gint id = 0;
      GdkPixbuf* pixbuf;

      g_object_get(group,"id", &id,
		   "group-name", &name, "group-label", &label,
		   "group-owner", &owner, "group-pixmap", &pixmap, 
		   "enabled", &enabled, NULL);

      
   
      pixbuf = gtk_icon_theme_load_icon (theme, pixmap, 22, 0, NULL);
      if (!pixbuf)
	pixbuf = gdk_pixbuf_new_from_file_at_size (pixmap, 20, 20, &error);

      if (!pixbuf)
	g_warning("\ntheme_changed -- no pixbuf: %s", error->message);
      
      gtk_list_store_append(GTK_LIST_STORE(model), &iter);
      gtk_list_store_set(GTK_LIST_STORE(model), &iter, 
			 GROUP_PIXMAP_COLUMN,   pixbuf, 
			 GROUP_NAME_COLUMN,     name,
			 GROUP_LABEL_COLUMN,    label,
			 GROUP_OWNER_COLUMN,    owner,
			 GROUP_PIXMAP_PATH,     pixmap,
			 GROUP_ID,              id,
			 -1);
    }  
}


static void 
groups_view_populate_view(RubricaGroupsView* view, RGroupBox* box,
			  RubricaGroupViewType type)
{
  GtkTreeView* tree;
  GtkTreeModel* model;
  GtkTreeViewColumn* column;
  GtkCellRenderer *renderer;
  GtkTreeIter iter;  
  RGroup* group;
  GtkIconTheme* theme;
  
#ifdef ENABLE_DEBUG
  g_log(G_LOG_DOMAIN, G_LOG_LEVEL_DEBUG, "groups_view_populate_view");
#endif

  theme = gtk_icon_theme_get_default(); 
  tree  = GTK_TREE_VIEW(rubrica_view_get_tree(RUBRICA_VIEW(view)));
  model = gtk_tree_view_get_model(tree);

  if (type == FORM_GROUPS)
    {
      renderer = gtk_cell_renderer_toggle_new();
      gtk_tree_view_insert_column_with_attributes(tree, -1, _("Belongs"), 
						  renderer, "active", 
						  GROUP_BELONG_COLUMN,
						  NULL); 
      
      g_signal_connect(G_OBJECT(renderer), "toggled", 
		       G_CALLBACK(model_toggled), model);  
    }

  column = gtk_tree_view_column_new();
  gtk_tree_view_column_set_title(column, _("Groups"));
  
  renderer = gtk_cell_renderer_pixbuf_new();
  gtk_tree_view_column_pack_start(column, renderer, FALSE);
  gtk_tree_view_column_set_attributes (column, renderer, 
				       "pixbuf", GROUP_PIXMAP_COLUMN, NULL);
  
  renderer = gtk_cell_renderer_text_new();
  gtk_tree_view_column_pack_start(column, renderer, TRUE);
  gtk_tree_view_column_set_attributes (column, renderer, 
				       "text", GROUP_LABEL_COLUMN,  NULL);

  gtk_tree_view_append_column (GTK_TREE_VIEW(tree), column);
  

  r_group_box_reset(box);
  group = r_group_box_get_group(box);

#ifdef ENABLE_DEBUG
  g_log(G_LOG_DOMAIN, G_LOG_LEVEL_DEBUG, "cross groups and build group's tree/list view");
  g_log(G_LOG_DOMAIN, G_LOG_LEVEL_DEBUG, "Group %-4d %-25s %-25s", 
	"id", "label", "pixmap");
#endif

  for(; group; group = r_group_box_get_next_group(box))
    {
      GError *error = NULL;
      GdkPixbuf* pixbuf = NULL;
      gboolean enabled = FALSE;
      gchar* name   = NULL;
      gchar* label  = NULL;
      gchar* owner  = NULL;
      gchar* pixmap = NULL;
      gint id = 0;

      g_object_get(group,"id", &id,
		   "group-name", &name, "group-label", &label,
		   "group-owner", &owner, "group-pixmap", &pixmap, 
		   "enabled", &enabled, NULL);
      
#ifdef ENABLE_DEBUG
      g_log(G_LOG_DOMAIN, G_LOG_LEVEL_DEBUG, "%-4d %-25s %-25s", 
	    id, label, pixmap);
#endif

      if (pixmap)
	{
	  pixbuf = gtk_icon_theme_load_icon (theme, pixmap, 22, 0, NULL);
	  if (!pixbuf)
	    pixbuf = gdk_pixbuf_new_from_file_at_size (pixmap, 20, 20, &error);
	}
      
      //      if (pixbuf)
      //	{
	  gtk_list_store_append(GTK_LIST_STORE(model), &iter);
	  gtk_list_store_set(GTK_LIST_STORE(model), &iter, 
			     GROUP_BELONG_COLUMN,   enabled, 		 
			     GROUP_PIXMAP_COLUMN,   pixbuf, 
			     GROUP_NAME_COLUMN,     name,
			     GROUP_LABEL_COLUMN,    label,
			     GROUP_OWNER_COLUMN,    owner,
			     GROUP_PIXMAP_PATH,     pixmap,
			     GROUP_ID,              id,
			     -1);
	  /*
	}
      else
	g_warning("\ngroups_view_populate_view -- \n\tpixmap: %s\t pixbuf: %p\terror: %s", 
		  pixmap, pixbuf, error->message); 
	  */  
    }
    
  g_signal_connect(G_OBJECT(theme), "changed", 
		   G_CALLBACK(theme_changed), view);   
}




GType
rubrica_groups_view_get_type()
{
  static GType groups_view_type = 0;
  
  if (!groups_view_type)
    {
      static const GTypeInfo groups_view_info =
	{
	  sizeof(RubricaGroupsViewClass),
	  NULL,
	  NULL,
	  (GClassInitFunc) rubrica_groups_view_class_init,
	  NULL,
	  NULL,
	  sizeof(RubricaGroupsView),
	  0,
	  (GInstanceInitFunc) rubrica_groups_view_init
	};

      groups_view_type = g_type_register_static (RUBRICA_VIEW_TYPE, 
						"RubricaGroupsView", 
						&groups_view_info, 0);
    }
  
  return groups_view_type;
}


static void 
rubrica_groups_view_dispose (RubricaGroupsView* self)
{
  g_return_if_fail(IS_RUBRICA_GROUPS_VIEW(self));
  
  if (self->priv->dispose_has_run)
    return;

  self->priv->dispose_has_run = TRUE;  
}


static void 
rubrica_groups_view_finalize (RubricaGroupsView* self)
{
  g_return_if_fail(IS_RUBRICA_GROUPS_VIEW(self));
}


static void
rubrica_groups_view_class_init(RubricaGroupsViewClass* klass)
{
  GObjectClass *object_class = G_OBJECT_CLASS (klass);
  GParamSpec* pspec;
  
  parent_class = g_type_class_peek_parent (klass);

  object_class->finalize = (GObjectFinalizeFunc) rubrica_groups_view_finalize;
  object_class->dispose  = (GObjectFinalizeFunc) rubrica_groups_view_dispose;

  object_class->set_property = (gpointer) rubrica_groups_view_set_property;
  object_class->get_property = (gpointer) rubrica_groups_view_get_property;

  g_type_class_add_private (klass, sizeof(RubricaGroupsViewPrivate));

  /* class property 
   */
  /**
   * RubricaGroupsView:name
   *
   * the name of the addressbook displayed
   */
  pspec = g_param_spec_object("group-box", 
			      "group box", 
			      "the group box",  
			      R_GROUP_BOX_TYPE,
			      G_PARAM_CONSTRUCT | G_PARAM_READWRITE);
  g_object_class_install_property(object_class, GROUP_BOX, pspec);  
}


static void
rubrica_groups_view_init(RubricaGroupsView* self)
{
  GtkListStore* model;
  
  g_return_if_fail(IS_RUBRICA_GROUPS_VIEW(self));
  
  self->priv = RUBRICA_GROUPS_VIEW_GET_PRIVATE(self);

  model = gtk_list_store_new(GROUP_LAST_COLUMN,
			     G_TYPE_INT,        /* group check (belong) */
			     GDK_TYPE_PIXBUF,   /* group pixbuf */
			     G_TYPE_STRING,     /* group name   */
			     G_TYPE_STRING,     /* group label  */
			     G_TYPE_STRING,     /* group owner  */
			     G_TYPE_STRING,     /* pixmap path  */
			     G_TYPE_INT);       /* group's id   */ 
  
  rubrica_view_set_model(RUBRICA_VIEW(self), GTK_TREE_MODEL(model));

  self->priv->dispose_has_run = FALSE;
}


static void 
rubrica_groups_view_set_property (GObject* obj, guint property_id,
				 GValue* value, GParamSpec* spec)
{
  RubricaGroupsView* self = (RubricaGroupsView*) obj;
  RubricaGroupsViewPrivate* priv = RUBRICA_GROUPS_VIEW_GET_PRIVATE(self);

  switch (property_id) 
    {
    case GROUP_BOX:
      priv->box = g_value_get_object(value);
      break;

    default: 
      break; 
    } 
} 
 

static void 
rubrica_groups_view_get_property (GObject* obj, guint property_id,
				 GValue* value, GParamSpec* spec)
{
  RubricaGroupsView* self = (RubricaGroupsView*) obj;
  RubricaGroupsViewPrivate* priv = RUBRICA_GROUPS_VIEW_GET_PRIVATE(self);

  switch (property_id) 
    {      
    case GROUP_BOX:
      g_value_set_object(value, priv->box);
      break;

    default:
      break;  
    }  
}




/*   Public
*/
/**
 * rubrica_groups_view_new
 *
 * create a #RubricaGroupsView
 *
 * returns: a new #RubricaGroupsView*
 */
GtkWidget*
rubrica_groups_view_new(RGroupBox *groups, RubricaGroupViewType type)
{
  GtkWidget* view;

  view = GTK_WIDGET(g_object_new(RUBRICA_GROUPS_VIEW_TYPE, 
				 "group-box", groups, 
				 NULL));

  groups_view_populate_view (RUBRICA_GROUPS_VIEW(view), groups, type);

  g_signal_connect(G_OBJECT (groups), "group_added",
		   G_CALLBACK (group_added_cb), view);

  g_signal_connect(G_OBJECT (groups), "group_removed",
		   G_CALLBACK (group_removed_cb), view); 
 
  g_signal_connect(G_OBJECT (groups), "group_modifyed",
		   G_CALLBACK (group_modifyed_cb), view);   
  return view;
}

RGroupBox*
rubrica_groups_view_get_group_box (RubricaGroupsView* view)
{
  RubricaGroupsViewPrivate* priv = RUBRICA_GROUPS_VIEW_GET_PRIVATE(view);

  g_return_val_if_fail(IS_RUBRICA_GROUPS_VIEW(view), NULL);
  
  return priv->box;
}



void 
rubrica_groups_view_add_group(RubricaGroupsView* view, RGroup* group)
{
  RubricaGroupsViewPrivate* priv = RUBRICA_GROUPS_VIEW_GET_PRIVATE(view);
  
  r_group_box_add_group(R_GROUP_BOX(priv->box), group);
}


void 
rubrica_groups_view_del_group(RubricaGroupsView* view, RGroup* group)
{
  RubricaGroupsViewPrivate* priv = RUBRICA_GROUPS_VIEW_GET_PRIVATE(view);
  
  r_group_box_delete_group(R_GROUP_BOX(priv->box), group);
}


