/*
*  program: Rubrica
*  file: popmenu.c
*
*  
*  Copyright (C) 2000-2005 Nicola Fragale <nicolafragale@libero.it>
*
*  This program is free software; you can redistribute it and/or modify
*  it under the terms of the GNU General Public License as published by
*  the Free Software Foundation; either version 3 of the License
*
*  This program is distributed in the hope that it will be useful,
*  but WITHOUT ANY WARRANTY; without even the implied warranty of
*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*  GNU General Public License for more details.
*
*  You should have received a copy of the GNU General Public License
*  along with this program; if not, write to the Free Software
*  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/


#include <gtk/gtk.h>
#include <gconf/gconf-client.h>
#include <glade/glade.h>

#include "libral.h"

#include "app.h"
#include "trash_view.h"
#include "callbacks.h"
#include "search_dialog.h"
#include "preferences.h"
#include "search.h"
#include "types.h"
#include "themes.h"
 

static gboolean mark_row              (GtkTreeModel *model, GtkTreePath *path,
				       GtkTreeIter *iter, gpointer data);



static void popup_item_set_image       (GtkWidget* item, const gchar* name);

static void on_column_toggled          (GtkWidget* button, gpointer data);
static void on_advanced_activate       (GtkWidget* button, gpointer data);
static void on_recovery_card_activate  (GtkWidget* button, gpointer data);
static void on_destroy_card_activate   (GtkWidget* button, gpointer data);
static void on_empty_trash_activate    (GtkWidget* button, gpointer data);

static void on_show_all_activate       (GtkWidget* button, gpointer data);
static void on_show_men_activate       (GtkWidget* button, gpointer data);
static void on_show_women_activate     (GtkWidget* button, gpointer data);
static void on_show_personal_activate  (GtkWidget* button, gpointer data);
static void on_show_company_activate   (GtkWidget* button, gpointer data);

static void on_show_by_rate1_ativate   (GtkWidget* button, gpointer data);
static void on_show_by_rate2_ativate   (GtkWidget* button, gpointer data);
static void on_show_by_rate3_ativate   (GtkWidget* button, gpointer data);
static void on_show_by_rate4_ativate   (GtkWidget* button, gpointer data);
static void on_show_by_rate5_ativate   (GtkWidget* button, gpointer data);

static void on_mark2print_cb           (GtkWidget* button, gpointer data);
static void on_mark_all_cb             (GtkWidget* button, gpointer data);
static void on_unmark_all_cb           (GtkWidget* button, gpointer data);
static void on_show_printable_cb       (GtkWidget* button, gpointer data);
static void on_hide_printable_cb       (GtkWidget* button, gpointer data);


GtkWidget* unmark;


static gboolean 
mark_row (GtkTreeModel *model, GtkTreePath *path, GtkTreeIter *iter, 
	  gpointer data)
{
  gboolean bool = (gboolean) data;

  gtk_list_store_set(GTK_LIST_STORE(model), iter, PRINT_COLUMN, bool, -1); 
  
  return FALSE;
}


static void     
popup_item_set_image (GtkWidget* item, const gchar* name)
{
  GtkIconTheme* theme;
  GdkPixbuf* pixbuf;
  GtkWidget* image;  

  g_return_if_fail(name != NULL);
  
  theme  = gtk_icon_theme_get_default(); 

  pixbuf = gtk_icon_theme_load_icon (theme, name, 24, 0, NULL);
  image  = gtk_image_new_from_pixbuf(pixbuf);
  gtk_image_menu_item_set_image(GTK_IMAGE_MENU_ITEM(item), image);  
}


/* Trash menu callbacks
 */

/* recovery the selected card (move to the original addressbook) 
 */
static void 
on_recovery_card_activate (GtkWidget* button, gpointer data)
{
  RubricaApp* app = (RubricaApp*) data;
  GtkWidget* trash;
  
  trash = rubrica_app_get_trash(app);
  rubrica_trash_view_recovery_card(RUBRICA_TRASH_VIEW(trash));
}


/* destroy the selected card 
 */
static void 
on_destroy_card_activate (GtkWidget* button, gpointer data)
{
  RubricaApp* app = (RubricaApp*) data;
  GtkWidget* trash;
  
  trash = rubrica_app_get_trash(app);
  rubrica_trash_view_destroy_card(RUBRICA_TRASH_VIEW(trash));
}


/* empty trash, destroy all cards in the trash list 
 */
static void 
on_empty_trash_activate  (GtkWidget* button, gpointer data)
{
  RubricaApp* app = (RubricaApp*) data;
  GtkWidget* trash;

  trash = rubrica_app_get_trash(app);
  rubrica_trash_view_empty_trash(RUBRICA_TRASH_VIEW(trash));
}
/* Trash menu callbacks  */



/* Show/hide columns in the cards list 
*/
static void 
on_column_toggled (GtkWidget* button, gpointer data)
{
  GConfClient* client;  
  gchar* key = (gchar*) data;
  gboolean bool;
  
  client = gconf_client_get_default();
  bool   = gtk_check_menu_item_get_active (GTK_CHECK_MENU_ITEM(button));
  
  gconf_client_set_bool(client, key, bool, NULL);

  g_object_unref(client);
}


/* Search on callbacks
 */
static void
on_advanced_activate (GtkWidget* button, gpointer data)
{
  rubrica_search_dialog((RubricaApp*) data);  
}


static void     
on_show_all_activate (GtkWidget* button, gpointer data)
{
  rubrica_app_display_cards_by_type((RubricaApp*) data, "all");
}

static void     
on_show_personal_activate (GtkWidget* button, gpointer data)
{
  rubrica_app_display_cards_by_type((RubricaApp*) data, "personal");
}

static void     
on_show_company_activate (GtkWidget* button, gpointer data)
{
  rubrica_app_display_cards_by_type((RubricaApp*) data, "company");
}


static void     
on_show_men_activate (GtkWidget* button, gpointer data)
{
  rubrica_app_display_cards_by_genre((RubricaApp*) data, "male");
}


static void     
on_show_women_activate (GtkWidget* button, gpointer data)
{
  rubrica_app_display_cards_by_genre((RubricaApp*) data, "female");
}


static void 
on_show_by_rate1_ativate (GtkWidget* button, gpointer data)
{
  rubrica_app_display_cards_by_rate((RubricaApp*) data, R_RATE_NONE);
}

static void 
on_show_by_rate2_ativate (GtkWidget* button, gpointer data)
{
  rubrica_app_display_cards_by_rate((RubricaApp*) data, R_RATE_FEW);
}

static void 
on_show_by_rate3_ativate (GtkWidget* button, gpointer data)
{
  rubrica_app_display_cards_by_rate((RubricaApp*) data, R_RATE_NORMAL);
}

static void 
on_show_by_rate4_ativate (GtkWidget* button, gpointer data)
{
  rubrica_app_display_cards_by_rate((RubricaApp*) data, R_RATE_HIGH);
}

static void 
on_show_by_rate5_ativate (GtkWidget* button, gpointer data)
{
  rubrica_app_display_cards_by_rate((RubricaApp*) data, R_RATE_VERY_HIGH);
}
/* *************** */


/*  Print callbacks
*/
static void
on_print_card_cb (GtkWidget* button, gpointer data)
{
  GtkTreeModel *model;
  GtkTreeSelection *selection; 
  GtkTreeIter iter;

  selection = g_object_get_data(G_OBJECT(button), "selection");
  if (gtk_tree_selection_get_selected(selection, &model, &iter))
    {
      gtk_list_store_set(GTK_LIST_STORE(model), &iter, PRINT_COLUMN, TRUE, -1);
      rubrica_app_print(RUBRICA_APP(data));
    }

  gtk_list_store_set(GTK_LIST_STORE(model), &iter, PRINT_COLUMN, FALSE, -1);  
}


static void 
on_mark2print_cb (GtkWidget* button, gpointer data)
{
  RubricaApp* app = (RubricaApp*) data;
  GtkTreeModel *model;
  GtkTreeSelection *selection; 
  GtkTreeIter iter;
  
  selection = g_object_get_data(G_OBJECT(button), "selection");
  if (gtk_tree_selection_get_selected(selection, &model, &iter))
    {
      GConfClient* client;  

      gtk_list_store_set(GTK_LIST_STORE(model), &iter, PRINT_COLUMN, TRUE, -1);
      
      client = gconf_client_get_default();
      gconf_client_set_bool(client, RUBRICA_GCONF_PRINT_CARD, TRUE, NULL);
      g_object_unref(client);
    }
  
  gtk_widget_set_sensitive(unmark, TRUE);
}



static void 
on_mark_all_cb (GtkWidget* button, gpointer data)
{
  RubricaApp* app = (RubricaApp*) data;
  GConfClient* client;  
  GtkTreeView* tree;
  GtkTreeModel *model;
  GtkTreeSelection *selection; 
  RAbook* abook;

  client = gconf_client_get_default();
  gconf_client_set_bool(client, RUBRICA_GCONF_PRINT_CARD, TRUE, NULL);
  g_object_unref(client);

  selection = g_object_get_data(G_OBJECT(button), "selection");
  tree      = gtk_tree_selection_get_tree_view (selection);
  model     = gtk_tree_view_get_model(GTK_TREE_VIEW(tree));
  
  gtk_tree_model_foreach(GTK_TREE_MODEL(model), mark_row, (gpointer) TRUE);

  abook = rubrica_app_get_active_addressbook(app);
  g_object_set(abook, "marked-cards", r_abook_get_items(abook), NULL);

  gtk_widget_set_sensitive(unmark, TRUE);
}


static void 
on_unmark_all_cb (GtkWidget* button, gpointer data)
{
  RubricaApp* app = (RubricaApp*) data;
  GConfClient* client;  
  GtkTreeView* tree;
  GtkTreeModel *model;
  GtkTreeSelection *selection; 
  RAbook* abook;

  client = gconf_client_get_default();
  gconf_client_set_bool(client, RUBRICA_GCONF_PRINT_CARD, TRUE, NULL);
  g_object_unref(client);

  selection = g_object_get_data(G_OBJECT(button), "selection");
  tree      = gtk_tree_selection_get_tree_view (selection);
  model     = gtk_tree_view_get_model(GTK_TREE_VIEW(tree));
  
  gtk_tree_model_foreach(GTK_TREE_MODEL(model), mark_row, (gpointer) FALSE);

  abook = rubrica_app_get_active_addressbook(app);
  g_object_set(abook, "marked-cards", 0, NULL);

  gtk_widget_set_sensitive(unmark, FALSE);
}



static void 
on_show_printable_cb (GtkWidget* button, gpointer data)
{
  GConfClient* client;
  
  client = gconf_client_get_default();

  gconf_client_set_bool(client, RUBRICA_GCONF_PRINT_CARD, TRUE, NULL);  
  g_object_unref(client);
}


static void 
on_hide_printable_cb (GtkWidget* button, gpointer data)
{
  GConfClient* client;
  
  client = gconf_client_get_default();

  gconf_client_set_bool(client, RUBRICA_GCONF_PRINT_CARD, FALSE, NULL);  
  g_object_unref(client);  
}


/*    Public
 */
GtkMenu* 
rubrica_menu_do_card_popup(RubricaApp* app, GtkTreeView* tree, GdkEvent *event)
{  
  GladeXML* gui;
  RAbook* abook;
  GdkEventButton *ev = (GdkEventButton *)event;
  GtkTreeModel *model;
  GtkTreeSelection *selection; 
  GtkTreePath *path = NULL;
  GSList* radio_group = NULL;
  GtkMenu* menu;  
  GtkWidget* cut;
  GtkWidget* copy;
  GtkWidget* paste;
  GtkWidget* modify;
  GtkWidget* delete;
  GtkWidget* markall;
  GtkWidget* show_printable;
  GtkWidget* hide_printable;
  GtkWidget* item;
  gint marked = 0;

  abook = rubrica_app_get_active_addressbook(app);
  g_object_get(abook, "marked-cards", &marked, NULL);

  gui       = glade_xml_new (RUBRICA_GUI_DIR"/Menu.glade", NULL, NULL);    
  model     = gtk_tree_view_get_model(GTK_TREE_VIEW(tree));
  selection = gtk_tree_view_get_selection(tree);

  menu = (GtkMenu*) glade_xml_get_widget(gui, "menu1");  

  /* callbacks functions are in callbacks.c 
   */
  cut = glade_xml_get_widget(gui, "cut");
  g_signal_connect(G_OBJECT(cut), "activate", 
		   G_CALLBACK(on_cut_cb), app);
  
  copy = glade_xml_get_widget(gui, "copy");
  g_signal_connect(G_OBJECT(copy), "activate", 
		   G_CALLBACK(on_copy_cb), app);
  
  paste = glade_xml_get_widget(gui, "paste");
  g_signal_connect(G_OBJECT(paste), "activate", 
		   G_CALLBACK(on_paste_cb), app);
  
  item = glade_xml_get_widget(gui, "print");
  g_object_set_data(G_OBJECT(item), "selection", selection);
  g_signal_connect(G_OBJECT(item), "activate", 
		   G_CALLBACK(on_print_card_cb), app);

  item = glade_xml_get_widget(gui, "mark2print");
  g_object_set_data(G_OBJECT(item), "selection", selection);
  g_signal_connect(G_OBJECT(item), "activate", 
		   G_CALLBACK(on_mark2print_cb), app);

  markall = glade_xml_get_widget(gui, "markall");
  g_object_set_data(G_OBJECT(markall), "selection", selection);
  g_signal_connect(G_OBJECT(markall), "activate", 
		   G_CALLBACK(on_mark_all_cb), app);

  unmark = glade_xml_get_widget(gui, "unmark");
  g_object_set_data(G_OBJECT(unmark), "selection", selection);
  g_signal_connect(G_OBJECT(unmark), "activate", 
		   G_CALLBACK(on_unmark_all_cb), app);

  show_printable = glade_xml_get_widget(gui, "show_printable");
  g_signal_connect(G_OBJECT(show_printable), "activate", 
		   G_CALLBACK(on_show_printable_cb), app);  
  
  hide_printable = glade_xml_get_widget(gui, "hide_printable");
  g_signal_connect(G_OBJECT(hide_printable), "activate", 
		   G_CALLBACK(on_hide_printable_cb), app);

  item = glade_xml_get_widget(gui, "add_personal");
  g_signal_connect(G_OBJECT(item), "activate", 
		   G_CALLBACK(on_add_personal_cb), app);
  
  item = glade_xml_get_widget(gui, "add_company");
  g_signal_connect(G_OBJECT(item), "activate", 
		   G_CALLBACK(on_add_company_cb), app);
  
  modify = glade_xml_get_widget(gui, "modify_card");
  g_signal_connect(G_OBJECT(modify), "activate", 
		   G_CALLBACK(on_modify_card_cb), app);
  
  delete = glade_xml_get_widget(gui, "delete_card");
  g_signal_connect(G_OBJECT(delete), "activate", 
		   G_CALLBACK(on_delete_card_cb), app);
  
  /* submenu Show Columns (callbacks functions are in this file)
   */
  item = glade_xml_get_widget(gui, "card"); 
  g_signal_connect(G_OBJECT(item), "toggled", 
		   G_CALLBACK(on_column_toggled), RUBRICA_GCONF_CARD);
  rubrica_preferences_setup_menu_check(item, RUBRICA_GCONF_CARD);
  
  item = glade_xml_get_widget(gui, "fn"); 
  g_signal_connect(G_OBJECT(item), "toggled", 
		   G_CALLBACK(on_column_toggled), RUBRICA_GCONF_FIRST_NAME);
  rubrica_preferences_setup_menu_check(item, RUBRICA_GCONF_FIRST_NAME);
  
  item = glade_xml_get_widget(gui, "ln"); 
  g_signal_connect(G_OBJECT(item), "toggled", 
		   G_CALLBACK(on_column_toggled), RUBRICA_GCONF_LAST_NAME);
  rubrica_preferences_setup_menu_check(item, RUBRICA_GCONF_LAST_NAME);
  
  item = glade_xml_get_widget(gui, "prof"); 
  g_signal_connect(G_OBJECT(item), "toggled", 
		   G_CALLBACK(on_column_toggled), RUBRICA_GCONF_PROFESSION);
  rubrica_preferences_setup_menu_check(item, RUBRICA_GCONF_PROFESSION);
  
  item = glade_xml_get_widget(gui, "city"); 
  g_signal_connect(G_OBJECT(item), "toggled", 
		   G_CALLBACK(on_column_toggled), RUBRICA_GCONF_CITY);
  rubrica_preferences_setup_menu_check(item, RUBRICA_GCONF_CITY);
  
  item = glade_xml_get_widget(gui, "count"); 
  g_signal_connect(G_OBJECT(item), "toggled", 
		   G_CALLBACK(on_column_toggled), RUBRICA_GCONF_COUNTRY);
  rubrica_preferences_setup_menu_check(item, RUBRICA_GCONF_COUNTRY);
  
  item = glade_xml_get_widget(gui, "web"); 
  g_signal_connect(G_OBJECT(item), "toggled", 
		   G_CALLBACK(on_column_toggled), RUBRICA_GCONF_WEB);
  rubrica_preferences_setup_menu_check(item, RUBRICA_GCONF_WEB);
  
  item = glade_xml_get_widget(gui, "email"); 
  g_signal_connect(G_OBJECT(item), "toggled", 
		   G_CALLBACK(on_column_toggled), RUBRICA_GCONF_EMAIL);
  rubrica_preferences_setup_menu_check(item, RUBRICA_GCONF_EMAIL);
  
  item = glade_xml_get_widget(gui, "phone"); 
  g_signal_connect(G_OBJECT(item), "toggled", 
		   G_CALLBACK(on_column_toggled), RUBRICA_GCONF_TELEPHONE);
  rubrica_preferences_setup_menu_check(item, RUBRICA_GCONF_TELEPHONE);
  
  item  = glade_xml_get_widget(gui, "company"); 
  g_signal_connect(G_OBJECT(item), "toggled", 
		   G_CALLBACK(on_column_toggled), RUBRICA_GCONF_COMPANY);
  rubrica_preferences_setup_menu_check(item, RUBRICA_GCONF_COMPANY);

  item = glade_xml_get_widget(gui, "role"); 
  g_signal_connect(G_OBJECT(item), "toggled", 
		   G_CALLBACK(on_column_toggled), RUBRICA_GCONF_ASSIGNMENT);
  rubrica_preferences_setup_menu_check(item, RUBRICA_GCONF_ASSIGNMENT);
  

  /* submenu Search on
   */    
  item = glade_xml_get_widget(gui, "rm_card");
  g_object_set_data(G_OBJECT(item), "name", "card");
  g_signal_connect(G_OBJECT(item), "toggled", 
		   G_CALLBACK(rubrica_preferences_set_search_on), 
		   GINT_TO_POINTER(CARD_COLUMN));
  
  /* get the radio group of the "search on" buttons's */
  radio_group = gtk_radio_menu_item_get_group(GTK_RADIO_MENU_ITEM(item));

  item = glade_xml_get_widget(gui, "rm_first");
  g_object_set_data(G_OBJECT(item), "name", "first name");
  g_signal_connect(G_OBJECT(item), "toggled", 
		   G_CALLBACK(rubrica_preferences_set_search_on), 
		   GINT_TO_POINTER(FIRST_NAME_COLUMN));
  
  item = glade_xml_get_widget(gui, "rm_last");
  g_object_set_data(G_OBJECT(item), "name", "last name");
  g_signal_connect(G_OBJECT(item), "toggled", 
		   G_CALLBACK(rubrica_preferences_set_search_on), 
		   GINT_TO_POINTER(LAST_NAME_COLUMN));
  
  item = glade_xml_get_widget(gui, "rm_prof");
  g_object_set_data(G_OBJECT(item), "name", "profession");
  g_signal_connect(G_OBJECT(item), "toggled", 
		   G_CALLBACK(rubrica_preferences_set_search_on), 
		   GINT_TO_POINTER(PROFESSION_COLUMN));
  
  item = glade_xml_get_widget(gui, "rm_city");
  g_object_set_data(G_OBJECT(item), "name", "city");
  g_signal_connect(G_OBJECT(item), "toggled", 
		   G_CALLBACK(rubrica_preferences_set_search_on), 
		   GINT_TO_POINTER(CITY_COLUMN));
  
  item = glade_xml_get_widget(gui, "rm_country");
  g_object_set_data(G_OBJECT(item), "name", "country");
  g_signal_connect(G_OBJECT(item), "toggled", 
		   G_CALLBACK(rubrica_preferences_set_search_on),
		   GINT_TO_POINTER(COUNTRY_COLUMN));
  
  item = glade_xml_get_widget(gui, "rm_web");
  g_object_set_data(G_OBJECT(item), "name", "web");
  g_signal_connect(G_OBJECT(item), "toggled", 
		   G_CALLBACK(rubrica_preferences_set_search_on), 
		   GINT_TO_POINTER(WEB_CARD_COLUMN));
  
  item = glade_xml_get_widget(gui, "rm_email");
  g_object_set_data(G_OBJECT(item), "name", "emails");
  g_signal_connect(G_OBJECT(item), "toggled", 
		   G_CALLBACK(rubrica_preferences_set_search_on), 
		   GINT_TO_POINTER(EMAIL_CARD_COLUMN));
  
  item = glade_xml_get_widget(gui, "rm_tel");
  g_object_set_data(G_OBJECT(item), "name", "telephone");
  g_signal_connect(G_OBJECT(item), "toggled", 
		   G_CALLBACK(rubrica_preferences_set_search_on), 
		   GINT_TO_POINTER(TELEPHONE_COLUMN));
  
  item = glade_xml_get_widget(gui, "rm_role");
  g_object_set_data(G_OBJECT(item), "name", "assignment");
  g_signal_connect(G_OBJECT(item), "toggled", 
		   G_CALLBACK(rubrica_preferences_set_search_on), 
		   GINT_TO_POINTER(ASSIGNMENT_COLUMN));
  
  item = glade_xml_get_widget(gui, "rm_company");
  g_object_set_data(G_OBJECT(item), "name", "company");
  g_signal_connect(G_OBJECT(item), "toggled", 
		   G_CALLBACK(rubrica_preferences_set_search_on),
		   GINT_TO_POINTER(COMPANY_COLUMN));
  
  /* setup radio group with preferences */
  rubrica_preferences_setup_radio_group(radio_group);
  

  /* Advanced search (call search gui) */
  item = glade_xml_get_widget(gui, "advanced");
  g_signal_connect(G_OBJECT(item), "activate", 
		   G_CALLBACK(on_advanced_activate), app);
  

  /* Submenu Show cards 
   */
  item   = glade_xml_get_widget(gui, "show_all");
  popup_item_set_image(item, "all");
  g_signal_connect(G_OBJECT(item), "activate",  
		   G_CALLBACK(on_show_all_activate), app);  

  item = glade_xml_get_widget(gui, "show_men");
  //  popup_item_set_image(item, "man");
  g_signal_connect(G_OBJECT(item), "activate", 
		   G_CALLBACK(on_show_men_activate), app);  

  item = glade_xml_get_widget(gui, "show_women");
  //  popup_item_set_image(item, "woman");
  g_signal_connect(G_OBJECT(item), "activate",  
		   G_CALLBACK(on_show_women_activate), app);   

  item = glade_xml_get_widget(gui, "show_personal");
  popup_item_set_image(item, "contacts");
  g_signal_connect(G_OBJECT(item), "activate", 
		   G_CALLBACK(on_show_personal_activate), app);  

  item = glade_xml_get_widget(gui, "show_company");
  popup_item_set_image(item, "puzzle");
  g_signal_connect(G_OBJECT(item), "activate", 
		   G_CALLBACK(on_show_company_activate), app);  


  /* Submenu Show by rate 
   */  
  item = glade_xml_get_widget(gui, "rate1");
  popup_item_set_image(item, "rate1");
  g_signal_connect(G_OBJECT(item), "activate", 
		   G_CALLBACK(on_show_by_rate1_ativate), app);
  
  item = glade_xml_get_widget(gui, "rate2");
  popup_item_set_image(item, "rate2");
  g_signal_connect(G_OBJECT(item), "activate", 
		   G_CALLBACK(on_show_by_rate2_ativate), app);
  
  item = glade_xml_get_widget(gui, "rate3");
  popup_item_set_image(item, "rate3");
  g_signal_connect(G_OBJECT(item), "activate", 
		   G_CALLBACK(on_show_by_rate3_ativate), app);
  
  item = glade_xml_get_widget(gui, "rate4");
  popup_item_set_image(item, "rate4");
  g_signal_connect(G_OBJECT(item), "activate", 
		   G_CALLBACK(on_show_by_rate4_ativate), app);
  
  item = glade_xml_get_widget(gui, "rate5");
  popup_item_set_image(item, "rate5");
  g_signal_connect(G_OBJECT(item), "activate", 
		   G_CALLBACK(on_show_by_rate5_ativate), app);

 
  /*  item = glade_xml_get_widget(gui, "");
      popup_item_set_image(item, "");
      g_signal_connect(G_OBJECT(), "", G_CALLBACK(), );
  */

  if (marked)
    gtk_widget_set_sensitive(unmark, TRUE);
  else
    gtk_widget_set_sensitive(unmark, FALSE);

  if (!rubrica_app_get_stack_len(app))
    gtk_widget_set_sensitive(paste, FALSE); 

  if (!gtk_tree_view_get_path_at_pos(tree, ev->x, ev->y, &path, NULL,
				     NULL, NULL))
    {
      gtk_widget_set_sensitive(cut, FALSE);
      gtk_widget_set_sensitive(copy, FALSE);
      gtk_widget_set_sensitive(modify, FALSE);
      gtk_widget_set_sensitive(delete, FALSE);
    }
  
  return menu;
}


GtkMenu* 
rubrica_menu_do_trash_popup(RubricaApp* app, GtkTreeView* tree, 
			    GdkEvent *event)
{  
  GladeXML* gui;
  GtkTreeModel *model;
  GtkTreeSelection *selection; 
  GtkMenu* menu;  

  GtkWidget* recovery;
  GtkWidget* destroy;
  GtkWidget* empty;

  gui       = glade_xml_new (RUBRICA_GUI_DIR"/MenuTrash.glade", NULL, NULL);
  model     = gtk_tree_view_get_model(GTK_TREE_VIEW(tree));
  selection = gtk_tree_view_get_selection(tree);

  menu = (GtkMenu*) glade_xml_get_widget(gui, "menu");  

  recovery = glade_xml_get_widget(gui, "recovery_card");
  g_signal_connect(G_OBJECT(recovery), "activate", 
		   G_CALLBACK(on_recovery_card_activate), app);
  
  destroy = glade_xml_get_widget(gui, "destroy_card");
  g_signal_connect(G_OBJECT(destroy), "activate", 
		   G_CALLBACK(on_destroy_card_activate), app);
  
  empty = glade_xml_get_widget(gui, "empty_trash");
  g_signal_connect(G_OBJECT(empty), "activate", 
		 G_CALLBACK(on_empty_trash_activate), app);


  return menu;
}
