/*
*  Program: Rubrica
*  file: preferences.c
*
*  
*  Copyright (C) 2000-2005 Nicola Fragale <nicolafragale@libero.it>
*
*  This program is free software; you can redistribute it and/or modify
*  it under the terms of the GNU General Public License as published by
*  the Free Software Foundation; either version 3 of the License
*
*  This program is distributed in the hope that it will be useful,
*  but WITHOUT ANY WARRANTY; without even the implied warranty of
*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*  GNU General Public License for more details.
*
*  You should have received a copy of the GNU General Public License
*  along with this program; if not, write to the Free Software
*  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#include <glib/gi18n-lib.h>
#include <gtk/gtk.h>
#include <glade/glade.h>
#include <gconf/gconf-client.h>
#include "libral.h"

#include "app.h"
#include "search.h"
#include "models.h"
#include "preferences.h"
#include "themes.h"
#include "types.h"


enum {
  FILES_COLUMN,
  TYPE_COLUMN,
  FILES_LAST_COLUMN
};


static GConfEnumStringPair toolbar_pairs[] = {
  {RUB_ICONS, "only icons"         },
  {RUB_TEXT,  "only text"          },
  {RUB_BOTH,  "both icons and text"},
  {RUB_GNOME, "gnome default"      },
  {0, NULL}  
};

static GConfEnumStringPair search_pairs[] = {
  {PRINT_COLUMN,      "print"     },
  {CARD_COLUMN,       "card"      },
  {FIRST_NAME_COLUMN, "first name"},
  {LAST_NAME_COLUMN,  "last name" },
  {PROFESSION_COLUMN, "profession"},
  {CITY_COLUMN,       "city"      },
  {COUNTRY_COLUMN,    "country"   },
  {WEB_CARD_COLUMN,   "web"       },
  {EMAIL_CARD_COLUMN, "emails"    },
  {TELEPHONE_COLUMN,  "telephone" },
  {ASSIGNMENT_COLUMN, "assignment"},
  {COMPANY_COLUMN,    "company"   },
  {0, NULL}  
};


static GConfEnumStringPair browser_pairs[] = {
  {RUB_EPIPHANY,      "epiphany"  },
  {RUB_FIREFOX,       "firefox"   },
  {RUB_GALEON,        "galeon"    },
  {RUB_KONQUEROR,     "konqueror" },
  {RUB_MOZILLA,       "mozilla"   },
  {RUB_SEAMONKEY,     "seamonkey" },
  {RUB_OPERA,         "opera"     },
  {RUB_OTHER_BROWSER, "other"     },
  {0, NULL}  
};

static GConfEnumStringPair emailer_pairs[] = {
  {RUB_BALSA,          "balsa"      },
  {RUB_EVOLUTION,      "evolution"  },
  {RUB_MOZILLA_MAIL,   "mozilla"    },
  {RUB_SEAMONKEY_MAIL, "seamonkey"  },  // seamonkey -mail
  {RUB_THUNDERBIRD,    "thunderbird"},
  {RUB_OTHER_EMAILER,  "other"      },
  {0, NULL}  
};

static GConfEnumStringPair irc_pairs[] = {
  {RUB_PIDGIN,    "pidgin"    },
  {RUB_GAIM,      "gaim"    },
  {RUB_GABBER,    "gabber"  },
  {RUB_GAJIM,     "gajim"   },
  {RUB_GADU,      "GNU gadu"},
  {RUB_GOSSIP,    "gossip"  },
  {RUB_OTHER_IRC, "other"   },
  {0, NULL}  
};
 


static RubricaThemedItem  toolbar[] = {
  {N_("Only icons"),          "", NULL},
  {N_("Only text"),           "", NULL},
  {N_("Both icons and text"), "", NULL},
  {N_("Icons and text alongside each other"), "", NULL},
  {N_("Use Gnome settings"),  "", NULL},
  {NULL}
};


static RubricaThemedItem browser[] = {
  {"Epiphany",  "epiphany",       "epiphany"    },   
  {"Firefox",   "firefox",        "firefox"     },
  {"Galeon",    "galeon",         "galeon"      },      
  {"Konqueror", "konqueror",      "konqueror"   },
  {"Mozilla",   "mozilla-icon",   "mozilla"     },
  {"Seamonkey", "seamonkey-icon", "seamonkey"   },
  {"Opera",     "opera",          "opera"       },
  {N_("Other"), "application-x-executable", NULL},
  {NULL}
};

static RubricaThemedItem emailer[] = {
  {"Balsa",          "balsa_icon",               "balsa"      },
  {"Evolution",      "evolution",                "evolution"  },
  {"Mozilla mail",   "mozilla_mail",             "mozilla"    },
  {"Seamonkey mail", "seamonkey-mail-icon",      "seamonkey"  },
  {"Thunderbird",    "thunderbird",              "thunderbird"},
  {N_("Other"),      "application-x-executable", NULL},
  {NULL}
};

static RubricaThemedItem ircs[] = {
  {"Pidgin",    "pidgin",  "pidgin"},
  {"Empathy",   "empathy", "empathy"},
  {"Gabber",    "gabber",  "gabber"},
  /*
  {"Gajim",     "application-x-executable", ""}, 
  {"GNU Gadu",  "application-x-executable", ""}, 
  {"Gossip",    "application-x-executable", ""},
  */
  {N_("Other"), "application-x-executable", NULL},
  {NULL}
};


typedef struct _prefer {
  GtkWidget* dialog;            

  GtkWidget* folder_chooser;

  GtkWidget* file_chooser;
  GtkWidget* load_check;
  GtkWidget* load_vbox;
  GtkWidget* load_view;
  GtkWidget* add;
  GtkWidget* del;

  GtkWidget* compress;
  GtkWidget* autosave; 
  GtkWidget* backup;

  GtkWidget* toolbar_combo;
  GtkWidget* font_button;
  GtkWidget* exit;

  GtkWidget* web_combo;
  GtkWidget* email_combo;
  GtkWidget* irc_combo;
  GtkWidget* ekiga;

  GtkWidget* web_box;
  GtkWidget* email_box;
  GtkWidget* irc_box;
  GtkWidget* web_entry;
  GtkWidget* email_entry;
  GtkWidget* irc_entry;  
} RubricaPrefer;



void on_load_check_cb           (GtkWidget* check, gpointer data);
void on_load_view_event         (GtkTreeView* tree, GdkEvent *event, 
				 gpointer data);
void on_folder_chooser_cb       (GtkFileChooser *chooser, gpointer data);
void on_remove_addressbook      (GtkWidget* button, gpointer data);
void on_add_cb                  (GtkWidget* button, gpointer data);

void on_compress_cb             (GtkWidget* spin,  gpointer data);
void on_autosave_cb             (GtkWidget* check, gpointer data); 
void on_backup_cb               (GtkWidget* check, gpointer data); 
void on_exit_cb                 (GtkWidget* check, gpointer data);

void on_toolbar_style_cb        (GtkWidget* combo, gpointer data);
void on_font_set_cb             (GtkFontButton *widget, gpointer data);
// void on_search_changed_cb       (GtkWidget* combo, gpointer data);
void on_browser_changed_cb      (GtkWidget* combo, gpointer data);
void on_emailer_changed_cb      (GtkWidget* combo, gpointer data);
void on_irc_changed_cb          (GtkWidget* combo, gpointer data);

void on_web_entry_activate_cb   (GtkWidget* combo, gpointer data);
void on_email_entry_activate_cb (GtkWidget* combo, gpointer data);
void on_irc_entry_activate_cb   (GtkWidget* combo, gpointer data);

void on_ekiga_cb                (GtkWidget* check, gpointer data);


void rubrica_preferences_change_toolbar (GConfClient *client, guint cnxn_id, 
					 GConfEntry *entry, gpointer data);
void rubrica_preferences_show_column    (GConfClient *client, guint cnxn_id, 
					 GConfEntry *entry, gpointer data);

void rubrica_preferences_font_changed   (GConfClient *client, guint cnxn_id, 
					 GConfEntry *entry, gpointer data);

void rubrica_preferences_gui_setup     (RubricaPrefer* prefer, gpointer data);
void rubrica_preferences_set_status    (RubricaPrefer* prefer, 
					const gchar* key);

void
on_load_check_cb(GtkWidget* check, gpointer data)
{
  GConfClient* client;
  RubricaPrefer* prefer = (RubricaPrefer*) data;
  gboolean bool;

  client = gconf_client_get_default();
  bool   = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON(check));

  gtk_widget_set_sensitive(prefer->load_vbox, bool);  
  gconf_client_set_bool(client, RUBRICA_GCONF_LOAD_FILES, bool, NULL);

  g_object_unref(client);
}


void
on_load_view_event(GtkTreeView* tree, GdkEvent *event, gpointer data)
{
  GdkEventButton *ev  = (GdkEventButton *) event;
  GtkTreeModel *model; 
  GtkTreeSelection *selection;  
  GtkTreePath *path = NULL; 
  GtkTreeIter iter; 

  model      = gtk_tree_view_get_model(GTK_TREE_VIEW(tree)); 
  selection  = gtk_tree_view_get_selection(tree); 
 
  if (gtk_tree_view_get_path_at_pos(tree, ev->x, ev->y, &path,
				    NULL, NULL, NULL))
    {
      gtk_tree_selection_select_path(selection, path);
      gtk_tree_model_get_iter(model, &iter, path);
      gtk_tree_path_free(path);

      if ((ev->type == GDK_BUTTON_PRESS) && (ev->button == 3))
	{
	  GtkMenu* menu;
	  GtkWidget *remove;
	  
	  menu = (GtkMenu*) gtk_menu_new();
	  
	  remove = gtk_menu_item_new_with_mnemonic(_("_remove the addressbook"));
	  gtk_widget_show (remove);
	  gtk_container_add (GTK_CONTAINER (menu), remove);
	  
	  g_signal_connect (G_OBJECT(remove), "activate",
			    G_CALLBACK (on_remove_addressbook), tree);
	  
	  gtk_widget_show_all(GTK_WIDGET(menu));
	  gtk_menu_popup (menu, NULL, NULL, NULL, NULL, ev->button, ev->time);
	}
    }
}



void 
on_remove_addressbook(GtkWidget* button, gpointer data)
{
  GtkTreeView* tree = (GtkTreeView*) data;  
  GtkTreeModel *model;
  GtkTreeSelection *selection; 
  GtkTreeIter iter;
  GConfClient *client;

  model      = gtk_tree_view_get_model(GTK_TREE_VIEW(tree));
  selection  = gtk_tree_view_get_selection(tree);  
  client     = gconf_client_get_default();

  if (gtk_tree_selection_get_selected(selection, &model, &iter))    
    {
      gchar* file = NULL;
      GSList* files = NULL;
      GSList* l = NULL;
      
      gtk_tree_model_get(model, &iter, FILES_COLUMN, &file, -1);
      gtk_list_store_remove(GTK_LIST_STORE(model), &iter);  
      
      files = gconf_client_get_list(client, RUBRICA_GCONF_FILES, 
				    GCONF_VALUE_STRING, NULL);
      
      for(l = files; l; l = l->next)
	{
	  if (g_ascii_strcasecmp(file, (gchar*) l->data) == 0)
	    {
	      files = g_slist_remove_link(files, l);
	      g_free(file);
	      g_slist_free_1(l);
	      break;
	    }
	}
      
      gconf_client_set_list(client, RUBRICA_GCONF_FILES, GCONF_VALUE_STRING,
			    files, NULL);      
    }
  else
    {
      g_warning("\nNo addressbook selected");
    }
}


void 
on_folder_chooser_cb (GtkFileChooser *chooser, gpointer data)
{
  gchar* folder = NULL;
  GConfClient* client;
  
  client = gconf_client_get_default();
  folder = gtk_file_chooser_get_current_folder (chooser);

  if (folder)
    gconf_client_set_string(client, RUBRICA_GCONF_FOLDER, folder, NULL);
  
  g_object_unref(client);
}




void
on_add_cb (GtkWidget* button, gpointer data)
{
  gchar* file = NULL;
  GConfClient* client;
  GtkTreeModel* model;
  GtkTreeIter   iter;
  RubricaPrefer* prefer = (RubricaPrefer*) data;
  
  client = gconf_client_get_default();

  file = gtk_file_chooser_get_filename(GTK_FILE_CHOOSER(prefer->file_chooser));
  if (file)
    {
      GSList* files = NULL;
      
      files = gconf_client_get_list(client, RUBRICA_GCONF_FILES, 
				    GCONF_VALUE_STRING, NULL);
      files = g_slist_append(files, file);
      
      model = gtk_tree_view_get_model(GTK_TREE_VIEW(prefer->load_view));
      gtk_list_store_append(GTK_LIST_STORE(model), &iter);
      gtk_list_store_set(GTK_LIST_STORE(model), &iter, FILES_COLUMN, file, -1);

      gconf_client_set_list(client, RUBRICA_GCONF_FILES, GCONF_VALUE_STRING,
			    files, NULL);
    }  

  g_object_unref(client);
}

void 
on_del_cb(GtkWidget* button, gpointer data)
{
  RubricaPrefer* prefer = (RubricaPrefer*) data;

  on_remove_addressbook(NULL, GTK_TREE_VIEW(prefer->load_view));
}


void 
on_compress_cb (GtkWidget* spin, gpointer data)
{
  GConfClient* client;
  gint val;
  
  client = gconf_client_get_default();
  val = (gint) gtk_spin_button_get_value(GTK_SPIN_BUTTON(spin));
  gconf_client_set_int(client, RUBRICA_GCONF_RATE, val, NULL);

  g_object_unref(client);
}


void 
on_autosave_cb (GtkWidget* check, gpointer data)
{
  GConfClient* client;
  gboolean bool;

  client = gconf_client_get_default();
  bool   = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON(check));  
  gconf_client_set_bool(client, RUBRICA_GCONF_AUTOSAVE, bool, NULL);

  g_object_unref(client);
}


void 
on_backup_cb (GtkWidget* check, gpointer data)
{
  GConfClient* client;
  gboolean bool;

  client = gconf_client_get_default();
  bool   = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON(check));
  gconf_client_set_bool(client, RUBRICA_GCONF_BACKUP, bool, NULL);

  g_object_unref(client);
}


/* notificare 
void 
on_tooltips_cb (GtkWidget* check, gpointer data)
{
  GConfClient* client;
  gboolean bool;

  client = gconf_client_get_default();
  bool   = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON(check));
  gconf_client_set_bool(client, RUBRICA_GCONF_TOOLTIPS, bool, NULL);  

  g_object_unref(client);
}
*/

void 
on_exit_cb (GtkWidget* check, gpointer data)
{
  GConfClient* client;
  gboolean bool;

  client = gconf_client_get_default();
  bool   = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON(check));
  gconf_client_set_bool(client, RUBRICA_GCONF_CONFIRM_EXIT, bool, NULL);  

  g_object_unref(client);
}


void on_ekiga_cb (GtkWidget* check, gpointer data)
{
  GConfClient* client;
  gboolean bool;

  client = gconf_client_get_default();
  bool   = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON(check));
  gconf_client_set_bool(client, RUBRICA_GCONF_EKIGA, bool, NULL);  

  g_object_unref(client);
}


void 
on_font_set_cb (GtkFontButton *button, gpointer data)
{
  const gchar* font = NULL;
  GConfClient* client;
  
  client = gconf_client_get_default();
  font =  gtk_font_button_get_font_name(GTK_FONT_BUTTON(button));

  if (font)
    gconf_client_set_string(client, RUBRICA_GCONF_FONT, font, NULL);
  
  g_object_unref(client);
  
}


void 
on_toolbar_style_cb (GtkWidget* combo, gpointer data)
{
  GConfClient* client;
  gint active;
  const gchar* val;

  client = gconf_client_get_default();
  active = gtk_combo_box_get_active(GTK_COMBO_BOX(combo));
  val    = gconf_enum_to_string (toolbar_pairs, active);
  gconf_client_set_string(client, RUBRICA_GCONF_TOOLBAR, val, NULL);

  g_object_unref(client);
}



void 
on_browser_changed_cb (GtkWidget* combo, gpointer data)
{
  RubricaPrefer* prefer = (RubricaPrefer*) data;
  GConfClient* client;
  gint active;
  const gchar* val;

  client = gconf_client_get_default();
  active = gtk_combo_box_get_active(GTK_COMBO_BOX(combo));
  val    = gconf_enum_to_string (browser_pairs, active);
  gconf_client_set_string(client, RUBRICA_GCONF_BROWSER, val, NULL);

  if (active == RUB_OTHER_BROWSER)
    gtk_widget_set_sensitive(prefer->web_box, TRUE);
  else
    gtk_widget_set_sensitive(prefer->web_box, FALSE);  

  g_object_unref(client);
}


void 
on_emailer_changed_cb (GtkWidget* combo, gpointer data)
{
  RubricaPrefer* prefer = (RubricaPrefer*) data;
  GConfClient* client;
  gint active;
  const gchar* val;

  client = gconf_client_get_default();
  active = gtk_combo_box_get_active(GTK_COMBO_BOX(combo));
  val    = gconf_enum_to_string (emailer_pairs, active);
  gconf_client_set_string(client, RUBRICA_GCONF_EMAILS, val, NULL);

  if (active == RUB_OTHER_EMAILER)
    gtk_widget_set_sensitive(prefer->email_box, TRUE);
  else
    gtk_widget_set_sensitive(prefer->email_box, FALSE);

  g_object_unref(client);
}


void 
on_irc_changed_cb (GtkWidget* combo, gpointer data)
{
  RubricaPrefer* prefer = (RubricaPrefer*) data;
  GConfClient* client;
  gint active;
  const gchar* val;

  client = gconf_client_get_default();
  active = gtk_combo_box_get_active(GTK_COMBO_BOX(combo));
  val    = gconf_enum_to_string (irc_pairs, active);
  gconf_client_set_string(client, RUBRICA_GCONF_IRC, val, NULL);

  if (active == RUB_OTHER_IRC)
    gtk_widget_set_sensitive(prefer->irc_box, TRUE);
  else
    gtk_widget_set_sensitive(prefer->irc_box, FALSE);

  g_object_unref(client);
}


void
on_web_entry_activate_cb   (GtkWidget* combo, gpointer data)
{
  RubricaPrefer* prefer = (RubricaPrefer*) data;
  GConfClient* client;
  const gchar* str;
  
  client = gconf_client_get_default();
  str = gtk_entry_get_text(GTK_ENTRY(prefer->web_entry));
  gconf_client_set_string(client, RUBRICA_GCONF_BROWSER_UD, str, NULL);

  g_object_unref(client);
}

void 
on_email_entry_activate_cb (GtkWidget* combo, gpointer data)
{
  RubricaPrefer* prefer = (RubricaPrefer*) data;
  GConfClient* client;
  const gchar* str;

  client = gconf_client_get_default();  
  str = gtk_entry_get_text(GTK_ENTRY(prefer->email_entry));
  gconf_client_set_string(client, RUBRICA_GCONF_EMAILS_UD, str, NULL);
  
  g_object_unref(client);
}


void 
on_irc_entry_activate_cb   (GtkWidget* combo, gpointer data)
{
  RubricaPrefer* prefer = (RubricaPrefer*) data;
  GConfClient* client;
  const gchar* str;

  client = gconf_client_get_default();
  str = gtk_entry_get_text(GTK_ENTRY(prefer->irc_entry));
  gconf_client_set_string(client, RUBRICA_GCONF_IRC_UD, str, NULL);

  g_object_unref(client);
}



void
rubrica_preferences_set_status(RubricaPrefer* prefer, const gchar* key)
{
  GConfClient* client;
  gchar* str;
  gint value;

  client = gconf_client_get_default();

  if (g_ascii_strcasecmp(key, RUBRICA_GCONF_TOOLBAR) == 0)
    {
      str = gconf_client_get_string(client, RUBRICA_GCONF_TOOLBAR, NULL);
      gconf_string_to_enum(toolbar_pairs, str, &value);
      gtk_combo_box_set_active(GTK_COMBO_BOX(prefer->toolbar_combo), value);   
    }

  if (g_ascii_strcasecmp(key, RUBRICA_GCONF_BROWSER) == 0) 
    {
      str = gconf_client_get_string(client, RUBRICA_GCONF_BROWSER, NULL);
      gconf_string_to_enum(browser_pairs, str, &value);
      gtk_combo_box_set_active(GTK_COMBO_BOX(prefer->web_combo), value);

      if (value == RUB_OTHER_BROWSER)
	{
	  str = gconf_client_get_string(client, RUBRICA_GCONF_BROWSER_UD, 
					NULL);
	  gtk_entry_set_text(GTK_ENTRY(prefer->web_entry), str);
	  gtk_widget_set_sensitive(prefer->web_box, TRUE);
	}
      else
	gtk_widget_set_sensitive(prefer->web_box, FALSE);	
    }

  if (g_ascii_strcasecmp(key, RUBRICA_GCONF_EMAILS) == 0) 
    {
      str = gconf_client_get_string(client, RUBRICA_GCONF_EMAILS, NULL);
      gconf_string_to_enum(emailer_pairs, str, &value);
      gtk_combo_box_set_active(GTK_COMBO_BOX(prefer->email_combo), value);   

      if (value == RUB_OTHER_EMAILER)
	{
	  str = gconf_client_get_string(client, RUBRICA_GCONF_EMAILS_UD, NULL);
	  gtk_entry_set_text(GTK_ENTRY(prefer->email_entry), str);
	  gtk_widget_set_sensitive(prefer->email_box, TRUE);
	}
      else
	gtk_widget_set_sensitive(prefer->email_box, FALSE);	
    }

  if (g_ascii_strcasecmp(key, RUBRICA_GCONF_IRC) == 0) 
    {
      str = gconf_client_get_string(client, RUBRICA_GCONF_IRC, NULL);
      gconf_string_to_enum(irc_pairs, str, &value);
      gtk_combo_box_set_active(GTK_COMBO_BOX(prefer->irc_combo), value);   

      if (value == RUB_OTHER_IRC)
	{
	  str = gconf_client_get_string(client, RUBRICA_GCONF_IRC_UD, NULL);
	  gtk_entry_set_text(GTK_ENTRY(prefer->irc_entry), str);
	  gtk_widget_set_sensitive(prefer->irc_box, TRUE);
	}
      else
	gtk_widget_set_sensitive(prefer->irc_box, FALSE);	
    }

  g_object_unref(client);
}


void 
rubrica_preferences_change_toolbar(GConfClient *client, guint cnxn_id, 
				   GConfEntry *entry, gpointer data)
{
  RubricaApp* app = (RubricaApp*) data;
  gchar* str;
  gint style;
  
  str = gconf_client_get_string(client, RUBRICA_GCONF_TOOLBAR, NULL);
  gconf_string_to_enum(toolbar_pairs, str, &style);      
  rubrica_app_set_toolbar_style(app, style);
}


void 
rubrica_preferences_show_column(GConfClient *client, guint cnxn_id, 
				GConfEntry *entry, gpointer data)
{
  gboolean bool;  
  GtkTreeViewColumn* column = (GtkTreeViewColumn*) data;
  
  bool = gconf_value_get_bool (entry->value);
  gtk_tree_view_column_set_visible(column, bool);
}

void 
rubrica_preferences_font_changed (GConfClient *client, guint cnxn_id, 
				  GConfEntry *entry, gpointer data)
{
  RubricaApp* app = (RubricaApp*) data;
  gchar* str;
  
  str = gconf_client_get_string(client, RUBRICA_GCONF_FONT, NULL);
  rubrica_app_set_font(app, str);  
}



void 
rubrica_preferences_gui_setup(RubricaPrefer* prefer, gpointer data)
{
  RubricaApp* app = (RubricaApp*) data;
  GConfClient* client;
  GtkTreeModel* model;
  GtkTreeIter  iter;
  GtkTreeSelection* selection;
  GtkCellRenderer *renderer;
  GSList* files = NULL;
  gboolean bool;
  gint rate;
  gchar* folder;
  gchar* font = NULL;

  client = gconf_client_get_default();
  
  /* Addressbook page */
  /* addressbooks files list view */
  model  = (GtkTreeModel*) gtk_list_store_new(FILES_LAST_COLUMN,
					      G_TYPE_STRING,
					      G_TYPE_STRING);
  gtk_tree_sortable_set_sort_column_id(GTK_TREE_SORTABLE(model),
				       FILES_COLUMN, GTK_SORT_ASCENDING);
  gtk_tree_view_set_model(GTK_TREE_VIEW(prefer->load_view), model); 
  selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(prefer->load_view));
  renderer  = gtk_cell_renderer_text_new();
  gtk_tree_view_insert_column_with_attributes(GTK_TREE_VIEW(prefer->load_view),
					      -1, _("Addressbook"), 
					      renderer, "text", 
					      FILES_COLUMN, NULL);
  
  renderer  = gtk_cell_renderer_text_new();
  gtk_tree_view_insert_column_with_attributes(GTK_TREE_VIEW(prefer->load_view),
					      -1, _("Type"), 
					      renderer, "text", 
					      TYPE_COLUMN, NULL);
  
  /* default addressbooks's folder */
  folder = gconf_client_get_string(client, RUBRICA_GCONF_FOLDER, NULL);
  gtk_file_chooser_set_current_folder(GTK_FILE_CHOOSER(prefer->folder_chooser),
				      folder);

  /* addressbooks to load at startup */
  bool = gconf_client_get_bool(client, RUBRICA_GCONF_LOAD_FILES, NULL);  
  gtk_widget_set_sensitive(prefer->load_vbox, bool);  
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(prefer->load_check), bool);

  /* addressbooks list */
  files  = gconf_client_get_list(client, RUBRICA_GCONF_FILES, 
				 GCONF_VALUE_STRING, NULL);
  for (; files; files = files->next) 
    {
      gtk_list_store_append(GTK_LIST_STORE(model), &iter);
      gtk_list_store_set(GTK_LIST_STORE(model), &iter, 
			 FILES_COLUMN, (gchar*) files->data,
			 -1);
    }
  
  /* combo buttons */
  rubrica_themes_create_themed_list(GTK_COMBO_BOX(prefer->toolbar_combo), 
				    toolbar);
  rubrica_themes_create_themed_list(GTK_COMBO_BOX(prefer->web_combo), browser);
  rubrica_themes_create_themed_list(GTK_COMBO_BOX(prefer->email_combo),
				    emailer);
  rubrica_themes_create_themed_list(GTK_COMBO_BOX(prefer->irc_combo), ircs);
  
  rubrica_preferences_set_status (prefer, RUBRICA_GCONF_TOOLBAR);
  rubrica_preferences_set_status (prefer, RUBRICA_GCONF_BROWSER);
  rubrica_preferences_set_status (prefer, RUBRICA_GCONF_EMAILS);
  rubrica_preferences_set_status (prefer, RUBRICA_GCONF_IRC);
  rubrica_preferences_set_status (prefer, RUBRICA_GCONF_SEARCH_ON);
  
  /* toolbar settings change */
  gconf_client_notify_add(client, RUBRICA_GCONF_TOOLBAR,
			  rubrica_preferences_change_toolbar,
			  app, NULL, NULL);

  /* font button */
  font = gconf_client_get_string(client, RUBRICA_GCONF_FONT, NULL);
  gtk_font_button_set_font_name (GTK_FONT_BUTTON(prefer->font_button), font);
  
  gconf_client_notify_add(client, RUBRICA_GCONF_FONT,
			  rubrica_preferences_font_changed, app, NULL, NULL);

  /* compression rate*/ 
  rate = gconf_client_get_int(client, RUBRICA_GCONF_RATE, NULL);
  gtk_spin_button_set_value(GTK_SPIN_BUTTON(prefer->compress), rate);

  /* autosave */
  bool = gconf_client_get_bool (client, RUBRICA_GCONF_AUTOSAVE, NULL);
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(prefer->autosave), bool);

  /* bakcup */
  bool = gconf_client_get_bool (client, RUBRICA_GCONF_BACKUP, NULL);
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(prefer->backup), bool);

  /* tooltips 
  bool = gconf_client_get_bool (client, RUBRICA_GCONF_TOOLTIPS, NULL);
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(prefer->tooltips), bool);
  */

  /* confirm exit */
  bool = gconf_client_get_bool (client, RUBRICA_GCONF_CONFIRM_EXIT, NULL);
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(prefer->exit), bool);

  /* ekiga */
  bool = gconf_client_get_bool (client, RUBRICA_GCONF_EKIGA, NULL);
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(prefer->ekiga), bool);
}






/* ************  Public
 */
gint 
rubrica_preferences_get_gnome_default(void)
{
  GConfClient* client;
  const gchar* val = NULL;
  gint ret = GTK_TOOLBAR_BOTH;

  client = gconf_client_get_default();
  val    = gconf_client_get_string(client, RUBRICA_GCONF_GNOME_TOOLBAR, NULL);
  
  if (g_ascii_strcasecmp(val, "icons") == 0)
    ret = GTK_TOOLBAR_ICONS;

  if (g_ascii_strcasecmp(val, "text") == 0)
    ret = GTK_TOOLBAR_TEXT;

  if (g_ascii_strcasecmp(val, "both") == 0)
    ret = GTK_TOOLBAR_BOTH;
  
  if (g_ascii_strcasecmp(val, "both-horiz") == 0)
    ret = GTK_TOOLBAR_BOTH_HORIZ;
  
  return ret;
}



void 
rubrica_preferences_set_search_on (GtkWidget* button, gpointer data)
{
  GConfClient* client;
  const gchar* val;  
  gint column = GPOINTER_TO_INT(data);
  
  client = gconf_client_get_default();
  val    = gconf_enum_to_string (search_pairs, column);

  gconf_client_set_string(client, RUBRICA_GCONF_SEARCH_ON, val, NULL);

  g_object_unref(client);     
}


void 
on_preferences_search_notify (GConfClient *client, guint cnxn_id, 
			      GConfEntry *entry, gpointer data)
{
  gint column;
  const gchar* value;
  GtkTreeView* view = (GtkTreeView*) data;  
  GtkTreeViewColumn* col;
  
  value  = gconf_value_get_string (entry->value);
  gconf_string_to_enum (search_pairs, value, &column);
   
  gtk_tree_view_set_search_column (view, column);
  col = gtk_tree_view_get_column(view, column);

  gtk_tree_view_scroll_to_cell(view, NULL, col, TRUE, .5, .5);
}


void
rubrica_preferences_setup_search_on(GtkTreeView* view)
{
  GConfClient* client;
  const gchar* val = NULL;   
  gint column;
  
  client = gconf_client_get_default();
  val    = gconf_client_get_string(client, RUBRICA_GCONF_SEARCH_ON, NULL);
  if (!val)
    {
      val = gconf_enum_to_string (search_pairs, CARD_COLUMN);
      gconf_client_set_string(client, RUBRICA_GCONF_SEARCH_ON, val, NULL);
      gtk_tree_view_set_search_column (view, CARD_COLUMN);
    }
  
  gconf_string_to_enum (search_pairs, val, &column);
  gtk_tree_view_set_search_column (view, column);
  
  gconf_client_notify_add(client, RUBRICA_GCONF_SEARCH_ON, 
			  on_preferences_search_notify,
			  view, NULL, NULL);
  
  g_object_unref(client);    
}


void 
rubrica_preferences_setup_radio_group (GSList* group)
{
  GConfClient* client;
  GtkWidget* radio;
  gchar* value;
  
  client = gconf_client_get_default();
  value  = gconf_client_get_string(client, RUBRICA_GCONF_SEARCH_ON, NULL);

  for (; group; group = g_slist_next(group))
    {
      gchar* name;

      radio = (GtkWidget*) group->data;
      name = g_object_get_data(G_OBJECT(radio), "name");

      if (g_ascii_strcasecmp(name, value) == 0)
	{
	  gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(radio), TRUE);
	  break;
	}
    }
  
  g_object_unref(client);  
}



void 
rubrica_preferences_setup_colum (GtkTreeViewColumn* column, gchar* key)
{
  GConfClient* client;
  gboolean bool;
  
  client = gconf_client_get_default();

  bool = gconf_client_get_bool(client, key, NULL);
  gtk_tree_view_column_set_visible(column, bool);

  gconf_client_notify_add(client, key, rubrica_preferences_show_column,
			  column, NULL, NULL);
  
  g_object_unref(client);  
}



void
rubrica_preferences_setup_menu_check(GtkWidget* item, gchar* key)
{
  GConfClient* client;
  gboolean bool;
  
  client = gconf_client_get_default();

  bool = gconf_client_get_bool(client, key, NULL);
  gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM(item), bool);

  g_object_unref(client);      
}

gint
rubrica_preferences_get_toolbar_style()
{
  gchar* str;
  gint style;
  GConfClient* client;

  client = gconf_client_get_default();
  str    = gconf_client_get_string(client, RUBRICA_GCONF_TOOLBAR, NULL);

  gconf_string_to_enum(toolbar_pairs, str, &style); 

  return style;
}


void
rubrica_preferences_run_gui(GtkWidget* button, gpointer data)
{
  RubricaApp* app = (RubricaApp*) data;
  GladeXML* gui;  
  GConfClient* client;
  RubricaPrefer* prefer;
  GtkWidget* chooser;
  gchar* folder = NULL;
  
  gui = glade_xml_new (RUBRICA_GUI_DIR"/Preferences.glade", NULL, NULL);  
  if (!gui)
    g_error("\nCan't load %s gui", RUBRICA_GUI_DIR"/Preferences.glade");
  
  prefer = g_malloc0(sizeof(RubricaPrefer));

  prefer->dialog = glade_xml_get_widget(gui, "preferences");

  prefer->folder_chooser = glade_xml_get_widget(gui, "folder_chooser");
  prefer->file_chooser   = glade_xml_get_widget(gui, "file_chooser");
  prefer->load_check     = glade_xml_get_widget(gui, "load_check");
  prefer->load_vbox      = glade_xml_get_widget(gui, "load_vbox");
  prefer->load_view      = glade_xml_get_widget(gui, "load_view");
  prefer->add            = glade_xml_get_widget(gui, "add");
  prefer->del            = glade_xml_get_widget(gui, "del");

  prefer->compress       = glade_xml_get_widget(gui, "compress");
  prefer->autosave       = glade_xml_get_widget(gui, "autosave");
  prefer->backup         = glade_xml_get_widget(gui, "backup");

  prefer->toolbar_combo  = glade_xml_get_widget(gui, "toolbar_combo");
  prefer->font_button    = glade_xml_get_widget(gui, "fontbutton");
  prefer->exit           = glade_xml_get_widget(gui, "exit");

  prefer->web_combo      = glade_xml_get_widget(gui, "web_combo");
  prefer->email_combo    = glade_xml_get_widget(gui, "email_combo");
  prefer->irc_combo      = glade_xml_get_widget(gui, "irc_combo");
  prefer->ekiga          = glade_xml_get_widget(gui, "ekiga");
  
  prefer->web_box        = glade_xml_get_widget(gui, "web_box");
  prefer->email_box      = glade_xml_get_widget(gui, "email_box");
  prefer->irc_box        = glade_xml_get_widget(gui, "irc_box");
  prefer->web_entry      = glade_xml_get_widget(gui, "web_entry");
  prefer->email_entry    = glade_xml_get_widget(gui, "email_entry");
  prefer->irc_entry      = glade_xml_get_widget(gui, "irc_entry");

  g_object_unref(gui);
 
  g_signal_connect(G_OBJECT (prefer->load_check) , "toggled",
		   G_CALLBACK(on_load_check_cb), prefer);

  g_signal_connect(G_OBJECT (prefer->add), "clicked", 
		   G_CALLBACK(on_add_cb), prefer);
  g_signal_connect(G_OBJECT (prefer->del), "clicked", 
		   G_CALLBACK(on_del_cb), prefer);

  g_signal_connect(G_OBJECT (prefer->compress), "value-changed", 
		   G_CALLBACK(on_compress_cb), prefer);
  g_signal_connect(G_OBJECT (prefer->autosave), "toggled",
		   G_CALLBACK(on_autosave_cb), prefer);
  g_signal_connect(G_OBJECT (prefer->backup), "toggled", 
		   G_CALLBACK(on_backup_cb), prefer);
  
  g_signal_connect(G_OBJECT (prefer->toolbar_combo), "changed", 
		   G_CALLBACK(on_toolbar_style_cb), prefer);
  g_signal_connect(G_OBJECT (prefer->font_button), "font-set", 
		   G_CALLBACK(on_font_set_cb), prefer);
  
  g_signal_connect(G_OBJECT (prefer->web_combo), "changed", 
		   G_CALLBACK(on_browser_changed_cb), prefer);
  g_signal_connect(G_OBJECT (prefer->email_combo), "changed", 
		   G_CALLBACK(on_emailer_changed_cb), prefer);
  g_signal_connect(G_OBJECT (prefer->irc_combo), "changed",
		   G_CALLBACK(on_irc_changed_cb), prefer);
  
  g_signal_connect(G_OBJECT (prefer->web_entry), "changed",
		   G_CALLBACK(on_web_entry_activate_cb), prefer);
  g_signal_connect(G_OBJECT (prefer->email_entry), "changed",
		   G_CALLBACK(on_email_entry_activate_cb), prefer);
  g_signal_connect(G_OBJECT (prefer->irc_entry), "changed",
		   G_CALLBACK(on_irc_entry_activate_cb), prefer);

  g_signal_connect(G_OBJECT (prefer->exit), "toggled", 
		   G_CALLBACK(on_exit_cb), prefer);

  g_signal_connect(G_OBJECT (prefer->ekiga), "toggled", 
		   G_CALLBACK(on_ekiga_cb), prefer);

  g_signal_connect(G_OBJECT (prefer->load_view), "button_press_event", 
		   G_CALLBACK(on_load_view_event), prefer);
  /*
  chooser = gtk_file_chooser_dialog_new("Open", 
					GTK_WINDOW(app->window),
					GTK_FILE_CHOOSER_ACTION_OPEN,
					GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
					GTK_STOCK_OPEN, GTK_RESPONSE_ACCEPT,
					NULL);
  */
  rubrica_preferences_gui_setup(prefer, app);

  gtk_dialog_run(GTK_DIALOG(prefer->dialog));

  client  = gconf_client_get_default();
  chooser = prefer->folder_chooser;
  folder  = gtk_file_chooser_get_current_folder (GTK_FILE_CHOOSER(chooser));

  if (folder)
    gconf_client_set_string(client, RUBRICA_GCONF_FOLDER, folder, NULL);
  
  g_object_unref(client);  

  gtk_widget_destroy(prefer->dialog);  
}



























