/*
 *  Program: Rubrica
 *  file: print.c
 *  
 *  Copyright (C) Nicola Fragale <nicolafragale@libero.it>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 3 of the License
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
#include <math.h>
#include <string.h>
#include <gtk/gtk.h>
#include <glib/gi18n-lib.h>
#include <libral.h>

#include "app.h"
#include "print.h"


#define HEADER_HEIGHT 20.0
#define HEADER_GAP 8.5


enum {
  PROP_0,
  PRINTING_ADDRESSBOOK,
  PRINTING_CARDS,
  PRINTING_FONT,
  PRINTING_FONT_SIZE,
};



struct _RubricaPrintPrivate { 
  RAbook* book;
  GList* ids;

  gchar* font;
  gdouble font_size;
  
  gint lines_per_page;
  gint chars_per_line;
  gint total_lines;
  gint total_pages;
  gchar** lines;

  gboolean dispose_has_run;
};


#define RUBRICA_PRINT_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE((o),   \
                                      RUBRICA_PRINT_TYPE,                \
                                      RubricaPrintPrivate))


static GObjectClass *parent_class = NULL;


static void rubrica_print_class_init   (RubricaPrintClass* klass);
static void rubrica_print_init         (RubricaPrint* obj);

static void rubrica_print_finalize     (RubricaPrint* self);
static void rubrica_print_dispose      (RubricaPrint* self);


static void rubrica_print_set_property (GObject* obj, guint property_id,
					const GValue* value, GParamSpec* spec);
static void rubrica_print_get_property (GObject* obj, guint property_id,
					GValue* value, GParamSpec* spec);

static void write_header  (RCard* card, GString* buffer);
static void write_groups  (RCard* card, GString* buffer);
static void write_address (RCard* card, GString* buffer);
static void write_work    (RCard* card, GString* buffer);
static void write_net     (RCard* card, GString* buffer);
static void write_phone   (RCard* card, GString* buffer);
static void write_notes   (RCard* card, GString* buffer);


static void begin_print   (GtkPrintOperation* operation, 
			   GtkPrintContext* context, gpointer data);
static void draw_page     (GtkPrintOperation* operation, 
			   GtkPrintContext* context, gint page_nr, 
			   gpointer data);
static void end_print     (GtkPrintOperation* operation, 
			   GtkPrintContext* context, gpointer data);




GType
rubrica_print_get_type()
{
  static GType print_type = 0;
  
  if (!print_type)
    {
      static const GTypeInfo print_info =
	{
	  sizeof(RubricaPrintClass),
	  NULL,
	  NULL,
	  (GClassInitFunc) rubrica_print_class_init,
	  NULL,
	  NULL,
	  sizeof(RubricaPrint),
	  0,
	  (GInstanceInitFunc) rubrica_print_init
	};

      print_type = g_type_register_static (G_TYPE_OBJECT, "RubricaPrint", 
					   &print_info, 0);
    }
  
  return print_type;
}


static void 
rubrica_print_dispose (RubricaPrint* self)
{
  g_return_if_fail(IS_RUBRICA_PRINT(self));
  
  if (self->priv->dispose_has_run)
    return;

  self->priv->dispose_has_run = TRUE;  
}


static void 
rubrica_print_finalize (RubricaPrint* self)
{
  g_return_if_fail(IS_RUBRICA_PRINT(self));
  
}


static void
rubrica_print_class_init(RubricaPrintClass* klass)
{
  GObjectClass *object_class = G_OBJECT_CLASS (klass);
  GParamSpec* pspec;
  
  parent_class = g_type_class_peek_parent (klass);

  object_class->finalize     = (GObjectFinalizeFunc) rubrica_print_finalize;
  object_class->dispose      = (GObjectFinalizeFunc) rubrica_print_dispose;

  object_class->set_property = (gpointer) rubrica_print_set_property;
  object_class->get_property = (gpointer) rubrica_print_get_property;

  g_type_class_add_private (klass, sizeof(RubricaPrintPrivate));

  /* class property 
   */
  pspec = g_param_spec_pointer("printing-addressbook",  
			       "printing addressbook",  
			       "the addressbook to print from",  
			       G_PARAM_READWRITE);
  g_object_class_install_property(object_class, PRINTING_ADDRESSBOOK, pspec);

  pspec = g_param_spec_pointer("printing-cards",  
			       "printing cards",  
			       "list of cards to print",  
			       G_PARAM_READWRITE);
  g_object_class_install_property(object_class, PRINTING_CARDS, pspec);


  /**
   * RubricaPrint:printing-font
   *
   * the font that will be used to print
   */
  pspec = g_param_spec_string("printing-font",  
			      "printing font",  
			      "the font that will be used to print",  
			      NULL,
			      G_PARAM_READWRITE);
  g_object_class_install_property(object_class, PRINTING_FONT, pspec);

  /**
   * RubricaPrint:printing-font
   *
   * the font that will be used to print
   */
  pspec = g_param_spec_int("printing-font-size",  
			   "printing font size",  
			   "the font's size",  
			   0,
			   1000,
			   12,
			   G_PARAM_READWRITE);
  g_object_class_install_property(object_class, PRINTING_FONT_SIZE, pspec);
}


static void
rubrica_print_init(RubricaPrint* self)
{
  RubricaPrintPrivate* priv;
  
  priv = RUBRICA_PRINT_GET_PRIVATE(self);  
  
  self->operation = NULL;
  priv->book      = NULL;
  priv->ids       = NULL;

  priv->font      = NULL;
  priv->font_size = 0L;

  priv->lines_per_page = 0;
  priv->total_lines    = 0;
  priv->total_pages    = 0;
  priv->lines          = NULL;

  priv->dispose_has_run = FALSE;
}


static void 
rubrica_print_set_property (GObject* obj, guint property_id,
			    const GValue* value, GParamSpec* spec)
{
  RubricaPrint* self = (RubricaPrint*) obj;
  RubricaPrintPrivate* priv = RUBRICA_PRINT_GET_PRIVATE(self);

 
    switch (property_id) 
    {
    case PRINTING_ADDRESSBOOK:
      priv->book = g_value_get_pointer(value);
      break;

    case PRINTING_CARDS:
      priv->ids = g_value_get_pointer(value);
      break;

    case PRINTING_FONT:
      if (priv->font)
	g_free(priv->font);
      priv->font = g_strdup(g_value_get_string(value));
      break;

    case PRINTING_FONT_SIZE:
      priv->font_size = g_value_get_int(value);
      break;

    default: 
      break; 
    } 
} 
 

static void 
rubrica_print_get_property (GObject* obj, guint property_id,
			    GValue* value, GParamSpec* spec)
{
  RubricaPrint* self = (RubricaPrint*) obj;
  RubricaPrintPrivate* priv = RUBRICA_PRINT_GET_PRIVATE(self);

  switch (property_id) 
    {    
    case PRINTING_ADDRESSBOOK:
      g_value_set_pointer(value, priv->book);
      break;
      
    case PRINTING_CARDS:
      g_value_set_pointer(value, priv->ids);
      break;
      
    case PRINTING_FONT:
      g_value_set_string(value, priv->font);
      break;
      
    case PRINTING_FONT_SIZE:
      g_value_set_int(value, priv->font_size);
      break;

    default:
      break;  
    }  
}



static void
write_header(RCard* card, GString* buffer)
{
  RContact* contact;
  gchar* card_name;
  gchar *first = NULL, *last = NULL, *middle = NULL;
  gchar *nick = NULL, *title = NULL, *prefix = NULL;
  gchar *prof = NULL;  

  contact = r_personal_card_get_contact(R_PERSONAL_CARD(card));

  g_object_get(card, "card-name", &card_name, NULL);
  g_object_get(contact, 
	       "first-name",  &first,  "last-name", &last, 
	       "middle-name", &middle, NULL);

  g_string_append_printf(buffer, "<b><u>%s</u></b>: %s\n ", 
			 _("Card"), card_name);

  if (first || middle || last)  
    g_string_append_printf(buffer, "\t%s: %s %s %s\n", _("Name"), 
			   last   ? last   : "", first  ? first  : "", 
			   middle ? middle : "");
  
  if(r_contact_check(contact, "nick-name", &nick))
    g_string_append_printf(buffer, "\t%s: %s\n", _("Nickname"), nick);
    
  if(r_contact_check(contact, "title", &title))
    g_string_append_printf(buffer, "\t%s: %s\n", _("Title"), title);
    
  if(r_contact_check(contact, "prefix", &prefix))
    g_string_append_printf(buffer, "\t%s: %s\n", _("Prefix"), prefix); 
    
  if(r_contact_check(contact, "profession", &prof))
    g_string_append_printf(buffer, "\t%s: %s\n", _("Profession"), prof);
}


static void
write_groups(RCard* card, GString* buffer)
{
  gpointer group = NULL;

  group = r_card_get_group(R_CARD(card));
  if (group)
    {
      g_string_append_printf(buffer, "\t<b>%s:</b>\n", _("Card's groups"));
      
      for(;group; group = r_card_get_next_group(R_CARD(card)))
	{
	  gchar *name = NULL; 
	  
	  g_object_get(R_GROUP(group), "group-name",  &name, NULL);
	  g_string_append_printf(buffer, "\t\t%s\n", name);
	}       
    }
}

 
static void
write_address(RCard* card, GString* buffer)
{
  RAddress* address;

  address = r_card_get_address(R_CARD(card));
  if (address)
    {
      g_string_append_printf(buffer, "\t<b>%s:</b>\n", _("Addresses"));
      
      for  (; address; address = r_card_get_next_address(R_CARD(card)))
	{
	  RAddressType type;
	  gchar *adtype, *street, *number, *zip;
	  gchar *city, *province, *state, *country;
	  
	  g_object_get(address, "address-type", &type, 
		       "street", &street, "street-number", &number, 
		       "zip", &zip, "city", &city,
		       "province", &province, "state", &state, 
		       "country", &country, NULL);

	  if ((street && (g_ascii_strcasecmp(street, "") != 0))   ||
	      (zip && (g_ascii_strcasecmp(zip, "") != 0))         ||
	      (city && (g_ascii_strcasecmp(city, "") != 0))       ||
	      (state && (g_ascii_strcasecmp(state, "") != 0))     ||
	      (country && (g_ascii_strcasecmp(country, "") != 0)))
	    {
	      adtype = r_address_lookup_enum2str(type);
	      g_string_append_printf(buffer, "\t\t<b>%s</b>\n", adtype);
	      
	      if (street && (g_ascii_strcasecmp(street, "") != 0))
		{
		  g_string_append_printf(buffer, "\t\t%s", street);
		  
		  if (number && (g_ascii_strcasecmp(number, "") != 0))
		    g_string_append_printf(buffer, ", %s\n", number);
		  else
		    g_string_append_printf(buffer, "\n");
		}
	      
	      if ((zip && (g_ascii_strcasecmp(zip, "") != 0))   ||
		  (city && (g_ascii_strcasecmp(city, "") != 0)) ||
		  (province && (g_ascii_strcasecmp(province, "") != 0)))
		{	
		  gboolean comma = FALSE;

		  g_string_append_printf(buffer, "\t\t");

		  if (zip && (g_ascii_strcasecmp(zip, "") != 0)) 
		    {
		      g_string_append_printf(buffer, "%s ", zip);
		      comma = TRUE;
		    }
		  
		  if (city && (g_ascii_strcasecmp(city, "") != 0))
		    {
		      g_string_append_printf(buffer, "%s", city);
		      comma = TRUE;
		    }
		  
		  if (province && (g_ascii_strcasecmp(province, "") != 0))
		    {
		      if (comma)
			g_string_append_printf(buffer, ", (%s) ", province);
		      else
			g_string_append_printf(buffer, "%s", province);
		    }

		  g_string_append_printf(buffer, "\n");
		}

	      if ((state && (g_ascii_strcasecmp(state, "") != 0))   ||
		  (country && (g_ascii_strcasecmp(country, "") != 0)))
		{
		  g_string_append_printf(buffer, "\t\t");

		  if (state && (g_ascii_strcasecmp(state, "") != 0))
		    g_string_append_printf(buffer, "%s ", state);
		  
		  if (country && (g_ascii_strcasecmp(country, "") != 0))
		    g_string_append_printf(buffer, "%s",country);
		
		  g_string_append_printf(buffer, "\n");
		}
	    }	  
	}
    }
}

 
static void
write_work(RCard* card, GString* buffer)
{
  RWork* work = NULL;
  gchar *assignment = NULL, *organization = NULL;
  gchar *department = NULL, *subdep = NULL, *manager = NULL;
  gchar *collaborator = NULL, *mphone = NULL, *cphone = NULL;
  
  work = r_personal_card_get_work(R_PERSONAL_CARD(card));
  if (work && r_work_have_data(work))
    {   
      g_string_append_printf(buffer, "\t<b>%s:</b>\n", _("Work"));
      
      if(r_work_check(work, "assignment", &assignment))
	g_string_append_printf(buffer, "\t\t%s: %s\n", 
			       _("Assignment"), assignment);

      if(r_work_check(work, "organization", &organization))
	g_string_append_printf(buffer, "\t\t%s: %s\n", 
			       _("Organization"), organization );

      if(r_work_check(work, "department", &department))
	g_string_append_printf(buffer, "\t\t%s: %s\n", 
			       _("Department"), department);

      if(r_work_check(work, "sub-department", &subdep))
	g_string_append_printf(buffer, "\t\t%s: %s\n", 
			       _("Sub department"), subdep);

      if(r_work_check(work, "manager-name", &manager))
	g_string_append_printf(buffer, "\t\t%s: %s\n", _("Manager"), manager);

      if(r_work_check(work, "collaborator", &collaborator))
	g_string_append_printf(buffer, "\t\t%s: %s\n", 
			       _("Collaborator"), collaborator);

      if(r_work_check(work, "manager-phone", &mphone))
	g_string_append_printf(buffer, "\t\t%s: %s\n",
			       _("Manager phone"), mphone );

      if(r_work_check(work, "collaborator-phone", &cphone))
	g_string_append_printf(buffer, "\t\t%s: %s\n",
			       _("Collaborator phone"), cphone);
    }
}

 
static void
write_net(RCard* card, GString* buffer)
{
  RNetAddress* net = NULL;
  RNetAddressType type;
  gchar* url;

  net = r_card_get_net_address(R_CARD(card));
  if (net)
    g_string_append_printf(buffer, "\t<b>%s:</b>\n", _("Net"));

  for (; net; net = r_card_get_next_net_address(R_CARD(card)))
    {	  
      gchar* ttype;
     
      g_object_get(R_NET_ADDRESS(net), "url", &url, "url-type", &type, NULL); 
      ttype = r_net_address_decode_type(type);
      
      g_string_append_printf(buffer, 
			     "\t\t%s: <span foreground=\"blue\">"
			     "<u>%s</u></span>\n", ttype, url);
    }
}

 
static void
write_phone(RCard* card, GString* buffer)
{
  RTelephone* tel = NULL;

  tel = r_card_get_telephone(R_CARD(card));
  if (tel)
    g_string_append_printf(buffer, "\t<b>%s</b>:\n", _("Telephone"));
  
  for (; tel; tel = r_card_get_next_telephone(R_CARD(card)))
    {
      RTelephoneType ttype;
      gchar* num;
	  
      g_object_get(R_TELEPHONE(tel), 
		   "telephone-number", &num,
		   "telephone-type",   &ttype, NULL);
      
      g_string_append_printf(buffer, "\t\t%s: %s\n", 
			     r_telephone_lookup_enum2str(ttype), num);
    }
}

 
static void
write_notes(RCard* card, GString* buffer)
{
  RNotes* notes;
  gchar *partner, *other_notes, *pubkey;
  gboolean has_partner;
  
  notes = r_personal_card_get_notes(R_PERSONAL_CARD(card));
  if (notes && r_notes_have_data(notes))
    {       
      g_string_append_printf(buffer, "\t<b>%s</b>:\n", _("Notes"));

      g_object_get(notes, 
		   "has-partner",  &has_partner, 
		   "partner-name", &partner,
		   "other-notes",  &other_notes, 
		   "pubkey",       &pubkey, NULL);
      
      if (has_partner)
	{   
	  g_string_append_printf(buffer, "\t\t%s\n", 
				 _("Contact has a partner"));
	  
	  g_string_append_printf(buffer, "\t\t%s: %s\n", 
				 _("partner's name"), partner);
	  
	  if (r_notes_know_birthday(notes))
	    g_string_append_printf(buffer, "\t\t%s: %s\n", 
				   _("partner's birthday"),
				   r_notes_get_birth_date(notes));
	  
	  if (r_notes_know_anniversary(notes))
	    g_string_append_printf(buffer, "\t\t%s: %s\n", 
				   _("anniversary"),
				   r_notes_get_anniversary_date(notes));
	}
      
      if (pubkey)
	g_string_append_printf(buffer, "\t\t%s: %s\n", _("Public key"), 
			       pubkey);

      if(other_notes)
	g_string_append_printf(buffer, "\t\t%s: %s\n", 
			       _("Other notes"), other_notes);
    }  
}


static void 
begin_print(GtkPrintOperation* operation, GtkPrintContext* context, 
	    gpointer data)
{
  RubricaPrint* print = (RubricaPrint*) data;
  RubricaPrintPrivate* priv = RUBRICA_PRINT_GET_PRIVATE(print);
  GList *alias = NULL;
  GString* buffer;
  gdouble height, width;
  gint total_lines, lines_in_page;
  gchar** tmp;

  height = gtk_print_context_get_height(context) - HEADER_HEIGHT - HEADER_GAP;
  width  = gtk_print_context_get_width(context);

  priv->lines_per_page = floor(height / (priv->font_size + 3));
  priv->chars_per_line = floor(width / (priv->font_size));

  buffer = g_string_new(NULL);
  alias  = priv->ids;
  total_lines = 0;
  lines_in_page = 0;
  for (; alias; alias = alias->next)
    {
      glong id = (glong) alias->data;
      RCard* card = NULL;

      card = r_abook_get_card_by_id(priv->book, id);
      if (card)
	{
	  write_header(card, buffer);
	  write_groups(card, buffer);
	  write_address(card, buffer);
	  write_work(card, buffer);
	  write_net(card, buffer);
	  write_phone(card, buffer);
	  write_notes(card, buffer);
	  g_string_append_printf(buffer, "\n");	  
	}
    }
  
  priv->lines = g_strsplit(buffer->str, "\n", 0);
  tmp = priv->lines;
  for(; *tmp; tmp++)
    priv->total_lines++;

  priv->total_pages = (priv->total_lines - 1) / priv->lines_per_page + 1;
  
  gtk_print_operation_set_n_pages(print->operation, priv->total_pages);

  g_print("\nLinee per pagina: %d", priv->lines_per_page);
  g_print("\nPagine totali: %d", priv->total_pages);  
  
  g_string_free(buffer, TRUE);
}


gchar* escape_string(gchar* str)
{
  gchar* ret, *tmp, *tmp2;
  gint len;
  gint found = 0, i = 0;
  
  len = strlen(str);
  for (; i < len; i++)
    if (str[i] == '&')
      found++;
  
  tmp2 = ret = g_malloc0(sizeof(char) * (len + found*5 +1));
  for(tmp = str; *tmp; tmp++)
    {
      if (*tmp != '&')
	{
	  *tmp2 = *tmp;
	  tmp2++;
	}
      else
	{
	  *tmp2++ = '&';
	  *tmp2++ = 'a';
	  *tmp2++ = 'm';
	  *tmp2++ = 'p';
	  *tmp2++ = ';';
	}
    }

  return ret;
}

static void 
draw_page (GtkPrintOperation* operation, GtkPrintContext* context, 
	   gint page_nr, gpointer data)
{
  RubricaPrint* print = (RubricaPrint*) data;
  RubricaPrintPrivate* priv = RUBRICA_PRINT_GET_PRIVATE(print);
  PangoFontDescription* desc;
  PangoAttrList* attr_list;
  PangoLayout *layout;
  cairo_t *cr;
  gdouble width, text_height;
  gint text_width, layout_height;
  gchar* page_str, *text;
  gint i, line;
  GError* error = NULL;

  cr     = gtk_print_context_get_cairo_context(context);
  width  = gtk_print_context_get_width(context);
  layout = gtk_print_context_create_pango_layout(context);
  desc   = pango_font_description_from_string(priv->font);
  pango_font_description_set_size(desc, priv->font_size * PANGO_SCALE);
  
  /* header: nome del file e numero di pagina */
  pango_layout_set_font_description(layout, desc);
  //  pango_layout_set_text(layout, priv->file, -1);
  pango_layout_set_width(layout, -1);
  pango_layout_set_alignment(layout, PANGO_ALIGN_LEFT);
  pango_layout_get_size(layout, NULL, &layout_height);
  text_height = (gdouble) layout_height / PANGO_SCALE;

  cairo_move_to(cr, 0, (HEADER_HEIGHT - text_height) /2);
  pango_cairo_show_layout(cr, layout);
  
  page_str = g_strdup_printf("%d di %d", page_nr +1, priv->total_pages);
  pango_layout_set_text(layout, page_str, -1);
  pango_layout_get_size(layout, &text_width, NULL);
  pango_layout_set_alignment(layout, PANGO_ALIGN_RIGHT);
  
  cairo_move_to(cr, width - (text_width / PANGO_SCALE),
		(HEADER_HEIGHT - text_height) /2);
  pango_cairo_show_layout(cr, layout);

  /* disegna le pagine */
  cairo_move_to(cr, 0, HEADER_HEIGHT + HEADER_GAP);
  line = page_nr * priv->lines_per_page;
  for (i = 0; i < priv->lines_per_page && line < priv->total_lines; i++)
    {   
      gchar* buf;
      
      buf = escape_string(priv->lines[line]);
      pango_parse_markup(buf, -1, 0, &attr_list, &text, NULL, &error);
      g_free(buf);

      if (!error)
	{
	  g_print("\ntesto: %s", text);
	  pango_layout_set_text(layout, text, -1);   
	  pango_cairo_show_layout(cr, layout);
	  cairo_rel_move_to(cr, 0, priv->font_size + 3);
	  line++;
	}
      else
	{
	  g_warning("\n%s", error->message);
	  error = NULL;
	}
    }
  
  g_free(page_str);
  g_object_unref(layout);
  pango_font_description_free(desc);
}



static void 
end_print(GtkPrintOperation* operation, GtkPrintContext* context, 
	  gpointer data)
{
  RubricaPrint* print = (RubricaPrint*) data;
  RubricaPrintPrivate* priv = RUBRICA_PRINT_GET_PRIVATE(print);

  g_list_free(priv->ids);
  priv->ids = NULL;
}


RubricaPrint* 
rubrica_print_new (void)
{
  RubricaPrint* print;
  
  print = g_object_new(rubrica_print_get_type(), NULL);

  print->operation = gtk_print_operation_new();

  g_signal_connect(G_OBJECT(print->operation), "begin_print", 
		   G_CALLBACK(begin_print), print);
  
  g_signal_connect(G_OBJECT(print->operation), "draw_page", 
		   G_CALLBACK(draw_page), print);
  
  g_signal_connect(G_OBJECT(print->operation), "end_print", 
		   G_CALLBACK(end_print), print);  

  return print;  
}

