/*
 * Program: rubrica
 * file: search_view.h
*  
*  Copyright (C) 2000-2005 Nicola Fragale <nicolafragale@libero.it>
*
*  This program is free software; you can redistribute it and/or modify
*  it under the terms of the GNU General Public License as published by
*  the Free Software Foundation; either version 3 of the License
*
*  This program is distributed in the hope that it will be useful,
*  but WITHOUT ANY WARRANTY; without even the implied warranty of
*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*  GNU General Public License for more details.
*
*  You should have received a copy of the GNU General Public License
*  along with this program; if not, write to the Free Software
*  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#include <gtk/gtk.h>
#include <glade/glade.h>
#include <glib/gi18n-lib.h>
#include <time.h>

#include "search_dialog.h"
#include "calendar.h"
#include "types.h"
#include "utils.h"

#include "libral.h"


typedef struct {
  gchar* str;
} ComboLabel;

static ComboLabel combo_labels[] = {
  {N_("is")       },
  {N_("is before of")},
  {N_("is after of") },
  {N_("is between")  },
  {NULL}
};

typedef enum {
  SEARCH_DATE = 0,
  SEARCH_BEFORE,
  SEARCH_AFTER,
  SEARCH_BETWEEN,
} SearchOn;



typedef struct {
  GtkWidget* window;

  GtkWidget* search;         // entry for string to search for
  GtkWidget* date_check;     // check button (checked if search on date)
  GtkWidget* str_box;        // enabled if search is performed on string
  GtkWidget* date_box;       // enabled if search is performed on date
  GtkWidget* date2_box;      // enabled if search is between dates
  GtkWidget* creation;       // radio button
  GtkWidget* last_change;    // radio button
  GtkWidget* combo;          // date search type
  GtkWidget* combo_box;      // date search type
  GtkWidget* date1;          // entry for (first) date  
  GtkWidget* date2;          // entry for second date
  GtkWidget* calendar1_box;  // calendar button
  GtkWidget* calendar2_box;  // calendar button  
} RubricaSearchDialog;

RubricaSearch* search;


static void on_date_check         (GtkWidget* check, gpointer data);
static void on_radio_cb           (GtkWidget* radio, gpointer data);
static void on_combo_changed      (GtkWidget* combo, gpointer data);

static void on_get_first_date_cb  (GtkWidget* calendar, GDate* date,
				   gpointer data);
static void on_get_second_date_cb (GtkWidget* calendar, GDate* date,
				   gpointer data);



static void 
on_date_check(GtkWidget* button, gpointer data)
{
  RubricaSearchDialog* dialog = (RubricaSearchDialog*) data;
  SearchType active;

  if(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(dialog->date_check)))
    {
      gtk_widget_set_sensitive(dialog->date_box, TRUE);
      gtk_widget_set_sensitive(dialog->str_box, FALSE);
      
      if(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(dialog->creation)))
	search->type = SEARCH_ON_CREATION;
      else
	search->type = SEARCH_ON_CHANGE;
    }
  else
    {
      gtk_widget_set_sensitive(dialog->date_box, FALSE);
      gtk_widget_set_sensitive(dialog->str_box, TRUE);

      search->type = SEARCH_ON_STRING;      
    }

  active = gtk_combo_box_get_active(GTK_COMBO_BOX(dialog->combo));
  if (active == SEARCH_BETWEEN)
    gtk_widget_show_all(dialog->date2_box);
  else
    gtk_widget_hide_all(dialog->date2_box);
}


static void 
on_radio_cb (GtkWidget* radio, gpointer data)
{
  RubricaSearchDialog* dialog = (RubricaSearchDialog*) data;

  if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(dialog->creation)))
    search->type = SEARCH_ON_CREATION;
  else
    search->type = SEARCH_ON_CHANGE;
}


static void 
on_combo_changed (GtkWidget *combo, gpointer data)
{
  RubricaSearchDialog* dialog = (RubricaSearchDialog*) data;
  SearchOn active;

  gtk_widget_hide_all(dialog->date2_box);
  active = (SearchOn) gtk_combo_box_get_active(GTK_COMBO_BOX(combo));
  switch (active)
    {
    case SEARCH_BEFORE:
      search->second = 0;
      if  (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(dialog->creation)))
	search->type = SEARCH_CREATION_BEFORE;
      else
	search->type = SEARCH_CHANGE_BEFORE;
      break;
      
    case SEARCH_AFTER:
      search->second = 0;
      if  (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(dialog->creation)))
      	search->type = SEARCH_CREATION_AFTER;
      else
	search->type = SEARCH_CHANGE_AFTER;

      break;

    case SEARCH_BETWEEN:
      gtk_widget_show_all(dialog->date2_box);
	    
      if (search->type == SEARCH_ON_CREATION)
	search->type = SEARCH_CREATION_BETWEEN;
      else
	search->type = SEARCH_CHANGE_BETWEEN;
      break;
      
    default:
      break;
    }
}


static void 
on_get_first_date_cb(GtkWidget* calendar, GDate* date, gpointer data)
{
  struct tm tm;

  g_date_to_struct_tm(date, &tm);
  search->first = mktime(&tm);
}

static void 
on_get_second_date_cb(GtkWidget* calendar, GDate* date, gpointer data)
{
  struct tm tm;
  
  g_date_to_struct_tm(date, &tm);
  search->second = mktime(&tm);
}


/*   Public
 */
void
rubrica_search_dialog(RubricaApp* app)
{
  GladeXML* gui;  
  RubricaSearchDialog* dialog;
  RAbook* book;
  GList* result = NULL;
  gchar* string = NULL;
  gint response, i;

  gui = glade_xml_new (RUBRICA_GUI_DIR"/Search.glade", NULL, NULL);
  if (!gui)
    g_error("Can't load gui");  

  search = g_new0(RubricaSearch, 1);
  dialog = g_new0(RubricaSearchDialog, 1);

  dialog->window        = glade_xml_get_widget(gui, "RubricaSearch"); 
  dialog->search        = glade_xml_get_widget(gui, "search");
  dialog->date_check    = glade_xml_get_widget(gui, "date_check"); 
  dialog->date_box      = glade_xml_get_widget(gui, "date_box");
  dialog->str_box       = glade_xml_get_widget(gui, "str_box");
  dialog->creation      = glade_xml_get_widget(gui, "creation");
  dialog->last_change   = glade_xml_get_widget(gui, "change");
  dialog->combo_box     = glade_xml_get_widget(gui, "combo_box");
  dialog->date2_box     = glade_xml_get_widget(gui, "date2_box");
  dialog->calendar1_box = glade_xml_get_widget(gui, "calendar1_box");
  dialog->calendar2_box = glade_xml_get_widget(gui, "calendar2_box");
  
  dialog->combo = gtk_combo_box_new_text();
  gtk_box_pack_start(GTK_BOX(dialog->combo_box), dialog->combo, 
		     TRUE, TRUE, 6);
  for (i = 0; combo_labels[i].str; i++)
    gtk_combo_box_append_text(GTK_COMBO_BOX(dialog->combo), 
			      _(combo_labels[i].str));
  gtk_widget_show(dialog->combo);

  dialog->date1 = rubrica_calendar_new();
  gtk_box_pack_start(GTK_BOX(dialog->calendar1_box), dialog->date1, 
		     TRUE, TRUE, 6);
  gtk_widget_show(dialog->date1);

  dialog->date2 = rubrica_calendar_new();
  gtk_box_pack_start(GTK_BOX(dialog->calendar2_box), dialog->date2, 
		     TRUE, TRUE, 6);
  gtk_widget_hide_all(dialog->date2_box);

  g_object_unref(gui);  

  search->type   = SEARCH_ON_STRING;
  search->first  = 0;
  search->second = 0;
 
  gtk_combo_box_set_active(GTK_COMBO_BOX(dialog->combo), SEARCH_DATE);
  gtk_widget_set_sensitive(dialog->date_box, FALSE);

  g_signal_connect(G_OBJECT(dialog->date_check), "toggled",
		   G_CALLBACK(on_date_check), dialog);
   
  g_signal_connect(G_OBJECT(dialog->creation), "clicked", 
		   G_CALLBACK(on_radio_cb), dialog);

  g_signal_connect(G_OBJECT(dialog->last_change), "clicked", 
		   G_CALLBACK(on_radio_cb), dialog);

  g_signal_connect(G_OBJECT(dialog->combo), "changed",
		   G_CALLBACK(on_combo_changed), dialog);

  g_signal_connect(G_OBJECT(dialog->date1), "calendar-change",
		   G_CALLBACK(on_get_first_date_cb), dialog->date1); 
  
  g_signal_connect(G_OBJECT(dialog->date2), "calendar-change",
		   G_CALLBACK(on_get_second_date_cb), dialog->date2);  
  
  response = gtk_dialog_run (GTK_DIALOG(dialog->window));  
  switch(response)
    {
    case GTK_RESPONSE_ACCEPT:
      book = rubrica_app_get_active_addressbook(app);

      switch(search->type)
	{
	case SEARCH_ON_STRING:
	  string = g_strdup(gtk_entry_get_text(GTK_ENTRY(dialog->search)));
	
	  result = r_abook_search(book, string);	  
	  break;
	  
	case SEARCH_ON_CREATION:
	case SEARCH_CREATION_BEFORE:
	case SEARCH_CREATION_AFTER:
	case SEARCH_ON_CHANGE:
	case SEARCH_CHANGE_AFTER:
	case SEARCH_CHANGE_BEFORE:
	  result = r_abook_search_date(book, search->first, search->type);
	  
	  break;
	  
	case SEARCH_CREATION_BETWEEN:
	case SEARCH_CHANGE_BETWEEN:
	  result = r_abook_search_between(book, search->first, search->second,
					  search->type);
	  break;
	  
	default:
	  break;
	}
      
      rubrica_app_display_search_results(app, result);
      break;

    case GTK_RESPONSE_CLOSE:
      break;
    }
  
  gtk_widget_destroy(dialog->window);  
  g_free(string);
  g_free(dialog);
  g_free(search);
}

