/*
 * Program: rubrica
 * file: search_view.c
 *  
 *  Copyright (C) Nicola Fragale <nicolafragale@libero.it>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 3 of the License
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <gtk/gtk.h>
#include <glib/gi18n-lib.h>
#include <gconf/gconf-client.h>

#include "libral.h"

#include "search_view.h"
#include "preferences.h"



/*    properties enumeration 
 */
enum { 
  PROP_0,
};




struct _RubricaSearchViewPrivate {
  RAbook* book;

  gboolean dispose_has_run;
};


#define RUBRICA_SEARCH_VIEW_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE((o), \
                                            RUBRICA_SEARCH_VIEW_TYPE,        \
	  				    RubricaSearchViewPrivate))


static GObjectClass *parent_class = NULL;


static void rubrica_search_view_class_init   (RubricaSearchViewClass* klass);
static void rubrica_search_view_init         (RubricaSearchView* obj);

static void rubrica_search_view_finalize     (RubricaSearchView* self);
static void rubrica_search_view_dispose      (RubricaSearchView* self);



/* Private 
 */
static void search_view_add_columns (GtkTreeView *tree);
static void search_view_add_card    (GtkTreeView* tree, RAbook* book, 
				     RCard* card);


static void 
search_view_add_columns (GtkTreeView *tree)
{
  GtkCellRenderer *renderer;
  GtkTreeModel *model;

  model = gtk_tree_view_get_model(GTK_TREE_VIEW(tree));
  
  renderer = gtk_cell_renderer_text_new();
  gtk_tree_view_insert_column_with_attributes(tree, -1, _("Card"), 
					      renderer, "text", 
					      SEARCH_CARD_COLUMN, NULL);
}


static void 
search_view_add_card (GtkTreeView* tree, RAbook* book , RCard* card)
{
  GtkTreeIter  iter;
  GtkTreeModel *model;
  gchar* label;
  gint id;

  g_return_if_fail(IS_R_CARD(card));
  
  g_object_get(card, "card-id", &id, "card-name", &label, NULL);

  model = gtk_tree_view_get_model(GTK_TREE_VIEW(tree));      
  gtk_list_store_append(GTK_LIST_STORE(model), &iter);  
  gtk_list_store_set(GTK_LIST_STORE(model), &iter,
		     SEARCH_ID_COLUMN,   id,
		     SEARCH_CARD_COLUMN, label, 
		     SEARCH_BOOK_COLUMN, book,
		     -1);
}




GType
rubrica_search_view_get_type()
{
  static GType search_view_type = 0;
  
  if (!search_view_type)
    {
      static const GTypeInfo search_view_info =
	{
	  sizeof(RubricaSearchViewClass),
	  NULL,
	  NULL,
	  (GClassInitFunc) rubrica_search_view_class_init,
	  NULL,
	  NULL,
	  sizeof(RubricaSearchView),
	  0,
	  (GInstanceInitFunc) rubrica_search_view_init
	};

      search_view_type = g_type_register_static (RUBRICA_VIEW_TYPE, 
						 "RubricaSearchView",
						 &search_view_info, 0);
    }
  
  return search_view_type;
}


static void 
rubrica_search_view_dispose (RubricaSearchView* self)
{
  g_return_if_fail(IS_RUBRICA_SEARCH_VIEW(self));
  
  if (self->priv->dispose_has_run)
    return;

  self->priv->dispose_has_run = TRUE;  
}


static void 
rubrica_search_view_finalize (RubricaSearchView* self)
{
  g_return_if_fail(IS_RUBRICA_SEARCH_VIEW(self));

}


static void
rubrica_search_view_class_init(RubricaSearchViewClass* klass)
{
  GObjectClass *object_class = G_OBJECT_CLASS (klass);
 
  parent_class = g_type_class_peek_parent (klass);

  object_class->finalize = (GObjectFinalizeFunc) rubrica_search_view_finalize;
  object_class->dispose  = (GObjectFinalizeFunc) rubrica_search_view_dispose;

  g_type_class_add_private (klass, sizeof(RubricaSearchViewPrivate));
}


static void
rubrica_search_view_init(RubricaSearchView* self)
{
  GtkWidget*    tree;
  GtkTreeModel* model;

  self->priv = RUBRICA_SEARCH_VIEW_GET_PRIVATE(self);
  
  self->priv->book = NULL;

  model = GTK_TREE_MODEL(gtk_list_store_new(LAST_SEARCH_COLUMN, 
					    G_TYPE_INT,      /* card's id    */
					    G_TYPE_STRING,   /* card's label */
					    G_TYPE_POINTER)); 

  rubrica_view_set_model(RUBRICA_VIEW(self), model);
  tree = rubrica_view_get_tree(RUBRICA_VIEW(self));
   
  search_view_add_columns (GTK_TREE_VIEW(tree));

  gtk_widget_show(GTK_WIDGET(self));

  self->priv->dispose_has_run = FALSE;
}




/*   Public
 */
/**
 * rubrica_search_view_new
 *
 * create a #RubricaSearchView
 *
 * returns: a new #RubricaSearchView 
 */
GtkWidget*
rubrica_search_view_new()
{
  GtkWidget* view;

  view = GTK_WIDGET(g_object_new(RUBRICA_SEARCH_VIEW_TYPE, NULL));

  return view;
}



void 
rubrica_search_view_set_addressbook(RubricaSearchView* view, RAbook *book)
{
  g_return_if_fail(IS_RUBRICA_SEARCH_VIEW(view));
  g_return_if_fail(IS_R_ABOOK(book));
  
  view->priv->book = book;
}


void 
rubrica_search_view_display_results (RubricaSearchView* view, GList* results)
{
  RCard* card;
  GtkWidget* tree;
  GtkTreeModel *model;
  RubricaSearchViewPrivate* priv;
  gint items;
  GList* iter;
  gchar* message;
  
  g_return_if_fail(IS_RUBRICA_SEARCH_VIEW(view));

  priv  = RUBRICA_SEARCH_VIEW_GET_PRIVATE(view);
  tree  = rubrica_view_get_tree(RUBRICA_VIEW(view));
  model = gtk_tree_view_get_model(GTK_TREE_VIEW(tree));    

  gtk_list_store_clear(GTK_LIST_STORE(model));
  items = g_list_length(results);

  iter = results;
  for (; iter; iter = iter->next)
    {
      glong id = (glong) iter->data;

      card = r_abook_get_card_by_id(priv->book, id);      
      search_view_add_card(GTK_TREE_VIEW(tree), priv->book, card);
    }
  g_list_free(results);
  
  if (items == 0)
    message = g_strdup_printf(_("No items found"));    
  else
    {
      if (items == 1)
	message = g_strdup_printf(_("One item found"));    
      else
	message = g_strdup_printf(_("Items found: %-5d "), items);
    }
   
  g_object_set(view, "view-message", message, NULL);
  
  g_signal_emit_by_name(view, "view-changed", message, G_TYPE_STRING);
}

