/*
 *  Program: Rubrica
 *  file: tab.c
 *
 *  Copyright (C) Nicola Fragale <nicolafragale@libero.it>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 3 of the License
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <gtk/gtk.h>

#include "tab.h"

enum {
  PROP_0,
  TAB_LABEL,
  TAB_BROTHER
};



struct _RubricaTabPrivate {
  GtkWidget* label;
  GtkWidget* button;
  
  GtkWidget* brother;
  
  gboolean dispose_has_run;
};


#define RUBRICA_TAB_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE((o),       \
                                    RUBRICA_TAB_TYPE, RubricaTabPrivate))


static GObjectClass *parent_class = NULL;


static void rubrica_tab_class_init   (RubricaTabClass* klass);
static void rubrica_tab_init         (RubricaTab* obj);

static void rubrica_tab_dispose      (RubricaTab* obj);
static void rubrica_tab_finalize     (RubricaTab* obj);

static void rubrica_tab_set_property (GObject* obj, guint property_id,
				      GValue* value,  GParamSpec* spec);
static void rubrica_tab_get_property (GObject* obj, guint property_id,
				      GValue* value, GParamSpec* spec);



GType
rubrica_tab_get_type()
{
  static GType rubrica_tab_type = 0;
  
  if (!rubrica_tab_type)
    {
      static const GTypeInfo rubrica_tab_info =
	{
	  sizeof(RubricaTabClass),
	  NULL,
	  NULL,
	  (GClassInitFunc) rubrica_tab_class_init,
	  NULL,
	  NULL,
	  sizeof(RubricaTab),
	  0,
	  (GInstanceInitFunc) rubrica_tab_init
	};

      rubrica_tab_type = g_type_register_static (GTK_TYPE_HBOX, 
						 "RubricaTab",
						 &rubrica_tab_info, 0);
    }
  
  return rubrica_tab_type;
}


static void
rubrica_tab_class_init(RubricaTabClass* klass)
{
  GObjectClass *object_class;
  GParamSpec* pspec;
 
  parent_class = g_type_class_peek_parent (klass);
  object_class = G_OBJECT_CLASS (klass);

  object_class->dispose  = (GObjectFinalizeFunc) rubrica_tab_dispose;
  object_class->finalize = (GObjectFinalizeFunc) rubrica_tab_finalize;

  object_class->set_property = (gpointer) rubrica_tab_set_property;
  object_class->get_property = (gpointer) rubrica_tab_get_property;

  g_type_class_add_private (klass, sizeof(RubricaTabPrivate));

  /* class property 
   */
  /**
   * RubricaTab:label
   *
   * the string that will be displayed 
   */
  pspec = g_param_spec_string("tab-label", 
			      "tab label", 
			      "the label that will be displayed on the tab",  
			      NULL,
			      G_PARAM_CONSTRUCT | G_PARAM_READWRITE);
  g_object_class_install_property(object_class, TAB_LABEL, pspec);  

  /* class property 
   */
  /**
   * RubricaTab:brother
   *
   * the object that will be detroyed on button click
   */
  pspec = g_param_spec_object("tab-brother", 
			      "tab brother", 
			      "the object that will be destroyed "
			      "on button click",  
			      GTK_TYPE_WIDGET,
			      G_PARAM_READWRITE);
  g_object_class_install_property(object_class, TAB_BROTHER, pspec);  
}


static void
rubrica_tab_init(RubricaTab* self)
{ 
  GtkWidget* image;

  gtk_box_set_homogeneous(GTK_BOX(self), FALSE);
  gtk_box_set_spacing(GTK_BOX(self), 0);

  self->priv = RUBRICA_TAB_GET_PRIVATE(self);
  
  self->priv->label   = gtk_label_new(NULL);
  self->priv->button  = gtk_button_new();
  self->priv->brother = NULL;

  image = gtk_image_new_from_stock(GTK_STOCK_CLOSE, GTK_ICON_SIZE_MENU); 
  gtk_button_set_image(GTK_BUTTON(self->priv->button), image);
  gtk_button_set_relief(GTK_BUTTON(self->priv->button), GTK_RELIEF_NONE);

  gtk_box_pack_start(GTK_BOX(self), self->priv->label, TRUE, TRUE, 0); 
  gtk_box_pack_start(GTK_BOX(self), self->priv->button, FALSE, FALSE, 0); 

  gtk_widget_show(GTK_WIDGET(self));
  gtk_widget_show(self->priv->label);
  gtk_widget_show(self->priv->button);

  self->priv->dispose_has_run = FALSE;
}

 

static void 
rubrica_tab_dispose (RubricaTab* self)
{
  g_return_if_fail(IS_RUBRICA_TAB(self));
  
  if (self->priv->dispose_has_run)
    return;

  self->priv->dispose_has_run = TRUE;
}


static void 
rubrica_tab_finalize (RubricaTab* self)
{
  g_return_if_fail(IS_RUBRICA_TAB(self));
}


static void 
rubrica_tab_set_property (GObject* obj, guint property_id,
			   GValue* value,  GParamSpec* spec)
{
  RubricaTab* self = (RubricaTab*) obj;
  RubricaTabPrivate* priv = RUBRICA_TAB_GET_PRIVATE(self);

  switch (property_id) 
    {
    case TAB_LABEL:
      gtk_label_set_markup(GTK_LABEL(priv->label), g_value_get_string(value));
      break;

    case TAB_BROTHER:
      priv->brother = g_value_get_object(value);
      break;

    default: 
      break; 
    }   
}

static void 
rubrica_tab_get_property (GObject* obj, guint property_id,
			  GValue* value, GParamSpec* spec)
{
  RubricaTab* self = (RubricaTab*) obj;
  RubricaTabPrivate* priv = RUBRICA_TAB_GET_PRIVATE(self);

  switch (property_id) 
    {
    case TAB_LABEL:
      g_value_set_string(value, gtk_label_get_text(GTK_LABEL(priv->label)));
      break; 

    case TAB_BROTHER:
      g_value_set_object(value, priv->brother);
      break;

    default: 
      break; 
    }  
}



/*   ***************************** Public *****************************
*/


/**
 * rubrica_tab_new
 *
 * create a #RubricaTab
 *
 * Returns: a new #RubricaTab
 */
GtkWidget*
rubrica_tab_new(GtkWidget* brother)
{
  g_return_val_if_fail(brother != NULL, NULL);

  GtkWidget* tab;
  
  tab = g_object_new(rubrica_tab_get_type(), "tab-brother", brother, NULL);
  
  return tab;
}


/**
 * rubrica_tab_new_with_text
 *
 * create a #RubricaTab and initialize tab's label to the given string
 *
 * Returns: a new #RubricaTab
 */
GtkWidget* 
rubrica_tab_new_with_label (GtkWidget* brother, gchar* label)
{
  g_return_val_if_fail(brother != NULL, NULL);

  GtkWidget* tab;

  tab = g_object_new(rubrica_tab_get_type(), 
		     "tab-brother", brother, "tab-label", label, NULL);
  
  return tab;  
}


/**
 * rubrica_tab_free
 * @tab: a #RubricaTab
 * 
 * free the #RubricaTab
 */
void
rubrica_tab_free(RubricaTab* tab)
{
  g_return_if_fail(IS_RUBRICA_TAB(tab));

  g_object_unref(tab);   
}


/**
 * rubrica_tab_get_label
 *
 * get the tab's label object
 *
 * return a #GtkWidget
 */
GtkWidget* 
rubrica_tab_get_label (RubricaTab* tab)
{
  g_return_val_if_fail(IS_RUBRICA_TAB(tab), NULL);
  
  return tab->priv->label;
}


/**
 * rubrica_tab_get_button
 *
 * get the tab's button object
 *
 * return a #GtkWidget
 */
GtkWidget* 
rubrica_tab_get_button (RubricaTab* tab)
{
  g_return_val_if_fail(IS_RUBRICA_TAB(tab), NULL);
  
  return tab->priv->button;
}


void
rubrica_tab_set_brother(RubricaTab* tab, GtkWidget* brother)
{
  g_return_if_fail(IS_RUBRICA_TAB(tab));
  g_return_if_fail(brother != NULL);
 
  g_object_set(tab, "tab-brother", brother, NULL);
}


GtkWidget*
rubrica_tab_get_brother(RubricaTab* tab)
{
  GtkWidget* brother = NULL;

  g_return_val_if_fail(IS_RUBRICA_TAB(tab), NULL);
 
  g_object_get(tab, "tab-brother", &brother, NULL);
  
  return brother;
}
