/*
 *  Program: Rubrica
 *  file: ref.c
 *  
 *  Copyright (C) Nicola Fragale <nicolafragale@libero.it>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 3 of the License
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <gtk/gtk.h>
#include <glib/gi18n-lib.h>
#include <gconf/gconf-client.h>

#include "libral.h"

#include "trash_view.h"
#include "preferences.h"


/*    properties enumeration 
 */
enum { 
  PROP_0,
  TRASH_NAME,
};


struct _RubricaTrashViewPrivate {
  gint   items;

  GList* books; 

  gboolean dispose_has_run;
};

#define RUBRICA_TRASH_VIEW_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE((o), \
                                           RUBRICA_TRASH_VIEW_TYPE,         \
					   RubricaTrashViewPrivate))

static GObjectClass *parent_class = NULL;


static void rubrica_trash_view_class_init   (RubricaTrashViewClass* klass);
static void rubrica_trash_view_init         (RubricaTrashView* obj);

static void rubrica_trash_view_finalize     (RubricaTrashView* self);
static void rubrica_trash_view_dispose      (RubricaTrashView* self);



/* Private 
 */
static gboolean trash_has_book     (RubricaTrashView* view, RAbook* book);
static void     trash_add_book     (RubricaTrashView* view, RAbook* book);

static void trash_update_infos     (RubricaTrashView* view);
static void trash_view_add_columns (GtkTreeView *tree);



static gboolean 
trash_has_book (RubricaTrashView* view, RAbook* book)
{
  GList* alias;

  for (alias = view->priv->books; alias; alias = alias->next)
    if ((RAbook*) alias->data == book)
      return TRUE;
  
  return FALSE;
}

static void 
trash_add_book (RubricaTrashView* view, RAbook* book)
{
  view->priv->books = g_list_append(view->priv->books, book);
}


static void 
trash_update_infos(RubricaTrashView* view)
{
  gchar* message;
  
  message = g_strdup_printf(_("Cards in Trash: %-5d"), view->priv->items);
  g_object_set(view, "view-message", message, NULL);  
}


static void 
trash_view_add_columns (GtkTreeView *tree)
{
  GtkCellRenderer *renderer;
  GtkTreeModel *model;

  model = gtk_tree_view_get_model(GTK_TREE_VIEW(tree));
  
  renderer = gtk_cell_renderer_text_new();
  gtk_tree_view_insert_column_with_attributes(tree, -1, _("Card"), 
					      renderer, "text", 
					      TRASH_CARD_LABEL, NULL);
  
  renderer = gtk_cell_renderer_text_new();
  gtk_tree_view_insert_column_with_attributes (tree, -1, _("Information"),
					       renderer, "text", 
					       TRASH_LN_COMPANY, NULL);
  
  renderer = gtk_cell_renderer_text_new();
  gtk_tree_view_insert_column_with_attributes (tree, -1, _("Web"),
					       renderer, "text", 
					       TRASH_WEB, NULL);
  
  renderer = gtk_cell_renderer_text_new();
  gtk_tree_view_insert_column_with_attributes (tree, -1, _("Email"),
					       renderer, "text", 
					       TRASH_EMAIL, NULL);
  
  renderer = gtk_cell_renderer_text_new();
  gtk_tree_view_insert_column_with_attributes (tree, -1, _("Telephone"),
					       renderer, "text", 
					       TRASH_PHONE,  NULL);
}



GType
rubrica_trash_view_get_type()
{
  static GType trash_view_type = 0;
  
  if (!trash_view_type)
    {
      static const GTypeInfo trash_view_info =
	{
	  sizeof(RubricaTrashViewClass),
	  NULL,
	  NULL,
	  (GClassInitFunc) rubrica_trash_view_class_init,
	  NULL,
	  NULL,
	  sizeof(RubricaTrashView),
	  0,
	  (GInstanceInitFunc) rubrica_trash_view_init
	};

      trash_view_type = g_type_register_static (RUBRICA_VIEW_TYPE, 
						"RubricaTrashView",
						&trash_view_info, 0);
    }
  
  return trash_view_type;
}


static void 
rubrica_trash_view_dispose (RubricaTrashView* self)
{
  g_return_if_fail(IS_RUBRICA_TRASH_VIEW(self));
  
  RubricaTrashViewPrivate* priv = RUBRICA_TRASH_VIEW_GET_PRIVATE(self);

  if (priv->dispose_has_run)
    return;

  priv->dispose_has_run = TRUE;  
}


static void 
rubrica_trash_view_finalize (RubricaTrashView* self)
{
  g_return_if_fail(IS_RUBRICA_TRASH_VIEW(self));

}


static void
rubrica_trash_view_class_init(RubricaTrashViewClass* klass)
{
  GObjectClass *object_class = G_OBJECT_CLASS (klass);
  
  parent_class = g_type_class_peek_parent (klass);

  object_class->finalize = (GObjectFinalizeFunc) rubrica_trash_view_finalize;
  object_class->dispose  = (GObjectFinalizeFunc) rubrica_trash_view_dispose;

  g_type_class_add_private (klass, sizeof(RubricaTrashViewPrivate));
}


static void
rubrica_trash_view_init(RubricaTrashView* self)
{
  GtkWidget*    tree;
  GtkListStore* model;

  self->priv = RUBRICA_TRASH_VIEW_GET_PRIVATE(self);

  self->priv->books = NULL;

  model = gtk_list_store_new(TRASH_LAST_COLUMN, 
			     G_TYPE_INT,       /* card's id            */
			     G_TYPE_POINTER,   /* addressbook          */
			     G_TYPE_STRING,    /* card label           */
			     G_TYPE_STRING,    /* first name | company */ 
			     G_TYPE_STRING,    /* web                  */
			     G_TYPE_STRING,    /* email                */ 
			     G_TYPE_STRING);   /* telephone            */ 
  
  rubrica_view_set_model(RUBRICA_VIEW(self), GTK_TREE_MODEL(model));
  tree = rubrica_view_get_tree(RUBRICA_VIEW(self));

  trash_view_add_columns (GTK_TREE_VIEW(tree));

  gtk_widget_show(GTK_WIDGET(self));

  self->priv->dispose_has_run = FALSE;
}



/*   Public
 */
/**
 * rubrica_trash_view_new
 *
 * create a #RubricaTrashView
 *
 * returns: a new #RubricaTrashView 
 */
GtkWidget*
rubrica_trash_view_new()
{
  GtkWidget* view;

  view = GTK_WIDGET(g_object_new(RUBRICA_TRASH_VIEW_TYPE, 
				 "child-type", "Trash", NULL));  
  
  return view;
}



void 
rubrica_trash_view_add_card (RubricaTrashView* view, RAbook* book, RCard* card)
{
  RubricaTrashViewPrivate* priv;
  GtkWidget* tree;
  GtkTreeIter  iter;
  GtkTreeModel *model;
  RInfos *infos;
  gchar* str;
    
  g_return_if_fail(IS_RUBRICA_TRASH_VIEW(view));
  g_return_if_fail(IS_R_ABOOK(book));
  g_return_if_fail(IS_R_CARD(card));
  
  priv = RUBRICA_TRASH_VIEW_GET_PRIVATE(view);

  infos = r_card_get_infos(R_CARD(card));
  tree  = rubrica_view_get_tree(RUBRICA_VIEW(view));
  model = gtk_tree_view_get_model(GTK_TREE_VIEW(tree));    
  
  gtk_list_store_append(GTK_LIST_STORE(model), &iter);  
  if (r_card_is_personal(card))
    str = r_infos_get_last(infos);
  else
    str = r_infos_get_label(infos);
  
  gtk_list_store_set(GTK_LIST_STORE(model), &iter,
		     TRASH_ID,          r_infos_get_id(infos),
		     TRASH_ADDRESSBOOK, book,   
		     TRASH_CARD_LABEL,  r_infos_get_label(infos), 
		     TRASH_LN_COMPANY,  str,
		     TRASH_WEB,         r_infos_get_web(infos),
		     TRASH_EMAIL,       r_infos_get_email(infos),
		     TRASH_PHONE,       r_infos_get_telephone(infos),      
		     -1); 
  priv->items++;

  if (!trash_has_book(view, book))
    trash_add_book(view, book);
  
  r_card_free_infos(infos);
  trash_update_infos(view);
}


void rubrica_trash_view_destroy_card(RubricaTrashView* view)
{
  GtkWidget* tree;
  GtkTreeIter iter;  
  GtkTreeModel *model;
  GtkTreeSelection *selection; 

  g_return_if_fail(IS_RUBRICA_TRASH_VIEW(view));

  tree      = rubrica_view_get_tree(RUBRICA_VIEW(view));
  model     = gtk_tree_view_get_model(GTK_TREE_VIEW(tree));  
  selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(tree));
  
  if (gtk_tree_selection_get_selected(selection, NULL, &iter))
    {
      RAbook* book  = NULL;    
      gpointer card = NULL;	    
      gchar* name;
      glong id = 0;   
      
      /* get the id and the original addressbook of selected card */
      gtk_tree_model_get(model, &iter, 
			 TRASH_ID, &id,
			 TRASH_ADDRESSBOOK, &book,
			 -1);      

      g_object_get(book, "addressbook-name", &name, NULL); 
      g_print("\nDEBUG **** Destroing card id: %ld from addressbook %s (%p)", 
	      id, name, book);
  
      /* get and destroy the card */
      card = r_abook_get_card_by_id(R_ABOOK(book), id);   
      r_abook_destroy_card(book, card);
      
      /* update trash view */
      gtk_list_store_remove(GTK_LIST_STORE(model), &iter);      
    }    
}


void 
rubrica_trash_view_recovery_card (RubricaTrashView* view)
{
  GtkWidget* tree;
  GtkTreeIter iter;  
  GtkTreeModel* model;
  GtkTreeSelection* selection; 
  RubricaTrashViewPrivate* priv;

  g_return_if_fail(IS_RUBRICA_TRASH_VIEW(view));

  priv      = RUBRICA_TRASH_VIEW_GET_PRIVATE(view);
  tree      = rubrica_view_get_tree(RUBRICA_VIEW(view));
  model     = gtk_tree_view_get_model(GTK_TREE_VIEW(tree));  
  selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(tree));
  
  if (gtk_tree_selection_get_selected(selection, NULL, &iter))
    {
      RAbook* book  = NULL;    
      gpointer card = NULL;	    
      glong id = 0;   
      
      /* get the id and the original addressbook of selected card */
      gtk_tree_model_get(model, &iter, 
			 TRASH_ID, &id,
			 TRASH_ADDRESSBOOK, &book,
			 -1);  
      gtk_list_store_remove(GTK_LIST_STORE(model), &iter);
      
      /* get the card */
      card = r_abook_get_card_by_id(R_ABOOK(book), id);
      
      /* recovery it (signal "card_recovered" is emitted) */
      r_abook_recovery_card(book, card);

      priv->items--;
    }  
  
  trash_update_infos(view);
}



void 
rubrica_trash_view_empty_trash (RubricaTrashView* view)
{
  GList* alias;
  GtkWidget* tree;
  GtkTreeModel* model;

  g_return_if_fail(IS_RUBRICA_TRASH_VIEW(view));

  tree  = rubrica_view_get_tree(RUBRICA_VIEW(view));
  model = gtk_tree_view_get_model(GTK_TREE_VIEW(tree));  

  for (alias = view->priv->books; alias; alias = alias->next)
    r_abook_empty_trash(R_ABOOK(alias->data));
  
  gtk_list_store_clear(GTK_LIST_STORE(model));    
}
