/*
 *  Program: Rubrica
 *  file: view.c
 *  
 *  Copyright (C) Nicola Fragale <nicolafragale@libero.it>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 3 of the License
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <gtk/gtk.h>

#include "view.h"

enum {
  PROP_0,
  VIEW_MSG,
  VIEW_CHILD_TYPE,
  VIEW_CHILD_SELECTED,
};

/*    signals enumeration 
 */
enum {
  CHANGED,
  LAST_SIGNAL   
};

static guint view_signals[LAST_SIGNAL] = {0};


struct _RubricaViewPrivate {
  gchar* name;

  GtkWidget* scrollwin;
  GtkWidget* tree;

  gchar* message;
  gchar* child_type;
  gboolean selected;
  
  gboolean dispose_has_run;
};


#define RUBRICA_VIEW_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE((o),       \
                                     RUBRICA_VIEW_TYPE, RubricaViewPrivate))


static GObjectClass *parent_class = NULL;


static void rubrica_view_class_init   (RubricaViewClass* klass);
static void rubrica_view_init         (RubricaView* obj);

static void rubrica_view_finalize     (RubricaView* self);
static void rubrica_view_dispose      (RubricaView* self);


static void rubrica_view_set_property (GObject* obj, guint property_id,
				       const GValue* value,  GParamSpec* spec);
static void rubrica_view_get_property (GObject* obj, guint property_id,
				       GValue* value, GParamSpec* spec);




GType
rubrica_view_get_type()
{
  static GType view_type = 0;
  
  if (!view_type)
    {
      static const GTypeInfo view_info =
	{
	  sizeof(RubricaViewClass),
	  NULL,
	  NULL,
	  (GClassInitFunc) rubrica_view_class_init,
	  NULL,
	  NULL,
	  sizeof(RubricaView),
	  0,
	  (GInstanceInitFunc) rubrica_view_init
	};

      view_type = g_type_register_static (GTK_TYPE_VBOX, "RubricaView", 
					  &view_info, 0);
    }
  
  return view_type;
}


static void 
rubrica_view_dispose (RubricaView* self)
{
  g_return_if_fail(IS_RUBRICA_VIEW(self));
  
  if (self->priv->dispose_has_run)
    return;

  self->priv->dispose_has_run = TRUE;  
}


static void 
rubrica_view_finalize (RubricaView* self)
{
  g_return_if_fail(IS_RUBRICA_VIEW(self));
  
  if (self->priv->child_type)
    g_free(self->priv->child_type);
}


static void
rubrica_view_class_init(RubricaViewClass* klass)
{
  GObjectClass *object_class = G_OBJECT_CLASS (klass);
  GParamSpec* pspec;
  
  parent_class = g_type_class_peek_parent (klass);

  object_class->finalize     = (GObjectFinalizeFunc) rubrica_view_finalize;
  object_class->dispose      = (GObjectFinalizeFunc) rubrica_view_dispose;

  object_class->set_property = (gpointer) rubrica_view_set_property;
  object_class->get_property = (gpointer) rubrica_view_get_property;

  g_type_class_add_private (klass, sizeof(RubricaViewPrivate));

 
  /* class signals 
   */   
  /**
   * RubricaView::changed
   * @view: the #RubricaCardsView object that receives the signal
   *
   * the "changed" signal is emitted when the view changes (add/remove item)
   */
  view_signals[CHANGED] =
    g_signal_new("view-changed", 
		 RUBRICA_VIEW_TYPE,
		 G_SIGNAL_RUN_LAST,
		 G_STRUCT_OFFSET(RubricaViewClass, changed),
		 NULL,
		 NULL,
		 g_cclosure_marshal_VOID__STRING,
		 G_TYPE_NONE,            /* return type */
		 1,                      /* params      */
		 G_TYPE_STRING);         /* params type: error code */

  /**
   * RubricaView:msg
   *
   * a message to display in statusbar
   */
  pspec = g_param_spec_string("view-message", 
			      "message", 
			      "a message",  
			      NULL,
			      G_PARAM_READWRITE);
  g_object_class_install_property(object_class, VIEW_MSG, pspec);

  /* class property 
   */
  /**
   * RubricaView:child-type
   *
   * the view's child type
   */
  pspec = g_param_spec_string("child-type",  
			      "child-type",  
			      "the type of view child",  
			      NULL,
			      G_PARAM_READWRITE);
  g_object_class_install_property(object_class, VIEW_CHILD_TYPE, pspec);


  /**
   * RubricaView:view-selected
   *
   * the view's child type
   */
  pspec = g_param_spec_boolean("view-selected",
			       "view selected",
			       "was a view's item selected",
			       FALSE,
			       G_PARAM_CONSTRUCT | G_PARAM_READWRITE);
  g_object_class_install_property(object_class, VIEW_CHILD_SELECTED, pspec);
}


static void
rubrica_view_init(RubricaView* self)
{
  GtkWidget* scrollwin;
  GtkTreeSelection *selection;

  gtk_box_set_homogeneous(GTK_BOX(self), FALSE);
  gtk_box_set_spacing(GTK_BOX(self), 6);

  self->priv = RUBRICA_VIEW_GET_PRIVATE(self);

  self->priv->tree       = gtk_tree_view_new();
  self->priv->scrollwin  = gtk_scrolled_window_new (NULL, NULL);
  self->priv->child_type = NULL;
  self->priv->message    = NULL;
  self->priv->selected   = FALSE;

  gtk_box_pack_start(GTK_BOX(self), self->priv->scrollwin, TRUE, TRUE, 0);
  gtk_container_add(GTK_CONTAINER(self->priv->scrollwin), self->priv->tree); 

  scrollwin = self->priv->scrollwin;
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrollwin),
				 GTK_POLICY_AUTOMATIC,
				 GTK_POLICY_AUTOMATIC);
  gtk_scrolled_window_set_shadow_type(GTK_SCROLLED_WINDOW(scrollwin),
				      GTK_SHADOW_IN);

  selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(self->priv->tree));
  gtk_tree_selection_set_mode (selection, GTK_SELECTION_SINGLE);

  //  gtk_widget_show(GTK_WIDGET(self));
  gtk_widget_show(self->priv->scrollwin);
  gtk_widget_show(self->priv->tree);

  self->priv->dispose_has_run = FALSE;
}


static void 
rubrica_view_set_property (GObject* obj, guint property_id,
			   const GValue* value, GParamSpec* spec)
{
  RubricaView* self = (RubricaView*) obj;
  RubricaViewPrivate* priv = RUBRICA_VIEW_GET_PRIVATE(self);

  switch (property_id) 
    {
    case VIEW_CHILD_TYPE:
      g_free(priv->child_type); 
      
      priv->child_type = g_value_dup_string(value);
      break;

    case VIEW_MSG:
      if (priv->message)
	g_free(priv->message);
      
      priv->message = g_value_dup_string(value);
      break;

    case VIEW_CHILD_SELECTED:
      priv->selected = g_value_get_boolean(value);
      break;

    default: 
      break; 
    } 
} 
 

static void 
rubrica_view_get_property (GObject* obj, guint property_id,
			   GValue* value, GParamSpec* spec)
{
  RubricaView* self = (RubricaView*) obj;
  RubricaViewPrivate* priv = RUBRICA_VIEW_GET_PRIVATE(self);

  switch (property_id) 
    {    
    case VIEW_CHILD_TYPE:
      g_value_set_string(value, priv->child_type);
      break;      
      
    case VIEW_MSG:
      g_value_set_string(value, priv->message);
      break;   

    case VIEW_CHILD_SELECTED:
      g_value_set_boolean(value, priv->selected);
      break;

    default:
      break;  
    }  
}




/*   Public
*/
/**
 * rubrica_view_new
 *
 * create a #RubricaView
 *
 * returns: a new #RubricaView*
 */
GtkWidget*
rubrica_view_new()
{
  GtkWidget* view;

  view = GTK_WIDGET(g_object_new(RUBRICA_VIEW_TYPE, NULL));

  return view;
}



void 
rubrica_view_set_model(RubricaView* view, GtkTreeModel* model)
{
  RubricaViewPrivate* priv;
  
  g_return_if_fail(IS_RUBRICA_VIEW(view));

  priv = RUBRICA_VIEW_GET_PRIVATE(view);

  gtk_tree_view_set_model(GTK_TREE_VIEW(priv->tree), GTK_TREE_MODEL(model));
}


GtkWidget* 
rubrica_view_get_tree(RubricaView* view)
{
  RubricaViewPrivate* priv;

  g_return_val_if_fail(IS_RUBRICA_VIEW(view), NULL);

  priv = RUBRICA_VIEW_GET_PRIVATE(view);

  return priv->tree;
}


GtkTreeSelection*
rubrica_view_get_selection(RubricaView* view)
{
  RubricaViewPrivate* priv;

  g_return_val_if_fail(IS_RUBRICA_VIEW(view), NULL);

  priv = RUBRICA_VIEW_GET_PRIVATE(view);

  return gtk_tree_view_get_selection(GTK_TREE_VIEW(priv->tree));
}

 
GtkTreeModel* 
rubrica_view_get_model (RubricaView* view)
{
  RubricaViewPrivate* priv;

  g_return_val_if_fail(IS_RUBRICA_VIEW(view), NULL);

  priv = RUBRICA_VIEW_GET_PRIVATE(view);

  return gtk_tree_view_get_model(GTK_TREE_VIEW(priv->tree));
}
