/****************************************************************************
    Copyright (C) 1987-2005 by Jeffery P. Hansen

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    Last edit by hansen on Sun Sep 10 23:03:13 2006
****************************************************************************/
#include "tkgate.h"

static char *unitCodes[] = {"s", "ms", "us", "ns", "ps", "fs"};
static int numUnitCodes  = sizeof(unitCodes)/sizeof(unitCodes[0]);

void Timescale_decode(simtime_t n, int *num,char *units)
{
  int index = 0;

  while (n >= 1000) {
    n /= 1000;
    index++;
  }

  if (index >= numUnitCodes || (n != 1 && n != 10 && n != 100)) {
    /* Invalid code, so just default to 1ns */
    *num = 1;
    strcpy(units,"ns");
    return;
  }


  strcpy(units,unitCodes[numUnitCodes-1-index]);
  *num = n;
}

simtime_t Timescale_parse(int num,const char *units)
{
  unsigned long long U = 1;
  int i;

  for (i = numUnitCodes-1;i >= 0;i--) {
    if (strcmp(units, unitCodes[i]) == 0) break;
    U *= 1000;
  }

  if (i < 0) {
    return 0;
  }

  if (num != 1 && num != 10 && num != 100) {
    return 0;
  }

  U *= num;

  return U;
}

void Timescale_save(Timescale *ts, FILE *f)
{
  int n1,n2;
  char u1[STRMAX],u2[STRMAX];

  Timescale_decode(ts->ts_units, &n1, u1);
  Timescale_decode(ts->ts_precision, &n2, u2);

  fprintf(f,"`timescale %d%s/%d%s\n",n1,u1,n2,u2);
}

Circuit *new_Circuit()
{
  Circuit *C = (Circuit*) ob_malloc(sizeof(Circuit),"Circuit");
  extern Timescale defaultTimescale;

  C->currentFile = new_CurrentFile();
  C->fileVersion = ob_strdup("");
  C->title = 0;
  C->c_timescale = defaultTimescale;
  C->mid_mod = new_GModuleDef("<interfaces>");		/* Buffer for all interfaces */
  C->mid_display = new_GModuleDef("<display>");		/* Temporary buffer for editing an interface */
  C->mid_altMod = new_GModuleDef("<alternate>");	/* Buffer for all alternate interfaces */
  C->c_breakpoints = new_NHash();
  C->c_scripts = new_NHash();
  C->cut_buffer = 0;
  C->mg_selection = 0;
  C->numInitScripts = 0;
  C->initScripts = 0;
  C->zoom_factor = 1;
  C->discardChanges = 0;
  C->useExtBars = 1;
  C->simAutoStart = 0;
  C->modified_flags = 0;
  C->search = new_GSearchContext();
  C->es = new_EditState();
  C->moduleTable = new_SHash();
  C->c_gatePrefix = TKGATE_DEFAULT_PREFIX;
  C->simClockMode = 0;
  C->simClockName = 0;
  C->labelsel = new_GrabbedLabel();
  C->rot = 0;
  C->mode = MODE_MOVE;

  return C;
}

void Circuit_setCurrentFile(const char *name)
{
  CurrentFile_set(XGate.circuit->currentFile,name);
}

void Circuit_setCurrentFileVersion(const char *ver)
{
  ob_touch(XGate.circuit);
  if (XGate.circuit->fileVersion)
    ob_free(XGate.circuit->fileVersion);
  XGate.circuit->fileVersion = ob_strdup(ver);
}

void Circuit_setTitle(const char *title)
{
  ob_touch(XGate.circuit);
  if (XGate.circuit->title) ob_free(XGate.circuit->title);
  XGate.circuit->title = title ? ob_strdup(title) : 0;
}

void Circuit_setScripts(int nScripts,const char **scripts)
{
  int i;

  ob_touch(XGate.circuit);
  if (XGate.circuit->initScripts)
    ob_free(XGate.circuit->initScripts);

  XGate.circuit->numInitScripts = nScripts;
  if (nScripts == 0) {
    XGate.circuit->initScripts = 0;
    return;
  }
  XGate.circuit->initScripts = (char**) ob_malloc(sizeof(char**)*nScripts,"char**");
  for (i = 0;i < nScripts;i++)
    XGate.circuit->initScripts[i] = ob_strdup(scripts[i]);
}

#if 0
void Circuit_setLibraries(int nLibraries,const char **libraries)
{
  int i;

  ob_touch(XGate.circuit);
  if (XGate.circuit->libraries)
    ob_free(XGate.circuit->libraries);

  XGate.circuit->numLibraries = nLibraries;
  if (nLibraries == 0) {
    XGate.circuit->libraries = 0;
    return;
  }
  XGate.circuit->libraries = (char**) ob_malloc(sizeof(char**)*nLibraries,"char**");
  for (i = 0;i < nLibraries;i++)
    XGate.circuit->libraries[i] = ob_strdup(libraries[i]);
}
#endif

void Circuit_setClockName(const char *name)
{
  Circuit *C = XGate.circuit;
  char *p;

  ob_touch(C);
  if (C->simClockName) ob_free(C->simClockName);
  C->simClockName = 0;

  if (!name || !*name) return;

  C->simClockName = ob_strdup(name);

  for (p = C->simClockName;*p;p++) {
    if (!isalpha(*p) && !isdigit(*p) && !strchr("_.",*p))
      memmove(p,p+1,strlen(p));
  }

  if (!*C->simClockName) {
    ob_free(C->simClockName);
    C->simClockName = 0;
  }
}

/*****************************************************************************
 *
 * Change the root/top module.
 *
 *****************************************************************************/
void Circuit_changeRoot(GModuleDef *newR)
{
  GModuleDef *oldR = XGate.circuit->root_mod;

  /*
   * No change in root.
   */
  if (newR == oldR)
    return;

  ob_touch(XGate.circuit);
  XGate.circuit->root_mod = newR;

  /*
   * The current root shouldn't have an interface, but if it does, delete it.
   */
  if (oldR->m_interface) {
    gate_delete(oldR->m_interface, XGate.circuit->mid_mod,0);
    ob_touch(oldR);
    oldR->m_altInterface = 0;
  }

  /*
   * If the current root module has an alternate interface, move it to be the primary interface.
   */
  if (oldR->m_altInterface) {
    GCElement *g;

    ob_touch(oldR);
    g = (*oldR->m_altInterface->typeinfo->CopyGate)(XGate.circuit->mid_mod,oldR->m_altInterface,0,0,0);
    gate_delete(oldR->m_altInterface, XGate.circuit->mid_altMod,0);
    oldR->m_interface = g;
  } else
    modint_setInterface(oldR,0);

  /*
   * If the new root has an alternate interface, delete it.
   */
  if (newR->m_altInterface) {
    gate_delete(newR->m_altInterface, XGate.circuit->mid_altMod,0);
    ob_touch(newR);
    newR->m_interface = 0;
  }

  /*
   * If the new root module has an interface, move it to be the alternate interface.
   */
  if (newR->m_interface) {
    GCElement *g;

    ob_touch(newR);
    g = (*newR->m_interface->typeinfo->CopyGate)(XGate.circuit->mid_altMod,newR->m_interface,0,0,0);
    gate_delete(newR->m_interface, XGate.circuit->mid_mod,0);
    newR->m_altInterface = g;
  }

  if (XGate.circuit->es->isInterface) {
    modint_arrange(XGate.circuit->es);
  }

  SetModified(MF_MODULE);
}

void Circuit_clear()
{
  Circuit *c = XGate.circuit;
  extern Timescale defaultTimescale;

  BrkPtTable_flush(c->c_breakpoints);
  c->c_timescale = defaultTimescale;
}

/*****************************************************************************
 *
 * Test to see if there is anything selected in a circuit.
 *
 *****************************************************************************/
int Circuit_isSelection(Circuit *c)
{
  if (XGate.circuit->select || XGate.circuit->mg_selection || XGate.circuit->wnsel)
    return 1;
  else
    return 0;
}



void Circuit_loadLibrary(Circuit *c,const char *name)
{
  if (tkgate_currentMode() != MM_EDIT)
    tkgate_setMajorMode(MM_EDIT);

  if (VerilogOpenLibrary(name) < 0) {
    message(1,msgLookup("err.badlibopen"),name);		/* Unable to open library file '%s'. */
    return;
  }

  if (c->es->isInterface)
    modint_arrange(c->es);

  FlagRedraw();
}

void Circuit_unloadLibrary(Circuit *c,const char *name)
{
  HashElem *he;
  NHash *dhash;

  /*
   * Make sure "name" is a loaded library.
   */
  if (!SHash_find(XGate.libraries, name))
    return;

  dhash = new_NHash_noob();

  SHash_remove(XGate.libraries, name);

  /*
   * Get list of modules to be deleted.
   */
  for (he = Hash_first(XGate.circuit->moduleTable);he;he = Hash_next(XGate.circuit->moduleTable,he)) {
    GModuleDef *M = (GModuleDef*) HashElem_obj(he);
    if (M->m_isLib && strcmp(M->m_libName,name) == 0) {
      NHash_insert(dhash,(int)M,M);
    }
  }

  /*
   * Delete modules found in library
   */
  for (he = Hash_first(dhash);he;he = Hash_next(dhash,he)) {
    GModuleDef *M = (GModuleDef*) HashElem_obj(he);
    ob_touch(M);
    M->m_isLib = 0;
    M->m_protIntf = 0;
    M->m_protEdit = 0;
    env_delete(XGate.circuit->es,M->m_name);
  }
  delete_NHash(dhash);
}

void Circuit_unloadAllLibraries(Circuit *c)
{
  HashElem *he;
  const char *names[Hash_numElems(XGate.libraries)+1];
  int n,i;

  for (he = Hash_first(XGate.libraries), n = 0;he;he = Hash_next(XGate.libraries,he))
    names[n++] = SHashElem_key(he);

  for (i = 0;i < n;i++)
    Circuit_unloadLibrary(c, names[i]);
}


GrabbedLabel *new_GrabbedLabel()
{
  GrabbedLabel *gl = (GrabbedLabel *) ob_malloc(sizeof(GrabbedLabel),"GrabbedLabel");

  gl->net = 0;
  gl->label = 0;

  return gl;
}

void GrabbedLabel_draw(int x,int y)
{
  GrabbedLabel *gl = XGate.circuit->labelsel;
  GC gc;

  if (gl->net->n_nbits > 1)
    gc = XGate.selBusGC;
  else
    gc = XGate.selWireGC;

  if (gl->net)
    dce_DrawString(gc,x + gl->ox,y + gl->oy,gl->position,gl->label);
}

void GrabbedLabel_unset()
{
  GrabbedLabel *gl = XGate.circuit->labelsel;
  gl->net = 0;
  if (gl->label) ob_free(gl->label);
  gl->label = 0;
}

void GrabbedLabel_set(GNet *net,int ox,int oy,int p)
{
  GrabbedLabel *gl = XGate.circuit->labelsel;
  char label[STRMAX];

  GNet_getDisplayLabel(net,label,STRMAX, DLM_GET_ALWAYS);
  gl->net = net;
  gl->label = ob_strdup(label);
  gl->ox = ox;
  gl->oy = oy;
  gl->position = p;

}

