/****************************************************************************
    Copyright (C) 1987-2005 by Jeffery P. Hansen

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    Last edit by hansen on Sat Jan 27 17:43:32 2007
****************************************************************************/
#include "tkgate.h"

#define DEMUX_IN 0
#define DEMUX_EN 1
#define DEMUX_OUT 2

void Demux_Delete(GCElement *g,GModuleDef *env,int drawp);
void Demux_Draw(GCElement *g,int md);
GCElement *Demux_Make(EditState **es,GModuleDef *env,int GType,
		    int x,int y,int r,const char *Name,int noWires,const char**,int);
void Demux_PSWrite(GPrint *P,GModLayout*,GCElement *g);
void Demux_Rotate(GCElement *g, int centX, int centY,int rdir);
GCElement *MuxDemux_Copy(GModuleDef *M,GCElement *g,int x,int y,unsigned flags);
void Demux_SetProp(GCElement*,const char*,const void*);
void MuxDemux_EditProps(GCElement *g,int isLoadDialog);
void Demux_VersionDelta(GCElement*,Version*); 
GWireNode *Demux_wireSnap(GCElement *g,GWire *w,int *mod,int retry);
void Demux_AddOutput(EditState *es,GCElement *g);
void MuxDemux_VerSave(FILE*,GCElement*);
static void Demux_WriteCellDef(FILE *f,GCellSpec *gcs);

static iconDimensions demux_iconDims[] = {
  {29, 60, 59, 29, 30, 12},
  {0,  29, 29, 59, 12, 30},
  {0,  0,  59, 29, 30, 15},
  {60, 0,  29, 59, 15, 30},
};
static int demux_iconBoldOffset = 89;

GPadLoc demux_out_loc[] = {
	{-30,16,30,16,D_DOWN},
	{16,-30,16,30,D_RIGHT},
	{-30,-16,30,-16,D_UP},
	{-16,-30,-16,30,D_LEFT},
};

GPadLoc demux_in_loc[] = {
	{0,-13,0,-13,D_UP},
	{-13,0,-13,0,D_LEFT},
	{0,13,0,13,D_DOWN},
	{13,0,13,0,D_RIGHT},
};

GPadLoc demux_en_loc[] = {
	{-24,0,-24,0,D_LEFT},
	{0,22,0,22,D_DOWN},
	{22,0,22,0,D_RIGHT},
	{0,-24,0,-24,D_UP},
};

GPadLoc demux_en_altloc[] = {
	{22,0,22,0,D_RIGHT},
	{0,-24,0,-24,D_UP},
	{-24,0,-24,0,D_LEFT},
	{0,22,0,22,D_DOWN},
};

static char *psDemux[] = {
  "%",
  "% x y r demux",
  "%",
  "/demux {",
  "    180 sub neg 360 mod neg",
  "    startgate",
  "    8 rfont",
  "    -29.5 15.5 moveto",
  "    29.5 15.5 lineto",
  "    16.5 -12.5 lineto",
  "    -16.5 -12.5 lineto",
  "    closepath stroke",
  "    dup					% n n",
  "    1 add 58 exch div			% n d1",
  "    2 copy mul				% n d1 dn",
  "    3 -1 roll 1 sub 50 string cvs exch	% d1 (n) dn",
  "    -29 add 7 rCT				% d1",
  "    (0) exch -29 add 7 rCT",
  "    grestore",
  "} def",
  0
};

GGateInfo gate_demux_info = {
  0,
  "DEMUX",
  "demux",0x0,
  "demux",psDemux,
  2,2,

 { {"D 2",	{"gm.msi",0},		{"gm.msi.12dec",0,"xdemux",100},		"gat_make DEMUX "},
   {"D 3",	{0,0},			{0,0,0,0},				"gat_make DEMUX -pins Z=3"},
   {"D 4",	{"gm.msi",0},		{"gm.msi.14dec",0,"xdemux",200},		"gat_make DEMUX -pins Z=4"},
   {"D 5",	{0,0},			{0,0,0,0},				"gat_make DEMUX -pins Z=5"},
   {"D 6",	{0,0},			{0,0,0,0},				"gat_make DEMUX -pins Z=6"},
   {"D 7",	{0,0},			{0,0,0,0},				"gat_make DEMUX -pins Z=7"},
   {"D 8",	{"gm.msi",0},		{"gm.msi.18dec",0,"xdemux",300},		"gat_make DEMUX -pins Z=8"},
   {"D 9",	{0,0},			{0,0,0,0},				"gat_make DEMUX -pins Z=9"},
   {0}},

  demux_iconDims,

  3,{{"I",IN,1,1,demux_in_loc,0},
       {"E",IN,1,1,demux_en_loc,0},
       {"Z",OUT,1,2,demux_out_loc,1}},
  {{23,0,LJ},{0,40,CT},{-23,8,RJ},{0,-33,CT}},
  {1},
  
  {"Dez","Diz",0},

  Demux_Make,
  Demux_WriteCellDef,
  Generic_Init,
  Demux_Delete,
  Generic_GetExtents,
  Generic_HitDistance,
  Demux_Draw,
  Generic_Move,
  MuxDemux_Copy,
  Err_AddInput,
  Demux_AddOutput,
  Err_AddInOut,
  Demux_Rotate,
  Err_RemovePort,
  Err_ChangePin,
  Nop_SimStateFunc,
  Nop_SimHitFunc,
  Demux_PSWrite,
  MuxDemux_EditProps,
  MuxDemux_VerSave,
  Demux_SetProp,

  Demux_VersionDelta,
  Demux_wireSnap
};

/*
 * Adjust the wires on a mux to conform to the current setting of data_order
 */
static void Demux_adjustWiresData(GCElement *g)
{
  int N,L,i;
  double dx,dy,wx,wy;
  GWire *w;
  GPadLoc *pd = &g->typeinfo->Pad[DEMUX_OUT].Loc[g->orient];


  wx = wy = dx = dy = 0;
  N = wire_numOnPad(g->wires[DEMUX_OUT]);
  L = iabs(pd->x1-pd->x2)  + iabs(pd->y1-pd->y2); 	/* Length of pad (assumed non-diagonal) */

  wx = g->xpos + pd->x1;
  wy = g->ypos + pd->y1;

  switch (g->orient) {
  case 0 :
    dx = (double)L/(double)(N+1);
    if (g->u.mux.data_order) {
      wx = g->xpos + pd->x2;
      dx = -dx;
    }
    break;
  case 1 :
    dy = (double)L/(double)(N+1);
    if (!g->u.mux.data_order) {
      wy = g->ypos + pd->y2;
      dy = -dy;
    }
    break;
  case 2 :
    dx = (double)L/(double)(N+1);
    if (!g->u.mux.data_order) {
      wx = g->xpos + pd->x2;
      dx = -dx;
    }
    break;
  case 3 :
    dy = (double)L/(double)(N+1);
    if (g->u.mux.data_order) {
      wy = g->ypos + pd->y2;
      dy = -dy;
    }
    break;
  }
  
  for (i = 0,w = g->wires[DEMUX_OUT];w;i++, w = w->next) {
    wx += dx;
    wy += dy;
    wire_move(w->nodes,(int)wx-w->nodes->x,(int)wy-w->nodes->y,VERTICAL|HORIZONTAL);
  }
}

/*
 * Adjust the wires on a mux to conform to the current setting of select_side
 */
static void Demux_adjustWiresSelector(GCElement *g)
{
  int wx,wy;
  GPadLoc *pd = &g->typeinfo->Pad[DEMUX_EN].Loc[g->orient];
  GWire *w = g->wires[DEMUX_EN];
  GWire *ow = wire_other(w);

  if (g->u.mux.select_side)
    pd = &demux_en_altloc[g->orient];

  wx = g->xpos + pd->x2;
  wy = g->ypos + pd->y2;

  ob_touch(w);
  wire_move(w->nodes,wx-w->nodes->x,wy-w->nodes->y,VERTICAL|HORIZONTAL);

  /*
   * If wire end is not attached, compute mirror image.  Otherwise use brute force
   * to fix the wire.
   */
  if (w->orient != pd->dir) {
    w->orient = pd->dir;
    if (!ow->gate) {
      int cx = w->nodes->x;
      int cy = w->nodes->y;
      GWireNode *n;


      for (n = w->driver->nodes;n;n = n->out) {
	ob_touch(n);
	if (g->orient == 0 || g->orient == 2)
	  n->x = cx - (n->x - cx);
	else
	  n->y = cy - (n->y - cy);
      }
    } else
      GWire_snap(w->driver);
  }
}

/*
 * Adjust the wires on a mux.
 */
void Demux_adjustWires(GCElement *g)
{
  Demux_adjustWiresData(g);
  Demux_adjustWiresSelector(g);
}


GCElement *Demux_Make(EditState **es,GModuleDef *env,int GType,
		    int x,int y,int r,const char *Name,int noWires,const char **options,int nOptions)
{
  int nbits;
  GCElement *g;
  GWire *w;

  if (!(g = Generic_Make(es,env,GType,x,y,r,Name,1,options,nOptions)))
    return NULL;

  g->u.mux.select_side = 0;
  g->u.mux.data_order = 0;

  if (!noWires) {
    int i,j;
    const char *Invert,*Pins;
    int N = GCElement_numPads(g);

    
    Invert = seekOption("-invert",options,nOptions);
    Pins = seekOption("-pins",options,nOptions);

    for (i = 0;i < N;i++) {
      int Num = 0;

      if (Pins) {
	char buf[STRMAX];
	int N;
	if (sscanf(Pins,"%[^=]=%d",buf,&N) == 2 && strcmp(buf,GCElement_getPadName(g,i)) == 0)
	  Num = N;
      }
      if (!Num)
	Num = GCElement_getPadNum(g,i);

      for (j = 0;j < Num;j++) {
	if (Invert && strcmp(Invert,GCElement_getPadName(g,i)) == 0)
	  wire_addToGate(g,i,env,1);
	else
	  wire_addToGate(g,i,env,0);
      }
    }

    Demux_adjustWires(g);


    for (i = 0,w = g->wires[DEMUX_OUT];w;w = w->next,i++);

    nbits = required_bits(i);
    net_setSize(g->wires[DEMUX_IN]->net,nbits);
  }

  return g;
}

void Demux_Draw_Special(GCElement *g)
{
  GWire *w,*w0;
  char n[STRMAX];
  int i;

  if (g->selected)
    XSetFont(XGate.D,XGate.instGC,XGate.textbXF[XGate.circuit->zoom_factor]);
  else
    XSetFont(XGate.D,XGate.instGC,XGate.textXF[XGate.circuit->zoom_factor]);

  dce_DrawString(XGate.instGC,g->xpos,g->ypos+4,AtBottom|BetweenLeftAndRight,"D");

  if (!(w0 = g->wires[DEMUX_OUT])) return;
  for (i = 0,w = w0;w->next;w = w->next,i++);
  sprintf(n,"%d",i);

  if (g->selected)
    XSetFont(XGate.D,XGate.instGC,XGate.stextbXF[XGate.circuit->zoom_factor]);
  else
    XSetFont(XGate.D,XGate.instGC,XGate.stextXF[XGate.circuit->zoom_factor]);

  switch (g->orient) {
  case 0 :
    dce_DrawString(XGate.instGC,
		   w0->nodes->x,w0->nodes->y - 2,
		   AtBottom|BetweenLeftAndRight,"0");
    dce_DrawString(XGate.instGC,
		   w->nodes->x,w->nodes->y - 2,
		   AtBottom|BetweenLeftAndRight,n);
    break;
  case 1 :
    dce_DrawString(XGate.instGC,
		   w0->nodes->x - 2,w0->nodes->y,
		   BetweenTopAndBottom|AtRight,"0");
    dce_DrawString(XGate.instGC,
		   w->nodes->x - 2,w->nodes->y,
		   BetweenTopAndBottom|AtRight,n);
    break;
  case 2 :
    dce_DrawString(XGate.instGC,
		   w0->nodes->x,w0->nodes->y + 2,
		   AtTop|BetweenLeftAndRight,"0");
    dce_DrawString(XGate.instGC,
		   w->nodes->x,w->nodes->y + 2,
		   AtTop|BetweenLeftAndRight,n);
    break;
  case 3:
    dce_DrawString(XGate.instGC,
		   w0->nodes->x + 3,w0->nodes->y,
		   BetweenTopAndBottom|AtLeft, "0");
    dce_DrawString(XGate.instGC,
		   w->nodes->x + 3,w->nodes->y,
		   BetweenTopAndBottom|AtLeft,n);
    break;
  }
}


void Demux_Delete(GCElement *g,GModuleDef *env,int drawp)
{
  Generic_Delete(g,env,drawp);
}

void Demux_Draw(GCElement *g,int md)
{
  Generic_Draw(g,md);
  Demux_Draw_Special(g);
}


void Demux_Rotate(GCElement *g, int centX, int centY,int rdir)
{
  int x = g->xpos;
  int y = g->ypos;

  ob_touch(g);
  g->xpos = rotateX(x - centX,y - centY,rdir) + centX;
  g->ypos = rotateY(x - centX,y - centY,rdir) + centY;
  g->orient = (g->orient + 4 + rdir) % 4;

  Demux_adjustWires(g);
}


void Demux_PSWrite(GPrint *P,GModLayout *L,GCElement *g)
{
    int N;
    struct wire *w;

    Generic_PSLabels(P,g);

    for (N = 0, w = g->wires[DEMUX_OUT];w;w = w->next) N++;

    fprintf(P->p_f,"%d %d %d %d %s\n",
	N,g->xpos,g->ypos,-g->orient*90,g->typeinfo->psprint);
}


void Demux_SetProp(GCElement *g,const char *prop,const void *value)
{
  if (strcmp(prop,"/ss") == 0) g->u.mux.select_side = *((int*)value);
  if (strcmp(prop,"/do") == 0) g->u.mux.data_order = *((int*)value);

  if (g->wires[DEMUX_EN])
    g->wires[DEMUX_EN]->orient = (g->orient + 2 + (g->u.mux.select_side ? 2 : 0) ) % 4;
}

void Demux_VersionDelta(GCElement *g,Version *V)
{
  Version x1 = {"1.8.7", 1, 8, 7};

  if (VersionCmp(V,&x1) >= 0)
    return;

  switch (g->orient) {
  case D_RIGHT :
    g->u.mux.select_side = 0;
    g->u.mux.data_order = 0;
    g->wires[DEMUX_EN]->orient = D_LEFT;
    break;
  case D_UP :
    g->u.mux.select_side = 1;
    g->u.mux.data_order = 1;
    g->wires[DEMUX_EN]->orient = D_UP;
    break;
  case D_LEFT :
    g->u.mux.select_side = 0;
    g->u.mux.data_order = 1;
    g->wires[DEMUX_EN]->orient = D_RIGHT;
    break;
  case D_DOWN :
    g->u.mux.select_side = 1;
    g->u.mux.data_order = 0;
    g->wires[DEMUX_EN]->orient = D_DOWN;
    break;
  }
}

GWireNode *Demux_wireSnap(GCElement *g,GWire *w,int *mod,int retry)
{
  int p,n;

  if (posongate(w,w->gate,&p,&n) == 0) {
    GPadLoc *pd = &g->typeinfo->Pad[p].Loc[g->orient];
    if (p == DEMUX_EN && g->u.mux.select_side)
      pd = &demux_en_altloc[g->orient];
  
    *mod = wire_force(w,pd->dir,retry);
  }
  return w->nodes;
}

void Demux_AddOutput(EditState *es,GCElement *g)
{
  int i;
  int N = GCElement_numPads(g);

  for (i = 0;i < N;i++)
    if (GCElement_getPadDir(g,i) == OUT && GCElement_getPadCanAdd(g,i))
      break;
  
  if (i == N)
    return;
  
  if (es) SetModified(MF_GATE|MF_NET);
  
  gate_draw(g,GD_NORMAL);
  wire_addToGate(g,i,es->env,0);
  Demux_adjustWires(g);
  gate_draw(g,GD_NORMAL);
}

/*****************************************************************************
 * 
 * Generate primitive cell definition for demuxes.
 *
 * Parameters:
 *    f			File to write cell to.
 *    name		Name of cell to write.
 *
 *****************************************************************************/
static void Demux_WriteCellDef(FILE *f,GCellSpec *gcs)
{
  int multiPad = gcs->gc_multiPad;
  /*  int numBit = gcs->gc_numBits;*/
  /* int *delays = gcs->gc_delays;*/
  const char *invSpec = gcs->gc_invSpec;
  const char *iq = invSpec;
  int j;

  GCellSpec_writeBeginModule(f,gcs);

  fprintf(f,"input%s I;\n", bitrangeSpec(required_bits(multiPad)));
  fprintf(f,"input E;\n");
  fprintf(f,"output ");
  for (j = 0;j < multiPad;j++) {
    if (j > 0) fprintf(f,", ");
    fprintf(f,"Z%d",j);
  }
  fprintf(f,";\n");
  fprintf(f,"wire%s Zcat;\n",bitrangeSpec(multiPad));

  fprintf(f,"  assign Zcat = %d'b1 << I;\n",multiPad);

  for (j = 0;j < multiPad;j++) {
    int isInv = (*iq == 'N');
    fprintf(f,"  assign #1 Z%d = %s(E & Zcat[%d]);\n",j,(isInv?"~":""),j);
    if (*iq && iq[1]) iq++;
  }

  GCellSpec_writeEndModule(f,gcs);
}

void init_demux()
{
  Pixmap P;

  P = Pixmap_find("mux");
  gateinfo_iconInit(&gate_demux_info,P,demux_iconDims,demux_iconBoldOffset);
  RegisterGate(&gate_demux_info);
}
