/****************************************************************************
    Copyright (C) 1987-2005 by Jeffery P. Hansen

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    Last edit by hansen on Thu Aug  2 22:43:43 2007
****************************************************************************/
/*

This module implements a subset of html for use in tkgate comments.

The implemented elements are:
  <value-of name="name">			Special variable value
  <a href="url"></a>				Hyperlink
  <a name="tag">				Hyperlink tag
  <b></b>					Bold
  <i></i>					Italic
  <h3></h3>					Heading 3
  <h2></h2>					Heading 2
  <h1></h1>					Heading 1
  <small></small>				Small font
  <big></big>					Big font
  <tt></tt>					Fixed-width font
  <pre></pre>					Preformated
  <br>						Break
  <p>						Paragraph
  <font size=n color=color face=face></font>	Set font characteristics
  <hr>						Horizontal rule
  &??;						Special character

Special pre-processed tags:
  <tutorial-navigation>                         Generates html for navigation bar in tutorials

 */
#include <stdio.h>
#include <ctype.h>
#include <sys/time.h>
#include <stdarg.h>
#include "tkgate.h"


static int html_isPostscript = 0;	/* Non-zero if we are formatting postscript */

/*
 * Step size for extending the array of html tag options.
 */
#define HTML_OSTEP 8

void expandSpecialDirs(char *file);

typedef void HtmlHandlerFunc(Html *h,HtmlTag *tag);

/*
 * Private class for specifying functions to handle html tags
 */
typedef struct {
  char			*tag;		/* Tag */
  HtmlHandlerFunc	*func;	        /* Handling function for tag */
} HtmlHandler;

typedef struct {
  char			*spec;		/* Special character specifier */
  char			*text;		/* Raw text for replacement */
  char			*pretag;	/* Html code to preceed text */	
  char			*posttag;	/* Html code to follow text */	
} HtmlSpecialSpec;

void Html_handle_basic(Html *h, HtmlTag *tag);
void Html_handle_valueOf(Html *h, HtmlTag *tag);
void Html_handle_a(Html *h, HtmlTag *tag);
void Html_handle_heading(Html *h, HtmlTag *tag);
void Html_handle_img(Html *h, HtmlTag *tag);

/*
 * Table of html tags that are recognized by tkgate
 */
HtmlHandler htmlHandlers[] = {
  {"a",		Html_handle_a },
  {"b",		Html_handle_basic },
  {"tt",	Html_handle_basic },
  {"i",		Html_handle_basic },
  {"big",	Html_handle_basic },
  {"small",	Html_handle_basic },
  {"font",	Html_handle_basic },
  {"h1",	Html_handle_heading },
  {"h2",	Html_handle_heading },
  {"h3",	Html_handle_heading },
  {"value-of",	Html_handle_valueOf },
  {"img",	Html_handle_img },
  {0, 0}
};

/*
 * Table of html special characters that are recognized by tkgate
 */
HtmlSpecialSpec htmlSpecialSpecs[] = {
  {"&nbsp;",	" ",	0,	0},
  {"&gt;",	">",	0,	0},
  {"&lt;",	"<",	0,	0},
  {"&amp;",	"&",	0,	0},


  {"&Alpha;",	"A",	"<font face=symbol>",	"</font>"},
  {"&Beta;",	"B",	"<font face=symbol>",	"</font>"},
  {"&Gamma;",	"G",	"<font face=symbol>",	"</font>"},
  {"&Delta;",	"D",	"<font face=symbol>",	"</font>"},
  {"&Epsilon;",	"E",	"<font face=symbol>",	"</font>"},
  {"&Zeta;",	"Z",	"<font face=symbol>",	"</font>"},
  {"&Eta;",	"H",	"<font face=symbol>",	"</font>"}, 	
  {"&Theta;",	"Q",	"<font face=symbol>",	"</font>"}, 
  {"&Iota;",	"I",	"<font face=symbol>",	"</font>"}, 	
  {"&Kappa;",	"K",	"<font face=symbol>",	"</font>"}, 
  {"&Lambda;",	"L",	"<font face=symbol>",	"</font>"},
  {"&Mu;",	"M",	"<font face=symbol>",	"</font>"}, 	
  {"&Nu;",	"N",	"<font face=symbol>",	"</font>"}, 	
  {"&Xi;",	"X",	"<font face=symbol>",	"</font>"}, 	
  {"&Omicron;",	"O",	"<font face=symbol>",	"</font>"},
  {"&Pi;",	"P",	"<font face=symbol>",	"</font>"}, 	
  {"&Rho;",	"R",	"<font face=symbol>",	"</font>"}, 	
  {"&Sigma;",	"S",	"<font face=symbol>",	"</font>"}, 
  {"&Tau;",	"T",	"<font face=symbol>",	"</font>"}, 	
  {"&Upsilon;",	"U",	"<font face=symbol>",	"</font>"},
  {"&Phi;",	"F",	"<font face=symbol>",	"</font>"}, 	
  {"&Chi;",	"C",	"<font face=symbol>",	"</font>"}, 	
  {"&Psi;",	"Y",	"<font face=symbol>",	"</font>"}, 	
  {"&Omega;",	"W",	"<font face=symbol>",	"</font>"}, 
  {"&alpha;",	"a",	"<font face=symbol>",	"</font>"}, 
  {"&beta;",	"b",	"<font face=symbol>",	"</font>"}, 	
  {"&gamma;",	"g",	"<font face=symbol>",	"</font>"}, 
  {"&delta;",	"d",	"<font face=symbol>",	"</font>"}, 
  {"&epsilon;",	"e",	"<font face=symbol>",	"</font>"},
  {"&zeta;",	"z",	"<font face=symbol>",	"</font>"}, 	
  {"&eta;",	"h",	"<font face=symbol>",	"</font>"}, 	
  {"&theta;",	"q",	"<font face=symbol>",	"</font>"}, 
  {"&iota;",	"i",	"<font face=symbol>",	"</font>"}, 	
  {"&kappa;",	"k",	"<font face=symbol>",	"</font>"}, 
  {"&lambda;",	"l",	"<font face=symbol>",	"</font>"},
  {"&mu;",	"m",	"<font face=symbol>",	"</font>"}, 
  {"&nu;",	"n",	"<font face=symbol>",	"</font>"}, 
  {"&xi;",	"x",	"<font face=symbol>",	"</font>"}, 
  {"&omicron;",	"o",	"<font face=symbol>",	"</font>"}, 
  {"&pi;",	"p",	"<font face=symbol>",	"</font>"}, 	
  {"&rho;",	"r",	"<font face=symbol>",	"</font>"}, 	
  {"&sigmaf;",	"V",	"<font face=symbol>",	"</font>"},
  {"&sigma;",	"s",	"<font face=symbol>",	"</font>"},
  {"&tau;",	"t",	"<font face=symbol>",	"</font>"}, 	
  {"&upsilon;",	"u",	"<font face=symbol>",	"</font>"},
  {"&phi;",	"f",	"<font face=symbol>",	"</font>"}, 	
  {"&chi;",	"c",	"<font face=symbol>",	"</font>"}, 	
  {"&psi;",	"y",	"<font face=symbol>",	"</font>"}, 	
  {"&omega;",	"w",	"<font face=symbol>",	"</font>"},
  {"&thetasym;","J",	"<font face=symbol>",	"</font>"},
  {"&upsih;",	"\241",	"<font face=symbol>",	"</font>"}, 
  {"&piv;",	"v",	"<font face=symbol>",	"</font>"}, 

  {"&bull;",	"\267",	"<font face=symbol>",	"</font>"},
  {"&hellip;",	"\274",	"<font face=symbol>",	"</font>"},
  {"&prime;",	"\242",	"<font face=symbol>",	"</font>"},
  {"&Prime;",	"\262",	"<font face=symbol>",	"</font>"},
  {"&oline;",	"\140",	"<font face=symbol>",	"</font>"},
  {"&frasl;",	"\244",	"<font face=symbol>",	"</font>"},
  {"&weierp;",	"\303",	"<font face=symbol>",	"</font>"},
  {"&image;",	"\301",	"<font face=symbol>",	"</font>"},
  {"&real;",	"\302",	"<font face=symbol>",	"</font>"},
  {"&trade;",	"\324",	"<font face=symbol>",	"</font>"},
  {"&alefsym;",	"\300",	"<font face=symbol>",	"</font>"},
  {"&larr;",	"\254",	"<font face=symbol>",	"</font>"},
  {"&uarr;",	"\255",	"<font face=symbol>",	"</font>"},
  {"&rarr;",	"\256",	"<font face=symbol>",	"</font>"},
  {"&darr;",	"\257",	"<font face=symbol>",	"</font>"},
  {"&harr;",	"\253",	"<font face=symbol>",	"</font>"},
  {"&crarr;",	"\277",	"<font face=symbol>",	"</font>"},
  {"&lArr;",	"\334",	"<font face=symbol>",	"</font>"},
  {"&uArr;",	"\335",	"<font face=symbol>",	"</font>"},
  {"&rArr;",	"\336",	"<font face=symbol>",	"</font>"},
  {"&dArr;",	"\337",	"<font face=symbol>",	"</font>"},
  {"&hArr;",	"\333",	"<font face=symbol>",	"</font>"},
  {"&forall;",	"\042",	"<font face=symbol>",	"</font>"},
  {"&part;",	"\266",	"<font face=symbol>",	"</font>"},
  {"&exist;",	"\044",	"<font face=symbol>",	"</font>"},
  {"&empty;",	"\306",	"<font face=symbol>",	"</font>"},
  {"&nabla;",	"\321",	"<font face=symbol>",	"</font>"},
  {"&isin;",	"\316",	"<font face=symbol>",	"</font>"},
  {"&notin;",	"\317",	"<font face=symbol>",	"</font>"},
  {"&ni;",	"\267",	"<font face=symbol>",	"</font>"},	/* NOT CORRECT */
  {"&prod;",	"\325",	"<font face=symbol>",	"</font>"},
  {"&sum;",	"\345",	"<font face=symbol>",	"</font>"},
  {"&minus;",	"-",	"<font face=symbol>",	"</font>"},
  {"&lowast;",	"*",	"<font face=symbol>",	"</font>"},
  {"&radic;",	"\326",	"<font face=symbol>",	"</font>"},
  {"&prop;",	"\265",	"<font face=symbol>",	"</font>"},
  {"&infin;",	"\245",	"<font face=symbol>",	"</font>"},
  {"&ang;",	"\320",	"<font face=symbol>",	"</font>"},
  {"&and;",	"\331",	"<font face=symbol>",	"</font>"},
  {"&or;",	"\332",	"<font face=symbol>",	"</font>"},
  {"&cap;",	"\307",	"<font face=symbol>",	"</font>"},
  {"&cup;",	"\310",	"<font face=symbol>",	"</font>"},
  {"&int;",	"\362",	"<font face=symbol>",	"</font>"},
  {"&there4;",	"\134",	"<font face=symbol>",	"</font>"},
  {"&sim;",	"\176",	"<font face=symbol>",	"</font>"},
  {"&cong;",	"\100",	"<font face=symbol>",	"</font>"},
  {"&asymp;",	"\273",	"<font face=symbol>",	"</font>"},
  {"&ne;",	"\271",	"<font face=symbol>",	"</font>"},
  {"&equiv;",	"\272",	"<font face=symbol>",	"</font>"},
  {"&le;",	"\243",	"<font face=symbol>",	"</font>"},
  {"&ge;",	"\263",	"<font face=symbol>",	"</font>"},
  {"&sub;",	"\314",	"<font face=symbol>",	"</font>"},
  {"&sup;",	"\311",	"<font face=symbol>",	"</font>"},
  {"&nsub;",	"\313",	"<font face=symbol>",	"</font>"},
  {"&sube;",	"\315",	"<font face=symbol>",	"</font>"},
  {"&supe;",	"\312",	"<font face=symbol>",	"</font>"},
  {"&oplus;",	"\305",	"<font face=symbol>",	"</font>"},
  {"&otimes;",	"\304",	"<font face=symbol>",	"</font>"},
  {"&perp;",	"\267",	"<font face=symbol>",	"</font>"},	/* NOT CORRECT */
  {"&sdot;",	"\327",	"<font face=symbol>",	"</font>"},
  {"&lceil;",	"\351",	"<font face=symbol>",	"</font>"},
  {"&rceil;",	"\371",	"<font face=symbol>",	"</font>"},
  {"&lfloor;",	"\353",	"<font face=symbol>",	"</font>"},
  {"&rfloor;",	"\373",	"<font face=symbol>",	"</font>"},
  {"&lang;",	"\341",	"<font face=symbol>",	"</font>"},
  {"&rang;",	"\361",	"<font face=symbol>",	"</font>"},
  {"&loz;",	"\340",	"<font face=symbol>",	"</font>"},
  {"&spades;",	"\252",	"<font face=symbol>",	"</font>"},
  {"&clubs;",	"\247",	"<font face=symbol>",	"</font>"},
  {"&hearts;",	"\251",	"<font face=symbol>",	"</font>"},
  {"&diams;",	"\250",	"<font face=symbol>",	"</font>"},

  {0,0,0,0}
};


#define DATA_STEP_SIZE			512

static HtmlContext default_context = {
  {FF_HELVETICA, FP_ROMAN, FS_NORMAL},
  0,
  0,
  0,
  1,
  0
};


static Tk_Font tkfonts[FF_MAX][FP_MAX][FS_MAX][ZOOM_MAX+1] = {{{{0}}}};
static Font xfonts[FF_MAX][FP_MAX][FS_MAX][ZOOM_MAX+1] = {{{{0}}}};

static char *font_family_names[FF_MAX] = {
  "courier",
  "helvetica",
  "times",
  "symbol",
  "kanji",
};

static int font_sizes[FS_MAX] = {8, 8, 10, 12, 14, 18, 24};
static int kanji_font_sizes[FS_MAX] = {9, 9, 11, 13, 15, 18, 25};

int istruevalue(const char *s)
{
  int n;

  if (sscanf(s,"%d",&n) == 1) {
    return (n != 0);
  }

  if (strcasecmp(s,"t") == 0) return 1;
  if (strcasecmp(s,"y") == 0) return 1;
  if (strcasecmp(s,"true") == 0) return 1;
  if (strcasecmp(s,"yes") == 0) return 1;

  return 0;
}


int HtmlFont_isEqual(const HtmlFont *a,const HtmlFont *b)
{
  if (a->family != b->family) return 0;
  if (a->props != b->props) return 0;
  if (a->size != b->size) return 0;
  if (a->points != b->points) return 0;
  return 1;
}

void HtmlFont_updatePoints(HtmlFont *font)
{
  if (font->family == FF_KANJI)
    font->points = kanji_font_sizes[font->size];
  else
    font->points = font_sizes[font->size];
}


HtmlFont *HtmlFont_init(HtmlFont *font,fontfamily_t family,fontprop_t props,fontsize_t points)
{
  int i;

  ob_touch(font);

  font->family = family;
  font->props = props;
  font->points = points;

  if (family != FF_KANJI) {
    for (i = 0;i < FS_MAX;i++)
      if (font_sizes[i] >= points) break;
  } else {
    for (i = 0;i < FS_MAX;i++)
      if (kanji_font_sizes[i] >= points) break;
  }
  if (i >= FS_MAX) i = FS_MAX - 1;
  font->size = i;

  return font;
}


/*
   Find a font from the specified variable name.  Issue error message and
   exit if not found.
*/
static Tk_Font LoadTkFont(char *name)
{
  Tk_Window root = Tk_MainWindow(XGate.tcl);
  Tk_Font F;

  if (!(F = Tk_GetFont(XGate.tcl,root,name))) {
    fprintf(stderr,"tkgate: Can't find the font '%s'\n",name);
    exit(1);
  }

  return F;
}


Tk_Font *GetTkFonts(fontfamily_t ff,fontprop_t fp,fontsize_t fs)
{
  if (!tkfonts[ff][fp][fs][1]) {
    int z;

    for (z = 1;z <= ZOOM_MAX;z++) {
      tkfonts[ff][fp][fs][z] = GetTkFont(ff,fp,fs,z);
    }
  }

  return tkfonts[ff][fp][fs];
}

Tk_Font GetTkFont(fontfamily_t ff,fontprop_t fp,fontsize_t fs,int zoom)
{
  char fullName[STRMAX];

  if (tkfonts[ff][fp][fs][zoom] == 0) {
    if (ff == FF_KANJI) {
      sprintf(fullName,
	      "-misc-fixed-medium-r-normal--*-%d-75-75-c-*-jisx0208.1983-0",
	      kanji_font_sizes[fs]*10*zoom);
      tkfonts[ff][fp][fs][zoom] = LoadTkFont(fullName);
    } else {
      char *font_weight = (fp & FP_BOLD) ? "bold" : "medium";
      char *font_posture;

      if (ff == FF_TIMES) 
	font_posture = (fp & FP_ITALIC) ? "i" : "r";
      else
	font_posture = (fp & FP_ITALIC) ? "o" : "r";

      sprintf(fullName,
	      "-adobe-%s-%s-%s-normal--*-%d-75-75-*-%s",
	      font_family_names[ff],
	      font_weight,
	      font_posture,
	      font_sizes[fs]*10*zoom,
	      XGate.fontCode);

      tkfonts[ff][fp][fs][zoom] = LoadTkFont(fullName);
    }
  }

  return tkfonts[ff][fp][fs][zoom];
}

Font GetXFont(fontfamily_t ff,fontprop_t fp,fontsize_t fs,int zoom)
{
  return Tk_FontId(GetTkFont(ff,fp,fs,zoom));
}

Font *GetXFonts(fontfamily_t ff,fontprop_t fp,fontsize_t fs)
{
  int z;

  for (z = 1;z <= ZOOM_MAX;z++) {
    xfonts[ff][fp][fs][z] = GetXFont(ff,fp,fs,z);
  }

  return xfonts[ff][fp][fs];
}

void UnloadAllFonts()
{
  fontfamily_t ff;
  fontprop_t fp;
  fontsize_t fs;
  int z;

  for (ff = 0;ff < FF_MAX; ff++) {
    for (fp = 0;fp < FP_MAX; fp++) {
      for (fs = 0;fs < FS_MAX; fs++) {
	for (z = 1;z <= ZOOM_MAX; z++) {
	  if (tkfonts[ff][fp][fs][z])
	    Tk_FreeFont(tkfonts[ff][fp][fs][z]);
	}
      }
    }
  }
}

HtmlTag *new_HtmlTag()
{
  HtmlTag *ht = (HtmlTag*) ob_malloc(sizeof(HtmlTag),"HtmlTag");

  ht->ht_name = 0;
  ht->ht_numOptions = 0;
  ht->ht_options = 0;

  return ht;
}

void delete_HtmlTag(HtmlTag *ht)
{
  if (ht->ht_name) ob_free(ht->ht_name);
  if (ht->ht_options) {
    int i;
    
    for (i = 0;i < ht->ht_numOptions;i++) {
      ob_free(ht->ht_options[i].hto_label);
     ob_free(ht->ht_options[i].hto_value);
    }

    ob_free(ht->ht_options);
  }
  ob_free(ht);
}


/*****************************************************************************
 * Break down an options string into tags and values.  Return the
 * number of tags parsed, or -1 if there was an error.
 *****************************************************************************/
HtmlTag *Html_parseTag(const char *tag)
{
  char name[STRMAX];
  char options[STRMAX];
  char ptag[STRMAX],pvalue[STRMAX];
  char *p;
  char c;
  HtmlTag *ht = 0;

  if (sscanf(tag,"< %[^ \t\n>] %[^>] %c",name,options,&c) == 3 && c == '>') {
    /* Tag with options */
  } else if (sscanf(tag,"< %[^ \t\n>] %c",name,&c) == 2 && c == '>') {
    /* Tag with no options */
    *options = 0;
  } else {
    /* Ignore badly formatted tag */
    return 0;
  }

  ht = new_HtmlTag();
  ht->ht_name = ob_strdup(name);

  p = options;
  while (*p && isspace(*p)) p++;
  while (*p) {

    if (sscanf(p,"%[^=] = \"%[^\"]",ptag,pvalue) == 2) {
      /* ok - advance past option */
      while (*p && *p != '=') p++;
      while (*p && *p != '"') p++;
      if (*p) p++;
      while (*p && *p != '"') p++;
      if (*p) p++;
    } else if (sscanf(p,"%[^=] = %s",ptag,pvalue) == 2) {
      /* ok - advance past option */
      while (*p && *p != '=') p++;
      if (*p) p++;
      while (*p && isspace(*p)) p++;
      while (*p && !isspace(*p)) p++;
    } else {
      while (*p && isspace(*p)) p++;		/* Ignore unparsable tag */
      while (*p && !isspace(*p)) p++;
      continue;
    }

    if ((ht->ht_numOptions % HTML_OSTEP) == 0) {
      if (ht->ht_numOptions == 0)
	ht->ht_options = (HtmlTagOpt*) ob_malloc(sizeof(HtmlTagOpt)*HTML_OSTEP,"HtmlTagOpt");
      else
	ht->ht_options = (HtmlTagOpt*) ob_realloc(ht->ht_options,sizeof(HtmlTagOpt)*(ht->ht_numOptions + HTML_OSTEP));
    }

    ht->ht_options[ht->ht_numOptions].hto_label = ob_strdup(ptag);
    ht->ht_options[ht->ht_numOptions].hto_value = ob_strdup(pvalue);
    ht->ht_numOptions++;

    while (*p && isspace(*p)) p++;
  }

  return ht;
}

static int  HtmlContext_stringWidth(HtmlContext *hc,const char *text,int len)
{
  if (html_isPostscript) {
    return PSStringWidth(&hc->hc_font,text,len);
  } else {
    if (hc->hc_is16bit)
      return XTextWidth16(hc->hc_xstrFont,(XChar2b*)text,len/2);
    else
      return Tk_TextWidth(hc->hc_tkFont,text,len);
  }
}

static void HtmlContext_activateFont(HtmlContext *hc)
{
  ob_touch(hc);

  if (html_isPostscript) {
    hc->hc_tkFont = 0;
    hc->hc_xFont = 0;
    hc->hc_xstrFont = 0;
    hc->hc_is16bit = (hc->hc_font.family == FF_KANJI);

    if (hc->hc_is16bit)
      hc->hc_spaceWidth = PSStringWidth(&hc->hc_font,"  ",2);
    else
      hc->hc_spaceWidth = PSStringWidth(&hc->hc_font," ",1);

    hc->hc_metrics.linespace = hc->hc_font.points*1.2;
    hc->hc_metrics.descent = hc->hc_font.points*0.2;
    hc->hc_metrics.ascent = hc->hc_metrics.linespace - hc->hc_metrics.descent;


  } else {
    hc->hc_tkFont = GetTkFont(hc->hc_font.family, hc->hc_font.props, hc->hc_font.size, XGate.circuit->zoom_factor);
    hc->hc_xFont  = GetXFont(hc->hc_font.family, hc->hc_font.props, hc->hc_font.size, XGate.circuit->zoom_factor);
    hc->hc_xstrFont = XQueryFont(XGate.D, hc->hc_xFont);
    hc->hc_is16bit = (hc->hc_font.family == FF_KANJI);

    if (hc->hc_is16bit)
      hc->hc_spaceWidth = XTextWidth16(hc->hc_xstrFont, (XChar2b*)"  ", 1);
    else
      hc->hc_spaceWidth = Tk_TextWidth(hc->hc_tkFont, " ", 1);

    Tk_GetFontMetrics(hc->hc_tkFont, &hc->hc_metrics);
  }
}

static HtmlContext *new_HtmlContext(HtmlContext *base)
{
  HtmlContext *hc = (HtmlContext*) ob_malloc(sizeof(HtmlContext),"HtmlContext");

  if (base)
    *hc = *base;
  else {
    *hc = default_context;
    hc->hc_pixel = XGate.comment_pixel;
  }

  if (hc->hc_link) hc->hc_link = ob_strdup(hc->hc_link);
  if (hc->hc_tag)  hc->hc_tag = ob_strdup(hc->hc_tag);
  hc->hc_next = 0;
  
  HtmlFont_updatePoints(&hc->hc_font);
  HtmlContext_activateFont(hc);

  return hc;
}

static void delete_HtmlContext(HtmlContext *hc)
{
  if (hc->hc_link) ob_free(hc->hc_link);
  if (hc->hc_tag) ob_free(hc->hc_tag);
}

static HtmlUnit *new_HtmlUnit(const char *text,int len,HtmlContext *hc)
{
  HtmlUnit *hu = ob_malloc(sizeof(HtmlUnit),"HtmlUnit");

  hu->hu_type = HU_TEXT;
  hu->hu_text = ob_malloc(len+1,"char*");
  strncpy(hu->hu_text,text,len);
  hu->hu_text[len] =0;
  hu->hu_x = hu->hu_y = 0;
  hu->hu_context = hc;
  hu->hu_image = 0;

  return hu;
}

static HtmlUnit *new_HtmlUnit_T(int htype,HtmlContext *hc)
{
  HtmlUnit *hu = ob_malloc(sizeof(HtmlUnit),"HtmlUnit");

  hu->hu_type = htype;
  hu->hu_text =0;
  hu->hu_x = hu->hu_y = 0;
  hu->hu_context = hc;
  hu->hu_image = 0;

  return hu;
}

static void delete_HtmlUnit(HtmlUnit *hu)
{
  if (hu->hu_image)
    Tk_FreeImage(hu->hu_image);

  if (hu->hu_text)
    ob_free(hu->hu_text);
  ob_free(hu);
}

Html *new_Html(int reqwidth)
{
  Html *h = (Html*) ob_malloc(sizeof(Html),"Html");

  h->h_reqWidth = reqwidth;
  h->h_width = reqwidth;
  h->h_height = 0;
  h->h_dataLen = 0;
  h->h_data = 0;
  h->h_head = 0;
  h->h_tail = 0;
  h->h_context = new_HtmlContext(0);
  h->h_contextPool = 0;
  h->h_zoom = XGate.circuit->zoom_factor;

  return h;
}

void delete_Html(Html *h)
{
  HtmlUnit *u,*next_u;
  HtmlContext *hc,*next_hc;

  if (h->h_data) ob_free(h->h_data);

  for (u = h->h_head;u;u = next_u) {
    next_u = u->hu_next;
    delete_HtmlUnit(u);
  }

  for (hc = h->h_context;hc;hc = next_hc) {
    next_hc = hc->hc_next;
    delete_HtmlContext(hc);
  }
  for (hc = h->h_contextPool;hc;hc = next_hc) {
    next_hc = hc->hc_next;
    delete_HtmlContext(hc);
  }


  ob_free(h);
}

const char *Html_makeTutorialNavigationLine_bymodule(char *line)
{
  char *p = line;
  GModuleDef *M = XGate.circuit->es->env;
  int cur_pnum,max_pnum,i;
  int spaces = 45;

  for (max_pnum = 0;;) {
    char name[STRMAX];

    sprintf(name,"PAGE%d",max_pnum+1);
    if (!env_findModule(name)) break;
    max_pnum++;
  }

  if (max_pnum > 1) {
    if (sscanf(M->m_name,"PAGE%d",&cur_pnum) != 1 || cur_pnum < 1 || cur_pnum > max_pnum)
      return "[no-controls]";
  
    if (cur_pnum == 1) {
      int i;

      p += sprintf(p,"<font color=gray>&lt;%s</font>   ",msgLookup("tutorial.prev"));
    } else {
      p += sprintf(p,"<a href=\"#/PAGE%d\">&lt;%s</a>   ",cur_pnum-1,msgLookup("tutorial.prev"));
    }

    for (i = 1;i <= max_pnum;i++) {
      if (i == cur_pnum)
	p += sprintf(p," <b size=5>%d</b>",i);
      else
	p += sprintf(p," <a href=\"#/PAGE%d\">%d</b>",i,i);

      if (i < 10)
	spaces -= 2;
      else
	spaces -= 3;
    }

    if (cur_pnum == max_pnum)
      p += sprintf(p,"    <font color=gray>%s></font>",msgLookup("tutorial.next"));
    else
      p += sprintf(p,"    <a href=\"#/PAGE%d\">%s></a>",cur_pnum+1,msgLookup("tutorial.next"));

    for (i = 0;i < spaces;i++)
      p += sprintf(p," ");

    p += sprintf(p,"   ");
  }

  p += sprintf(p,"<a href=\"index.v\">&lang;%s&rang;</a>     <a href=\"%s#/PAGE%d\">&lang;%s&rang;</a>",
	       msgLookup("tutorial.chapter"),
	       CurrentFile_path(XGate.circuit->currentFile),cur_pnum,
	       msgLookup("tutorial.reload"));

  return line;
}

const char *Html_makeTutorialNavigationLine_byfile(char *line)
{
  char *p = line;
  int cur_pnum,max_pnum,i;
  int spaces = 45;
  const char *curDirName = CurrentFile_getDir(XGate.circuit->currentFile);
  const char *curFileName = CurrentFile_getBase(XGate.circuit->currentFile);
  char baseName[STRMAX], extension[STRMAX], path[STRMAX];

  if (sscanf(curFileName,"%[^0123456789]%d%s",baseName,&cur_pnum,extension) != 3 || strcmp(extension,".v") != 0)
    return "[no-controls - invalid file name]";

#if 1
  sprintf(path,"%s/%s",curDirName,baseName);
#endif

  for (max_pnum = 0;;) {
    char name[STRMAX];
    struct stat sb;

    sprintf(name,"%s%d.v",path,max_pnum+1);

    if (stat(name, &sb) != 0) break;

    max_pnum++;
  }

  if (max_pnum > 1) {
    if (cur_pnum == 1) {
      p += sprintf(p,"<font color=gray>&lt;%s</font>   ",msgLookup("tutorial.prev"));
    }  else
      p += sprintf(p,"<a href=\"%s%d.v\">&lt;%s</a>   ",baseName,cur_pnum-1,msgLookup("tutorial.prev"));

    for (i = 1;i <= max_pnum;i++) {
      if (i == cur_pnum)
	p += sprintf(p," <b size=5>%d</b>",i);
      else
	p += sprintf(p," <a href=\"%s%d.v\">%d</b>",baseName,i,i);

      if (i < 10)
	spaces -= 2;
      else
	spaces -= 3;
    }

    if (cur_pnum == max_pnum)
      p += sprintf(p,"    <font color=gray>%s></font>",msgLookup("tutorial.next"));
   else
      p += sprintf(p,"    <a href=\"%s%d.v\">%s></a>",baseName,cur_pnum+1,msgLookup("tutorial.next"));

    for (i = 0;i < spaces;i++)
      p += sprintf(p," ");

    p += sprintf(p,"   ");
  }

  p += sprintf(p,"<a href=\"index.v\">&lang;%s&rang;</a>     <a href=\"%s%d.v\">&lang;%s&rang;</a>",
	       msgLookup("tutorial.chapter"),
	       baseName,cur_pnum,
	       msgLookup("tutorial.reload"));

  return line;
}

/*****************************************************************************
 *
 * Generate the HTML resulting from the <tutorial-navigation> tag.
 *
 *****************************************************************************/
const char *Html_makeTutorialNavigationLine(HtmlTag *navtag)
{
  static char line[STRMAX];
  int byfile = 0;
  int i;

  for (i = 0;i < navtag->ht_numOptions;i++) {
    if (strcasecmp(navtag->ht_options[i].hto_label, "byfile") == 0)
      byfile = istruevalue(navtag->ht_options[i].hto_value);
  }

  if (byfile)
    Html_makeTutorialNavigationLine_byfile(line);
  else
    Html_makeTutorialNavigationLine_bymodule(line);

#if HAVE_ICONV_H
  if (XGate.japaneseMode) {
    char qbuf[1024];
    DoTcl("iconv decode \"%s\"",quoteChars(qbuf,line,TCL_SPECIALCHARS));
    strcpy(line,XGate.tcl->result);
  }
#endif

  return line;
}

/******************************************************************************
 *
 * Add a line of data to the html object
 *
 *****************************************************************************/
void Html_addLine(Html *h,const char *line)
{
  /* Current # of allocated bytes */
  int curLen = ((h->h_dataLen+DATA_STEP_SIZE-1)/DATA_STEP_SIZE)*DATA_STEP_SIZE;
  int l;

  ob_touch(h);

  //
  // This is a kludge to automatically create the tutorial navigation buttons
  //
  if (strncasecmp(line,"<tutorial-navigation",20) == 0) {
    HtmlTag *navtag = Html_parseTag(line);

    line = Html_makeTutorialNavigationLine(navtag);
  }

  l = strlen(line);


  /*
   * Check to see if we need to increase length of data buffer.  Be sure to 
   * make space for the newline and the null character.
   */
  if (h->h_dataLen + l + 2 > curLen) {
    int newLen = ((h->h_dataLen + l + 2 + DATA_STEP_SIZE-1)/DATA_STEP_SIZE)*DATA_STEP_SIZE;

    if (h->h_data) {
        h->h_data = ob_realloc(h->h_data,newLen);
     } else
       h->h_data = ob_malloc(newLen,"char*");
  }

  /*
   * Append the data plus a newline.
   */
  strcpy(h->h_data + h->h_dataLen,line);
  h->h_dataLen += l;
  h->h_data[h->h_dataLen++] = '\n';
  h->h_data[h->h_dataLen] = 0;
}

/******************************************************************************
 *
 * Delete all html units from an html object.
 *
 *****************************************************************************/
static void Html_flushUnits(Html *h)
{
  HtmlUnit *hu,*next_hu;

  ob_touch(h);

  for (hu = h->h_head;hu;hu = next_hu) {
    next_hu = hu->hu_next;
    delete_HtmlUnit(hu);
  }
  h->h_head = h->h_tail = 0;
}

static void Html_addUnit(Html *h,HtmlUnit *hu)
{
  ob_touch(h);

  if (h->h_tail) {
    h->h_tail->hu_next = hu;
    hu->hu_prev = h->h_tail;
    h->h_tail = hu;
    hu->hu_next = 0;
  } else {
    h->h_head = h->h_tail = hu;
    hu->hu_prev = 0;
    hu->hu_next = 0;
  }
}

static void Html_pushContext(Html *h,HtmlContext *hc)
{
  ob_touch(hc);
  ob_touch(h);

  hc->hc_next = h->h_context;
  h->h_context = hc;
}

static void Html_popContext(Html *h)
{
  HtmlContext *hc = h->h_context;

  if (!hc->hc_next) return;		/* Don't pop last context */

  ob_touch(hc);
  ob_touch(h);

  h->h_context = h->h_context->hc_next;  

  hc->hc_next = h->h_contextPool;
  h->h_contextPool = hc;
}

/*****************************************************************************
 *
 * Handle a <img> element.
 *
 *****************************************************************************/
void Html_handle_img(Html *h, HtmlTag *tag)
{
  HtmlContext *hc = new_HtmlContext(h->h_context);
  HtmlUnit *hu = new_HtmlUnit_T(HU_IMAGE,hc);
  int width = 16, height = 16;
  const char *gifFile = "blk_copy.gif";
  int i;

  hc->hc_pixel = -1;

  ob_touch(hu);

  for (i = 0;i < tag->ht_numOptions;i++) {
    if (strcasecmp(tag->ht_options[i].hto_label, "src") == 0) {
      char buf[STRMAX];

      strcpy(buf,tag->ht_options[i].hto_value);
      expandSpecialDirs(buf);
      gifFile = ob_strdup(buf);
    } else if (strcasecmp(tag->ht_options[i].hto_label, "bgcolor") == 0) {
      ob_touch(hc);
      hc->hc_pixel = Tkg_GetColor(tag->ht_options[i].hto_value);
      
    }
  }

  Html_pushContext(h,hc);
  Html_popContext(h);
  Html_addUnit(h,hu);

  ob_touch(hu);
  ob_touch(hc);

  DoTcl("gifI %s",gifFile);
  hu->hu_image = Tk_GetImage(XGate.tcl, Tk_MainWindow(XGate.tcl), XGate.tcl->result, 0, 0);
  if (hu->hu_image)
    Tk_SizeOfImage(hu->hu_image, &width, &height);

  hu->hu_text = 0;
  hu->hu_width = width;
  hc->hc_metrics.descent = h->h_context->hc_metrics.descent;
  hc->hc_metrics.ascent = height - hc->hc_metrics.descent;
}

/*****************************************************************************
 *
 * Handle a <value-of> element.
 *
 *****************************************************************************/
void Html_handle_valueOf(Html *h, HtmlTag *tag)
{
  extern char *release_date;
  char *name = 0;
  const char *text = 0;
  int i;

  for (i = 0;i < tag->ht_numOptions;i++)
    if (strcasecmp(tag->ht_options[i].hto_label,"name") == 0)
      name = tag->ht_options[i].hto_value;

  if (!name) return;

  if (strcasecmp(name,"tkgate-version") == 0) {
    text = TKGATE_FULL_VERSION;
  } else if (strcasecmp(name,"tkgate-homepage") == 0) {
    text = TKGATE_HOMEPAGE;
  } else if (strcasecmp(name,"tkgate-mailcontact") == 0) {
    text = TKGATE_MAILCONTACT;
  } else if (strcasecmp(name,"tkgate-copyright") == 0) {
    text = TKGATE_COPYRIGHT;
  } else if (strcasecmp(name,"tkgate-release-date") == 0) {
    text = release_date;
  }

  if (text)
    Html_addUnit(h,new_HtmlUnit(text,strlen(text),h->h_context));
}

/*****************************************************************************
 *
 * Handle font modifiers in a tag
 *
 *****************************************************************************/
void HtmlContext_handle_modifiers(HtmlContext *hc,HtmlTag *tag)
{
  int i;

  for (i = 0;i < tag->ht_numOptions;i++) {
    const char *label = tag->ht_options[i].hto_label;
    const char *value = tag->ht_options[i].hto_value;

    if (strcasecmp(label,"face") == 0) {
      int j;

      ob_touch(hc);
      
      for (j = 0;j < FF_MAX;j++)
	if (strcasecmp(font_family_names[j],value) == 0)
	  hc->hc_font.family = j;
    } else if (strcasecmp(label,"size") == 0) {
      /*
       * Scan font size.  We decrement the scanned size to convert from the
       * html specification to our internal size code.
       */
      if (sscanf(value,"%d",&hc->hc_font.size) == 1) hc->hc_font.size--;
      HtmlFont_updatePoints(&hc->hc_font);
    } else if (strcasecmp(label,"color") == 0) {
      ob_touch(hc);
      hc->hc_pixel = Tkg_GetColor(value);
    }
  }
}


/*****************************************************************************
 *
 * Handler for basic open/close html tags which modify the context.
 *
 *****************************************************************************/
void Html_handle_basic(Html *h, HtmlTag *tag)
{
  HtmlContext *hc = new_HtmlContext(h->h_context);

  ob_touch(hc);
  if (strcasecmp(tag->ht_name,"b") == 0) {
    hc->hc_font.props |= FP_BOLD;
    HtmlContext_handle_modifiers(hc,tag);
  } else if (strcasecmp(tag->ht_name,"i") == 0) {
    hc->hc_font.props |= FP_ITALIC;
    HtmlContext_handle_modifiers(hc,tag);
  } else if (strcasecmp(tag->ht_name,"tt") == 0) {
    hc->hc_font.family = FF_COURIER;
    HtmlContext_handle_modifiers(hc,tag);
  } else if (strcasecmp(tag->ht_name,"big") == 0) {
    if (hc->hc_font.size + 1  < FS_MAX)
      hc->hc_font.size++;
    HtmlFont_updatePoints(&hc->hc_font);
    HtmlContext_handle_modifiers(hc,tag);
  } else if (strcasecmp(tag->ht_name,"small") == 0) {
    if (hc->hc_font.size - 1  >= 0)
      hc->hc_font.size--;
    HtmlFont_updatePoints(&hc->hc_font);
    HtmlContext_handle_modifiers(hc,tag);
  } else if (strcasecmp(tag->ht_name,"font") == 0) {
    HtmlContext_handle_modifiers(hc,tag);
  }

  HtmlContext_activateFont(hc);
  Html_pushContext(h,hc);
}

/*****************************************************************************
 *
 * Handler for heading tags
 *
 *****************************************************************************/
void Html_handle_heading(Html *h, HtmlTag *tag)
{
  HtmlContext *hc = new_HtmlContext(h->h_context);

  Html_addUnit(h,new_HtmlUnit_T(HU_BREAK,h->h_context));

  ob_touch(hc);

  hc->hc_font.props |= FP_BOLD;

  if (strcasecmp(tag->ht_name,"h1") == 0) {
    hc->hc_font.size = FS_XHUGE;
  } else if (strcasecmp(tag->ht_name,"h2") == 0) {
    hc->hc_font.size = FS_HUGE;
  } else if (strcasecmp(tag->ht_name,"h3") == 0) {
    hc->hc_font.size = FS_LARGE;
  }

  HtmlFont_updatePoints(&hc->hc_font);

  HtmlContext_handle_modifiers(hc,tag);

  HtmlContext_activateFont(hc);
  Html_pushContext(h,hc);
}

void Html_handle_a(Html *h, HtmlTag *ht)
{
  HtmlContext *hc = new_HtmlContext(h->h_context);
  int i;

  ob_touch(hc);
  hc->hc_pixel = XGate.hyperlink_pixel;
  HtmlContext_activateFont(hc);

  for (i = 0;i < ht->ht_numOptions;i++) {
    if (strcasecmp(ht->ht_options[i].hto_label, "href") == 0) {
      hc->hc_link = ob_strdup(ht->ht_options[i].hto_value);
    } else if (strcasecmp(ht->ht_options[i].hto_label, "name") == 0) {
      hc->hc_tag = ob_strdup(ht->ht_options[i].hto_value);
    }
  }


  Html_pushContext(h,hc);
}

/*****************************************************************************
 *
 * Process the html tag that starts at 'tag' and is 'len' characters long.
 *
 *****************************************************************************/
static void Html_processTag(Html *h, const char *tag,int len)
{
  HtmlTag *ht;
  int i;

  ht = Html_parseTag(tag);
  if (!ht) return;


  if (*ht->ht_name == '/') {
    Html_popContext(h);
    return;
  }

  for (i = 0;htmlHandlers[i].tag;i++)
    if (strcasecmp(ht->ht_name,htmlHandlers[i].tag) == 0)
      (*htmlHandlers[i].func)(h,ht);
}

/*****************************************************************************
 *
 * Process character entity references.
 *
 *****************************************************************************/
static void Html_processSpecialChar(Html *h,char *spec,int len)
{
  char *text = 0;
  char *pretag = 0;
  char *posttag = 0;
  int i;

  for (i = 0;htmlSpecialSpecs[i].spec;i++) {
    if (strncmp(spec,htmlSpecialSpecs[i].spec,len) == 0) {
      text = htmlSpecialSpecs[i].text;
      pretag = htmlSpecialSpecs[i].pretag;
      posttag = htmlSpecialSpecs[i].posttag;
    }
  }

  if (text) {
    if (pretag) Html_processTag(h,pretag,strlen(pretag));
    Html_addUnit(h,new_HtmlUnit(text,strlen(text),h->h_context));
    if (posttag) Html_processTag(h,posttag,strlen(posttag));
  } else
    Html_addUnit(h,new_HtmlUnit(spec,len,h->h_context));
}


/******************************************************************************
 *
 * Parition an html object into pieces.  Each piece is one of:
 *   * An html tag
 *   * An html special character sequence
 *   * A new line
 *   * A text string
 *
 *****************************************************************************/
void Html_partition(Html *h,char *data)
{
  char *p,*q;

  Html_flushUnits(h);

  if (!data) return;				/* No data */

  p = q = data;
  while (*p) {
    if (*p == '<') {
      /******************************************************************
       *
       * This is an html tag.  Include everything up to end of tag.
       *
       ******************************************************************/
      q = strchr(p,'>');
      if (!q) q = p + strlen(p);

      Html_processTag(h,p,q-p+1);
      if (!*q) break;
      p = q+1;
    } else if (*p == '&') {
      /******************************************************************
       *
       * This is special character.  Parse until ';' and convert to the
       * specified character.
       *
       ******************************************************************/
      q = strchr(p,';');
      if (!q) q = p + strlen(p);

      Html_processSpecialChar(h,p,q-p+1);
      if (!*q) break;
      p = q+1;
    } else if (*p == '\n') {
      /******************************************************************
       *
       * This is newline.  Insert a newline unit if we are in preformat
       * mode, otherwise just insert a space.
       *
       ******************************************************************/
      if (h->h_context->hc_preformat) {
	HtmlUnit *hu = new_HtmlUnit_T(HU_NEWLINE,h->h_context);
	Html_addUnit(h,hu);
      } else
	Html_addUnit(h,new_HtmlUnit(" ",1,h->h_context));
      p++;
    } else if ((*p & 0x80) && XGate.japaneseMode) {
      /******************************************************************
       *
       * This is Kanji text.  Scan until we find a non-8-bit character.
       *
       ******************************************************************/
      HtmlContext *hc;
      HtmlUnit *hu;
      int i;

      for (q = p;*q;q++)
	if (!(*q & 0x80))
	  break;


      hc = new_HtmlContext(h->h_context);
      hc->hc_font.family = FF_KANJI;
      HtmlFont_updatePoints(&hc->hc_font);
      HtmlContext_activateFont(hc);
      Html_pushContext(h,hc);
      Html_popContext(h);
      hu = new_HtmlUnit(p,q-p,hc);
      Html_addUnit(h,hu);
      for (i = 0;hu->hu_text[i];i++) hu->hu_text[i] &= 0x7f;

      if (!*q) break;
      p = q;
    } else {
      /******************************************************************
       *
       * This is a regular non-kanji text string
       *
       ******************************************************************/
      HtmlUnit *hu;

      for (q = p;*q;q++)
	if (strchr("<&\n",*q) != 0 || ((*q & 0x80) && XGate.japaneseMode))
	  break;
      hu = new_HtmlUnit(p,q-p,h->h_context);
      Html_addUnit(h,hu);

      if (!*q) break;
      p = q;
    }
  }
}

/******************************************************************************
 *
 * Format an html object.
 *
 * Parameters:
 *     h		Html object to be formatted.
 *     isPs		Non-zero to format for postscript
 *
 *****************************************************************************/
void Html_format(Html *h,int isPs)
{
  HtmlUnit *hu,*hu2;
  HtmlUnit *linestart_hu;
  int max_ascent = 0;
  int max_descent = 0;
  int max_width = 0;
  int x = 0, y = 0;					/* Current text position */
  char *p;

  ob_touch(h);

  html_isPostscript = isPs;

  Html_partition(h,h->h_data);

  h->h_isVisible = 0;

  linestart_hu = h->h_head;				/* This is the text unit for the current line */
  for (hu = h->h_head;hu;hu = hu->hu_next) {
    HtmlContext *hc = hu->hu_context;			/* Get context of this unit */

    /*
     * Update line height metrics
     */
    if (hc->hc_metrics.ascent > max_ascent)
      max_ascent = hc->hc_metrics.ascent;
    if (hc->hc_metrics.descent > max_descent)
      max_descent = hc->hc_metrics.descent;

    switch (hu->hu_type) {
    case HU_BREAK :
      if (x == 0) break;		/* Do a newline only if we are not already on a newline */
      /* fall through */
    case HU_NEWLINE :
      y += max_ascent;
      for (hu2 = linestart_hu;hu2 != hu;hu2 = hu2->hu_next) {
	ob_touch(hu2);
	hu2->hu_y = y;
      }
      y += max_descent;
      x = 0;

      /*
       * Reset metrics and advance linestart_hu to start of next line.
       */
      max_ascent = 0;
      max_descent = 0;
      linestart_hu = hu->hu_next;
      break;
    case HU_TEXT :
      ob_touch(hu);

      hu->hu_x = x;

      hu->hu_width = HtmlContext_stringWidth(hc,hu->hu_text,strlen(hu->hu_text));

      x += hu->hu_width;
      if (x > max_width) max_width = x;

      if (!h->h_isVisible) {
	for (p = hu->hu_text;*p;p++)
	  if (!isspace(*p)) {
	    h->h_isVisible = 1;
	  }
      }
      break;
    case HU_IMAGE :
      hu->hu_x = x;
      x += hu->hu_width;
      if (x > max_width) max_width = x;
      h->h_isVisible = 1;
      break;
    }
  }

  h->h_width = max_width;
  h->h_height = y + max_descent;
}

void Html_psPrint(Html *h,GPrint *P,int x,int y)
{
  HtmlUnit *hu;
  HtmlContext *last_hc = 0;

  for (hu = h->h_head;hu;hu = hu->hu_next) {
    HtmlContext *hc = hu->hu_context;			/* Get context of this unit */

    /*
     * Update properties only if there was a change.
     */
    if (hc != last_hc) {
      // Tkg_changeColor(gc, GXxor, hc->hc_pixel);
      last_hc = hc;
    }

    switch (hu->hu_type) {
    case HU_TEXT :
      PSDrawText(P,&hc->hc_font,hu->hu_x + x,hu->hu_y + y,hu->hu_text,AtBaseline|AtLeft);
      break;
    case HU_IMAGE :
      break;
    default :
      break;
    }
  }
}

/*****************************************************************************
 *
 * Draw a block of html text.
 *
 *****************************************************************************/
void Html_draw(Html *h,int x,int y)
{
  GC gc = XGate.commentGC;
  HtmlUnit *hu;
  HtmlContext *last_hc = 0;

  x = ctow_x(x)*XGate.circuit->zoom_factor;
  y = ctow_y(y)*XGate.circuit->zoom_factor;

  for (hu = h->h_head;hu;hu = hu->hu_next) {
    HtmlContext *hc = hu->hu_context;			/* Get context of this unit */

    /*
     * Update properties only if there was a change.
     */
    if (hc != last_hc) {
      XSetFont(XGate.D,gc,hc->hc_xFont);
      if (hc->hc_pixel >= 0)
	Tkg_changeColor(gc, GXxor, hc->hc_pixel);
      last_hc = hc;
    }

    switch (hu->hu_type) {
    case HU_TEXT :
      if (hc->hc_font.family == FF_KANJI) {
	XDrawString16(XGate.D,XGate.W,gc,hu->hu_x + x,hu->hu_y + y,(XChar2b*)hu->hu_text,strlen(hu->hu_text)/2);
      } else
	XDrawString(XGate.D,XGate.W,gc,hu->hu_x + x,hu->hu_y + y,hu->hu_text,strlen(hu->hu_text));
      
      break;
    case HU_IMAGE :
      {
	int width = hu->hu_width;
	int height = hc->hc_metrics.descent  + hc->hc_metrics.ascent;
	int base_x = hu->hu_x + x;
	int base_y = hu->hu_y + y - hc->hc_metrics.ascent;
#if 0
	ZDrawRectangle(XGate.D,XGate.W,gc, base_x,base_y, width, height);
#endif
	if (hc->hc_pixel >= 0)
	  ZFillRectangle(XGate.D,XGate.W,gc, base_x,base_y, width, height);
	Tk_RedrawImage(hu->hu_image, 0, 0, width, height, XGate.W, base_x, base_y);
      }
      break;
    default :
      break;
    }
  }
}

int Html_isHit(Html *h,int x,int y)
{
  HtmlUnit *hu;

  for (hu = h->h_head;hu;hu = hu->hu_next) {
    HtmlContext *hc = hu->hu_context;			/* Get context of this unit */

    if (x >= hu->hu_x && x <= (hu->hu_x + hu->hu_width)
	&& y <= (hu->hu_y + hc->hc_metrics.descent) && y >= (hu->hu_y - hc->hc_metrics.ascent)) {

      return 1;
    }
  }

  return 0;
}

const char *Html_getLink(Html *h,int x,int y)
{
  HtmlUnit *hu;

  for (hu = h->h_head;hu;hu = hu->hu_next) {
    HtmlContext *hc = hu->hu_context;			/* Get context of this unit */

    if (!hc->hc_link) continue;

    if (hu->hu_type == HU_IMAGE) {
      int width = hu->hu_width;
      int height = hc->hc_metrics.descent  + hc->hc_metrics.ascent;
      int base_x = hu->hu_x;
      int base_y = hu->hu_y - hc->hc_metrics.ascent;

      if (x >= base_x && x <= (base_x + width)
	  && y >= base_y && y <= (base_y + height)) {

	return hc->hc_link;
      }
    } else {
      if (x >= hu->hu_x && x <= (hu->hu_x + hu->hu_width)
	  && y <= (hu->hu_y + hc->hc_metrics.descent) && y >= (hu->hu_y - hc->hc_metrics.ascent)) {

	return hc->hc_link;
      }
    }
  }

  return 0;
}

