#   Copyright (C) 1987-2004 by Jeffery P. Hansen
#
#   This program is free software; you can redistribute it and/or modify
#   it under the terms of the GNU General Public License as published by
#   the Free Software Foundation; either version 2 of the License, or
#   (at your option) any later version.
#
#   This program is distributed in the hope that it will be useful,
#   but WITHOUT ANY WARRANTY; without even the implied warranty of
#   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#   GNU General Public License for more details.
#
#   You should have received a copy of the GNU General Public License
#   along with this program; if not, write to the Free Software
#   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
#
# Last edit by hansen on Thu Sep  8 22:55:33 2005
#############################################################################
#
# Tabboxes are used to select from a group of windows
#
#	new w args		Create a new tabbox
#	select w i args		Select the tab with index i
#	getactive w		Get the active tab name

namespace eval TabBox {
  variable details
  variable active
  variable disabledTabs 

  proc select {w i args} {
    variable details
    variable active
    variable disabledTabs

    if {[lsearch -exact $disabledTabs($w) $i] >= 0} return

    set reopen 0
    set docommand 1
    parseargs $args {-docommand -reopen}

    if { $active($w) == $i && !$reopen } return

    set mydetails $details($w)
    set bd [assoc bd $mydetails]
    set tabheight [assoc tabheight $mydetails]
    set tabwidth [assoc tabwidth $mydetails]
    set activecolor [assoc activecolor $mydetails]
    set inactivecolor [assoc inactivecolor $mydetails]
    set command [assoc command $mydetails]
    set tabs [assoc tabs $mydetails]
    set dodestroy [assoc dodestroy $mydetails]
    set passive [assoc passive $mydetails]

    if { $command != "" } {
      if {$dodestroy} { catch { destroy $w.f.f } }
      if {$docommand } { $command $w.f.f [lindex $tabs $i] }
      if {$dodestroy} { catch { pack $w.f.f -anchor center -expand 1 -fill both} }
    }

    if { ! $passive} {
      setactive $w [lindex $tabs $i]
    }
  }

  proc getactive {w} {
    variable details
    variable active

    set mydetails $details($w)
    set tabs [assoc tabs $mydetails]
    set i $active($w)

    return [lindex $tabs $i]
  }

  proc getactiveidx {w} {
    variable active
    return $active($w)
  }
  
  proc setactive {w name} {
    variable details
    variable active

    set mydetails $details($w)
    set bd [assoc bd $mydetails]
    set tabheight [assoc tabheight $mydetails]
    set tabwidth [assoc tabwidth $mydetails]
    set width [assoc width $mydetails]
    set activecolor [assoc activecolor $mydetails]
    set inactivecolor [assoc inactivecolor $mydetails]
    set tabs [assoc tabs $mydetails]
    set taboffset [assoc taboffset $mydetails]
    set maxvisible [assoc maxvisible $mydetails]
    set tabside [assoc tabside $mydetails]

    set i [lsearch -exact $tabs $name]
    if {$i < 0} return

    set active($w) $i

    if { $taboffset > 0 } {
      set xstart [winfo reqwidth $w.leftjaggy]
    } else {
      set xstart 0
    }

    switch $tabside {
      top {
	set di [expr $i-$taboffset]
	if {$di < 0 || $di >= $maxvisible} {
	  # Move it out of sight.
	  $w coords hider [expr -2*$tabwidth] 0
	} else {
	  $w coords hider [expr $bd + $xstart + $di*$tabwidth] $tabheight
	}
      }
      left {
	set di [expr $i-$taboffset]
	$w coords hider $tabwidth [expr $bd + $di*($tabheight+$bd)]
      }
    }

    catch {
      for { set j 0 } { 1 } { incr j } {
	if { $j == $i } {
	  $w.b$j configure -background $activecolor
	  catch { $w.b$j.l configure -background $activecolor -font boldLabelFont }
	  catch { $w.b$j.img configure -background $activecolor }
	} else {
	  $w.b$j configure -background $inactivecolor
	  catch { $w.b$j.l configure -background $inactivecolor -font labelFont }
	  catch { $w.b$j.img configure -background $inactivecolor }
	}
      }
    }
  }

  proc reconfigure {w} {
    variable details 

    set mydetails $details($w)
    set tabheight [assoc tabheight $mydetails]
    set tabwidth [assoc tabwidth $mydetails]
    set tabside [assoc tabside $mydetails]
    set bd [assoc bd $mydetails]

    set height [winfo height $w]
    set width [winfo width $w]

    switch $tabside {
      top {
	$w.f configure -width $width  -height [expr $height - $tabheight]
      }
      left {

	$w.f configure -width [expr $width - $tabwidth]  -height $height
      }
    }
  }

  proc placeTabs {w} {
    variable details 
    set tabside [assoc tabside $details($w)]
    placeTabs_$tabside $w
  }

  #############################################################################
  #
  # Place tabs on left of tabbox.  The tab slider is not implemented for left
  # tab tab boxes.
  #
  proc placeTabs_left {w} {
    variable details 

    set mydetails $details($w)

    set width  [assoc width $mydetails]
    set tabheight [assoc tabheight $mydetails]
    set tabwidth  [assoc tabwidth $mydetails]
    set tabs [assoc tabs $mydetails]
    set bd [assoc bd $mydetails]
    set taboffset [assoc taboffset $mydetails]
    set maxvisible [assoc maxvisible $mydetails]
    set maxoffset  [assoc maxoffset $mydetails]

    set ystart 0

    set x $tabwidth
    set y $ystart
    set i 0
    set vi 0
    foreach tab $tabs {
      $w create window [expr $x + $bd] $y -anchor ne -window $w.b$i -tags [list tab$i tabs]
      set y [expr $y + $tabheight + $bd]

      incr i
      incr vi
    }
  }

  #############################################################################
  #
  # Standard tab boxes with tabs on the top.
  #
  proc placeTabs_top {w} {
    variable details 

    set mydetails $details($w)

    set width  [assoc width $mydetails]
    set tabheight [assoc tabheight $mydetails]
    set tabwidth  [assoc tabwidth $mydetails]
    set tabs [assoc tabs $mydetails]
    set bd [assoc bd $mydetails]
    set taboffset [assoc taboffset $mydetails]
    set maxvisible [assoc maxvisible $mydetails]
    set maxoffset  [assoc maxoffset $mydetails]

    if { $taboffset > 0 } {
      set xstart [winfo reqwidth $w.leftjaggy]
    } else {
      set xstart 0
    }

    set x $xstart
    set y $tabheight
    set i $taboffset
    set vi 0
    foreach tab $tabs {
      if {$vi >= $maxvisible} {
	break
      }

      $w create window $x [expr $y + $bd] -anchor sw -window $w.b$i -tags [list tab$i tabs]
      set x [expr $x + $tabwidth]

      incr i
      incr vi
    }

    if {$taboffset < $maxoffset } {
      $w create window $x [expr $y + $bd] -anchor sw -window $w.rightjaggy -tags rjag
    } else {
      $w delete rjag
    }

    if {$taboffset > 0 } {
      $w create window 0 [expr $y + $bd] -anchor sw -window $w.leftjaggy -tags ljag
    } else {
      $w delete ljag
    }

  }

  #############################################################################
  #
  # Called at initialization time and before/after moving the slider.
  #
  proc setSliderState {w} {
    variable details 
    variable active 

    set mydetails $details($w)
    set taboffset  [assoc taboffset $mydetails]
    set maxoffset  [assoc maxoffset $mydetails]

    if {$taboffset == 0 } {
      $w.slider.l configure -state normal
      $w.slider.r configure -state disabled
    } elseif {$taboffset == $maxoffset } {
      $w.slider.l configure -state disabled
      $w.slider.r configure -state normal
    } else {
      $w.slider.l configure -state normal
      $w.slider.r configure -state normal
    }
  }

  proc shiftLeft {w} {
    variable details 
    variable active 

    set mydetails $details($w)
    set tabs  [assoc tabs $mydetails]
    set taboffset  [assoc taboffset $mydetails]
    set maxoffset  [assoc maxoffset $mydetails]

    if {$taboffset < $maxoffset} {
      incr taboffset
    }
    
    set details($w) [assocset taboffset $mydetails $taboffset]

    $w delete tabs
    placeTabs $w
    setactive $w [lindex $tabs $active($w)]

    setSliderState $w
  }

  proc shiftRight {w} {
    variable details 
    variable active 

    set mydetails $details($w)
    set tabs  [assoc tabs $mydetails]
    set taboffset  [assoc taboffset $mydetails]
    if {$taboffset > 0} {
      incr taboffset -1
    }
    set details($w) [assocset taboffset $mydetails $taboffset]

    $w delete tabs
    placeTabs $w
    setactive $w [lindex $tabs $active($w)]

    setSliderState $w
  }

  #############################################################################
  #
  # User pressed right continuation 
  #
  proc rightJag {w} {
    variable details 

    shiftLeft $w

    set mydetails $details($w)
    set taboffset  [assoc taboffset $mydetails]
    set maxvisible [assoc maxvisible $mydetails]
    select $w [expr $taboffset + $maxvisible - 1 ]
  }

  #############################################################################
  #
  # User pressed left continuation 
  #
  proc leftJag {w} {
    variable details 

    shiftRight $w

    set mydetails $details($w)
    set taboffset  [assoc taboffset $mydetails]
    select $w $taboffset
  }


  #############################################################################
  #
  # Create everything needed to do tab sliding.
  #
  proc makeTabSlider {w} {
    variable details 

    set mydetails $details($w)

    set width  [assoc width $mydetails]
    set tabheight [assoc tabheight $mydetails]
    set bd [assoc bd $mydetails]
    set tabs [assoc tabs $mydetails]
    set tabwidth [assoc tabwidth $mydetails]
    set activecolor [assoc activecolor $mydetails]
    set inactivecolor [assoc inactivecolor $mydetails]
    set relief [assoc relief $mydetails]

    set y $tabheight

    frame $w.slider
    button $w.slider.l -image [gifI leftarrow.gif] -command "TabBox::shiftLeft $w"
    button $w.slider.r -image [gifI rightarrow.gif] -command "TabBox::shiftRight $w"
    pack $w.slider.l $w.slider.r -side left
    update
    set x [expr $width - [winfo reqwidth $w.slider]]
    $w create window $x [expr $y + $bd] -anchor sw -window $w.slider

    # Number of visible tabs
    set n [expr ($width - 60)/$tabwidth]
    set details($w) [assocset maxvisible $details($w) $n]

    set maxoffset [expr [llength $tabs] - $n]
    set details($w) [assocset maxoffset $details($w) $maxoffset]



    frame $w.rightjaggy -width 15 -height $tabheight -relief $relief -bd $bd -bg $inactivecolor
    label $w.rightjaggy.l -text ... -bg $inactivecolor
    frame $w.rightjaggy.v -height [expr $tabheight - 2*$bd] -width 0
    pack $w.rightjaggy.v $w.rightjaggy.l -side left

    frame $w.leftjaggy  -width 15 -height $tabheight -relief $relief -bd $bd -bg $inactivecolor
    label $w.leftjaggy.l -text ... -bg $inactivecolor
    frame $w.leftjaggy.v -height [expr $tabheight - 2*$bd] -width 0
    pack $w.leftjaggy.v $w.leftjaggy.l -side left

    bind $w.rightjaggy <Button-1> "TabBox::rightJag $w"
    bind $w.rightjaggy.l <Button-1> "TabBox::rightJag $w"
    bind $w.leftjaggy <Button-1> "TabBox::leftJag $w"
    bind $w.leftjaggy.l <Button-1> "TabBox::leftJag $w"

#    $w.rightjaggy create line 0 0 15 15

    setSliderState $w
  }

  #############################################################################
  #
  # Make a tab unusable
  #
  proc disabletab {w name} {
    variable disabledTabs
    variable details 

    set mydetails $details($w)
    set tabs [assoc tabs $mydetails]
    set i [lsearch -exact $tabs $name]
    if { $i < 0 } return

    if {[lsearch -exact $disabledTabs($w) $i] <= 0 } {
      lappend disabledTabs($w) $i
      $w.b$i.l configure -foreground gray60
    }
  }

  #############################################################################
  #
  # Make a tab usable again
  #
  proc enabletab {w name} {
    variable disabledTabs
    variable details 

    set mydetails $details($w)
    set tabs [assoc tabs $mydetails]
    set i [lsearch -exact $tabs $name]
    if { $i < 0 } return


    set j [lsearch -exact $disabledTabs($w) $i]
    if { $j >= 0 } {
      set disabledTabs($w) [lreplace $disabledTabs($w) $j $j]
      $w.b$i.l configure -foreground black
    }
  }


  #############################################################################
  #
  # Create a new tab box.
  #
  proc new {w args} {
    variable details
    variable active
    variable disabledTabs 

    set activecolor "#d9d9d9"
    set inactivecolor gray70
    set width 300
    set height 300
    set tabheight *25
    set tabwidth  *50
    set tabs {}
    set tablabels {}
    set bd 2
    set relief raised
    set command ""
    set startpage ""
    set dodestroy 1
    set passive 0
    set expand 0
    set imagemap ""
    set boxlabel ""
    set boxlabelfunc ""
    set noslider 0
    set tabside top

    set active($w) ""
    set disabledTabs($w) {}

    parseargs $args {-width -height -tabheight -tabwidth -bd -relief -tabs -activecolor -inactivecolor -command -tablabels -startpage -dodestroy -imagemap -expand -boxlabel -boxlabelfunc -passive -noslider -tabside}

    set width [rescale $width]
    set height [rescale $height]
    set tabwidth [rescale $tabwidth]
    set tabheight [rescale $tabheight]


    if { [llength $tablabels] == 0 } {
      set tablabels $tabs
    }
    if { [llength $tablabels] != [llength $tabs] } {
      error "-tablabels and -tabs must be of equal length"
    }

    set details($w) {}
    lappend details($w) [list width $width]
    lappend details($w) [list height $height]
    lappend details($w) [list tabwidth $tabwidth]
    lappend details($w) [list tabheight $tabheight]
    lappend details($w) [list activecolor $activecolor]
    lappend details($w) [list inactivecolor $inactivecolor]
    lappend details($w) [list bd $bd]
    lappend details($w) [list command $command]
    lappend details($w) [list tabs $tabs]
    lappend details($w) [list tablabels $tablabels]
    lappend details($w) [list dodestroy $dodestroy]
    lappend details($w) [list passive $passive]
    lappend details($w) [list imagemap $imagemap]
    lappend details($w) [list expand $expand]
    lappend details($w) [list boxlabel $boxlabel]
    lappend details($w) [list boxlabelfunc $boxlabelfunc]
    lappend details($w) [list taboffset 0]
    lappend details($w) [list maxoffset 0]
    lappend details($w) [list maxvisible 0]
    lappend details($w) [list relief $relief]
    lappend details($w) [list tabside $tabside]

    canvas $w -width $width -height $height -highlightthickness 0

    if { $expand } {
      bind $w <Configure> "TabBox::reconfigure $w"
    }

    set i 0
    foreach tab $tabs {
      set t [lindex $tablabels $i]

      frame $w.b$i -height $tabheight -width $tabwidth -bd $bd -relief $relief
      frame $w.b$i.v -height [expr $tabheight - 2*$bd] -width 0
      frame $w.b$i.h -height 0 -width [expr $tabwidth -2*$bd-1]

      if {$t != "" } {
	label $w.b$i.l -text $t
      }

      pack $w.b$i.v -side left
      pack $w.b$i.h

      set didImg 0
      catch {
	if { $imagemap != "" } {
	  global $imagemap
	  set img [lindex [array get $imagemap $tab] 1]
	  if { $img != "" } {
	    label $w.b$i.img -image $img
	    pack $w.b$i.img -side left -padx 2
	    set didImg 1
	  }
	}
      }

      if { $t != "" } {
	if { $didImg } {
	  pack $w.b$i.l -side left -padx 2
	} else {
	  pack $w.b$i.l
	}
      }

      bind $w.b$i <Button-1> "TabBox::select $w $i"
      if {$t != "" } {
	bind $w.b$i.l <Button-1> "TabBox::select $w $i"
      }
      catch { bind $w.b$i.img <Button-1> "TabBox::select $w $i" }

      incr i
    }

    if { !$noslider && [expr $tabwidth * [llength $tabs]] > [expr $width - 50] } {
      makeTabSlider $w
    } else {
      set details($w) [assocset maxvisible $details($w) [llength $tabs]]
    }
    
    placeTabs $w

    if { $boxlabel != "" } {
      global tkg_pastelHighlight

      set x [expr $tabwidth * [llength $tabs]]
      set y $tabheight

      set specialcolor $tkg_pastelHighlight
      frame $w.boxlabel -height [expr $tabheight  + $bd] -bg $specialcolor
      label $w.boxlabel.l -text $boxlabel -bg $specialcolor -width 50 -justify left -anchor w
      pack $w.boxlabel.l -padx 2 -pady 2 -fill both -expand 1 -anchor w
      $w create window $x [expr $y + $bd] -anchor sw -window $w.boxlabel
    } elseif { $boxlabelfunc != ""} {
      set x [expr $tabwidth * [llength $tabs]]
      set y $tabheight

      frame $w.boxlabel -height [expr $tabheight  + $bd]
      $boxlabelfunc $w.boxlabel.l
      pack $w.boxlabel.l -padx 2 -pady 2 -fill both -expand 1 -anchor w
      $w create window $x [expr $y + $bd] -anchor sw -window $w.boxlabel
    }


    set start_idx [lsearch -exact $tabs $startpage]
    if { $start_idx < 0 } {
      set start_idx 0
    }

    switch $tabside {
      top {
	frame $w.f -width $width  -height [expr $height - $tabheight] -bd $bd -relief $relief
	pack propagate $w.f 0

#	frame $w.f.v -height [expr $height - $tabheight - 2*$bd] -width 0
#	frame $w.f.h -height 0 -width [expr $width -2*$bd-1]
#	pack $w.f.v -side left
#	pack $w.f.h

	$w create window 0 $tabheight -anchor nw -window $w.f

	frame $w.hider -width [expr $tabwidth - 2*$bd] -height [expr 2*$bd+1]
	$w create window 0 0 -anchor w -window $w.hider -tags hider
      }
      left {
	frame $w.f -width [expr $width - $tabwidth]  -height $height -bd $bd -relief $relief
	pack propagate $w.f 0

#	frame $w.f.v -height [expr $height - 2*$bd] -width 0
#	frame $w.f.h -height 0 -width [expr $width - $tabwidth - 2*$bd-1]
#	pack $w.f.v -side left
#	pack $w.f.h

	$w create window $tabwidth 0 -anchor nw -window $w.f

	frame $w.hider -width [expr 2*$bd +1] -height [expr $tabheight - $bd]
	$w create window 0 0 -anchor n -window $w.hider -tags hider
      }
    }

    TabBox::select $w $start_idx

    set tabheight 30
  }
}
