/* Brutalcopy client half
 * (c) 2000 Karel Kulhavy, Clocksoft
 * Brutalcopy is a program for copying files through UDP
 */

/* All times are in microseconds.
 */
#include <errno.h>
#include <netdb.h>
#include <stdlib.h>
#include <netdb.h>
#include <fcntl.h>
#include <stdio.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <sys/time.h>
#include <unistd.h>
#include <string.h>
#include <signal.h>

#include "def.h"

struct in_addr ip; /* ip where the server runs */
int ip_ok;
u16 distant_port=1235; /* Port on which the server sits */
unsigned char *filename="bcp.dat"; /* Name of file to dump the data from */
int fd; /* Socket */
int input_fd; /* Output file descriptor */
u64 bps=4096; /* Bytes per second */
int update_new_values; /* After SIGHUP signal */
u64 new_bps;
u64 packet_distance; /* Distance between consecutive packets in microseconds */
struct sockaddr_in distant_sai;
int distant_sai_len;
unsigned char pkt[65536]; /* In first 4 bytes of the packet there is permanently the session_id. */
unsigned char opkt[65536]; /* In first 4 bytes of the packet there is permanently the session_id. */
int pkt_len;
unsigned long payload_size=256, file_len;
unsigned long info_cache_size=128;
unsigned long n_packets;
unsigned long remaining_packets;
unsigned char *bitmap;
u64 next_packet_time; /* Last time SOT or data packet was sent */
u64 sot_retransmit_time=1000000;
u32 packet_seq;
unsigned char *listenname;

void
throw_out_socket(void)
{
 if (fd) 
 {
  close(fd);
 }
}

void die()
{
 printf("Exiting on fatal error.\n");
 throw_out_socket();
 exit(1);
}

/* Returns time since 1970 in microseconds */
unsigned long long get_time(void)
{
 struct timeval tv;
 struct timezone tz={0,0};

 if (gettimeofday(&tv,&tz))
 {
  fprintf(stderr,"Can't getttimeofday.\n");
  perror("");
  die();
 }
 return ((unsigned long long)tv.tv_sec)*1000000+((unsigned long long)tv.tv_usec);
}

void
print_help(void)
{
 fprintf(stderr,"bcpc -- a Brutalcopy client half\n"
"Brutalcopy (c) 2000 Karel Kulhavy, Clocksoft\n\n"

"Brutalcopy is a program for transferring rather large files by means of UDP\n"
"protocol in order to bypass TCP tendency to screw up on bad lines. Brutalcopy is\n"
"based on manual flow control.\n\n"
"Usage: bcpc -l <server hostname> -p <server port> [-h] [-i\n"
"<filename>] [-s <payload_size>] [-r <sot_retransmit_time>] [-b <bytes_per_sec>]\n"
"-l <server>    DNS name or IP address (in dot notation) of the server.\n"
"-p <dist_port> Port must be some high number (>=1024). Default is 1235.\n"
"-h             Prints out help and ends.\n"
"-i <filename>  File to dump the received data from. Default is bcp.dat.\n"
"-s <payload>   Number of bytes as a data payload in a single packet.\n"
"-r <srt>       Number of microsecond after which is SOT retransmitted.\n"
"-b <Bps>       Bytes of payload per second\n");
}

void
get_socket(void)
{
 fd=socket(PF_INET,SOCK_DGRAM,IPPROTO_UDP);
 if (fd<0)
 {
  fprintf(stderr,"Can't open socket.\n");
  die();
 }
 fcntl(fd,F_SETFL,O_NONBLOCK);
}

void
parse_args(int argc, char **argv)
{
 int res;

 while(-1!=(res=getopt(argc, argv, "hp:i:l:s:b:")))
 {
  switch(res)
  {
   case '?':
   fprintf(stderr,"Invalid option char.\n");
   case 'h':
   print_help();
   exit(0);
   case 's':
   payload_size=atol(optarg);
   break;
   case 'b':
   bps=atof(optarg);
   case 'r':
   sot_retransmit_time=atof(optarg);
   break;
   case 'p':
   distant_port=strtoul(optarg,NULL,0);
   break;
   case 'i':
   filename=optarg;
   break;
   case 'l':
   {
    struct hostent *h;
    listenname=optarg;
    h=gethostbyname(optarg);
    if (!h)
    {
     fprintf(stderr,"Can't look up hostname %s\n",optarg);
     perror("");
     die();
    }
    ip=*((struct in_addr *)(h->h_addr_list[0]));
    ip_ok=1;
   }
   break;
  }
 }
 if (!ip_ok)
 {
  fprintf(stderr,"Missing hostname.\n");
  print_help();
  die();
 }
}

void
print_ip(struct in_addr ip)
{
 int a;
 printf("%d",*((unsigned char *)&ip));
 for (a=1;a<sizeof(ip);a++)
  printf(".%d",((unsigned char *)&ip)[a]);
}

/* Takes pkt, pkt_len, computes checksum of pkt_len bytes starting at pkt
 * and compares with 4 bytes starting at pkt+pkt_len. The checksum format is described
 * in bcp.txt.
 * retval: 0 checksum OK, 1 checksum bad.
 */
int
test_checksum(void)
{
 unsigned long checksum; 
 unsigned long packet_checksum;

 checksum=crc32(pkt,pkt_len);
 packet_checksum=pkt[pkt_len]+(pkt[pkt_len+1]<<8)+(pkt[pkt_len+2]<<16)+(pkt[pkt_len+3]<<24);
 return checksum!=packet_checksum;
}

/* If packet is from unwanted ip it throws it out and tries again.
 * Result is placed on pkt[4], pkt_len is set to length of received UDP packet without the trailing CRC32
 * If no packet is available, immediately returns.
 */
void
get_packet()
{
 int retval;

 again:
 pkt_len=0;
 distant_sai_len=sizeof(distant_sai);
 retval=recvfrom(fd,(void *)pkt+4,sizeof(pkt)-4,MSG_WAITALL
  ,(struct sockaddr*)&distant_sai, &distant_sai_len);
 if (retval>0)
 {
  if (distant_port!=distant_sai.sin_port) goto again;
  pkt_len=retval;
  if (pkt_len<4) goto again;
  if (test_checksum()) goto again;
  pkt_len-=4;
  return;
 }
 if (retval==0) goto again;
 if (errno==EWOULDBLOCK) return;
 if (errno==EINTR) goto again;
 fprintf(stderr,"Can't read from UDP socket.\n");
 perror("");
 die();
}

/* Packet must be at opkt[4]. Len long. CRC32 is added to the end and in opkt[0] there must
 * be session_id.
 */
void
send_packet(unsigned len)
{
 unsigned long crc;
 
 if (len+8>sizeof(opkt))
 {
  fprintf(stderr,"Internal error: output packet too long.\n");
  die();
 }
 crc=crc32(opkt,len+4);
 opkt[4+len]=crc&255;crc>>=8;
 opkt[5+len]=crc&255;crc>>=8;
 opkt[6+len]=crc&255;crc>>=8;
 opkt[7+len]=crc&255;
 len+=4; /* Len is length of UDP packet now */
 {
  struct sockaddr_in sai;
  int retval;
  
  sai.sin_family=AF_INET;
  sai.sin_port=distant_port;
  sai.sin_addr=ip;
  again:
  retval=sendto(fd,opkt+4,len,0,(struct sockaddr*)&sai,sizeof(sai));
  if (retval==len) return;
  if (retval==0) goto again;
  if (retval>0&&retval!=len)
  {
   fprintf(stderr,"sendto should have sent %d bytes but sent only %d bytes.\n",len,retval);
   die();
  }
  if (retval==-1)
  {
   if (errno==EAGAIN||errno==ENOBUFS||errno==EINTR||errno==ENOMEM) goto again;
  }
  fprintf(stderr,"Can't send %d-byte packet.\n",len);
  perror("");
  die();
 } 
}

/* Allocates bitmap enough big to accomodate n_packets bits. */
void
alloc_bitmap(void)
{
 int bytes=(n_packets+7)>>3;

 bitmap=calloc(bytes,1);
 if (!bitmap)
 {
  fprintf(stderr,"Can't malloc %d bytes for bitmap.\n",bytes);
  die();
 }
}

/* If bitmap is set from 0 to 1, remaining_packets is deced */
void
set_bitmap(unsigned n)
{
 unsigned bit=n&7;
 unsigned byte=n>>3;

 if (n>=n_packets)
 {
  fprintf(stderr,"Invalid packet seq# of %u from %lu came! It means data are corrupted. Quitting immediately.\n",n,n_packets);
  die();
 } 
 if ((bitmap[byte]>>bit)&1) return;
 remaining_packets--;
 bitmap[byte]|=1<<bit;
}

/* Returns value of the bit */
int
test_bitmap(unsigned n)
{
 unsigned bit=n&7;
 unsigned byte=n>>3;

 return (bitmap[byte]>>bit)&1;
}

void
write32(unsigned char *dest, u32 val)
{
 dest[0]=val;val>>=8;
 dest[1]=val;val>>=8;
 dest[2]=val;val>>=8;
 dest[3]=val;
}

void
write64(unsigned char *dest, u64 val)
{
 dest[0]=val;val>>=8;
 dest[1]=val;val>>=8;
 dest[2]=val;val>>=8;
 dest[3]=val;val>>=8;
 dest[4]=val;val>>=8;
 dest[5]=val;val>>=8;
 dest[6]=val;val>>=8;
 dest[7]=val;
}

/* Reads in LSB first */
u32
read32(unsigned char *src)
{
 u32 rv=0;

 rv|=src[3];
 rv<<=8;
 rv|=src[2];
 rv<<=8;
 rv|=src[1];
 rv<<=8;
 rv|=src[0];
 return rv;
}

/* Reads in LSB first */
u64
read64(unsigned char *src)
{
 u64 rv=0;
 
 rv|=src[7];
 rv<<=8; 
 rv|=src[6];
 rv<<=8;
 rv|=src[5];
 rv<<=8;
 rv|=src[4];
 rv<<=8;
 rv|=src[3];
 rv<<=8;
 rv|=src[2];
 rv<<=8;
 rv|=src[1];
 rv<<=8;
 rv|=src[0];
 return rv;
}

/* Returns 1 if got a SOT ACK
 * Eats all packets from the que.
 * returns 0 if all thrown out and SOT ACK not found.
 */  
int got_sot_ack(void)
{
 again:
 get_packet();
 if (!pkt_len) return 0;
 if (pkt_len!=1) goto again;
 if (pkt[4]!=4) goto again;
 return 1;
}

void
sleep_until(unsigned long long t)
{
 u64 u=get_time();
 
 if (u>=t) return;
 t-=u;
 {
  fd_set set;
  fd_set zero_set;
  struct timeval timeout;

  /*
  printf("Sleeping %lld usec\n",t);
  */
  timeout.tv_sec=t/1000000;
  timeout.tv_usec=t%1000000;
  FD_ZERO(&zero_set);
  FD_ZERO(&set);
  FD_SET(fd,&set);
  select(fd+1,&set,&zero_set,&zero_set,&timeout);
 }
}

void compute_packet_distance(void)
{
 double ppus=((double)bps/1000000/payload_size);
 packet_distance=1/ppus;
}

/* SOT ACK is received before return.
 * send_sot also leaves intput_filename open on input_fd.
 */
void
send_sot(void)
{
 
 int h;
 
 input_fd=open(filename,O_RDONLY);
 if (input_fd<0)
 {
  fprintf(stderr,"Unable to open file %s for reading.\n",filename);
  perror("");
  die();
 }
 {
  off_t off;
  off=lseek(input_fd,0,SEEK_END);
  if (off<0)
  {
   fprintf(stderr,"Can't seek at the end of %s. Can't know the size.\n",filename);
   perror("");
   die();
  }
  file_len=off;
  n_packets=(file_len+payload_size-1)/payload_size;
  alloc_bitmap();
  remaining_packets=n_packets;
 }
 compute_packet_distance();
 printf("Packet sending period=%lld usec\n",packet_distance);
 h=open("/dev/random",O_RDONLY);
 read(h,opkt,4); /* Random session ID */
 close(h);
 opkt[4]=0;
 memcpy(opkt+5,opkt,4);
 memcpy(pkt,opkt,4);
 write64(opkt+9,bps);
 write32(opkt+17,payload_size);
 write32(opkt+21,file_len);
 write32(opkt+25,info_cache_size);
 resend:
 next_packet_time=get_time()+sot_retransmit_time;
 send_packet(25);
 again:
 sleep_until(next_packet_time);
 if (got_sot_ack()) return;
 if (next_packet_time<=get_time()) goto resend;
 goto again;
}

/* Takes packet_seq and finds first zero under or behind it.
 */
void
find_first_zero(void)
{
 while(test_bitmap(packet_seq))
 {
  packet_seq++;
  if (packet_seq>=n_packets) packet_seq=0;
 }
}

/* Emits data packets using packet_seq and moves it appropriately. Updates also remaining_packets.
 * If nothing to send, returns.
 */
void
emit_data(void)
{
  int off=9;
  ssize_t retval;
  ssize_t my_payload_size;
  ssize_t packet_size;
  
  if (!remaining_packets) return;
  find_first_zero();
  opkt[4]=1;
  write32(opkt+5,packet_seq);
  if (packet_seq*payload_size!=lseek(input_fd,packet_seq*payload_size,SEEK_SET))
  {
   fprintf(stderr,"Can't seek to position %ld in file %s.\n",packet_seq*payload_size,filename);
   perror("");
   die();
  } 
  
  read_again:
  packet_size=payload_size;
  if (packet_seq==n_packets-1)
  {
   packet_size=file_len%payload_size;
   if (!packet_size) packet_size=payload_size;
  }
  my_payload_size=packet_size;
  retval=read(input_fd,opkt+off,my_payload_size);
  if (!retval) goto read_again;
  if (retval>my_payload_size)
  {
   fprintf(stderr,"read read more than I asked it to!\n");
   die();
  }
  if (retval==my_payload_size) goto done;
  if (retval>0)
  {
   off+=retval;
   my_payload_size-=retval;
  }
  if (errno==EINTR||errno==EAGAIN) goto read_again;
  fprintf(stderr,"Can't read from file %s.\n",filename);
  perror("");
  die();

  done:
  send_packet(5+packet_size);
  if (update_new_values){
	  update_new_values=0;
	  bps=new_bps;
	  compute_packet_distance();
  }
  if (packet_size==payload_size)
  {
   next_packet_time+=packet_distance;
  }
  else
  {
   next_packet_time+=packet_distance*packet_size/payload_size;
  }
  packet_seq++;
  if (packet_seq==n_packets) packet_seq=0;
}

/* On pkt[5] info packet begins with seq#. proces_info_packet must not modify the packet seq#. */
void
process_info_packet(void)
{
 unsigned char *ptr;
 if (pkt_len&3) return; /* Broken packet */
 
 pkt_len>>=2;
 pkt_len--;
 for (ptr=pkt+9;pkt_len;pkt_len--,ptr+=4)
 {
  u32 bit=read32(ptr);
  set_bitmap(bit);
 }
}

/* Processes incoming packets. Returns 1 if EOT seen, 0 otherwise. Return as soon as no packets are available
 * for reading.
 */
int
check_for_incoming_packets(void)
{
 again:
 get_packet();
 if (!pkt_len) return 0; /* Nothing available */
 pkt_len--;
 switch(pkt[4])
 {
  case 2:
  process_info_packet();
  opkt[4]=5;
  memcpy(opkt+5,pkt+5,4);
  send_packet(5);
  break;
  case 3:
  return 1;
 }
 goto again;
}

/* When EOT is received, this function returns. Otherwise it generates the transmitted data and receives
 * info packets, sends info acks, fills in the bitmap and emits the data. Also checks for EOT.
 */
void
transmit(void)
{
 next_packet_time=get_time();
 goto emit_now;
 
 again:
 sleep_until(next_packet_time);
 if (next_packet_time>get_time()) goto bypass;
 emit_now:
 emit_data();
 bypass:
 if (check_for_incoming_packets()) return;
 goto again;
}

void
open_session(void)
{
 get_socket();
 send_sot();
}

void
close_session(void)
{
 throw_out_socket();
 close(input_fd);
}

void *
xalloc(size_t len)
{
 void *r;
 
 if (!len) return 0;
 r=malloc(len);
 if (!r)
 {
  fprintf(stderr,"Can't malloc %d bytes.\n",len);
  die();
 }
 return r;
}

void *
xcalloc(size_t len)
{
 void *r;
 
 if (!len) return 0;
 r=calloc(len,1);
 if (!r)
 {
  fprintf(stderr,"Can't calloc %d bytes.\n",len);
  die();
 }
 return r;
}

void hup_handler(int sig)
{
	FILE *f=fopen(".speed","r");

	if (!f) goto ending;
	if (!fscanf(f,"%llu",&new_bps)) goto ending;
	fclose(f);
	update_new_values=1;
	ending:
	signal(1,hup_handler);
}

int
main(int argc, char **argv)
{
 
 signal(1,hup_handler);
 parse_args(argc,argv);
 printf("Starting bcp client, dumping from %s, sending to %s, IP ", filename,listenname);
 print_ip(ip);
 printf(", port %d.\n",distant_port);
 open_session();
 printf("Session established.\n");
 /* Here we know all about the accepted session. */
 transmit();
 /* EOT received. We end :-) */
 close_session();
 return 0;
}

