#include <unistd.h>
#include <dirent.h>
#include <getopt.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>
#include <errno.h>
#include <fcntl.h>
#include <sys/time.h>

#include "cpdu.h"
#include "cipher.h"
#include "secmem.h"
#include "sys_linux.h"
#include "var.h"
#include "opts.h"
#include "error.h"
#include "ciphervar.h"
#include "sha1.h"
#include "zlf.h"

typedef enum
{	cpdu_archive=0,
	cpdu_encrypted=1,
	cpdu_none =2
} cpdu_file_type;

/* -- start main -- */

//#define size_t unsigned long long

/* -- function prototypes -- */
void chmod_secure_file(char *file);
void chmod_unsecure_file(char *file);
void chmod_secure_mykeydir();
void chmod_unsecure_mykeydir();
char *make_mykeydir();
char *make_mydir();
char **getdir(char *dirname, int recurse);
size_t init_file_encrypt(void* pctx, FILE* in, FILE* out);
size_t init_file_decrypt(void* pctx, FILE* in, FILE* out);
int getpassword(int ciphermode, char *prompt);
int read_stdin_to_fd(int fd_out);
int write_fd_to_stdout(int fd_in);
void _secureclearpassbuf();
void secure_local_key_set();
void secure_local_key_get();
char *get_current_dir_name(){return getenv("PWD");}
int cpdu_archive_unload(char* larchive_name);
int cpdu_archive_load(char* larchive_name);
unsigned char* sha1_get_file_hash(char* file);
char *make_myrecoverdir();
char *rm_relative_path(char* name);
int copy_file(char *file, char* new);
char *mk_recoverdb_entry(char *entry);
int init_log(int argc, char** argv);
int addkey_to_masterkeylist();
int getkey_from_masterkeylist(char *digest);
void status_print_session_done();
void cpdu_signal_sigint(int signal);
cpdu_file_type cpdu_query_file_type(char *file);
int generate_keyfile(char *keyfile, size_t fsize);
unsigned char *crunch_key(unsigned char *xkey, unsigned long xkeysize, unsigned char *keybuf, unsigned long outsize);
int read_key_from_keyfile(char *keyfile, unsigned char *keybuf, unsigned long keysize);
/* -- function prototypes -- */

/* -- cpdu.c variables -- */
unsigned char* passbuf=NULL;
unsigned long* passbufsz=NULL;
int margc;
char **margv;
char *archive_name = NULL;
static int file_processed_count = 0;
unsigned char check_hash[SHA1_DIGESTSIZE];
/* -- cpdu.c variables -- */

/* -- cpdu.c constants -- */
char cpdu_archive_hdr[] = "cpdu_encrypted_archive";
char cpdu_encrypt_hdr[] = "cpdu_encrypted_file";
/* -- cpdu.c constants -- */

/* -- main -- */
#include <sys/mman.h>
#include <signal.h>
int main(int argc, char **argv)
{
	struct stat statbuf;
	struct sigaction act_sigint;
	char dostdin=0;

	opts = parse_options(argc, argv);

	if ( mlockall(MCL_CURRENT) == -1 ) {
	char *user;
	user = getenv("USER");
		if ( opts->verbose || !strcmp(user, "root") ) 
		fprintf(stderr, "%s: error locking process virtual memory space: %s\n", argv[0], strerror(errno));
	}

	act_sigint.sa_handler = cpdu_signal_sigint;
	sigemptyset(&act_sigint.sa_mask);
	act_sigint.sa_flags = 0;
	sigaction(SIGINT, &act_sigint, 0);
	init_system();

	ciphermode = opts->ciphermode; /* global variable 'ciphermode' set for the session */
	dostdin = opts->rstdin; /* once stdin is finished 'dostdin' variable is set to 0 */

/* other option processing: we first handle to see if other options are set, if so we process them */
	if ( opts->slkeyset ) {
		secure_local_key_set();
		exit(0);
	}

	if ( opts->genkeyfile ) {
		generate_keyfile(opts->keyfile, opts->gensize);
		chmod_secure_file(opts->keyfile);
		exit(0);
	}

	if ( ciphermode == CIPHER_MODE_ENCRYPT )
	{   cipherctx = determineCipherContext(opts->cipher);
    		if ( !cipherctx ) {
                        merror("not able to determine cipher context.\n");
                        opts->destructor(opts);
                        close_system();
                        exit(0);
                }
	initcbcblock = (unsigned char*) smalloc(curblocksize);
	initcbcblocksize = curblocksize;
    }

/* we recurse always in archive mode for reasons of safety */
if ( ciphermode==CIPHER_MODE_ENCRYPT && opts->doarchive ){
opts->recurse = 1;
opts->zlib = 1;
}

archivelist = (char**) malloc(sizeof(char*));

if ( ciphermode==CIPHER_MODE_DECRYPT && opts->doarchive ){
	cpdu_file_type ft;
	if ( (ft = cpdu_query_file_type (opts->archive)) == cpdu_archive ) {
		archivelist = (char**) realloc(archivelist, archivelist_size +=sizeof(char*));
		archivelist[archivelist_count] = opts->archive;
		archivelist_count++;
	} else { merror("not a cpdu archive: %s\n", opts->archive); }
}

/* file name processing: we update the file list and count variables here, if there are no files we exit */
{
	int i;
	struct stat sb;
	char *entry;
	cpdu_file_type ft;
	argvlist = opts->argv;
for(i=0; argvlist[i]; i++) {
		if ( stat(argvlist[i], &sb) == -1 )
		{ 
			merror("%s: %s\n", argvlist[i], strerror(errno));
			continue;
		}
		else
		{
#if 0
		if ( ciphermode == CIPHER_MODE_ENCRYPT && strstr(argvlist[i], make_myrecoverdir()) ) {
			merror("tring to encrypt a file in the recovery database. skipping file.\n");
			continue;
		} else {
			char *svl = strdup(get_current_dir_name()), *pragma;
			chdir("..");
			pragma = strdup(get_current_dir_name());
			if ( strstr(pragma, make_myrecoverdir()) ) {
				merror("tring to encrypt file in the recovery database. skipping file.\n");
				chdir(svl); free(svl); free(pragma);
				continue;
			}
			chdir(svl); free(svl); free(pragma);
		}
#endif
		if ( chmod(argvlist[i], sb.st_mode) == -1 ) {
			merror("insufficient permissions: %s: %s\n", argvlist[i], strerror(errno));
			continue;
		}

		if ( (ciphermode == CIPHER_MODE_DECRYPT) && !S_ISDIR(sb.st_mode) )
		{
			ft = cpdu_query_file_type (argvlist[i]);
			if ( ft == cpdu_none ) {
				merror("file not encrypted: %s\n", argvlist[i]);
				continue;
			} else if ( ft == cpdu_archive ) {
				archivelist = (char**) realloc(archivelist, archivelist_size +=sizeof(char*));
				archivelist[archivelist_count] = strdup(argvlist[i]);
				archivelist_count++;
				continue;
			} else if ( ft == cpdu_encrypted ) {
				filecount++;
			}
		}
		} /* else if stat */

	if ( ciphermode == CIPHER_MODE_ENCRYPT && !S_ISDIR(sb.st_mode) ) {
		if ( opts->rmrpath ) {
		static int* n;
		rmrpathlist = realloc(rmrpathlist, rmrpathlistsz+=sizeof(int*));
		n = malloc(sizeof(int));
		*n = i;
		rmrpathlist[rmrpathcount++] = n;
		}
		filecount++;
	}
	entrylist = (char**) realloc(entrylist, entrylist_size+=sizeof(char*));
	entrylist[entrylist_count] = strdup(argvlist[i]);
	entrylist_count++;

	if ( S_ISDIR(sb.st_mode) ) {
		if ( opts->rmrpaths ) rmrpathsnoe++;
		getdir(argvlist[i],opts->recurse);
	}
} /* for */

	if ( opts->rstdin ) filecount++;

	entrylist = (char**) realloc(entrylist, entrylist_size+=sizeof(char*));
	entrylist[entrylist_count] = NULL;
	listcounter=0;
	entrycounter=0;
	if ( archivelist ) {
		archivelist = (char**) realloc(archivelist, archivelist_size +=sizeof(char*));
		archivelist[archivelist_count] = NULL;
		archivecounter = 0;
	}
	/* check to see first if there are files to process, if not we print message and exit */
	if ( !filecount && !archivelist_count) { /*FIXME: THIS THING IS ANNOYING! LOL merror("nothing to do\n");*/
	status_print_session_done();		
	opts->destructor(opts); if ( cipherctx ) freeCipherContext(cipherctx); close_system(); exit(0);
	}
}

if ( !opts->keylist ) {
	if ( opts->slkey ) {
		secure_local_key_get();
	}
	else if ( !opts->slkey && !opts->usekeyfile )
	{ int r;
		r = getpassword(ciphermode, "Passphrase:");
		if ( r == -1 ) { _secureclearpassbuf(); sfree(passbuf); sfree(passbufsz); opts->destructor(opts); if ( cipherctx ) freeCipherContext(cipherctx); 				close_system(); exit(0);
		}
	} else 	if ( ciphermode == CIPHER_MODE_ENCRYPT && opts->usekeyfile ) {
		int ret;
		passbuf = (unsigned char*) smalloc(curkeysize);
		passbufsz = (unsigned long*) smalloc(sizeof(unsigned long));
		*passbufsz = curkeysize;
		ret = read_key_from_keyfile(opts->keyfile, passbuf, curkeysize);
		if ( ret == -1 ) {
			merror("could not retrieve key from keyfile %s\n", opts->keyfile);
			opts->destructor(opts); if ( cipherctx ) freeCipherContext(cipherctx); close_system();
			exit(0);
		}
	}
}

	if ( ciphermode == CIPHER_MODE_ENCRYPT ) {
		int ret;
		ret = addkey_to_masterkeylist(); /* adds the current passbuf */
		if ( ret != 0 ) {
			merror("WARNING: could not add key to master key list!: security error. aborting encryption session.\n");
			opts->destructor(opts); if ( cipherctx ) freeCipherContext(cipherctx); close_system(); exit(0);
		}
	}

_do_n_archive:

while(entrylist[entrycounter]||dostdin)
{
	FILE *in, *out;
	char* string;
	mode_t svdmode;

if ( dostdin ) {
	char fstdin[] = ".cpdu_stdin";
	int fd=0; fd = open(fstdin, O_CREAT|O_WRONLY, S_IRUSR|S_IWUSR|S_IXUSR);
	read_stdin_to_fd(fd);
	close(fd);
	curfile = &fstdin[0];
	chmod_unsecure_file(curfile);	
} else {

	if (entrylist[entrycounter]) {
		curfile = entrylist[entrycounter++];
	{ int r; struct stat sb; r = stat(curfile, &sb); if ( S_ISDIR(sb.st_mode) || r==-1 ) continue; }
	} else { break; }

}
	stat(curfile, &statbuf);
	curfilesize = statbuf.st_size;
	svdmode = statbuf.st_mode;

	if ( ciphermode == CIPHER_MODE_ENCRYPT && opts->registerrdb ) {
		int r;
		chmod(make_myrecoverdir(), S_IRUSR|S_IWUSR|S_IXUSR);
		char* rentry = mk_recoverdb_entry(curfile);
		r = copy_file(curfile, rentry);
			if ( r == -1 ) {
				merror("WARNING: could not copy file %s to recover database, aborting encryption\n", curfile);
				continue;
			}
		chmod_secure_file(make_myrecoverdir());
	}
		
	{ int n = strlen(curfile)+10+4;
	string = (char*) malloc(n);
	strcpy(string, curfile);
	strcat(string, ".encrypted\0");
	curfileout = string;
	}

	if ( ciphermode == CIPHER_MODE_ENCRYPT && opts->zlib )
	{
	char mode[4];
	strcpy(mode, "wb");
	strncat(mode, opts->zlib_level_string, 1);
	strcat(mode,"\0");
	if ( file_compress(curfile, string, mode) == -1 ) {
		merror("could not compress file %s, aborting encryption.\n", curfile);
		continue;
	}
	rename(string, curfile);
	}

	stat(curfile, &statbuf);
	curfilesize = statbuf.st_size;

	in = fopen(curfile, "rm");
	if ( ferror(in) ) { merror("file open error on file %s: %s", curfile, strerror(errno)); continue;}
	out = fopen(string, "w");
	if ( ferror(out) ) { merror("file open error on file %s: %s", string, strerror(errno)); continue;}
	setvbuf(in, NULL, _IONBF, 0);
	setvbuf(out, NULL, _IONBF, 0);

	cipher_block_mode = CIPHER_MODE_CBC;

if ( ciphermode == CIPHER_MODE_DECRYPT )
{	
	rwoffset = init_file_decrypt(cipherctx, in, out);
		if ( rwoffset == -1 ) { fclose(in); fclose(out); remove(string); fprintf(stderr, "\n"); continue; }
	(*pCreateWorkContext)(cipherctx, passbuf, *passbufsz, CIPHER_MODE_DECRYPT, initcbcblock, NULL, NULL);
}
else
{
	(*pCreateWorkContext)(cipherctx, passbuf, *passbufsz, CIPHER_MODE_ENCRYPT, initcbcblock, &get_random_bytes, NULL);
	originalfilesize = curfilesize;
	rwoffset = init_file_encrypt(cipherctx, in, out);
		if ( rwoffset == -1 ) { fclose(in); fclose(out); remove(string); fprintf(stderr, "\n"); continue; }
}

fprintf(stderr, "%s", curfile);

{ int n;
	n = do_file_transform(cipherctx, in, out);
		if ( n == -1 ) {
			merror("\nerror transforming file %s\n", curfile);
			fclose(in); fclose(out);
			remove(string);
			dostdin = 0;
			continue;
		}
}

	fclose(in);
	fclose(out);

	if ( ciphermode == CIPHER_MODE_DECRYPT ) {
		unsigned char* ldg;
		ldg = sha1_get_file_hash(string);
		if ( memcmp(ldg, check_hash, SHA1_DIGESTSIZE) == 0 )
		{ } else {
			remove(string);
			merror("\n%s: file hashes do not match\n", curfile);
			continue;
		}
	}

	if ( ciphermode == CIPHER_MODE_ENCRYPT && opts->wipe ) {
		wipefile(curfile);
	}

	rename(string, curfile);
	remove(string);

	if ( ciphermode == CIPHER_MODE_DECRYPT && decompress )
	{
		file_uncompress(curfile);
		decompress = 0;
	}

	chmod(curfile, svdmode);	

	if ( dostdin ) {
		int fd; fd = open(curfile, O_RDONLY);
		write_fd_to_stdout(fd);
		close(fd);
		remove(curfile);
		dostdin = 0;
	}

	file_processed_count++;
{fprintf(stderr, "\n");}
} /* while */

	if ( opts->doarchive && (ciphermode == CIPHER_MODE_ENCRYPT)) {
		if ( cpdu_archive_load(opts->archive) == -1 ) {
		merror("error loading archive %s with session files\n", opts->archive);
		}
	}

	if ( entrylist ) /* free the file names in the entry list */
	{ int i; for(i=0;i<entrylist_count;i++) free(entrylist[i]); entrylist[i] = NULL;}
	entrylist = NULL;
	entrylist_size = 0;
	entrylist_count = 0;
	entrycounter = 0;

if ( (ciphermode == CIPHER_MODE_DECRYPT) && archivelist_count ) {
	static int counter=0, r;
	for(; counter < archivelist_count;) {
		r = cpdu_archive_unload(archivelist[counter]);
		if ( r == -1 ) {
			if ( entrylist ) /* free the file names in the entry list */
			{ int i; for(i=0;i<entrylist_count;i++) free(entrylist[i]); entrylist[i] = NULL;
			entrylist = NULL;
			entrylist_size = 0;
			entrylist_count = 0;
			entrycounter = 0;
			}
			merror("error unloading archive %s. aborting.\n", archivelist[counter]);
			counter++;
			continue;
		}
	counter++;
	goto _do_n_archive;
	}
}

	status_print_session_done();	
	chmod_secure_mykeydir();
	opts->destructor(opts);
	_secureclearpassbuf();
	freeCipherContext(cipherctx);
	close_system();

	return 0;
}
/*! -- main -- */

/* -- end main -- */

/* -- start cpdu.c functions -- */
unsigned char* sha1_get_file_hash(char* file)
{
	int n;
	PSHA1CTX shactx;
	FILE *fs;
	unsigned char* buffer;
	static unsigned char dg[SHA1_DIGESTSIZE];
	fs = fopen(file, "r");
		if (!fs) {
			merror("could not open file %s for digest: %s\n", file, strerror(errno));
			return NULL;
		}
	buffer = (unsigned char*) smalloc(4096);
	if ( !buffer ) {
		merror("error allocating buffer for digest: %s\n", strerror(errno));
		fclose(fs);
		return NULL;
	}
	shactx = SHA1_Create();
	while ( ((n = fread(buffer, 1, 4096, fs)) != 0) && !ferror(fs) )
	{
		SHA1_Update(shactx, buffer, n);
	}
	if ( ferror(fs) ) {
		merror("error reading from %s stream: %s\n", file, strerror(errno));
		sfree(buffer);
		fclose(fs);
		return NULL;
	}

	SHA1_Final(dg, shactx);
	if ( ciphermode != CIPHER_MODE_DECRYPT )
	memcpy(check_hash, dg, SHA1_DIGESTSIZE);

	sfree(buffer);
	fclose(fs);
	return &dg[0];
}

#define ENCRYPT (cpdu_encrypt_hdr)
size_t init_file_encrypt(void* pctx, FILE *in, FILE *out)
{
	size_t r=0, n=0;
	unsigned char *vencrypt;
	PSHA1CTX shactx;
	WORD8 dg[SHA1_DIGESTSIZE];

	shactx = SHA1_Create();
	SHA1_Update(shactx, passbuf, *passbufsz);
	SHA1_Final(dg, shactx);

	sha1_get_file_hash(curfile);

	n = sizeof(ENCRYPT)+sizeof(size_t)+initcbcblocksize+(SHA1_DIGESTSIZE*2)+1+1;
	vencrypt = (unsigned char*) malloc(n);
	memcpy(vencrypt, ENCRYPT, sizeof(ENCRYPT));
	vencrypt[sizeof(ENCRYPT)] = curcode;
	vencrypt[sizeof(ENCRYPT)+1] = opts->zlib ? '1' : '0';
	memcpy(vencrypt+sizeof(ENCRYPT)+1+1, initcbcblock, initcbcblocksize);
	memcpy(vencrypt+sizeof(ENCRYPT)+1+1+initcbcblocksize, &curfilesize, sizeof(size_t));
	memcpy(vencrypt+sizeof(ENCRYPT)+1+1+initcbcblocksize+sizeof(size_t), dg, SHA1_DIGESTSIZE);
	memcpy(vencrypt+sizeof(ENCRYPT)+1+1+initcbcblocksize+sizeof(size_t)+SHA1_DIGESTSIZE, check_hash, SHA1_DIGESTSIZE);
	r += fwrite(vencrypt, 1, n, out);

	return r;
}

size_t init_file_decrypt(void* pctx, FILE* in, FILE* out)
{
	int ret;
	size_t r=0, n=0, s=0;
	unsigned char *vencrypt;
	PSHA1CTX shactx;
	WORD8 dg[SHA1_DIGESTSIZE], tempdg[SHA1_DIGESTSIZE];

	n = sizeof(ENCRYPT)+1+1;
	s = n;
	vencrypt = (unsigned char*) smalloc(n); 
	r = fread(vencrypt, 1, n, in);
	if ( strncmp(vencrypt, ENCRYPT, sizeof(ENCRYPT)) ) { merror("file not encrypted: %s", curfile); return -1; }
	curcode = vencrypt[sizeof(ENCRYPT)];
	codeset = 1;
	if ( cipherctx ) sfree(cipherctx);
	cipherctx = determineCipherContext(NULL);
	if ( initcbcblock ) sfree(initcbcblock);
	initcbcblocksize = curblocksize;
	initcbcblock = (unsigned char*) smalloc(initcbcblocksize);
	if ( vencrypt[sizeof(ENCRYPT)+1] == '1' ) decompress = 1; else decompress = 0;
	n = initcbcblocksize+sizeof(size_t)+(SHA1_DIGESTSIZE*2);
	sfree(vencrypt);
	vencrypt = (unsigned char*) smalloc(n);
	r = fread(vencrypt, 1, n, in);
	s+=n;
	memcpy(initcbcblock, vencrypt, initcbcblocksize);
	memcpy(&originalfilesize, vencrypt+initcbcblocksize, sizeof(size_t));
	memcpy(tempdg, vencrypt+initcbcblocksize+sizeof(size_t), SHA1_DIGESTSIZE);
	if ( opts->keylist ) {
		ret = getkey_from_masterkeylist(tempdg); /* FIXME: currently, we have no internal security mechanism for the master keylist */
		if ( ret != 0 ) {
			merror("could not retrieve key for file %s. key is not in the current master list. aborting decryption.\n", curfile);
			sfree(vencrypt);
			return -1;
		}
	}
	if ( opts->usekeyfile ) {
		int ret;
		if ( passbuf ) _secureclearpassbuf();
		passbuf = (unsigned char*) smalloc(curkeysize);
		passbufsz = (unsigned long*) smalloc(sizeof(unsigned long));
		*passbufsz = curkeysize;
		ret = read_key_from_keyfile(opts->keyfile, passbuf, curkeysize);
		if ( ret == -1 ) {
			merror("could not retrieve key from keyfile %s\n", opts->keyfile);
			sfree(vencrypt);
			return -1;
		}
	}
	shactx = SHA1_Create();
	SHA1_Update(shactx, passbuf, *passbufsz);
	SHA1_Final(dg, shactx);
	if ( memcmp(tempdg, dg, SHA1_DIGESTSIZE) == 0 ) {}
	else { merror("invalid password: %s", curfile); return -1; }
	memcpy(check_hash, vencrypt+initcbcblocksize+sizeof(size_t)+SHA1_DIGESTSIZE, SHA1_DIGESTSIZE);
	sfree(vencrypt);
	
	return s;
}

int do_file_transform(void* pctx, FILE* in, FILE* out)
{
	struct timeval tv; struct timezone tz;
	time_t stmsec; suseconds_t stmusec;
	unsigned char *buf = NULL, *obuf = NULL, *sbuf = NULL;
	size_t bufsize = 4096, bytes=curfilesize, toread=bufsize, towrite=0, r=0, w=0, cr=0, cw=0, rresized=0;
	unsigned long long i=0, j=0;

	if ( opts->progress ) {
	gettimeofday(&tv,&tz); 
	stmsec = tv.tv_sec; stmusec = tv.tv_usec;
	}

	buf = (unsigned char *) smalloc(bufsize+curblocksize);
		if ( !buf ) {
			merror("could not allocate %d bytes from a secure heap", bufsize);
			return -1;
		}
	obuf = (unsigned char *) smalloc(bufsize+curblocksize);
		if ( !obuf ) {
			merror("could not allocate %d bytes from a secure heap", bufsize);
			return -1;
		}

	if ( ciphermode == CIPHER_MODE_DECRYPT ) {
	curfilesize -= rwoffset;
	bytes = curfilesize;
	}

	r = bufsize;

	j = curfilesize;
	i=0;
	cw=0;

while ( bytes > 0 ) {

	do {
	r = fread(buf, 1, bufsize, in);
	} while ( r == -1 && errno == EINTR );

	cr += r;
	i+=r;

	if ( r % curblocksize )
	r += curblocksize;

	if ( r == 0 ) break;

	if ( ferror(in) ) { merror("file read error: %s", strerror(errno)); return -1; }

	if ( ciphermode == CIPHER_MODE_ENCRYPT ) {
	if ( cr == curfilesize && curfilesize % curblocksize )
			r += curblocksize;
		(*pEncryptBuffer)(pctx, buf, obuf, r);
	} else
	if ( ciphermode == CIPHER_MODE_DECRYPT ) {
		(*pDecryptBuffer)(pctx, buf, obuf, r, NULL);
	}

	if ( ciphermode == CIPHER_MODE_DECRYPT ) {
		if ( cw + r >= originalfilesize ) {
			r = originalfilesize-(cw);
			rresized = 1;
			i = j = 1;
		}
	}

	do {
	w = fwrite(obuf, 1, r, out);
	} while ( w == -1 && errno == EINTR );

	if ( ferror(out) ) { merror("file write error: %s", strerror(errno)); return -1; }

	cw+=w;

	if ( opts->progress )
	{
	static char sbuf[2000] = {' '};
	static unsigned long long percent;
	static char pb[41];
	pb[0] = '[';
	pb[39] = ']';
	pb[40] = '\0';
	memset(pb+1, ' ', 38);
	memset(pb+1, ciphermode == 1 ? '+' : '-', i*38/j);
	percent = i*100/j;
	if ( percent > 100 ) percent = 100;
	sprintf(sbuf, "\r[%s][%d%c]%s %s\0", opts->cipher, (int) percent, '%', pb, curfile);
	fprintf(stderr, "%s", sbuf);
	if ( j == i ) { //int x; fputc('\r', stderr);
	//for(x=0; x<strlen(sbuf); x++) fputc(' ', stderr);
//	fprintf(stderr, "\r[%s][100%c] %s", opts->cipher, '%', curfile);
#if 0
		gettimeofday(&tv,&tz);
		stmsec = tv.tv_sec - stmsec; stmusec = tv.tv_usec - stmusec;
		stmsec = stmsec < 0 ? 0:stmsec; stmusec = stmusec < 0 ? 0:stmusec;
		fprintf(stderr, "\r|%s||%d%c|%s %s |%d.%d|", opts->cipher, (int) percent, '%',pb, curfile, stmsec, stmusec);
#endif
	}
	}

	bytes-=w;
}

	sfree(buf);
	sfree(obuf);
	return 0;
}

char **getdir(char *dirname, int recurse)
{
	struct dirent *direntt = NULL;
	DIR *dir = NULL;
	char *entry=NULL;
	struct stat sb;
	cpdu_file_type ft;

	stat(dirname, &sb);
	if ( chmod(dirname, sb.st_mode) == -1 ) {
			merror("insufficient permissions on directory: %s: %s\n", dirname, strerror(errno));
			return entrylist;
	}

	dir = opendir(dirname);
		if (!dir) {
			merror("could not open directory %s: %s", dirname, strerror(errno));
			goto _nullify_;
		}

	while(direntt = readdir(dir)) {
		if ( !strcmp(direntt->d_name, ".") || !strcmp(direntt->d_name, "..") )
				continue;

/* build our entry */
{ size_t len, i, n=0; char nbs=0;
		len = strlen(dirname)+strlen(direntt->d_name);
		if ( dirname[strlen(dirname)] != '/' ) { nbs = 1; len += 1; }
		entry = (char*) malloc(len+1); /* including null space */
		for (i = 0 ; i < strlen(dirname); i++)
			entry[i] = dirname[i];
		if ( nbs ) entry[i++] = '/';
		n+=i;
		for (i = 0 ; i < strlen(direntt->d_name); i++)
			entry[i+n] = direntt->d_name[i];
		n+=i;
		entry[n] = '\0';
}
	stat(entry, &sb);

	if ( chmod(entry, sb.st_mode) == -1 ) {
			merror("insufficient permissions: %s: %s\n", entry, strerror(errno));
			free(entry);
			continue;
	}

if ( ciphermode == CIPHER_MODE_DECRYPT && !S_ISDIR(sb.st_mode) ) {
	ft = cpdu_query_file_type (entry);
	if ( ft == cpdu_none ) {
		merror("file not encrypted: %s\n", entry);
		free(entry);
		continue;
	} else if ( ft == cpdu_archive ) {
			archivelist = (char**) realloc(archivelist, archivelist_size +=sizeof(char*));
			archivelist[archivelist_count] = entry;
			archivelist_count++;
			continue;
	}
}

		entrylist = (char**) realloc(entrylist, entrylist_size +=sizeof(char*));
		entrylist[entrylist_count] = entry;
		entrylist_count++;

		if ( S_ISDIR(sb.st_mode) ) {
		if ( opts->rmrpaths ) rmrpathsnoe++;
		if ( recurse ) {
			getdir(entry, recurse);
			continue;
		}
		} else { filecount++; }

	} /* while */

	closedir(dir);

	_nullify_:
	return entrylist;
}

int wipefile(char* file)
{
	unsigned char buffer[4096];
	int size, i=0, j, n;
	FILE *out;
	struct stat statbuf;

	stat(file, &statbuf);
	size = statbuf.st_size;

	out = fopen(file, "r+");
	if ( !out || ferror(out) ) {
		merror("open failed on %s: %s", file, strerror(errno));
		return -1;
	}

	setvbuf(out, NULL, _IONBF, 0);

	for(j = 0; j<opts->wtimes; j++) {
		while ( size > 0 ) {
			i = gather_random_fast(buffer, 4096);
			do {
				n = fwrite(buffer, i, 1, out);
				fflush(out);
				fsync(fileno(out));
			} while ( n == -1 && errno == EINTR );
			fflush(out);
			size -= i;
		}
		rewind(out);
	}

	fclose(out);
}

void _securereadpassbuf(int fd)
{
	int i=0;
	int *r;
	unsigned char *n;
	r = (int*) smalloc(1);
	n = (unsigned char*) smalloc(1);
	passbufsz = (unsigned long*) smalloc(sizeof(unsigned long));
	passbuf = NULL;
	*passbufsz = 0;
	do {
		do {
			*r = read(fd, n, 1);
		} while ( *r == -1 && errno == EINTR );
		if ( *r != 1 ) continue;
		if ( *n == '\n') { if ( *passbufsz == 0 ) { passbuf = (unsigned char*) smalloc(1); memset(passbuf, 0x00, 1); } break; }
		*passbufsz+=1;
		if ( !passbuf )
			passbuf = (unsigned char*) smalloc(1);
		else
			passbuf =  (unsigned char*) srealloc(passbuf, *passbufsz);		
		passbuf[*passbufsz-1] = *n;
	} while (1);
	memset(r, 0xff, sizeof(int));
	memset(r, 0xaa, sizeof(int));
	memset(r, 0x55, sizeof(int));
	sfree(r);
	memset(n, 0xff, sizeof(unsigned char));
	memset(n, 0xaa, sizeof(unsigned char));
	memset(n, 0x55, sizeof(unsigned char));
	sfree(n);
}


#include <termios.h>
int ttyfd;
struct termios initialrsettings;
int _getpassword(char* prompt)
{
	struct termios newrsettings;
	ttyfd=open("/dev/tty",O_RDONLY);
		if ( ttyfd == -1 ) {
			merror("could not open /dev/tty for reading: %s", strerror(errno));
			return -1;
		}
	tcgetattr(ttyfd, &initialrsettings);
	newrsettings = initialrsettings;
	newrsettings.c_lflag &= ~ECHO;
	if(tcsetattr(ttyfd, TCSAFLUSH, &newrsettings) != 0) {
		merror("Could not set /dev/tty attributes");
		return -1;
	} else {
		fprintf(stderr, "%s", prompt);
		_securereadpassbuf(ttyfd);
		tcsetattr(ttyfd, TCSANOW, &initialrsettings);
	}
	close(ttyfd);
	return 0;
}

int getpassword(int cmode, char *prompt)
{
	struct termios newrsettings;
	int r=0;
	unsigned char *tbuf = NULL;
	unsigned long *tbsz = NULL;

	r = _getpassword(prompt);
		if ( r == -1 ) {
		memset(passbuf, 0xff, *passbufsz);
		memset(passbuf, 0xaa, *passbufsz);
		memset(passbuf, 0x55, *passbufsz);
		memset(passbufsz, 0xff, sizeof(unsigned long));
		memset(passbufsz, 0xaa, sizeof(unsigned long));
		memset(passbufsz, 0x55, sizeof(unsigned long));
		return -1;
		}

	if ( cmode == CIPHER_MODE_ENCRYPT ) {
		tbsz = (unsigned long*) smalloc(sizeof(unsigned long));
		*tbsz = *passbufsz;
		tbuf = (unsigned char*) smalloc(*passbufsz);
		memset(tbuf, 0x00, *tbsz);
		memcpy(tbuf, passbuf, *passbufsz);
		fprintf(stderr, "\r");
		fprintf(stderr,"Reenter ");
		r = _getpassword(prompt);
			if (r == -1) {
				goto error_;
			}
		if ( (*tbsz != *passbufsz) || (memcmp(passbuf, tbuf, *tbsz)) ) {
			r = -1;
			fputc('\n', stderr); merror("Passphrases do not match");
			goto error_;
		}
	}
	
	if ( *passbufsz < maxkeysize ) {
		passbuf = (unsigned char*) srealloc(passbuf, maxkeysize);
		memset(passbuf+*passbufsz, 0x00, maxkeysize-*passbufsz);
	}

	goto return_;

	error_:
		_secureclearpassbuf();

	return_:
if(tbuf) {
		memset(tbuf, 0xff, *tbsz);
		memset(tbuf, 0xaa, *tbsz);
		memset(tbuf, 0x55, *tbsz);
		memset(tbuf, 0xff, *tbsz);
		memset(tbuf, 0xaa, *tbsz);
		memset(tbuf, 0x55, *tbsz);
		memset(tbuf, 0xff, *tbsz);
		memset(tbuf, 0xaa, *tbsz);
		memset(tbuf, 0x55, *tbsz);
		memset(tbsz, 0xff, sizeof(unsigned long));
		memset(tbsz, 0xaa, sizeof(unsigned long));
		memset(tbsz, 0x55, sizeof(unsigned long));
		memset(tbsz, 0xff, sizeof(unsigned long));
		memset(tbsz, 0xaa, sizeof(unsigned long));
		memset(tbsz, 0x55, sizeof(unsigned long));
		memset(tbsz, 0xff, sizeof(unsigned long));
		memset(tbsz, 0xaa, sizeof(unsigned long));
		memset(tbsz, 0x55, sizeof(unsigned long));
		sfree(tbuf);
		sfree(tbsz);
}

{fprintf(stderr, "\n");}
	return r;
}

int write_fd_to_stdout(int fd_in)
{
	long bytes;
	unsigned int r, n;
	struct stat statbuf;
	unsigned char *buffer;
	int fd_out = fileno(stdout);

	buffer = (unsigned char *) smalloc(4096);
	if ( !buffer ) {
		merror("could not allocate 4096 bytes from a secure heap");
		return -1;
	}

	fstat(fd_in, &statbuf);

	bytes = statbuf.st_size;

	while ( bytes > 0 ) {

	do {
		r = read( fd_in, buffer, 4096);
	} while ( r == -1 && errno == EINTR );
	if (r < 0) {
		merror("read from stdin support file failed: %s", strerror(errno));
		return -1;
	}

	bytes = bytes - r;

	do {
		n = write(fd_out, buffer, r);
	} while ( n == -1 && errno == EINTR );
	if (n < 0) {
		merror("write to stdout failed: %s", strerror(errno));
		return -1;
	}

	} /* while */

	sfree(buffer);
	return 0;
}
char *make_myrecoverdir()
{
	static char *home=NULL;
	static char dir[255] = {0};
	if ( !home ) {
		home = getenv("HOME");
		strcpy(dir, home);
		strcat(dir, "/.cpdu/.recoverdb");
		mkdir(dir, S_IRUSR|S_IWUSR|S_IXUSR);
		chmod(dir, S_IRUSR|S_IWUSR|S_IXUSR);
	}
	return dir;
}

char *rm_relative_path(char* name)
{
	char *s, *p;
	size_t n;
	struct stat sb;
	stat(name, &sb);
	if ( S_ISDIR(sb.st_mode) ) return name;
	s = strrchr(name, '/');
	if ( !s ) return name;
	n = strlen(s+1);
	p = (char *) malloc(n+1);
	strncpy(p, s+1, n);
	p[n] = '\0';
	return p;
}

char *mk_recoverdb_entry(char *pentry)
{
	char* nentry;
	char* entry;
	entry = rm_relative_path(pentry);
	nentry = (char*) malloc(strlen(make_myrecoverdir())+strlen(entry)+2);
	strncpy(nentry, make_myrecoverdir(), strlen(make_myrecoverdir()));
	strncpy(nentry+strlen(make_myrecoverdir()), "/", 1);
	strncpy(nentry+strlen(make_myrecoverdir())+1, entry, strlen(entry));
	nentry[strlen(make_myrecoverdir())+strlen(entry)+1] = '\0';
	{
		struct stat sb;
		char *dup, *o=NULL;
		int chgn=0;
		retry:
			chmod(nentry, S_IWUSR|S_IRUSR);
			if ( stat(nentry, &sb) == 0 ) {
				if ( chgn == 0 ) o = strdup(nentry);
				chgn++;
				dup = chgn ? o : strdup(nentry);
				nentry = (char*) realloc(nentry, strlen(nentry)+4);
				snprintf(nentry, strlen(nentry)+4,"%s(%d)\0", dup, chgn);
				if( o != dup ) { free(dup); dup = NULL; }
				goto retry;
			}
		if ( o ) free(o);
	}

	return nentry;
}

int copy_file(char *file, char* new)
{
	long bytes;
	unsigned int r, n;
	struct stat sb;
	unsigned char *buffer;
	int fd_in, fd_out;
	fd_in = open(file, O_RDONLY, S_IRUSR);
	fd_out = open(new, O_CREAT|O_WRONLY, S_IWUSR);
	if ( fd_in == -1 || fd_out == -1 ) {
		fd_in = fd_in > 0 ? close(fd_in) : 0;
		fd_out = fd_out > 0 ? close(fd_out) : 0;
		merror("could not open file %s for copying to %s: %s\n", file, new, strerror(errno));
		return -1;
	}

	buffer = (unsigned char *) smalloc(4096);
	if ( !buffer ) {
		merror("could not allocate 4096 bytes from a secure heap");
		close(fd_in);
		close(fd_out);
		return -1;
	}

	fstat(fd_in, &sb);

	bytes = sb.st_size;

	while ( bytes > 0 ) {

	do {
		r = read(fd_in, buffer, 4096);
	} while ( r == -1 && errno == EINTR );
	if (r < 0) {
		merror("read from stdin failed: %s", strerror(errno));
		return -1;
	}

	bytes = bytes - r;

	do {
		n = write(fd_out, buffer, r);
	} while ( n == -1 && errno == EINTR );
	if (n < 0) {
		merror("write to stdin support file failed: %s", strerror(errno));
		return -1;
	}

	} /* while */

	sfree(buffer);
	close(fd_in);
	close(fd_out);
	chmod_secure_file(new);
	return 0;
}

int read_stdin_to_fd(int fd_out)
{
	long bytes;
	unsigned int r, n;
	struct stat statbuf;
	unsigned char *buffer;
	int fd_in = fileno(stdin);

	buffer = (unsigned char *) smalloc(4096);
	if ( !buffer ) {
		merror("could not allocate 4096 bytes from a secure heap");
		return -1;
	}

	fstat(fd_in, &statbuf);

	bytes = statbuf.st_size;

	while ( bytes > 0 ) {

	do {
		r = read(fd_in, buffer, 4096);
	} while ( r == -1 && errno == EINTR );
	if (r < 0) {
		merror("read from stdin failed: %s", strerror(errno));
		return -1;
	}

	bytes = bytes - r;

	do {
		n = write(fd_out, buffer, r);
	} while ( n == -1 && errno == EINTR );
	if (n < 0) {
		merror("write to stdin support file failed: %s", strerror(errno));
		return -1;
	}

	} /* while */

	sfree(buffer);
	return 0;
}

char *make_mydir()
{
	static char *home=NULL;
	static char dir[255] = {0};
	if ( !home ) {
		home = getenv("HOME");
		strcpy(dir, home);
		strcat(dir, "/.cpdu");
		mkdir(dir, S_IRUSR|S_IWUSR|S_IXUSR);
	}
	return dir;
}

char *make_mykeydir()
{
	static char *home=NULL;
	static char dir[255] = {0};
	if ( !home ) {
		home = &dir[0];
		strcpy(dir, make_mydir());
		strcat(dir, "/.keystore");
		mkdir(dir, S_IRUSR|S_IWUSR|S_IXUSR);
	}
	return dir;
}

void chmod_unsecure_mykeydir()
{ 
	if ( -1 == chmod(make_mykeydir(), S_IRUSR|S_IWUSR|S_IXUSR) )
		merror("%s: could not change permissions", make_mykeydir());
}

void chmod_secure_mykeydir()
{
	if ( -1 == chmod(make_mykeydir(), 0) )
		merror("could not change the permissions of the local key directory");
}

void chmod_secure_file(char *file)
{
	chmod(file, 0);
}

void chmod_unsecure_file(char *file)
{
	if ( chmod(file, S_IRUSR|S_IWUSR) == -1 )
		merror("could not change read and write permissions on file %s: %s", file, strerror(errno));
}

void _secureclearpassbuf()
{
if ( passbuf ) {
memset(passbuf, 0xff, passbufsz ? *passbufsz : 0);
memset(passbuf, 0xaa, passbufsz ? *passbufsz : 0);
memset(passbuf, 0x55, passbufsz ? *passbufsz : 0);
memset(passbuf, 0xff, passbufsz ? *passbufsz : 0);
memset(passbuf, 0xaa, passbufsz ? *passbufsz : 0);
memset(passbuf, 0x55, passbufsz ? *passbufsz : 0);
memset(passbuf, 0xff, passbufsz ? *passbufsz : 0);
memset(passbuf, 0xaa, passbufsz ? *passbufsz : 0);
memset(passbuf, 0x55, passbufsz ? *passbufsz : 0);
}
if ( passbufsz ) {
memset(passbufsz, 0xff, sizeof(unsigned long));
memset(passbufsz, 0xaa, sizeof(unsigned long));
memset(passbufsz, 0x55, sizeof(unsigned long));
memset(passbufsz, 0xff, sizeof(unsigned long));
memset(passbufsz, 0xaa, sizeof(unsigned long));
memset(passbufsz, 0x55, sizeof(unsigned long));
memset(passbufsz, 0xff, sizeof(unsigned long));
memset(passbufsz, 0xaa, sizeof(unsigned long));
memset(passbufsz, 0x55, sizeof(unsigned long));
}
sfree(passbuf);
sfree(passbufsz);
passbuf = NULL;
passbufsz = NULL;
}

#if 0
char* getfile_from_recoverylist(char *digest)
{
	static char dgsv[SHA1_DIGESTSIZE];
	static char keyfile[200];
	static int isopen = 0;
	static FILE* f;
	int r;
	struct stat sb;

	if ( !isopen ) {
	strcpy(keyfile, make_mykeydir());
	strcat(keyfile, "/.masterkeylist");
	chmod(make_mykeydir(), S_IRUSR|S_IWUSR|S_IXUSR);
	r = stat(keyfile, &sb);
	if ( r == -1 ) {
		merror("could not stat master key list: %s\n", strerror(errno));
		chmod_secure_file(make_mykeydir());
		return -1;
	}
	chmod_unsecure_file(keyfile);
	f = fopen(keyfile, "r");
	}

	if ( isopen ) {
	int ret; ret = memcmp(dgsv, digest, sizeof(dgsv));
	if ( ret == 0 )
		return 0;
	}

	r = _seek_key(f, digest);
	if ( r != 0 ) {
		merror("did not find key in master key list\n");
		return 1;
	}

	passbufsz = (unsigned long*) smalloc(sizeof(unsigned long));
	fread((unsigned char*)passbufsz, 1, sizeof(unsigned long), f);
	passbuf = (unsigned char*) smalloc(*passbufsz);
	fread(passbuf, 1, *passbufsz, f);
	memcpy(dgsv, digest, sizeof(dgsv));
        if ( *passbufsz < maxkeysize ) {
                passbuf = (unsigned char*) srealloc(passbuf, maxkeysize);
                memset(passbuf+*passbufsz, 0x00, maxkeysize-*passbufsz);
        }
	chmod_secure_file(keyfile);
	chmod_secure_file(make_mykeydir());
	isopen = 1;
	return 0;
}

int _seek_recover_list(FILE *f, char *digest)
{
	off_t r;
	unsigned long *sz;
	char buffer[SHA1_DIGESTSIZE];
	sz = (unsigned long*) smalloc(sizeof(unsigned long));
	fseek(f, 0, SEEK_SET);
	do {
	fread(buffer, 1, SHA1_DIGESTSIZE, f);
	if ( feof(f) ){ sfree(sz); return 1; }
	if ( 0 == memcmp(buffer, digest, sizeof(buffer)) )
	{ sfree(sz); return 0; }
	else {
		fread((unsigned char*)sz, 1, sizeof(unsigned long), f);
		{ unsigned char* buf = smalloc(*sz);	
		fread(buf, 1, *sz, f);
		sfree(buf);
		}
	}
	} while (1);
}

int addfile_to_recoverylist()
{
	WORD8 *dg;
	char file[200];
	strcpy(file, make_mydir());
	strcat(file, "/.recoverlist");
	dg = sha1_get_file_hash(curfile);
	FILE *f = fopen(file, "a");
	fclose(f);
	f = fopen(file, "r");
	if ( _seek_recover_list(f, dg) != 0 ) {
	fclose(f);
	f = fopen(keyfile, "a");
	fwrite(dg, 1, SHA1_DIGESTSIZE, f);
	fwrite((unsigned char*)passbufsz, 1, sizeof(unsigned long), f);
	fwrite(passbuf, 1, *passbufsz, f);
	}
	fclose(f);
	chmod_secure_file(keyfile);
	return 0;
}
#endif

int _seek_key(FILE *f, char *digest)
{
	off_t r;
	unsigned long *sz;
	char buffer[SHA1_DIGESTSIZE];
	sz = (unsigned long*) smalloc(sizeof(unsigned long));
	fseek(f, 0, SEEK_SET);
	do {
	r = fread(buffer, 1, SHA1_DIGESTSIZE, f);
	if ( feof(f) ){ sfree(sz); return 1; }
	if ( 0 == memcmp(buffer, digest, sizeof(buffer)) )
	{ sfree(sz); return 0; }
	else {
		r = fread((unsigned char*)sz, 1, sizeof(unsigned long), f);
		{ unsigned char* buf = smalloc(*sz);	
		r = fread(buf, 1, *sz, f);
		sfree(buf);
		}
	}
	} while (1);
}

int getkey_from_masterkeylist(char *digest)
{
	static char dgsv[SHA1_DIGESTSIZE];
	static char keyfile[200];
	static int isopen = 0;
	static FILE* f;
	int r;
	struct stat sb;

	if ( !isopen ) {
	strcpy(keyfile, make_mykeydir());
	strcat(keyfile, "/.masterkeylist");
	chmod(make_mykeydir(), S_IRUSR|S_IWUSR|S_IXUSR);
	r = stat(keyfile, &sb);
	if ( r == -1 ) {
		merror("could not stat master key list: %s\n", strerror(errno));
		chmod_secure_file(make_mykeydir());
		return -1;
	}
	chmod_unsecure_file(keyfile);
	f = fopen(keyfile, "r");
	}

	if ( isopen ) {
	int ret; ret = memcmp(dgsv, digest, sizeof(dgsv));
	if ( ret == 0 )
		return 0;
	}

	r = _seek_key(f, digest);
	if ( r != 0 ) {
		merror("did not find key in master key list\n");
		return 1;
	}

	passbufsz = (unsigned long*) smalloc(sizeof(unsigned long));
	r = fread((unsigned char*)passbufsz, 1, sizeof(unsigned long), f);
	passbuf = (unsigned char*) smalloc(*passbufsz);
	r = fread(passbuf, 1, *passbufsz, f);
	memcpy(dgsv, digest, sizeof(dgsv));
        if ( *passbufsz < maxkeysize ) {
                passbuf = (unsigned char*) srealloc(passbuf, maxkeysize);
                memset(passbuf+*passbufsz, 0x00, maxkeysize-*passbufsz);
        }
	chmod_secure_file(keyfile);
	chmod_secure_file(make_mykeydir());
	isopen = 1;
	return 0;
}

int addkey_to_masterkeylist()
{
	PSHA1CTX shactx;
	WORD8 dg[SHA1_DIGESTSIZE];
	char keyfile[200];
	int r;
	strcpy(keyfile, make_mykeydir());
	strcat(keyfile, "/.masterkeylist");
	chmod(make_mykeydir(), S_IRUSR|S_IWUSR|S_IXUSR);
	chmod_unsecure_file(keyfile);
	shactx = SHA1_Create();
	SHA1_Update(shactx, passbuf, *passbufsz);
	SHA1_Final(dg, shactx);
	FILE *f = fopen(keyfile, "a");
	fclose(f);
	f = fopen(keyfile, "r");
	if ( _seek_key(f, dg) != 0 ) {
	fclose(f);
	f = fopen(keyfile, "a");
	r = fwrite(dg, 1, SHA1_DIGESTSIZE, f);
	r = fwrite((unsigned char*)passbufsz, 1, sizeof(unsigned long), f);
	r = fwrite(passbuf, 1, *passbufsz, f);
	}
	fclose(f);
	chmod_secure_file(keyfile);
	return 0;
}

void secure_local_key_set()
{
	struct stat statbuf;
	char slkeyfile[200];
	strcpy(slkeyfile, make_mykeydir());
	strcat(slkeyfile, "/.keyfile");
	chmod_unsecure_mykeydir();

	if ( stat(slkeyfile, &statbuf) == 0 ) {
		merror("secure local keyfile already exists, terminating...\n");
		close_system();
		opts->destructor(opts);
		exit(0);
	}

	fprintf(stderr,
		"- Secure Local Keyfile Passphrase Entry -\n"
		"Secure Local Key Directory: [%s]\n"
		"Secure Local Key File: [%s]\n",
		make_mykeydir(), slkeyfile
	);

	{ int r=0;
	r = getpassword(CIPHER_MODE_ENCRYPT, "Secure Local Key Phrase:");
	if ( r == -1 ) {
		r = 0;
		merror("could not retrieve password");
		_secureclearpassbuf();
		opts->destructor(opts);
		chmod_secure_mykeydir();
		close_system();
		exit(0);
	}
	}

	{ int fd, r;
		fd = open(slkeyfile, O_CREAT|O_WRONLY, S_IRUSR|S_IWUSR|S_IXUSR);
		r = write(fd, passbuf, *passbufsz);
		close(fd); fd = -1;
	}

	_secureclearpassbuf();
	chmod_secure_file(slkeyfile);
	chmod_secure_mykeydir();
	opts->destructor(opts);
	close_system();
}

void secure_local_key_get()
{ int fd, r; struct stat sb; char slkeyfile[200];
	strcpy(slkeyfile, make_mykeydir());
	strcat(slkeyfile, "/.keyfile\0");
	chmod_unsecure_mykeydir();
	chmod_unsecure_file(slkeyfile);
	if ( stat(slkeyfile, &sb) == -1 ) { merror("secure local key file error: stat error: %s", strerror(errno));
	chmod_secure_file(slkeyfile);
	chmod_secure_mykeydir();
	opts->destructor(opts);
	close_system();
	exit(1);
	}
	passbufsz = (unsigned long*) smalloc(sizeof(unsigned long));
	*passbufsz = sb.st_size;
	passbuf = (unsigned char*) smalloc(*passbufsz);
	fd = open(slkeyfile, O_RDONLY);
	r = read(fd, passbuf, *passbufsz);

	if ( *passbufsz < maxkeysize ) {
		passbuf = (unsigned char*) srealloc(passbuf, maxkeysize);
		memset(passbuf+*passbufsz, 0x00, maxkeysize-*passbufsz);
	}

	close(fd); fd = -1;
	chmod_secure_file(slkeyfile);
	chmod_secure_mykeydir();
}

int cpdu_archive_load(char* larchive_name)
{
	FILE* archive, *file;
	struct stat sb;
	int i, count;
	unsigned char buffer[4096];
	size_t size, r;
	PSHA1CTX shactx;
	WORD8 dg[SHA1_DIGESTSIZE];

	archive_name = larchive_name;

	if ( stat(archive_name, &sb) == 0 ) { merror("%s: archive already exists\n", archive_name); return -1; }
	archive = fopen(archive_name, "wb");
	r = fwrite(cpdu_archive_hdr, 1, sizeof(cpdu_archive_hdr), archive);
	shactx = SHA1_Create();
	SHA1_Update(shactx, passbuf, *passbufsz);
	SHA1_Final(dg, shactx);
	r = fwrite(dg, 1, SHA1_DIGESTSIZE, archive);
	count = entrylist_count - (opts->rmrpaths ? rmrpathsnoe : 0);
	fprintf(stderr, "archive %s: %d files\n", archive_name, count);
	r = fwrite(&count, 1, sizeof(int), archive);

	for(i=0;entrylist[i]; i++) {
		size_t slen;
		static char *s;
		if ( stat(entrylist[i], &sb) == -1 ) continue;
		if ( opts->rmrpath ) {
			static int ce, cn=0;
			if ( cn < rmrpathcount)
			ce = *(rmrpathlist[cn]);
			if ( i == ce ) {
				s = rm_relative_path(entrylist[i]);
				cn++;
			} else { s = entrylist[i]; }
		} else if ( opts->rmrpaths ) {
			s = rm_relative_path(entrylist[i]);
			if ( S_ISDIR(sb.st_mode) ) continue;
		} else { s = entrylist[i]; } /* keep relative paths */
		slen = strlen(s);
		r = fwrite(&slen, 1, sizeof(size_t), archive);
		r = fwrite(s, 1, strlen(s), archive);
	}

	for(i=0;entrylist[i];i++) {
		if ( stat(entrylist[i], &sb) == -1 ) continue;
		if ( S_ISDIR(sb.st_mode) ) { if ( !opts->rmrpaths ) r = fwrite(&(sb.st_mode), 1, sizeof(mode_t), archive); continue; }
		file = fopen(entrylist[i], "rb");
		size = sb.st_size;
		r = fwrite(&(sb.st_mode), 1, sizeof(mode_t), archive);
		r = fwrite(&size, 1, sizeof(size_t), archive);
		fprintf(stderr, " deflating: %s\n", entrylist[i]);
		while( size > 0 ) {
			r = fread(buffer, 1, 1, file);
			r = fwrite(buffer, 1, 1, archive);
			size -= r;
		}
	fclose(file);
	}

	fclose(archive);

return 0;
}

int cpdu_archive_unload(char* larchive_name)
{
	FILE* archive, *file;
	struct stat sb;
	int i, count;
	unsigned char buffer[4096], hdr[sizeof(cpdu_archive_hdr)];
	size_t size, r;
	int n;
	char c;
	char *entry;
	char *curdir;
	mode_t mode;
	char *dirlist;
	int isdir=0;
	PSHA1CTX shactx;
	WORD8 dg[SHA1_DIGESTSIZE], tdg[SHA1_DIGESTSIZE];

	archive_name = larchive_name;

	if ( stat(archive_name, &sb) == -1 ) { merror("%s: %s\n", archive_name, strerror(errno)); return -1; }

	archive = fopen(archive_name, "rb");
	r = fread(hdr, 1, sizeof(cpdu_archive_hdr), archive);
	if ( memcmp(hdr, cpdu_archive_hdr, sizeof(cpdu_archive_hdr)) ) {
		merror("file %s is not cpdu archive\n", archive_name);
		fclose(archive);
		return -1;
	}
	r = fread(tdg, 1, SHA1_DIGESTSIZE, archive);
	if ( opts->keylist ) { int ret;
		ret = getkey_from_masterkeylist(tdg); /* FIXME: currently, we have no internal security mechanism for the master keylist */
		if ( ret != 0 ) {
			merror("could not retrieve key for archive %s. key is not in the current master list. aborting decryption.\n", archive);
			fclose(archive);
			return -1;
		}
	}
	shactx = SHA1_Create();
	SHA1_Update(shactx, passbuf, *passbufsz);
	SHA1_Final(dg, shactx);
	if ( memcmp(tdg, dg, SHA1_DIGESTSIZE) ) {
		merror("password for archive %s is incorrect\n", archive_name);
		fclose(archive);
		return -1;
	}
	r = fread(&count, 1, sizeof(int), archive);
	entrylist = (char**) malloc(sizeof(char*));

	do {
	{
	size_t slen;
	r = fread(&slen, 1, sizeof(size_t), archive);
	entry = (char*) malloc(slen+1);
	r = fread(entry, 1, slen, archive);
	entry[slen] = '\0';
	entrylist = (char**) realloc(entrylist, entrylist_size+=sizeof(char*));
	entrylist[entrylist_count] = entry;
	filecount++;
	entrylist_count++;
	}
	/* we check here to see if any of the archives files already exist, if one does we abort processing the complete archive */
	n = stat(entrylist[entrylist_count-1], &sb);
	if ( n == 0 ) { merror("%s already exists, aborting archive operation\n", entrylist[entrylist_count-1]); return -1; }

	} while(entrylist_count<count);

	fprintf(stderr, "archive %s: %d files\n", archive_name, entrylist_count);

	i=0;
	do {
		n = 0;
		r = fread(&mode, 1, sizeof(mode_t), archive);
		if ( S_ISDIR(mode) ) {
			curdir = entrylist[i];
			mkdir(curdir, mode);
			i++;
			continue;
		}
		r = fread(&size, 1, sizeof(size_t), archive);
		fprintf(stderr, " inflating: %s\n", entrylist[i]);
	{
		char *dup, *o=NULL;
		int chgn=0;
		retry:
			if ( stat(entrylist[i], &sb) == 0 ) {
				if ( chgn == 0 ) o = strdup(entrylist[i]);
				chgn++;
				dup = chgn ? o : strdup(entrylist[i]);
				entrylist[i] = (char*) realloc(entrylist[i], strlen(entrylist[i])+4);
				snprintf(entrylist[i], strlen(entrylist[i])+4,"%s(%d)\0", dup, chgn);
				if( o != dup ) { free(dup); dup = NULL; }
				goto retry;
			}
		if ( o ) free(o);
	}
		file = fopen(entrylist[i], "wb");
			if ( !file ) {
				merror("in archive %s: file %s cannot be created: %s\n", archive_name, entrylist[i], strerror(errno));
				merror("aborting unloading archive %s\n", archive_name);
				fclose(archive);
				return -1;
			}
		while( size > 0 ) {
			r = fread(buffer, 1, 1, archive);
			r = fwrite(buffer, 1, 1, file);
			size -= r;
			n+=r;
		}
		fclose(file);
		chmod(entrylist[i], mode);
		i++;
	} while(i < count);

	fclose(archive);	
	return 0;
}

cpdu_file_type cpdu_query_file_type(char *file)
{
	cpdu_file_type ft = cpdu_none;
	size_t n;
	FILE *rd = fopen(file, "r");
	char cpdu_archive_s[sizeof(cpdu_archive_hdr)];
	char cpdu_encrypted_s[sizeof(cpdu_encrypt_hdr)];
	{ unsigned char buffer[4096], n; size_t r; r = fread(buffer, 1, 4096, rd); n = is_base64_encoded(buffer, r);
		if ( n ) { fclose(rd); return cpdu_encrypted; } else { ft = cpdu_none; }
	}
	fseek(rd, 0, SEEK_SET);
	n = fread(cpdu_archive_s, 1, sizeof(cpdu_archive_hdr), rd);
		if ( strncmp(cpdu_archive_s, cpdu_archive_hdr, sizeof(cpdu_archive_s)) ) {
			ft = cpdu_none;
		}
		else {
			ft = cpdu_archive;
			fclose(rd);
			return ft;
		}
	fseek(rd, 0, SEEK_SET);
	n = fread(cpdu_encrypted_s, 1, sizeof(cpdu_encrypt_hdr), rd);
		if ( strncmp(cpdu_encrypted_s, cpdu_encrypt_hdr, sizeof(cpdu_encrypted_s)) )
			ft = cpdu_none;
		else {
			ft = cpdu_encrypted;
			fclose(rd);
			return ft;
		}

	fclose(rd);
	return ft;
}

void cpdu_signal_sigint(int signal)
{
	fprintf(stderr, "\n");
	merror("signal SIGINT recieved. cleaning up and terminating...");

	if ( entrylist ) /* free the file names in the entry list */
	{ int i; for(i=0;i<entrylist_count;i++) if ( entrylist[i] ) free(entrylist[i]); }

	remove(curfileout);
	chmod_secure_mykeydir();
	opts->destructor(opts);
	_secureclearpassbuf();
	freeCipherContext(cipherctx);
	close_system();

	fprintf(stderr, "done.\n");

	exit(1);
}

void status_print_session_done()
{
fprintf(stderr, "%d files %s\n", file_processed_count, ciphermode == CIPHER_MODE_ENCRYPT ? "encrypted" : "decrypted");
}

#include "twofish.h"
int generate_keyfile(char *keyfile, size_t fsize)
{
	int fd=0, i=0, ret=0;
	void *ctx=NULL;
	unsigned char *key=NULL;
	unsigned char *fbuf=NULL;
	unsigned char *initdata=NULL;
	unsigned int rounds=0;
	char prompt_for_entropy[] = "This process may block until enough entropy is gathered to resolve a read\nfrom the random device file. You may move your mouse or type all you like\nuntil enough entropy is gathered to resolve this.";



	if ( fsize < TWOFISH_KEYSIZE ) {
		merror("the keyfile size specified is smaller than the\ntwofish algorithm keysize, it must be larger than 32 bytes");
		return -1;
	}

	ctx = (void*) smalloc(cs_twofish());
		if (!ctx) {
			merror("unable to allocate memory for cipher context");
			return -1;
		}
	memset(ctx, 0x00, cs_twofish());

	key = (unsigned char *) smalloc(TWOFISH_KEYSIZE);
		if ( !key ) {
			merror("unable to allocate memory for keyspace");
			goto _error_;
		}

	initdata = (unsigned char *) smalloc(sizeof(TWOFISH_BLOCKSIZE));
		if ( !key ) {
			merror("unable to allocate memory for cipher initdata");
			goto _error_;
		}

	fd = open(keyfile, O_WRONLY|O_CREAT|O_EXCL, S_IRUSR);
		if ( fd == -1 ) {
			merror("could not create keyfile %s: %s", keyfile,
strerror(errno));
			goto _error_;
		}

	fprintf(stderr, "\n%s\n", prompt_for_entropy);

	gather_random_slow(key, TWOFISH_KEYSIZE);
	gather_random_fast(&rounds, sizeof(int));
	gather_random_fast(initdata, TWOFISH_BLOCKSIZE);
	rounds = rounds % 2;

	Twofish_CreateWorkContext(ctx, key, TWOFISH_KEYSIZE, CIPHER_MODE_ENCRYPT, initdata,
			(Cipher_RandomGenerator*)get_random_bytes, NULL);

	fprintf(stderr, "Generating...");
{	
	char pb[41];
	unsigned long bufsiz=(fsize>4096?4096:fsize);
	long bytes=fsize;
	long r=0;
	int n=0;
	unsigned long long sofar = 0, total = bytes;

	pb[0] = '[';
	pb[39] = ']';
	pb[40] = '\0';

	fbuf = (unsigned char *) smalloc(bufsiz+TWOFISH_BLOCKSIZE);
		if ( !fbuf ) {
			merror("could not allocate 4096 byte buffer");
			return -1;
		}

	while ( bytes > 0 ) {
	
	for(i=0;i<rounds;i++)
		Twofish_EncryptBuffer(ctx,key,key,TWOFISH_KEYSIZE);
	
	gather_random_fast(fbuf, bufsiz);
	Twofish_EncryptBuffer(ctx,fbuf,fbuf,bufsiz);

	if (r >= fsize-bufsiz)
		n = fsize-r;
	else
		n=bufsiz;
	
	do {
		n = write(fd, fbuf, n);
	} while ( n == -1 && errno == EINTR );
	if (n < 0) {
		merror("keydata write to keyfile %s failed: %s", keyfile, strerror(errno));
		return -1;
	}
	r+=n;
	bytes = bytes - n;

	sofar+=n;
	memset(pb+1, ' ', 38);
	memset(pb+1, '#', (int) sofar*38/total);
{ int percent = ((int) sofar*100/total);
	if ( percent > 100 ) percent = 100;
	fprintf(stderr,"\r[%d%c]%s %s Generating...", percent, '%', &pb[0], keyfile);
}
	} /* while */
	memset(fbuf, 0x00, bufsiz);
	sfree(fbuf);
}
	fprintf(stderr, "done.\n");

	goto _success_;

_error_:
	ret = -1;

_success_:
	if (ctx) {
		Twofish_DestroyWorkContext(ctx);
		sfree(ctx);
	}
	if ( initdata ) {
		memset(initdata, 0x00, TWOFISH_BLOCKSIZE);
		memset(initdata, 0x00, TWOFISH_BLOCKSIZE);
		memset(initdata, 0x00, TWOFISH_BLOCKSIZE);
		sfree(initdata);
	}
	if ( key ) {
		memset(key, 0x00, TWOFISH_KEYSIZE);
		memset(key, 0x00, TWOFISH_KEYSIZE);
		memset(key, 0x00, TWOFISH_KEYSIZE);
		sfree(key);
	}

	close(fd);

	return ret;
}

unsigned char *crunch_key(unsigned char *xkey, unsigned long xkeysize, unsigned char *keybuf, unsigned long outsize)
{
	static void *ctx=NULL;
	static unsigned long r=0;
	static unsigned long n=0;
	static char init=0;

	if ( !xkey )
	{ init = 0; return NULL; }

	for(r=0;r<xkeysize;r++){
		if( init ) keybuf[r%outsize] ^= xkey[r];
		else keybuf[r%outsize] = xkey[r];
	}

	init = 1;

	return keybuf;
}

int read_key_from_keyfile(char *keyfile, unsigned char *keybuf, unsigned long keysize)
{
	int fd, i, bytes=0, r=0, n=0;
	unsigned char *buffer=NULL;
	unsigned char *key=NULL;
	struct stat statbuf;

	if ( !buffer )
		buffer = (unsigned char *) smalloc(4096);
	if ( !buffer ) {
		merror("unable to allocate 4096 bytes from secure memory");
		return -1;
	}

	chmod_unsecure_file(keyfile);
	fd = open(keyfile, O_RDONLY);
		if ( fd == -1 ) {
			merror("could not open keyfile %s: %s", keyfile, strerror(errno));
			return -1;
		}

	if ( stat(keyfile, &statbuf) == -1 ) {
		merror("could not stat keyfile %s: %s", keyfile, strerror(errno));
		close(fd);
		return -1;
	}

	bytes = statbuf.st_size;

	memset(keybuf, 0, keysize);

	while ( bytes > 0 ) {
	do {
		n = read(fd, buffer, 4096);
	} while ( n == -1 && errno == EINTR );
	if ( n == -1 ) {
		merror("read from keyfile %s failed: %s", keyfile, strerror(errno));
		return -1;
	}
	key = crunch_key(buffer, n, keybuf, keysize);
		if ( !key ) {
			crunch_key(NULL,0,NULL,0);
			sfree(buffer);
			return -1;
		}
	bytes -= n;
	}

	sfree(buffer);
	crunch_key(NULL,0,NULL,0);
	close(fd);
	chmod_secure_file(keyfile);
	return 0;
}

/* -- end cpdu.c functions -- */

