#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <unistd.h>
#include <getopt.h>
#include "opts.h"
#include "cipherdef.h"
#include "error.h"
#include "var.h"
#include "version.h"


void display_help(char *);
void display_version(void);

opts_t opts = NULL;

void parse_options_destroy(void *pobject)
{
	opts_t object = pobject;
	if (!object)
		return;
	if (object->argv)
		free(object->argv);
	free(object);
}

opts_t parse_options(int argc, char **argv)
{

	struct option long_options[] = {
		{"help", 0, 0, 'h'},
		{"version", 0, 0, 'V'},
		{"verbose", 0, 0, 'v'},
		{"zlib", 0, 0, 'z'},
		{"zlevel", 1, 0, 'l'},
		{"wipe", 1, 0, 'w'},
		{"stdin", 0, 0, 's'},
		{"recurse", 0, 0, 'r'},
		{"lkey", 0, 0, 'k'},
		{"lkeyset", 0, 0, 'S'},
		{"archive", 1, 0, 'a'},
		{"rmpath", 0, 0, 't'},
		{"rmapath", 0, 0, 'T'},
		{"cipher", 1, 0, 'c'},
		{"encrypt", 0, 0, 'e'},
		{"decrypt", 0, 0, 'd'},
		{"secure", 0, 0, 'f'},
		{"masterkey", 0, 0, 'm'},
		{"genkeyfile", 1, 0, 'g'},
		{"keyfile", 1, 0, 'K'},
		{"gensize", 1, 0, 'Z'},
		{0, 0, 0, 0}
	};
	int option_index = 0;
	char *short_options = "h?Vedc:rsw:zl:kSa:tTivCfmg:K:Z:";
	int c, numopts;
	opts_t options;

	if ( argc <= 1 ) {
		display_help(argv[0]);
		exit(0);
	}

	options = calloc(1, sizeof(*options));
	if (!options) {
		fprintf(stderr, "%s: option structure allocation failed (%s)", argv[0], strerror(errno));
		fprintf(stderr, "\n");
		return 0;
	}

	prog = argv[0];
	options->prog = argv[0];

	options->destructor = parse_options_destroy;

	options->argc = 0;
	options->argv = calloc(argc + 1, sizeof(char *));
	if (!options->argv) {
		fprintf(stderr, "%s: option structure argv allocation failed (%s)", argv[0], strerror(errno));
		fprintf(stderr, "\n");
		options->destructor(options);
		return 0;
	}

	numopts = 0;

	/* -- set default values */
	options->ciphermode = CIPHER_MODE_DECRYPT;
	options->cipher = "aes";
	options->zlib_level = 9;
	options->zlib_level_string = "9";
	options->log = 1;
	options->registerrdb = 1;
	options->progress = 1;
	options->wipe = 1;
	options->wtimes = 35;

	do {
		c = getopt_long(argc, argv,
				short_options, long_options,
				&option_index);

		if (c < 0)
			continue;

		switch (c) {
		case '?':
		goto DisplayHelp;
		break;
		case 'h':
		DisplayHelp:
			display_help(argv[0]);
			exit(0);
		break;
		case 'V':
			display_version();
			exit(0);
		break;
		case 'v':
//			options->verbose = atoi(optarg);
			options->verbose = 1;
		break;
		case 'm':
			options->keylist = 1;
		break;
		case 'e':
			options->ciphermode = CIPHER_MODE_ENCRYPT;
		break;
		case 'd':
			options->ciphermode = CIPHER_MODE_DECRYPT;
		break;
		case 'c':
			options->cipher = optarg;
		break;
		case 's':
			options->rstdin = 1;
		break;
		case 'z':
			options->zlib = 1;
		break;
		case 'l':
			if (optarg) options->zlib_level = atoi(optarg);
			if ( options->zlib_level >= 0 && options->zlib_level <= 9 )
			{} else { merror("invalid compression level, valid level is value 0 - 9\n");
						options->destructor(options);
						exit(0);
					}
			options->zlib_level_string = optarg;
		break;
		case 'k':
			options->slkey = 1;
		break;
		case 'N':
			options->log = 0;
		break;
		case 'r':
			options->recurse = 1;
		break;
		case 'S':
			options->slkeyset = 1;
		break;
		case 'a':
			options->doarchive = 1;
			options->archive = optarg;
		break;
		case 't':
			options->rmrpath = 1;
		break;
		case 'T':
			options->rmrpaths = 1;
		break;
		case 'w':
			options->wipe = 1;
			options->wtimes = atoi(optarg);
		break;
		case 'f':
			options->registerrdb = 0;
		break;
		case 'g':
			options->genkeyfile = 1;
			options->keyfile = optarg;
		break;
		case 'K':
			options->usekeyfile = 1;
			options->keyfile = optarg;
		break;
		case 'Z':
			options->gensize = atol(optarg);
			if ( options->gensize > 100000000 ) { merror("too many bytes for gen. data\n"); options->destructor(options); exit(0); }
		break;
		default:
			fprintf(stderr, "%s: Try '-?', '-h' or `--help' for more information.", argv[0]);
			fprintf(stderr, "\n");
			options->destructor(options);
			exit(0);
		break;
		}

	} while (c != -1);

	/* we make sure we have correct programmatic evaluation */
	if ( options->ciphermode == CIPHER_MODE_ENCRYPT ) {
		options->keylist = 0;
	}

	if ( options->genkeyfile && !options->gensize ) {
		merror("did not specify amount of random gen. data with the '-Z' flag\n");
		options->destructor(options);
		exit(0);
	}

	while (optind < argc) {
		options->argv[options->argc++] = argv[optind++];
	}

	return options;
}

#include "version.h"
void display_help(char *arg)
{
	fprintf(stderr,
		"Cryptographic Data Utility v%s\n"
		"Home: %s/.cpdu\n"
		"Usage: %s [options] [files...] [-d, --decrypt [archives]...]"
		"\nOptions:\n"
		" -e,  --encrypt          encrypt mode\n"
		" -d,  --decrypt          decrypt mode, default\n"
		" -r,  --recurse          recursive directory search\n"
		" -c,  --cipher  'cipher' session cipher, default=aes/rijndael\n"
		" -a,  --archive 'file'   load or unload archive\n"
		" -t,  --rmpath           for archive, remove relative paths of single files on command line\n"
		" -T,  --rmapath          for archive, remove all relative paths of files\n"
		" -z,  --zlib             zlib compress\n"
		" -l,  --zlevel  'l'      zlib compression level 0-9, default=9\n"
		" -k,  --lkey             local secure key for session\n"
		" -S,  --setlkey          local secure key set passphrase prompt\n"
		" -K,  --keyfile          file for session key\n"
		" -g,  --genkeyfile       generate a random key-file with bytes from random device\n"
		" -Z,  --gensize          size of generated key-file in bytes\n"
		" -m,  --masterkey        get session decryption key(s) from exportable master key database\n"
		" -f,  --secure           secure mode, disable recovery registry\n"
		" -w,  --wipe    't'      wipe file 't' times /w random data, default /w 35\n"
		" -v,  --verbose          verbose mode\n"
		" -s,  --stdin            stdin, stdout\n"
		" -V,  --version          show version and cipher info\n"
		" -h,  --help             prints help\n",
CPDU_VERSION, getenv("HOME"), arg);
}

void display_version(void)
{
	fprintf(stderr,
			"Cryptographic Data Utility | Cryptographic Package Distribution Utility\n"
			" version %s\n"
			"Zlib\n"
			" version 1.2.3\n"
			"Ciphers in CBC mode:          |Keysize|    |Blocksize|\n"
			" Aes/Rijndael (aes, rijndael)   32 256        16 128     (bytes/bits)\n"
			" Twofish      (tf, twofish)     32 256        16 128\n"
			" Blowfish     (bf, blowfish)    56 448         8 64\n"
			" Serpent      (spnt, serpent)   32 256        16 128\n"
			" Cast         (cast)            16 128         8 64\n"
			" Tripledes    (3des, des, tdes) 21 168         8 64\n"
			"Author: Richard Enciu - richardenciu@gmail.com\n",
			CPDU_VERSION);
}
