/* All times are in microseconds.
 */

#include <errno.h>
#include <netdb.h>
#include <stdlib.h>
#include <netdb.h>
#include <fcntl.h>
#include <stdio.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <sys/time.h>
#include <unistd.h>
#include <string.h>

#include "def.h"

struct qpkt
{
 u32 seq;
 unsigned char *ptr; /* Pointer to the block of info's, but without seq at the
                      * Beginning.
                      */
 u32 len; /* Number of unsigned char's in the *ptr */
 unsigned long long time; /* When it will be necessary to send it again. */
 struct qpkt *next;
};

u16 port=1235; /* port to sit on */
struct in_addr ip; /* ip to listen to -- in network order*/
u16 distant_port; /* Port on which the client sits */
unsigned char *filename="bcp.dat"; /* Name of file to dump the data to */
int tickers=0; /* 0= full tickers 1=No tickers 2=print nothing */
int fd; /* Socket */
int output_fd; /* Output file descriptor */
u64 bps; /* Bytes per second */
struct sockaddr_in distant_sai;
int distant_sai_len;
unsigned char pkt[65536]; /* In first 4 bytes of the packet there is permanently the session_id. */
unsigned char opkt[65536]; /* In first 4 bytes of the packet there is permanently the session_id. */
int pkt_len;
unsigned long payload_len, file_len;
unsigned long n_packets;
unsigned long remaining_packets;
unsigned number_of_eots=3;
unsigned char *bitmap;
unsigned info_cache_size; /* Length of info_cache in entries */
unsigned long *info_cache; /* info cache itself. Has 4* info_cache_size bytes.
                            */
unsigned long info_cache_ctr; /* Number of used entries in info_cache */
unsigned long long info_cache_time; /* When the cache must be flushed */
unsigned long long info_cache_timeout=10000000;
unsigned long long retransmit_timeout=2800000;
unsigned long long eot_period=100000;
unsigned long info_seq; /* info packet identifier */
struct qpkt *que;
int ignore_send_errors;
unsigned info_packets_sent;
unsigned qpkt_ctr;
unsigned info_acks_received;
u32 bytes_received;
u64 first_data_time;
unsigned superfluous_packets;
unsigned superfluous_bytes;

void
throw_out_socket(void)
{
 if (fd) 
 {
  close(fd);
 }
}

void die()
{
 printf("Exiting on fatal error.\n");
 throw_out_socket();
 exit(1);
}

/* Returns time since 1970 in microseconds */
unsigned long long get_time(void)
{
 struct timeval tv;
 struct timezone tz;

 bzero (&tz,sizeof(tz));
 if (gettimeofday(&tv,&tz))
 {
  fprintf(stderr,"Can't getttimeofday.\n");
  perror("");
  die();
 }
 return ((unsigned long long)tv.tv_sec)*1000000+((unsigned long long)tv.tv_usec);
}

void
print_help(void)
{
 fprintf(stderr,"bcps -- a Brutalcopy server half
Brutalcopy (c) 2000 Karel Kulhavy, Clocksoft

Brutalcopy is a program for transferring rather large files by means of UDP
protocol in order to bypass TCP tendency to screw up on bad lines. Brutalcopy
is based on manual flow control.

Usage: bcps -p <number_of_port_to_listen on> [-h] [-o
<filename>] [-s] [-e <number_of_eots>] [-r <eot_period>] [-1] [-2]
[-q <info_retransmit_timeout>]

-p <port>      Port must be some high number. Default is 1235.
-h             Prints out help and ends.
-o <filename>  File to dump the received data to. Default is bcp.dat.
-s             Print tickers
-e <n_eots>    Number of EOT packets sent at the end. Default is 3.
-r <eot_p>     Time distance between consecutive EOT packets in microseconds.
               Default is 100,000.
-q             Info packet retransmit timeout in usec.
-1             Do not print tickers
-2             Print nothing
");
}

void
get_socket(void)
{
 fd=socket(PF_INET,SOCK_DGRAM,IPPROTO_UDP);
 if (fd<0)
 {
  fprintf(stderr,"Can't open socket.\n");
  die();
 }
 {
  struct sockaddr_in sai;

  sai.sin_family=AF_INET;
  sai.sin_port=port;
  sai.sin_addr.s_addr=INADDR_ANY;
  if (bind(fd,(struct sockaddr *)&sai,sizeof(sai)))
  {
   fprintf(stderr,"Can't bind socket to port %d.\n",port);
   perror("");
   die ();
  }
  fcntl(fd,F_SETFL,O_NONBLOCK);
 }
}

/* legitimate args
h print help
l string dns name of computer to listen to
p port sit on port
*/

void
parse_args(int argc, char **argv)
{
 int res;

 while(-1!=(res=getopt(argc, argv, "12shp:o:e:r:")))
 {
  switch(res)
  {
   case '?':
   fprintf(stderr,"Invalid option char.\n");
   case 'h':
   print_help();
   exit(0);
   case '1':
   tickers=1;
   break;
   case '2':
   tickers=2;
   break;
   case 'p':
   port=strtoul(optarg,NULL,0);
   break;
   case 'o':
   filename=optarg;
   break;
   case 's':
   tickers=1;
   break;
   case 'q':
   retransmit_timeout=atol(optarg);
   break;
   case 'e':
   number_of_eots=strtoul(optarg,NULL,0);
   break;
   case 'r':
   eot_period=strtoul(optarg,NULL,0);
   break;
  }
 }
}

/* Takes pkt, pkt_len, computes checksum of pkt_len bytes starting at pkt
 * and compares with 4 bytes starting at pkt+pkt_len.
 * retval: 0 checksum OK, 1 checksum bad.
 */
int
test_checksum(void)
{
 unsigned long checksum; 
 unsigned long packet_checksum;

 checksum=crc32(pkt,pkt_len);
 packet_checksum=pkt[pkt_len]+(pkt[pkt_len+1]<<8)+(pkt[pkt_len+2]<<16)+(pkt[pkt_len+3]<<24);
 return checksum!=packet_checksum;
}

/* Access: 0 read access, 1 write access
 * If necessary waits to infinity
 */
void
wait_for_rw(int fd, int access)
{
 /* Let's sleep until something comes from the fd
  */
 fd_set set;
 fd_set zero_set;

 FD_ZERO(&set);
 FD_ZERO(&zero_set);
 FD_SET(fd,&set);
 if (access)
  select(fd+1,&zero_set,&set,&zero_set,NULL);
 else
  select(fd+1,&set,&zero_set,&zero_set,NULL);
}

/* Accepts every IP and port.
 * Resulting pkt_len is UDP length without 4 (CRC32).
 * The resulting packet is placed to pkt[4].
 * If no packets are available, waits.
 */
void
get_packet_no_test()
{
 int retval;

 again:
 pkt_len=0;
 distant_sai_len=sizeof(distant_sai);
 wait_for_rw(fd,0);
 retval=recvfrom(fd,(void *)pkt+4,sizeof(pkt)-4,MSG_WAITALL
  ,(struct sockaddr*) &distant_sai, &distant_sai_len);
 if (retval>0)
 {
  ip=distant_sai.sin_addr;
  distant_port=distant_sai.sin_port;
  pkt_len=retval;
  if (pkt_len<5) goto again; 
  memcpy(pkt,pkt+5,4);
  memcpy(opkt,pkt,4);
  if (test_checksum()) goto again;
  pkt_len-=4;
  return;
 }
 if (retval==0||errno==EWOULDBLOCK||errno==EINTR) goto again;
 fprintf(stderr,"Can't read from UDP socket.\n");
 perror("");
 die();
}

/* If packet is from unwanted ip it throws it out and tries again.
 * Result is placed on pkt[4], pkt_len is set to length of received UDP packet
 * without the trailing CRC32
 * If no packet is available, immediately returns and set pkt_len to 0.
 */
void
get_packet()
{
 int retval;

 again:
 pkt_len=0;
 distant_sai_len=sizeof(distant_sai);
 retval=recvfrom(fd,(void *)pkt+4,sizeof(pkt)-4,MSG_WAITALL
  ,(struct sockaddr*)&distant_sai, &distant_sai_len);
 if (retval>0)
 {
  if (distant_port!=distant_sai.sin_port) goto again;
  pkt_len=retval;
  if (pkt_len<4) goto again;
  if (test_checksum()) goto again;
  pkt_len-=4;
  return;
 }
 if (retval==0) goto again;
 if (errno==EAGAIN) return;
 if (errno==EINTR) goto again;
 fprintf(stderr,"Can't read from UDP socket.\n");
 perror("");
 die();
}

/* Packet must be at opkt[4]. Len long. CRC32 is added to the end and in opkt[0] there must
 * be session_id.
 */
void
send_packet(unsigned len)
{
 unsigned long crc;
 
 if (len+8>sizeof(opkt))
 {
  fprintf(stderr,"Internal error: output packet too long.\n");
  die();
 }
 crc=crc32(opkt,len+4);
 opkt[4+len]=crc&255;crc>>=8;
 opkt[5+len]=crc&255;crc>>=8;
 opkt[6+len]=crc&255;crc>>=8;
 opkt[7+len]=crc&255;
 len+=4; /* Len is length of UDP packet now */
 {
  struct sockaddr_in sai;
  int retval;
  
  sai.sin_family=AF_INET;
  sai.sin_port=distant_port;
  sai.sin_addr=ip;
  again:
  retval=sendto(fd,opkt+4,len,0,(struct sockaddr*)&sai,sizeof(sai));
  if (retval==len) return;
  if (retval==0) goto again;
  if (retval>0&&retval!=len)
  {
   fprintf(stderr,
    "sendto should have sent %d bytes but sent %d bytes.\n",len,retval);
   die();
  }
  if (retval<0)
  {
   if (errno==EAGAIN||errno==ENOBUFS||errno==EINTR||errno==ENOMEM) goto again;
  }
  if (ignore_send_errors) return;
  fprintf(stderr,"Can't send %d-byte packet.\n",len);
  perror("");
  die();
 } 
}

void
send_eot(void)
{
 opkt[4]=3;
 send_packet(1);
}

void
send_sot_ack(void)
{
 opkt[4]=4;
 send_packet(1);
}

/* Like malloc but never fails, on 0 returns NULL and the resulting memory is
 * always zeroed up.
 */
void *
xcalloc(size_t len)
{
 void *r;
 
 if (!len) return 0;
 r=calloc(len,1);
 if (!r)
 {
  fprintf(stderr,"Can't calloc %d bytes.\n",len);
  die();
 }
 return r;
}

/* If bitmap is set from 0 to 1, remaining_packets is deced */
void
set_bitmap(unsigned n)
{
 unsigned bit=n&7;
 unsigned byte=n>>3;

 if (n>=n_packets)
 {
  fprintf(stderr,"Invalid packet seq# came! It means data are corrupted. Quitting immediately.\n");
  die();
 } 
 bitmap[byte]|=1<<bit;
}

/* Returns value of the bit. */
int
test_bitmap(unsigned n)
{
 unsigned bit=n&7;
 unsigned byte=n>>3;

 return (bitmap[byte]>>bit)&1;
}

/* Writes LSB first 32 bits u32
 */
void
write32(unsigned char *dest, u32 val)
{
 dest[0]=val;val>>=8;
 dest[1]=val;val>>=8;
 dest[2]=val;val>>=8;
 dest[3]=val;
}

/* Writes 64 bits LSB first u64
 */
void
write64(unsigned char *dest, u64 val)
{
 dest[0]=val;val>>=8;
 dest[1]=val;val>>=8;
 dest[2]=val;val>>=8;
 dest[3]=val;val>>=8;
 dest[4]=val;val>>=8;
 dest[5]=val;val>>=8;
 dest[6]=val;val>>=8;
 dest[7]=val;
}

/* Reads in LSB first */
u32
read32(unsigned char *src)
{
 u32 rv;

 rv=src[3];
 rv<<=8;
 rv|=src[2];
 rv<<=8;
 rv|=src[1];
 rv<<=8;
 rv|=src[0];
 return rv;
}

/* Reads in LSB first */
u64
read64(unsigned char *src)
{
 u64 rv;
 
 rv=src[7];
 rv<<=8; 
 rv|=src[6];
 rv<<=8;
 rv|=src[5];
 rv<<=8;
 rv|=src[4];
 rv<<=8;
 rv|=src[3];
 rv<<=8;
 rv|=src[2];
 rv<<=8;
 rv|=src[1];
 rv<<=8;
 rv|=src[0];
 return rv;
}

#define flush fflush(stdout);

void
cls(void)
{
 printf("\033[2J");
 flush
}

/* top-left is 0,0, x is right, y is down. */
void
g0t0(int x, int y)
{
 printf("\033[%d;%dH",y+1,x+1);
 flush
}

void *
xalloc(size_t len)
{
 void *r;
 
 if (!len) return 0;
 r=malloc(len);
 if (!r)
 {
  fprintf(stderr,"Can't malloc %d bytes.\n",len);
  die();
 }
 return r;
}

/* Returns 0 if SOT ok, otherwise 1. */
int
test_sot(void)
{
 
 if (pkt_len!=25) return 1;
 if (pkt[4]!=0) return 1;
 bps=read64(pkt+9);
 cls();
 if (tickers<2)
 {
  g0t0(0,8);
  printf("Output to %s",filename);
  g0t0(40,0);
  printf("%lld bytes per second.",bps);
 }
 payload_len=read32(pkt+17);
 if (tickers<2)
 {
  g0t0(0,0);
  printf("Payload length %ld bytes.",payload_len);
  flush
 }
 if (!payload_len) return 1;
 file_len=read32(pkt+21);
 if (tickers<2)
 {
  g0t0(0,1);
  printf("File length %ld bytes.",file_len);
  flush
 }
 n_packets=(file_len+payload_len-1)/payload_len;
 if (tickers<2)
 {
  g0t0(0,2);
  printf("Total packets %ld.",n_packets);
  flush
 }
 remaining_packets=n_packets;
 if (!tickers)
 {
  g0t0(0,3);
  printf("Remaining packets %ld.",remaining_packets);
  flush
 }
 bitmap=xcalloc((n_packets+7)>>3);
 info_cache_size=read32(pkt+25);
 if (tickers<2)
 {
  g0t0(0,4);
  printf("Info cache size %d entries.",info_cache_size);
  flush
 }
 if (!info_cache_size) return 1;
 if ((info_cache_size<<2)+12>sizeof(opkt)) return 1;
 info_cache=xalloc(info_cache_size*sizeof(*info_cache));
 send_sot_ack();
 return 0;
}

void
open_session(void)
{
 get_socket();
 again:
 get_packet_no_test();
 if (test_sot()) goto again;
 output_fd=open(filename,O_CREAT|O_TRUNC|O_WRONLY,0644);
 if (output_fd<0)
 {
  fprintf(stderr,"Can't open output file %s.\n",filename);
  perror("");
  die();
 }
}

void
close_session(void)
{
 throw_out_socket();
 close(output_fd);
}

/* Builds packet from this and sends it. */
void
send_qpkt(struct qpkt *p)
{
 opkt[4]=2; /* Info packet */
 write32(opkt+5,p->seq);
 memcpy(opkt+9,p->ptr,p->len);
 send_packet(p->len+5);
 p->time=get_time()+retransmit_timeout;
 info_packets_sent++;
 if (!tickers)
 {
  g0t0(40,1);
  printf("Info packets sent %u.",info_packets_sent);
  flush
 }
}

void
print_qpkt_ctr(void)
{
 if (tickers) return;
 g0t0(40,3);
 printf("Info pkts waiting for ack %d.    ",qpkt_ctr);
 flush
}

void
insert_qpkt(struct qpkt *w)
{
 w->next=que;
 w->seq=info_seq++;
 que=w;
 qpkt_ctr++;
 print_qpkt_ctr();
}

void
print_info_cache_ctr(void)
{
 if (tickers) return;
 g0t0(40,2);
 printf("Info cache %ld entries.            ",info_cache_ctr);
 flush
}

/* Encodes the info cache and ques it into the packet que
 * Sets info_cache_ctr to zero.
 * Does nothing if info_cache is empty.
 */
void
flush_info_cache(void)
{
 struct qpkt *a;
 unsigned char *b;
 int c;
 
 if (!info_cache_ctr) return;
 a=xcalloc(sizeof(*a));
 b=xalloc(info_cache_ctr<<2);
 for (c=0;c<info_cache_ctr;c++)
  write32(b+(c<<2),info_cache[c]);
 a->ptr=b;
 a->len=info_cache_ctr<<2;
 insert_qpkt(a);
 send_qpkt(a);
 info_cache_ctr=0;
 print_info_cache_ctr();
}

/* When cache is full, it's immediately flushed by cache_info.
 * cache_info caches an info onto the info cache.
 */
void
cache_info(unsigned seq)
{
 if (!info_cache_ctr) info_cache_time=get_time()+info_cache_timeout;
 info_cache[info_cache_ctr]=seq;
 info_cache_ctr++;
 if (info_cache_ctr==info_cache_size) flush_info_cache();
 print_info_cache_ctr();
}

/* Returns 1 if the seq is hanging in the info cache, 0 otherwise */
int
in_info_cache(seq)
{
 int a;
 for (a=0;a<info_cache_ctr;a++)
  if (info_cache[a]==seq) return 1;
 return 0;
}

/* Takes the data packet at pkt[5], pkt_len long. */
void
process_data_packet(void)
{
 unsigned long seq;
 ssize_t retval;
 int inc=9;
 unsigned data_len;
 
 if (pkt_len<5) return;
 seq=read32(pkt+5);
 pkt_len-=4;
 if (seq>=n_packets) return; /* Nonsense */
 {
  unsigned long should;
  should=payload_len;
  if(seq==n_packets-1)
  {
   should=file_len%payload_len;
   if (!should) should=payload_len;
  } 
  if (pkt_len!=should) return; /* Nonsense */
 } 
 if (test_bitmap(seq)) 
 {
  if (in_info_cache(seq)) flush_info_cache();
  superfluous_packets++;
  superfluous_bytes+=pkt_len;
  if (!tickers)
  {
   g0t0(40,7);
   printf("Superfluous packets %u.",superfluous_packets);
   g0t0(40,8);
   printf("Superfluous bytes %u.",superfluous_bytes);
   flush
  }
  return;
 }
 set_bitmap(seq);
 lseek(output_fd,payload_len*seq,SEEK_SET);
 data_len=pkt_len;
 again:
 wait_for_rw(output_fd,1);
 retval=write(output_fd,pkt+inc,pkt_len);
 if (retval<0&&(errno==EINTR||errno==EAGAIN)) goto again;
 if (retval==pkt_len) goto written;
 if (retval>=0) 
 {
  if (retval>pkt_len)
  {
   fprintf(stderr,"write wrote more bytes than I asked!!!\n");
   die();
  }
  inc+=retval;
  pkt_len-=retval;
  goto again;
 }

 written:
 cache_info(seq);
 remaining_packets--;
 if (!tickers)
 {
  g0t0(18,3);
  printf("%ld.              ",remaining_packets);
  flush
 }
 if (remaining_packets==n_packets-1) 
  first_data_time=get_time();
 else
 {
  if (!tickers)
  {
   g0t0(40,5);
   printf("Average speed %.1f Bps    ",
    bytes_received*1e6/(get_time()-first_data_time));
   g0t0(40,6);
   printf("Average data loss %.2f%%     ",
    100-(bytes_received*1e8/bps/(get_time()-first_data_time)));
  }
 }
 bytes_received+=data_len;
 if (!tickers)
 {
  g0t0(0,5);
  printf("Bytes received %lu.",bytes_received);
  g0t0(0,6);
  printf("%.1f%% done.",(double)bytes_received*100/file_len);
  g0t0(0,7);
  printf("Time elapsed %lld sec.",(get_time()-first_data_time)/1000000);
  flush
 }
}

/* Each seq must be there only once. */
void
remove_qpkt(u32 seq)
{
 struct qpkt *q=que;

 while(q->next)
 {
  if (q->seq==seq)
  {
   struct qpkt * todel;
   
   todel=q->next;
   free(q->ptr);
   memcpy(q,todel,sizeof(*q));
   free(todel);
   qpkt_ctr--;
   print_qpkt_ctr();
   return;
  }
  q=q->next;
 }
}

/* Takes info ack in pkt+5, pkt_len long */
void
process_info_ack(void)
{
 u32 seq;
 
 if (pkt_len!=4) return; /* Nonsense */
 seq=read32(pkt+5);
 remove_qpkt(seq);
 info_acks_received++;
 if (!tickers)
 {
  g0t0(40,4);
  printf("Info acks received %d",info_acks_received);
  flush
 }
}

void
sleep_until(unsigned long long t)
{
 u64 u=get_time();
 
 if (u>=t) return;
 t-=u;
 {
  fd_set set;
  fd_set zero_set;
  struct timeval timeout;

  timeout.tv_sec=t/1000000;
  timeout.tv_usec=t%1000000;
  FD_ZERO(&zero_set);
  FD_ZERO(&set);
  FD_SET(fd,&set);
  select(fd+1,&set,&zero_set,&zero_set,&timeout);
 }
}

void
touch_info_cache(void)
{
 if (info_cache_ctr&&(get_time()>=info_cache_time)) flush_info_cache();
}

void
touch_que(void)
{
 unsigned long long t=get_time();
 
 struct qpkt *q=que;
 while (q->next)
 {
  if (q->time<t) 
  {
   send_qpkt(q);
  }
  q=q->next;
 }
}

#define min(x,y) ((x)<(y)?(x):(y))

unsigned long long
determine_next_action(void)
{
 unsigned long long next_action=0xffffffffffffffffULL;
 struct qpkt *q;

 if (info_cache_ctr) next_action=min(next_action,info_cache_time);
 q=que;
 while(q->next)
 {
  next_action=min(next_action,q->time);
  q=q->next;
 }
 return next_action;
}

void
process_packets(void)
{
 unsigned long long next_action;
 
 if (!remaining_packets) goto done;
 new_cycle:
 next_action=determine_next_action();
 sleep_until(next_action);
 again:
 touch_info_cache();
 touch_que();
 get_packet();
 if (!pkt_len) goto new_cycle;
 pkt_len--;
 switch(pkt[4])
 {
  case 0:
  send_sot_ack();
  case 1:
  process_data_packet();
  if (remaining_packets) goto again;
  done:
  ignore_send_errors=1;
  /* Here we have completed the file */
  for (;number_of_eots;number_of_eots--)
  {
   send_eot();
   usleep(eot_period);
  }
  g0t0(0,20);
  return;
  case 5:
  process_info_ack();
  break;
 }
 goto again;
}

void
init_que(void)
{

 que=(struct qpkt *)xcalloc(sizeof(*que));
}

int
main(int argc, char **argv)
{
 
 parse_args(argc,argv);
 if (tickers<2)
 {
  printf("Starting bcps server, dumping to %s.\n", filename);
  printf("Listening on port %d.\n",port);
 }
 init_que();
 open_session();
 /* Here we know all about the accepted session. */
 process_packets();
 /* We terminate after successful session. */
 close_session();
 return 0;
}

