/***************************************************************************
                          main.c  -  experimental frontend for the encrypting
                          decrypting engine.
                             -------------------
    begin                : Thu Nov 16 16:59:50 EST 2000
    copyright            : (C) 2000 by Brian Wagener and Katrina Illari
    email                : CptanPanic@users.sourceforge.net
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/stat.h>
#include "getopt.h"
#include <unistd.h>
#include <sys/types.h>
#include <globaldefs.h>
#include <crypt.h>

#define EXTENSION ".star"
#define OUTEXTENSION ".stardecrypted"

int interactive = 1;

void usage(){
  fprintf (stderr, "sectar: help\n"
	  "usage: [options] [file]\n"
	  "   --help : This help message\n"
	  "   --version : Show current version\n"
	  "   -c, --stdout : write to standard output, original files remain unchanged\n"
	  "   -v : show optional messages\n"
	  "   -k <keyfile> : set keyfile, if left off will look in stdin\n"
	  "   -e : create star file by encrypting\n"
	  "   -d : decrypt star file (either x or c required)\n"
	  "   -b <num> :  set blocksize in bits (optional) default=256\n"
	  "   -K <num> :  set keysize in bits (optional) default=256 \n"
	  "   -o <file> : set output file\n"
	  "   -z : use compression \n"
	  "   -i : Turn off interactivity, namely won't prompt you for keyname.\n"
	  "   -m <keyfile> : create keyfile (required) and exit\n");
}

struct option long_options[] =
{
  {"version", no_argument, NULL, 'V'},
  {"verbose",no_argument,NULL,'v'},
  {"help",no_argument, NULL, 'h'},
  {"encrypt", no_argument, NULL, 'e'},
  {"decrypt", no_argument, NULL, 'd'},
  {"key",required_argument, NULL, 'k'},
  {"make_key",required_argument, NULL, 'm'},
  {"blocksize",required_argument, NULL, 'b'},
  {"keysize",required_argument, NULL, 'K'},
  {"stdout",required_argument, NULL, 'c'},
  {0,0,0,0}
};

int main (int argc, char *argv[]) {
  char *file = 0, *outfile = 0, *keyfile = 0;
  stword32 nb = MAX_NB, nk = MAX_NK, rc;
  unsigned char *key;
  char optchar = 0;
  FILE *fin = 0, *fout = 0;
  int pip[2];
  pid_t child;
  int result;
  
  static int make_key = 0;
  static int show_help = 0;
  static int show_version = 0;
  static int do_decrypt = 0;
  static int do_encrypt = 0;
  static int set_file = 0;
  static int verbose = 0;
  static int std = 0;
  static char *use_compress_program_option = 0;
  
  if (argc == 1) {
    fprintf (stderr,"star: missing options( \"star --help\" for help\n");
    fprintf (stderr,"sample usage: star [options] \n");
    exit (EXIT_FAILURE);
  }
  
  while (optchar = getopt_long (argc, argv, "Vzivhcm:k:b:f:deK:o:", long_options, NULL),
         optchar != EOF)
    switch (optchar){
      
    case 'V':
      show_version = 1;
      break;
    case 'v':
      verbose = 1;
      break;
    case 'z' :
      use_compress_program_option = "gzip";
      break;
    case 'h':
      show_help = 1;
      break;
    case 'm':	// Create Key
      make_key = 1;
      keyfile = optarg;
      break;
    case 'k':	// Read in key file
      keyfile = optarg;
      break;
    case 'i':	// Read in key file
      interactive = 0;
      break; 
    case 'b':	// Block length
      if ( !optarg ){
	fprintf(stderr, "Error in getting block length\n"
		"usage: star -b <block length> [options]\n");
	exit(EXIT_FAILURE);
      }
      nb = (int) strtol(optarg, NULL, 10);
      if (nb != 128 && nb != 192 && nb != 256){
	fprintf(stderr, "Illegal block length option, "
		"must be 128, 192, or 256 bits\n");
	exit(EXIT_FAILURE);
      }
      break;
    case 'K':
      if ( !optarg ){
	fprintf(stderr, "Error in getting key length\n"
		"usage: star -K <key length> [options]\n");
	exit(EXIT_FAILURE);
      }
      nk = (int) strtol(optarg, NULL, 10);
      if (nk != 128 && nk != 192 && nk != 256){
	fprintf(stderr, "Illegal key length option, "
		"must be 128, 192, or 256 bits\n");
	exit(EXIT_FAILURE);
      }
      break;
    case 'c':
      std = 1;
      break;
    case 'd':
      do_decrypt = 1;
      break;
    case 'e':
      do_encrypt = 1;
      break;
    case 'o':
      set_file = 1;
      outfile = optarg;
      break;
    default:
      usage();
      exit(EXIT_FAILURE);
    }   // end of swich


  //Store filename argument, if no filename given, get from stdin.
  if( optind < argc )
    file = argv[optind];

  // Once arguments are read in, now start doing stuff.
  if(show_help ){
    usage();
    exit(EXIT_SUCCESS);
  }
  if( show_version) {
    fprintf(stderr,"Sectar version %s\n",VERSION);
    exit(EXIT_SUCCESS);
  }
  if( make_key ){
    key = malloc(nk/8*2); /* Note that the key is twice as big as the key length
			     because we store a binary byte using 2 ascii chars
			     as nibbles*/	
    if( (createKey(nk, key, keyfile) ) ){
      if (verbose) fprintf(stderr,"%d bit key created successfully\n", (int)nk);
      free(key);
      return SUCCESS;
    } else {
      fprintf(stderr,"Error creating key\n");
      exit( EXIT_FAILURE );
    }
  }
  if( do_encrypt ){

    if( file == 0 ){
      fin = stdin;
    } else if ((fin = fopen(file, "r"))==NULL){
      fprintf(stderr, "File not found for encrypting: %s\n",
	      file);
      return (FAILURE);
    }

    if( std){
      outfile = 0;
      fout = stdout;
    } else if( !outfile){
      outfile=malloc(sizeof (char) * (strlen(file) + strlen(EXTENSION) + 1));
      strcpy(outfile, file);
      strcat(outfile,EXTENSION);
    } 

    if ( !fout ){
      if( (fout = fopen(outfile, "w"))==NULL){
	fclose(fin);
	fprintf(stderr, "Unable to create file: %s for writing\n",
	      outfile);
	return (FAILURE);
      }
    }


    if( use_compress_program_option){
      
      pipe(pip);

      if( !fork() ){
	// Open compressor.
	// stdin = fin, stdout = pipe[1]
	close(pip[0]);
	dup2(fileno(fin),0);
	dup2(pip[1],1);

	execlp(use_compress_program_option,use_compress_program_option,NULL);
	fprintf(stderr,"Error running gzip...\n");
	
      } else {
	// Open Encryptor.
	// in = pipe[0], out = fout
	// fin = pipe[0], fout = fout

	close(pip[1]);
	fin = fdopen(pip[0],"r");

	rc=cryptEncode(nb, nk, keyfile, fin, fout);

	wait(&result);

      }
    } else {

      // Since we don't need any compression, it is easy.
      rc=cryptEncode(nb, nk, keyfile, fin, fout);

    }
    if( !set_file) free(outfile);
    
    if( !rc ) {
      if (verbose) fprintf(stderr,"Encryption operation completed successfully\n");
    } else
      fprintf(stderr, "Encryption operation failed\n");
    exit(EXIT_SUCCESS);
  }

  if ( do_decrypt ){
    char *pszTemp;

    if( file == 0 ){
      fin = stdin;
    } else if ((fin = fopen(file, "r"))==NULL){
      fprintf(stderr, "File not found for encrypting: %s\n",
              file);
      return (FAILURE);
    }

    if( std){
      outfile = 0;
      fout = stdout;
    } else if(!outfile){
      outfile=malloc(sizeof (char) * (strlen(file) + strlen(OUTEXTENSION)+1));
      if ((pszTemp=strstr(file, EXTENSION))==NULL){ 
	// FIXME: Doesn't decrypt if it doesn't have .star ending.
	fprintf(stderr,
		"Warning: file: %s does not have the standard extension: %s\n"
		"for %s. using %s extension for output instead\n",
		file, EXTENSION, argv[0], OUTEXTENSION);
	strcpy(outfile, file);
	strcat(outfile, OUTEXTENSION);
      }else{
	strncpy(outfile, file, pszTemp-file);
	/* strncpy does not automatically 0-terminate */
	outfile[pszTemp-file] = 0;
      }
    }	

    if ( !fout ){
      if( (fout = fopen(outfile, "w"))==NULL){
        fclose(fin);
        fprintf(stderr, "Unable to create file: %s for writing\n",
		outfile);
        return (FAILURE);
      }
    }


    if( use_compress_program_option){

      use_compress_program_option = "gunzip";

      pipe(pip);

      if( !fork() ){
        // Open compressor.
        // stdin = pipe[0], stdout = fout
        close(pip[1]);
        dup2(pip[0],0);
	dup2(fileno(fout),1);

        execlp(use_compress_program_option,use_compress_program_option,NULL);
        fprintf(stderr,"Error running gzip...\n");

      } else {
        // Open Encryptor.
        // in = fin, out = pipe[1]
        // fin = fin, fout = pipe[1]

        close(pip[0]);
        fout = fdopen(pip[1],"w");

	rc=cryptDecode(nk, keyfile, fin, fout);
        wait(&result);

      }
    } else {

      rc=cryptDecode(nk, keyfile, fin, fout);
    }

    if( !set_file )
      free(outfile);

    if( !rc ) {
      if(verbose) fprintf(stderr,"Decryption operation completed successful\n");
    } else
      fprintf(stderr, "Decryption operation failed\n");
    exit(EXIT_SUCCESS);
  }

  fprintf(stderr, "You forgot to put whether you want to encrypt or decrypt\n");
  usage();
  exit(EXIT_FAILURE);
}

