/***************************************************************************
 *   Copyright (C) 2005 by Felipe Sateler                                  *
 *   fsateler@gmail.com                                                    *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Steet, Fifth Floor, Boston, MA 02110-1301, USA.           *
 ***************************************************************************/

#include <qstring.h>
#include <qcstring.h>
#include <qfile.h>
#include <qfileinfo.h>
#include <klocale.h>
#include <kdesktopfile.h>

#include <sys/types.h>
#include <sys/stat.h>
#include <errno.h>

#include "appinfo.h"


AppInfo::AppInfo(const QString& name)
{
	this->name = name;
}

AppInfo::~AppInfo()
{
}

const QString& AppInfo::getName() const
{
	return name;
}

QString AppInfo::getStrType() const
{
	AppType type = getType();
	switch(type)
	{
	case DESKTOP:
		return i18n("Desktop File");
	case LINK:
		return i18n("Link");
	case SCRIPT:
		return i18n("Script");
	case UNKNOWN:
		return i18n("Unknown");
	}
	return QString::null;
}


DesktopAppInfo::DesktopAppInfo(const QString& name, const QString& command, const QString& comment)
		: AppInfo(name)
{
	this->command = command;
	this->comment = comment;
}

DesktopAppInfo::~DesktopAppInfo()
{
}


const QString& DesktopAppInfo::getCommand() const
{
	return command;
}

const QString& DesktopAppInfo::getComment() const
{
	return comment;
}

AppInfo::AppType DesktopAppInfo::getType() const
{
	return DESKTOP;
}

void DesktopAppInfo::save(const QString& path) const
{
	KDesktopFile file(path+name+".desktop");
	file.writeEntry("Name", name);
	file.writeEntry("Comment", comment);
	file.writeEntry("Exec",command);
	file.writeEntry("Type","Application");
	file.writeEntry("StartupNotify", false);
}


LinkAppInfo::LinkAppInfo(const QString& name, const QString& destination)
		:AppInfo(name)
{
	this->destination = destination;
}

LinkAppInfo::~LinkAppInfo()
{
}

const QString& LinkAppInfo::getDestination() const
{
	return destination;
}

AppInfo::AppType LinkAppInfo::getType() const
{
	return LINK;
}

void LinkAppInfo::save(const QString& path) const
{
	// Ugly trick
	QString target = "\""+destination+"\"";
	QString source = "\""+path+name+"\"";
	QString command = "ln -fs "+target+" "+source;
	system(command);
}


ScriptAppInfo::ScriptAppInfo(const QString& name, const QString& code)
		: AppInfo(name)
{
	this->code = code;
}

ScriptAppInfo::~ScriptAppInfo()
{
}

const QString& ScriptAppInfo::getCode() const
{
	return code;
}

AppInfo::AppType ScriptAppInfo::getType() const
{
	return SCRIPT;
}

QString ScriptAppInfo::getLang() const
{
	return i18n("Unknown");
}

void ScriptAppInfo::save(const QString& path) const
{
	QFile file(path+name);
	file.open(IO_WriteOnly);
	file.writeBlock(code,code.length());
	file.close();
	if( -1 == chmod( path + name, S_IRUSR|S_IWUSR|S_IXUSR|S_IRGRP) )
	{
		throw strerror( errno );
	}
}


UnknownAppInfo::UnknownAppInfo(const QString& name, const QByteArray& contents, mode_t fileMode)
		:AppInfo(name)
{
	this->contents = contents;
	this->mode = fileMode;
}

UnknownAppInfo::~UnknownAppInfo()
{
}

const QByteArray& UnknownAppInfo::getContents() const
{
	return contents;
}


AppInfo::AppType UnknownAppInfo::getType() const
{
	return UNKNOWN;
}

void UnknownAppInfo::save(const QString& path) const 
{
	QFile file(path+name);
	file.open(IO_WriteOnly);
	file.writeBlock(contents);
	file.close();
	if( -1 == chmod( path + name, mode ) )
	{
		throw strerror( errno );
	}
}
