/***************************************************************************
 *   Copyright (C) 2005 by Felipe Sateler                                  *
 *   fsateler@gmail.com                                                    *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Steet, Fifth Floor, Boston, MA 02110-1301, USA.           *
 ***************************************************************************/

#ifndef APPINFO_H
#define APPINFO_H

class QString;
template<class T> class QMemArray;
typedef QMemArray<char> QByteArray;
template<class T> class QPtrList;

/**
Common interface for all application types. It is an abstract class.

@author Felipe Sateler
*/
class AppInfo
{
public:
	
	typedef QPtrList<AppInfo> List;
	/** Specifies the kind of application the instance represents */
	enum AppType {
		UNKNOWN,	///< Anything that is not defined by this enum.
		DESKTOP,	///< Desktop file (*.desktop)
		LINK,		///< A symlink to an application
		SCRIPT		///< A script (python, etc)
	};
	
	/** The reason for this constructor is that all applications should have a name.
	 * @param name The application name
	 */
	AppInfo(const QString& name);
	virtual ~AppInfo();

	/// Accesor
	/** @return The application name */
	virtual const QString& getName() const;
	/// Application type
	/** Returns which application type this instance is. Useful for querying 
	 * type before a typecast.
	 * @return The application type.
	 */
	virtual AppType getType() const = 0;
	/// Application type
	/** Returns a string representation of the application type. */
	QString getStrType() const;
	
	/// Write file to disk
	/** Writes the application info into a valid file, according to it's type.
	 * @param path The directory where to write the file
	 */
	virtual void save(const QString& path) const = 0;
	
protected:
	QString name;
};

/** Specialization of AppInfo, for Desktop files.
 * This is only for desktop files with Type=Application
 */
class DesktopAppInfo : public AppInfo
{
public:
	/// Constructor
	/** @param name The file name
	 *  @param command The command to be executed
	 *  @param comment The comment embedded in the file.
	 */
	DesktopAppInfo(const QString& name, const QString& command, const QString& comment);
	virtual ~DesktopAppInfo();
	
	/// Accesor
	/** @return The command to be executed */
	const QString& getCommand() const;
	/// Accesor
	/** @return The comment embedded in the file */
	const QString& getComment() const;
	
	/// Application type
	/** @return The application type. It is always AppType::DESKTOP */
	virtual AppType getType() const;

	/// Write file to disk
	/** Writes the application info into a valid desktop file.
	 * @param path The directory where to write the file
	 */
	virtual void save(const QString& path) const;

protected:
	QString command;
	QString comment;
};

/** Specialization of AppInfo, for link files (symlinks). */
class LinkAppInfo : public AppInfo
{
public:
	/// Constructor
	/** @param name The link name.
	 * @param destination Where the link points
	 */
	LinkAppInfo(const QString& name, const QString& destination);
	virtual ~LinkAppInfo();

	/// Accesor
	/** @return Where the link points */
	const QString& getDestination() const;

	/// Application type
	/** @return The application type. It is always AppType::LINK */
	virtual AppType getType() const;

	/// Write file to disk
	/** Writes the link in disk.
	 * @param path The directory where to write the file
	 */
	virtual void save(const QString& path) const;

protected:
	QString destination;
};

/** Specialization of AppInfo, for scripts. */
class ScriptAppInfo : public AppInfo
{
public:
	/// Constructor
	/** @param name The script name
	 * @param code The code contained in the file.
	 */
	ScriptAppInfo(const QString& name, const QString& code);
	virtual ~ScriptAppInfo();

	/// Accesor
	/** @return The code contained in the file. */
	const QString& getCode() const;
	/// Language of the script
	/** @return The language the script is written in (eg: bash, perl) */
	QString getLang() const;
	
	/// Application type
	/** @return The application type. It is always AppType::SCRIPT */
	virtual AppType getType() const;

	/// Write file to disk
	/** Writes the application info into a script file.
	 * @param path The directory where to write the file
	 */
	virtual void save(const QString& path) const;

protected:
	QString code;
} ;

/** Specialization of AppInfo, for unknown file types. */
class UnknownAppInfo : public AppInfo
{
public:
	/// Constructor
	/** @param name The file name.
	 * @param contents The contents of the file.
	 */
	UnknownAppInfo(const QString& name, const QByteArray& contents, mode_t fileMode);
	virtual ~UnknownAppInfo();

	/// Accesor
	/** @return The contents of the file */
	const QByteArray& getContents() const;

	/// Application type
	/** @return The application type. It is always AppType::UNKNOWN */
	virtual AppType getType() const;

	/// Write file to disk
	/** Writes the dile contents into a file.
	 * @param path The directory where to write the file
	 */
	virtual void save(const QString& path) const;

protected:
	QByteArray contents;
	mode_t mode;
} ;


#endif
